clear all; clc
% written by Allison Shaw, Contact ashaw@umn.edu for assistance.

%----- SETUP--------------------------------------------------------------%
    % define initial conditions
    N = 3000;          % population size
    I_init = 1;         % initial infected    
    S_init = N-I_init;  % intial susceptible
    E_init = 0;         % initial exposed
    R_init = 0;         % initial removed
    IC = [S_init E_init I_init R_init];

    tf = 600; % number of days to run for

    Tc = 1/24; % fraction of day commuting each day
    Tw = 8/24; % fraction of the day at work
    Th = 1-2*Tc-Tw; % fraction of the day at home

    theta = 1; % fraction of the population commuting to campus to work
%----- SETUP--------------------------------------------------------------%


%-----INFECTION PARAMETERS------------------------------------------------%
    % set infection parameters
    gamma = 1/9.5; % recovery rate = 1 / infection duration
                  %    units of per time
    
    sigma = 1/4.2; % rate of moving from exposed to infected
    
    % infection parameters in each environment
    R0_base = 2.5; % current estimate of R0 in normal setting
    R0_h = R0_base*0.5; % current estimate, with physical distancing
     
    % transmission rate for frequency-dependent, units of per time
    beta_h = R0_h*gamma;  % transmission at home
%-----INFECTION PARAMETERS------------------------------------------------%

% consider three scenarios:
% [1] low risk of going to work
    R0_c = 0.5*R0_base; % commuting: none
    R0_w = 0.5*R0_base; % at work: same as at home
    beta_c = R0_c*gamma;  % transmission commuting
    beta_w = R0_w*gamma;  % transmission at work
    [S_series1,E_series1,I_series1,R_series1] = run_SEIR_spatial(IC,beta_c,beta_h,beta_w,sigma,gamma,Tc,Th,Tw,tf,theta);

% [2] medium risk of going to work
    R0_c = R0_base; % commuting: unmitigated
    R0_w = 0.75*R0_base; % at work: in between at home and unmitigated
    beta_c = R0_c*gamma;  % transmission commuting
    beta_w = R0_w*gamma;  % transmission at work
    [S_series2,E_series2,I_series2,R_series2] = run_SEIR_spatial(IC,beta_c,beta_h,beta_w,sigma,gamma,Tc,Th,Tw,tf,theta);

    % [3] high risk of going to work
    R0_c = 2*R0_base; % commuting: double unmitigated (elevated)
    R0_w = R0_base; % at work: unmitigated
    beta_c = R0_c*gamma;  % transmission commuting
    beta_w = R0_w*gamma;  % transmission at work
    [S_series3,E_series3,I_series3,R_series3] = run_SEIR_spatial(IC,beta_c,beta_h,beta_w,sigma,gamma,Tc,Th,Tw,tf,theta);

    clear R0_w beta_w R0_c beta_c


    disp('R-end   I-max')
    disp([R_series1(end)/N max(I_series1)/N])
    disp([R_series2(end)/N max(I_series2)/N])
    disp([R_series3(end)/N max(I_series3)/N])
    

% simulate basic SEIR model
params = [beta_h sigma gamma];
options = odeset('NonNegative',1:4);
[T_out,N_out] = ode45(@SEIR,[1 tf],[S_init E_init I_init R_init],options,params);    
% rename variables to make more intuitive
S = N_out(:,1);
E = N_out(:,2);
I = N_out(:,3);
R = N_out(:,4);
%%

% define figure settings
lw = 2;
fs1 = 10;  % axes labels
fs3 = 9; % axis numbering
sx = 0.06;
sy = 0.19;
d2 = 0.09;
w = 0.25;
hei = 0.7;
figure(1); clf
hh = gcf;
set(hh,'PaperUnits','inches');
set(hh,'Units','inches');
width = 7.5; height = 2.2;
xpos = 5;
ypos = 4;
set(gcf,'Position',[xpos ypos width height])

xx=1; % grain of x-axis plotting

axes('position',[sx sy w hei])
    hold on
    plot(T_out,S/N,'b-','LineWidth',lw);
    plot(T_out,E/N,'c-','LineWidth',lw);
    plot(T_out,I/N,'r-','LineWidth',lw);
    plot(T_out,R/N,'k-','LineWidth',lw);
    plot(1:xx:tf,S_series1(1:xx:end)/N,'b--','LineWidth',lw);
    plot(1:xx:tf,E_series1(1:xx:end)/N,'c--','LineWidth',lw);
    plot(1:xx:tf,I_series1(1:xx:end)/N,'r--','LineWidth',lw);
    plot(1:xx:tf,R_series1(1:xx:end)/N,'k--','LineWidth',lw);
    box on
    xlabel('Time (days)','FontSize',fs1)
    ylabel('Fraction of population','FontSize',fs1)
    title('Dynamics (low risk)','FontSize',fs1)
    legend('S','E','I','R','Location','West')
    axis([0 tf 0 1])
    set(gca,'FontSize',fs3);

axes('position',[sx+d2+w sy w hei])
    hold on
    plot(T_out,S/N,'b-','LineWidth',lw);
    plot(T_out,E/N,'c-','LineWidth',lw);
    plot(T_out,I/N,'r-','LineWidth',lw);
    plot(T_out,R/N,'k-','LineWidth',lw);
    plot(1:xx:tf,S_series2(1:xx:end)/N,'b--','LineWidth',lw);
    plot(1:xx:tf,E_series2(1:xx:end)/N,'c--','LineWidth',lw);
    plot(1:xx:tf,I_series2(1:xx:end)/N,'r--','LineWidth',lw);
    plot(1:xx:tf,R_series2(1:xx:end)/N,'k--','LineWidth',lw);
    box on
    xlabel('Time (days)','FontSize',fs1)
    ylabel('Fraction of population','FontSize',fs1)
    title('Dynamics (moderate risk)','FontSize',fs1)
    axis([0 tf 0 1])
    set(gca,'FontSize',fs3);

axes('position',[sx+2*d2+2*w sy w hei])
    hold on
    plot(T_out,S/N,'b-','LineWidth',lw);
    plot(T_out,E/N,'c-','LineWidth',lw);
    plot(T_out,I/N,'r-','LineWidth',lw);
    plot(T_out,R/N,'k-','LineWidth',lw);
    plot(1:xx:tf,S_series3(1:xx:end)/N,'b--','LineWidth',lw);
    plot(1:xx:tf,E_series3(1:xx:end)/N,'c--','LineWidth',lw);
    plot(1:xx:tf,I_series3(1:xx:end)/N,'r--','LineWidth',lw);
    plot(1:xx:tf,R_series3(1:xx:end)/N,'k--','LineWidth',lw);
    box on
    xlabel('Time (days)','FontSize',fs1)
    ylabel('Fraction of population','FontSize',fs1)
    title('Dynamics (high risk)','FontSize',fs1)
    axis([0 tf 0 1])
    set(gca,'FontSize',fs3);

% label subpanels
    axes('position',[0 0 1 1],'visible','off')
    hold on
    text(0       ,0.96,'a)','FontSize',fs1);
    text(d2+w    ,0.96,'b)','FontSize',fs1);
    text(2*d2+2*w,0.96,'c)','FontSize',fs1);
axis([0 1 0 1])


% Backup previous settings
prePaperType = get(hh,'PaperType');
prePaperPosition = get(hh,'PaperPosition');
prePaperSize = get(hh,'PaperSize');

% Make changing paper type possible
set(hh,'PaperType','<custom>');
% Set the page size and position to match the figure's dimensions
position = get(hh,'Position');
set(hh,'PaperPosition',[0,0,position(3:4)]);
set(hh,'PaperSize',position(3:4));

saveas(1,'fig2.jpg')
print(1,'-dtiff','fig2.tiff','-r300')

% Restore the previous settings
set(hh,'PaperType',prePaperType);
set(hh,'PaperPosition',prePaperPosition);
set(hh,'PaperSize',prePaperSize);





