function dndt = SEIR(t,n,params)
% Usage: dndt = SEIR(t,n,params)
% written by Allison Shaw, Contact ashaw@umn.edu for assistance.
% last updated 3 June 2020
%
% Calculate instantaneous rate of change for susceptibles (S), exposed (E),
% infected (I), and removed (R) in a simple SEIR model with
% frequency-dependent transmission dynamics:
%     dy=[- beta*S*I/(S+E+I+R);
%           beta*S*I/(S+E+I+R) - sigma*E
%           sigma*E - gamma*I
%         - gamma*I];
%
% INPUTS:
%   t = amount of time to run dynamics for
%   S = starting number of susceptible
%   E = starting number of exposed
%   I = starting number of infected
%   R = starting number of removed
%   beta = infection rate
%   sigma = rate of moving from exposed to infected
%   gamma = recovery rate
%
% OUTPUTS:
%   dndt = ending number of susceptible, exposed, infected, and recovered
%   individual

beta  = params(1); % infection rate
sigma = params(2); % rate of moving from exposed to infected
gamma = params(3); % recovery rate

S = n(1);
E = n(2);
I = n(3);
R = n(4);

dndt(1,1) = -beta*S*I/(S+E+I+R);
dndt(2,1) =  beta*S*I/(S+E+I+R) -sigma*E;
dndt(3,1) =  sigma*E            -gamma*I;
dndt(4,1) =  gamma*I;
