
#############################################################################
####  FITTING SPATIOTEMPORAL DISEASE MODELS TO THE APHID ARENA EXPERIMENT ####
##############################################################################

library(deSolve)
library(bbmle)
library(plyr)
library(gplots)
library(plot3D)


##########################################################################
###################   A FEW DIFFERENT COMPETING MODELS   #################
##########################################################################

#  lagged dispersal model AKA global dispersal AKA N=2
modN2 <- function(t,y,params){
  V1 <- y[1]; V2 <- y[2]; W1 <- y[3]; W2 <- y[4];
  S1 <- y[5]; S2 <- y[6]; I1 <- y[7]; I2 <- y[8]
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I2*V2*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*W1) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2)))) 
    dW2 <- (I2*V2*beta_hv) - (d*W2) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    res <- c(dV1, dV2, dW1, dW2, dS1, dS2, dI1, dI2)
    list(res)})}


##############################################################
###################     READ IN THE DATA   ###################
##############################################################

data <- read.csv("AA1.csv") 
data$juv[is.na(data$juv)] <- 0; data$adults[is.na(data$adults)] <- 0; data$wings[is.na(data$wings)] <- 0 # fill in 0s:
data$aph <- data$juv + data$adults + data$wings; data$logaph <- log(data$aph+1)
data$pwings <- data$wings / data$aph
data$day <- data$week*7-14 # convert weeks to days

# add treatment IDs
data$fert <- ifelse(data$ttmt==1 | data$ttmt==2 | data$ttmt==3, 0, 1) # keeping numeric so i can 'average' in ddply
data$inf <- ifelse(data$ttmt==1 | data$ttmt==4, -1, ifelse(data$ttmt==2 | data$ttmt==5, 0, 1))
data <- data[data$ring != "Lost aphids" ,] # aphids that were on tray but don't know which plant they came from

# identifiers for different ways to split and summarize data:
data$dID <- paste(data$block, data$ttmt) # dorm-ID 
data$dwID <- paste(data$block, data$ttmt, data$week) # dorm-week ID 
data$twrID <- paste(data$ttmt, data$week, data$ring) # treatment-week ID
inf.data <- read.csv("inf.data.csv") # read in infection data
data <- merge(data, inf.data, by=c("week", "block", "ttmt", "ring"), all.x=T) # merge
data$I[is.na(data$I)] <- 0 # fill in 0s:
data$ring <- 4-as.numeric(data$ring) # convert to 0,1,2,3, where 0 is the center
data$ttmt <- as.factor(data$ttmt) # don't want as numeric

# average by dorm by week (for plotting)
dwdata <- ddply(data, .(dwID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], blk=block[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                prev=sum(I)/4, pwings=mean(pwings)) 

dwdata$dorm <- paste(dwdata$blk, dwdata$ttmt) # dorm-ID 
dwdata$tdID <- paste(dwdata$ttmt, dwdata$day) # treatment-week ID

# average by treatment by week (for plotting)
twdata <- ddply(dwdata, .(tdID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), p.sd=sd(prev),
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), p=mean(prev),
                pwings=mean(pwings)) 
twdata <- twdata[order(twdata$day),] 

# average by treatment-week (i.e., to get prevalence at each ring)
twrdata <- ddply(data, .(twrID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1], ring=ring[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                prev=sum(I)/5) 

fitdata <- data[data$ttmt!=1 & data$ttmt!=4 , ] # omit the aphid free treatments
treats <- unique(fitdata$ttmt) # to loop through each treatment


########################################################################
######################   LIKELIHOOD FUNCTIONS   ########################
########################################################################

# notice the if/else statements in the parameter lists inside the function
# they let me set parameters to zero if i want to try a simpler model

#########################
### GLOBAL DISPERSAL ####
#########################

# #parameters to check that it is working, and also to doublecheck
# #that the mathematical right-up of likelihood is correct
# #(see dbinom and dnbinom by hand, commented out below)
# r_D0R0=2.2; r_D0R1=0; r_D1R0=0; r_D1R1=0;
# K_D0R0=100; K_D0R1=0; K_D1R0=0; K_D1R1=0;
# d_D0R0=0.05; d_D0R1=0; d_D1R0=0; d_D1R1=0; 
# beta_vh_R0=0.005; beta_vh_R1=0;
# theta_D0R0=0.03; theta_D0R1=0; theta_D1R0=0; theta_D1R1=0;
# beta_hv=0.68


ll.func.N2 <- function(r_D0R0, r_D0R1, r_D1R0, r_D1R1, K_D0R0, K_D0R1, K_D1R0, K_D1R1,
                       d_D0R0, d_D0R1, d_D1R0, d_D1R1, 
                       beta_vh_R0, beta_vh_R1, theta_D0R0, theta_D0R1, theta_D1R0, theta_D1R1,
                       beta_hv){
  
  # which parameters used for each simulation
  # need a bunch of if/else statements for backwards model selection.
  par_D0R0 <- c(r=r_D0R0, K=K_D0R0, d=d_D0R0, beta_vh=0, theta=theta_D0R0, beta_hv=beta_hv)
  par_D1R0 <- c(r=ifelse(r_D1R0>0, r_D1R0, r_D0R0), K=ifelse(K_D1R0>0, K_D1R0, K_D0R0), 
                d=ifelse(d_D1R0>0, d_D1R0, d_D0R0), beta_hv=beta_hv,
                beta_vh=beta_vh_R0, theta=ifelse(theta_D1R0>0, theta_D1R0, theta_D0R0))
  par_D0R1 <- c(r=ifelse(r_D0R1>0, r_D0R1, r_D0R0), K=ifelse(K_D0R1>0, K_D0R1, K_D0R0),
                d=ifelse(d_D0R1>0, d_D0R1, d_D0R0), beta_hv=beta_hv, 
                beta_vh=0, theta=ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))
  par_D1R1 <- c(r=ifelse(r_D1R1>0, r_D1R1, ifelse(r_D1R0>0, r_D1R0, ifelse(r_D0R1>0, r_D0R1, r_D0R0))),
                K=ifelse(K_D1R1>0, K_D1R1, ifelse(K_D1R0>0, K_D1R0, ifelse(K_D0R1>0, K_D0R1, K_D0R0))),
                d=ifelse(d_D1R1>0, d_D1R1, ifelse(d_D1R0>0, d_D1R0, ifelse(d_D0R1>0, d_D0R1, d_D0R0))),
                beta_vh=ifelse(beta_vh_R1>0, beta_vh_R1, beta_vh_R0), beta_hv=beta_hv,
                theta=ifelse(theta_D1R1>0, theta_D1R1, ifelse(theta_D1R0>0, theta_D1R0, ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))))
  par_lists <- list(par_D0R0, par_D1R0, par_D0R1, par_D1R1) # order must match treatments
  
  # for all simulations:
  start_D0 <- c(V1=2.5, V2=0, W1=0, W2=0, S1=4, S2=96, I1=0, I2=0)
  start_D1 <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)
  t <- seq(from=1, to=60, by=1)
  
  treat.lls <- data.frame(treats=treats)
  for(i in 1:length(treats)){
    tdata <- data[data$ttmt==treats[i],]
    tdata <- tdata[!(is.na(tdata$aph)),] # omit cases with unknown aphid #
    if (max(tdata$inf)>0){start=start_D1} else {start=start_D0}  
    sim <- data.frame(lsoda(y=start, times=t, func=modN2, parms=as.list(par_lists[[i]])))
    res.df <- merge(data.frame(time=tdata$day, aph=tdata$aph, ring=tdata$ring, inf=tdata$I), sim, by="time")
    res.df$ll.aph <- dnbinom(res.df$aph, mu=res.df$V2+res.df$W2, size=1/par_lists[[i]]["theta"] , log=T)
    # dnbinom by hand:
    # res.df$ll.aph <- log(choose(res.df$aph+(1/par_lists[[i]]["theta"])-1,res.df$aph)*
    #                            (((res.df$V2+res.df$W2)/(res.df$V2+res.df$W2+(1/par_lists[[i]]["theta"])))^res.df$aph)*
    #                            (((1/par_lists[[i]]["theta"])/((1/par_lists[[i]]["theta"])+res.df$V2+res.df$W2))^(1/par_lists[[i]]["theta"])))
    res.df$ll.inf <- dbinom(res.df$inf, size=1, prob=pmin(res.df$I2/(res.df$S2+res.df$I2),1), log=T)
    # dbinom by hand (Bernoulli):
    # res.df$ll.inf <- log((res.df$I2/(res.df$S2+res.df$I2))^res.df$inf*
    #   (1-(res.df$I2/(res.df$S2+res.df$I2)))^(1-res.df$inf))
    res.df$ll.inf <- ifelse(res.df$ll.inf==-Inf, -100, res.df$ll.inf) # replace -inf (from rounding error) with big negative
    treat.lls$ll[i] <- -sum(res.df$ll.aph)+-sum(res.df$ll.inf)
  }
  print(treat.lls$ll)
}




################################################################
#######  READ IN PARAMETERS OF BEST FIT; THEN CREATE   #########
####### LIST TO LOOP THROUGH AND CALCULATE LIKELIHOODS #########
################################################################

fit <- read.csv("model.competition.phase.B.csv")
fit <- fit[fit$X=="N2.4d",]

levels <- 31 # should be odd number, so fitted point estimate is in middle
range1 <- 0.05 # as a proportion of fitted value 
range2 <- 0.5
range3 <- 0.1

# range on r's
r_D0R0_levels <- seq(fit$r_D0R0-fit$r_D0R0*range1, fit$r_D0R0+fit$r_D0R0*range1, length.out=levels)
r_D0R1_levels <- seq(fit$r_D0R1-fit$r_D0R1*range1, fit$r_D0R1+fit$r_D0R1*range1, length.out=levels)
r_D1R0_levels <- seq(fit$r_D1R0-fit$r_D1R0*range1, fit$r_D1R0+fit$r_D1R0*range1, length.out=levels)
r_D1R1_levels <- seq(fit$r_D1R1-fit$r_D1R1*range1, fit$r_D1R1+fit$r_D1R1*range1, length.out=levels)

# range on d's
d_D0R0_levels <- seq(fit$d_D0R0-fit$d_D0R0*range2, fit$d_D0R0+fit$d_D0R0*range2, length.out=levels)
d_D0R1_levels <- seq(fit$d_D0R1-fit$d_D0R1*range2, fit$d_D0R1+fit$d_D0R1*range2, length.out=levels)
d_D1R0_levels <- seq(fit$d_D1R0-fit$d_D1R0*range2, fit$d_D1R0+fit$d_D1R0*range2, length.out=levels)
d_D1R1_levels <- seq(fit$d_D1R1-fit$d_D1R1*range2, fit$d_D1R1+fit$d_D1R1*range2, length.out=levels)

# range on K's
K_D0R0_levels <- seq(fit$K_D0R0-fit$K_D0R0*range3, fit$K_D0R0+fit$K_D0R0*range3, length.out=levels)
K_D0R1_levels <- seq(fit$K_D0R1-fit$K_D0R1*range3, fit$K_D0R1+fit$K_D0R1*range3, length.out=levels)
K_D1R0_levels <- seq(fit$K_D1R0-fit$K_D1R0*range3, fit$K_D1R0+fit$K_D1R0*range3, length.out=levels)
K_D1R1_levels <- seq(fit$K_D1R1-fit$K_D1R1*range3, fit$K_D1R1+fit$K_D1R1*range3, length.out=levels)


################
# r vs. d space:

rd.loop=data.frame(r_D0R0=rep(r_D0R0_levels, levels), r_D0R1=rep(r_D0R1_levels, levels),
                   r_D1R0=rep(r_D1R0_levels, levels), r_D1R1=rep(r_D1R1_levels, levels),
                   K_D0R0=fit$K_D0R0, K_D0R1=fit$K_D0R1, K_D1R0=fit$K_D1R0, K_D1R1=fit$K_D1R1,
                   d_D0R0=sort(rep(d_D0R0_levels, levels)), d_D0R1=sort(rep(d_D0R1_levels, levels)), 
                   d_D1R0=sort(rep(d_D1R0_levels, levels)), d_D1R1=sort(rep(d_D1R1_levels, levels)),
                   beta_vh_R0=fit$beta_vh_R0, beta_vh_R1=fit$beta_vh_R1,
                   theta_D0R0=fit$theta_D0R0, theta_D0R1=fit$theta_D0R1, theta_D1R0=fit$theta_D1R0, theta_D1R1=fit$theta_D1R1,
                   beta_hv=fit$beta_hv)

for(j in 1:nrow(rd.loop)){
  print(j)
  ll <- ll.func.N2(r_D0R0=rd.loop$r_D0R0[j], r_D0R1=rd.loop$r_D0R1[j], r_D1R0=rd.loop$r_D1R0[j], r_D1R1=rd.loop$r_D1R1[j], 
                   K_D0R0=rd.loop$K_D0R0[j], K_D0R1=rd.loop$K_D0R1[j], K_D1R0=rd.loop$K_D1R0[j], K_D1R1=rd.loop$K_D1R1[j], 
                   d_D0R0=rd.loop$d_D0R0[j], d_D0R1=rd.loop$d_D0R1[j], d_D1R0=rd.loop$d_D1R0[j], d_D1R1=rd.loop$d_D1R1[j], 
                   beta_vh_R0=rd.loop$beta_vh_R0[j], beta_vh_R1=rd.loop$beta_vh_R1[j], 
                   theta_D0R0=rd.loop$theta_D0R0[j], theta_D0R1=rd.loop$theta_D0R1[j], theta_D1R0=rd.loop$theta_D1R0[j], theta_D1R1=rd.loop$theta_D1R1[j], 
                   beta_hv=rd.loop$beta_hv[j])
  rd.loop$nll_D0R0[j] <- ll[1]
  rd.loop$nll_D1R0[j] <- ll[2]
  rd.loop$nll_D0R1[j] <- ll[3]
  rd.loop$nll_D1R1[j] <- ll[4]
}

################
# r vs. K space:

rK.loop=data.frame(r_D0R0=rep(r_D0R0_levels, levels), r_D0R1=rep(r_D0R1_levels, levels),
                   r_D1R0=rep(r_D1R0_levels, levels), r_D1R1=rep(r_D1R1_levels, levels),
                   K_D0R0=sort(rep(K_D0R0_levels, levels)), K_D0R1=sort(rep(K_D0R1_levels, levels)), 
                   K_D1R0=sort(rep(K_D1R0_levels, levels)), K_D1R1=sort(rep(K_D1R1_levels, levels)),
                   d_D0R0=fit$d_D0R0, d_D0R1=fit$d_D0R1, d_D1R0=fit$d_D1R0, d_D1R1=fit$d_D1R1,
                   beta_vh_R0=fit$beta_vh_R0, beta_vh_R1=fit$beta_vh_R1,
                   theta_D0R0=fit$theta_D0R0, theta_D0R1=fit$theta_D0R1, theta_D1R0=fit$theta_D1R0, theta_D1R1=fit$theta_D1R1,
                   beta_hv=fit$beta_hv)

for(j in 1:nrow(rK.loop)){
  print(j)
  ll <- ll.func.N2(r_D0R0=rK.loop$r_D0R0[j], r_D0R1=rK.loop$r_D0R1[j], r_D1R0=rK.loop$r_D1R0[j], r_D1R1=rK.loop$r_D1R1[j], 
                   K_D0R0=rK.loop$K_D0R0[j], K_D0R1=rK.loop$K_D0R1[j], K_D1R0=rK.loop$K_D1R0[j], K_D1R1=rK.loop$K_D1R1[j], 
                   d_D0R0=rK.loop$d_D0R0[j], d_D0R1=rK.loop$d_D0R1[j], d_D1R0=rK.loop$d_D1R0[j], d_D1R1=rK.loop$d_D1R1[j], 
                   beta_vh_R0=rK.loop$beta_vh_R0[j], beta_vh_R1=rK.loop$beta_vh_R1[j], 
                   theta_D0R0=rK.loop$theta_D0R0[j], theta_D0R1=rK.loop$theta_D0R1[j], theta_D1R0=rK.loop$theta_D1R0[j], theta_D1R1=rK.loop$theta_D1R1[j], 
                   beta_hv=rK.loop$beta_hv[j])
  rK.loop$nll_D0R0[j] <- ll[1]
  rK.loop$nll_D1R0[j] <- ll[2]
  rK.loop$nll_D0R1[j] <- ll[3]
  rK.loop$nll_D1R1[j] <- ll[4]
}

################
# d vs. K space:

dK.loop=data.frame(r_D0R0=fit$r_D0R0, r_D0R1=fit$r_D0R1, r_D1R0=fit$r_D1R0, r_D1R1=fit$r_D1R1,
                   K_D0R0=sort(rep(K_D0R0_levels, levels)), K_D0R1=sort(rep(K_D0R1_levels, levels)), 
                   K_D1R0=sort(rep(K_D1R0_levels, levels)), K_D1R1=sort(rep(K_D1R1_levels, levels)),
                   d_D0R0=rep(d_D0R0_levels, levels), d_D0R1=rep(d_D0R1_levels, levels),
                   d_D1R0=rep(d_D1R0_levels, levels), d_D1R1=rep(d_D1R1_levels, levels),
                   beta_vh_R0=fit$beta_vh_R0, beta_vh_R1=fit$beta_vh_R1,
                   theta_D0R0=fit$theta_D0R0, theta_D0R1=fit$theta_D0R1, theta_D1R0=fit$theta_D1R0, theta_D1R1=fit$theta_D1R1,
                   beta_hv=fit$beta_hv)

for(j in 1:nrow(dK.loop)){
  print(j)
  ll <- ll.func.N2(r_D0R0=dK.loop$r_D0R0[j], r_D0R1=dK.loop$r_D0R1[j], r_D1R0=dK.loop$r_D1R0[j], r_D1R1=dK.loop$r_D1R1[j], 
                   K_D0R0=dK.loop$K_D0R0[j], K_D0R1=dK.loop$K_D0R1[j], K_D1R0=dK.loop$K_D1R0[j], K_D1R1=dK.loop$K_D1R1[j], 
                   d_D0R0=dK.loop$d_D0R0[j], d_D0R1=dK.loop$d_D0R1[j], d_D1R0=dK.loop$d_D1R0[j], d_D1R1=dK.loop$d_D1R1[j], 
                   beta_vh_R0=dK.loop$beta_vh_R0[j], beta_vh_R1=dK.loop$beta_vh_R1[j], 
                   theta_D0R0=dK.loop$theta_D0R0[j], theta_D0R1=dK.loop$theta_D0R1[j], theta_D1R0=dK.loop$theta_D1R0[j], theta_D1R1=dK.loop$theta_D1R1[j], 
                   beta_hv=dK.loop$beta_hv[j])
  dK.loop$nll_D0R0[j] <- ll[1]
  dK.loop$nll_D1R0[j] <- ll[2]
  dK.loop$nll_D0R1[j] <- ll[3]
  dK.loop$nll_D1R1[j] <- ll[4]
}






################################################################
########  HEAT MAPS TO SHOW 2-D LIKELIHOOD SURFACES   ##########
################################################################

png("likelihood.surfaces.png", width = 6, height = 4, res = 600, units='in')

par(mfrow=c(3,4), mar=c(3,3,.5,.5), oma=c(0,0,0,0), las=1) # save 500w x 400t



##########################
# r-K likelihood heat maps

# r-K-D0R0
scatter2D(rK.loop$r_D0R0, rK.loop$K_D0R0, colvar=exp(-rK.loop$nll_D0R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D0R0-fit$r_D0R0*range1, fit$r_D0R0, 
                        fit$r_D0R0+fit$r_D0R0*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R0-fit$K_D0R0*range3, fit$K_D0R0, 
                        fit$K_D0R0+fit$K_D0R0*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# r-K-D1R0
scatter2D(rK.loop$r_D1R0, rK.loop$K_D0R0, colvar=exp(-rK.loop$nll_D1R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D1R0-fit$r_D1R0*range1, fit$r_D1R0, 
                        fit$r_D1R0+fit$r_D1R0*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R0-fit$K_D0R0*range3, fit$K_D0R0, 
                        fit$K_D0R0+fit$K_D0R0*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# r-K-D0R1
scatter2D(rK.loop$r_D0R1, rK.loop$K_D0R1, colvar=exp(-rK.loop$nll_D0R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D0R1-fit$r_D0R1*range1, fit$r_D0R1, 
                        fit$r_D0R1+fit$r_D0R1*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R1-fit$K_D0R1*range3, fit$K_D0R1, 
                        fit$K_D0R1+fit$K_D0R1*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# r-K-D1R1
scatter2D(rK.loop$r_D1R1, rK.loop$K_D0R1, colvar=exp(-rK.loop$nll_D1R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D1R1-fit$r_D1R1*range1, fit$r_D1R1, 
                        fit$r_D1R1+fit$r_D1R1*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R1-fit$K_D0R1*range3, fit$K_D0R1, 
                        fit$K_D0R1+fit$K_D0R1*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))


##########################
# r-d likelihood heat maps

# r-d-D0R0
scatter2D(rd.loop$r_D0R0, rd.loop$d_D0R0, colvar=exp(-rd.loop$nll_D0R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="d", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D0R0-fit$r_D0R0*range1, fit$r_D0R0, 
                        fit$r_D0R0+fit$r_D0R0*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$d_D0R0-fit$d_D0R0*range2, fit$d_D0R0, 
                        fit$d_D0R0+fit$d_D0R0*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.6,0))
     
# r-d-D1R0
scatter2D(rd.loop$r_D1R0, rd.loop$d_D1R0, colvar=exp(-rd.loop$nll_D1R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="d", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D1R0-fit$r_D1R0*range1, fit$r_D1R0, 
                        fit$r_D1R0+fit$r_D1R0*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$d_D1R0-fit$d_D1R0*range2, fit$d_D1R0, 
                        fit$d_D1R0+fit$d_D1R0*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# r-d-D0R1
scatter2D(rd.loop$r_D0R1, rd.loop$d_D0R1, colvar=exp(-rd.loop$nll_D0R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="d", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D0R1-fit$r_D0R1*range1, fit$r_D0R1, 
                        fit$r_D0R1+fit$r_D0R1*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$d_D0R1-fit$d_D0R1*range2, fit$d_D0R1, 
                        fit$d_D0R1+fit$d_D0R1*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# r-d-D1R1
scatter2D(rd.loop$r_D1R1, rd.loop$d_D1R1, colvar=exp(-rd.loop$nll_D1R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="r", ylab="d", colkey=list(plot=F))
axis(side=1, at=round(c(fit$r_D1R1-fit$r_D1R1*range1, fit$r_D1R1, 
                        fit$r_D1R1+fit$r_D1R1*range1),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$d_D1R1-fit$d_D1R1*range2, fit$d_D1R1, 
                        fit$d_D1R1+fit$d_D1R1*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.6,0))


##########################
# d-K likelihood heat maps

# d-K-D0R0
scatter2D(dK.loop$d_D0R0, dK.loop$K_D0R0, colvar=exp(-dK.loop$nll_D0R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="d", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$d_D0R0-fit$d_D0R0*range2, fit$d_D0R0, 
                        fit$d_D0R0+fit$d_D0R0*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R0-fit$K_D0R0*range3, fit$K_D0R0, 
                        fit$K_D0R0+fit$K_D0R0*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# d-K-D1R0
scatter2D(dK.loop$d_D1R0, dK.loop$K_D0R0, colvar=exp(-dK.loop$nll_D1R0),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="d", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$d_D1R0-fit$d_D1R0*range2, fit$d_D1R0, 
                        fit$d_D1R0+fit$d_D1R0*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R0-fit$K_D0R0*range3, fit$K_D0R0, 
                        fit$K_D0R0+fit$K_D0R0*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# d-K-D0R1
scatter2D(dK.loop$d_D0R1, dK.loop$K_D0R1, colvar=exp(-dK.loop$nll_D0R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="d", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$d_D0R1-fit$d_D0R1*range2, fit$d_D0R1, 
                        fit$d_D0R1+fit$d_D0R1*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R1-fit$K_D0R1*range3, fit$K_D0R1, 
                        fit$K_D0R1+fit$K_D0R1*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

# d-K-D1R1
scatter2D(dK.loop$d_D1R1, dK.loop$K_D0R1, colvar=exp(-dK.loop$nll_D1R1),
          cex=2, pch=15, xaxt="n", yaxt="n", 
          xlab="d", ylab="K", colkey=list(plot=F))
axis(side=1, at=round(c(fit$d_D1R1-fit$d_D1R1*range2, fit$d_D1R1, 
                        fit$d_D1R1+fit$d_D1R1*range2),3), cex.axis=0.8, labels=T, mgp=c(1,.3,0))
axis(side=2, at=round(c(fit$K_D0R1-fit$K_D0R1*range3, fit$K_D0R1, 
                        fit$K_D0R1+fit$K_D0R1*range3),0), cex.axis=0.8, labels=T, mgp=c(1,.6,0))

dev.off()
