##########################################################################
####  COMPETING LINKED LOGISTIC MODELS FOR APHID DISPERSAL AND GROWTH ####
##########################################################################

library(deSolve)
library(bbmle)
library(plyr)
library(gplots)


##########################################################################
###################   A FEW DIFFERENT COMPETING MODELS   #################
##########################################################################

# non-spatial model:
modN1 <- function(t,y,params){
  V <- y[1]; W <- y[2]; S <- y[3]; I <- y[4]
  with(as.list(params), {
    dV <- r*(V+W)*(1-((V+W)/K)) - (I*V*beta_hv)
    dW <- I*V*beta_hv
    dS <- -beta_vh*S*W
    dI <- beta_vh*S*W
    res <- c(dV, dW, dS, dI)
    list(res)})}


#  lagged dispersal model AKA global dispersal AKA N=2
modN2 <- function(t,y,params){
  V1 <- y[1]; V2 <- y[2]; W1 <- y[3]; W2 <- y[4];
  S1 <- y[5]; S2 <- y[6]; I1 <- y[7]; I2 <- y[8]
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I2*V2*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*W1) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2)))) 
    dW2 <- (I2*V2*beta_hv) - (d*W2) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    res <- c(dV1, dV2, dW1, dW2, dS1, dS2, dI1, dI2)
    list(res)})}


# traveling wave / local dispersal / N=5
modN5 <- function(t,y,params){
  V1<-y[1]; V2<-y[2]; V3<-y[3]; V4<-y[4]; V5<-y[5] 
  W1<-y[6]; W2<-y[7]; W3<-y[8]; W4<-y[9]; W5<-y[10] 
  S1<-y[11]; S2<-y[12]; S3<-y[13]; S4<-y[14]; S5<-y[15] 
  I1<-y[16]; I2<-y[17]; I3<-y[18]; I4<-y[19]; I5<-y[20] 
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                                        (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                                        (V1*(S2+I2)/(S1+I1+S2+I2))+
                                                        (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4)))) - (I2*V2*beta_hv)
    dV3 <- (r*(V3+W3)*(1-((V3+W3)/K))) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                                        (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                                        (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))- (I3*V3*beta_hv)
    dV4 <- (r*(V4+W4)*(1-((V4+W4)/K))) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                                        (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                                        V5*(S4+I4)/(S4+I4+S5+I5)))- (I4*V4*beta_hv)
    dV5 <- (r*(V5+W5)*(1-((V5+W5)/K))) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                                        (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5)))) - (I5*V5*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                            (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3))))
    dW2 <- (I2*V2*beta_hv) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                            (V1*(S2+I2)/(S1+I1+S2+I2))+
                                            (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4))))
    dW3 <- (I3*V3*beta_hv) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                            (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                            (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))
    dW4 <- (I4*V4*beta_hv) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                            (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                            V5*(S4+I4)/(S4+I4+S5+I5)))
    dW5 <- (I5*V5*beta_hv) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                            (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dS3 <- -beta_vh*S3*W3
    dS4 <- -beta_vh*S4*W4
    dS5 <- -beta_vh*S5*W5
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    dI3 <- beta_vh*S3*W3
    dI4 <- beta_vh*S4*W4
    dI5 <- beta_vh*S5*W5
    res <- c(dV1, dV2, dV3, dV4, dV5,
             dW1, dW2, dW3, dW4, dW5,
             dS1, dS2, dS3, dS4, dS5,
             dI1, dI2, dI3, dI4, dI5)
    list(res)})}


##############################################################
###################     READ IN THE DATA   ###################
##############################################################

data <- read.csv("AA1.csv") 
data$juv[is.na(data$juv)] <- 0; data$adults[is.na(data$adults)] <- 0; data$wings[is.na(data$wings)] <- 0 # fill in 0s:
data$aph <- data$juv + data$adults + data$wings; data$logaph <- log(data$aph+1)
data$pwings <- data$wings / data$aph
data$day <- data$week*7-14 # convert weeks to days

# add treatment IDs
data$fert <- ifelse(data$ttmt==1 | data$ttmt==2 | data$ttmt==3, 0, 1) # keeping numeric so i can 'average' in ddply
data$inf <- ifelse(data$ttmt==1 | data$ttmt==4, -1, ifelse(data$ttmt==2 | data$ttmt==5, 0, 1))
data <- data[data$ring != "Lost aphids" ,] # aphids that were on tray but don't know which plant they came from

# identifiers for different ways to split and summarize data:
data$dID <- paste(data$block, data$ttmt) # dorm-ID 
data$dwID <- paste(data$block, data$ttmt, data$week) # dorm-week ID 
data$twrID <- paste(data$ttmt, data$week, data$ring) # treatment-week ID
inf.data <- read.csv("inf.data.csv") # read in infection data
data <- merge(data, inf.data, by=c("week", "block", "ttmt", "ring"), all.x=T) # merge
data$I[is.na(data$I)] <- 0 # fill in 0s:
data$ring <- 4-as.numeric(data$ring) # convert to 0,1,2,3, where 0 is the center
data$ttmt <- as.factor(data$ttmt) # don't want as numeric

# average by dorm by week (for plotting)
dwdata <- ddply(data, .(dwID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], blk=block[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                prev=sum(I)/4, pwings=mean(pwings)) 

dwdata$dorm <- paste(dwdata$blk, dwdata$ttmt) # dorm-ID 
dwdata$tdID <- paste(dwdata$ttmt, dwdata$day) # treatment-week ID

# average by treatment by week (for plotting)
twdata <- ddply(dwdata, .(tdID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), p.sd=sd(prev),
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), p=mean(prev),
                pwings=mean(pwings)) 
twdata <- twdata[order(twdata$day),] 

# average by treatment-week (i.e., to get prevalence at each ring)
twrdata <- ddply(data, .(twrID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                 day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1], ring=ring[1],
                 aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                 aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                 prev=sum(I)/5) 

fitdata <- data[data$ttmt!=1 & data$ttmt!=4 , ] # omit the aphid free treatments
treats <- unique(fitdata$ttmt) # to loop through each treatment


##################################################################################
#############          PLOT BEST MODEL FIT TO EACH TREATMENT       ###############
##################################################################################

fit <- read.csv("best.model.w.CIs.csv")

t <- seq(from=1, to=60, by=1) # for simulations
start2V <- c(V1=2.5, V2=0, W1=0, W2=0, S1=4, S2=96, I1=0, I2=0)
start2W <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)

sim2 <- data.frame(lsoda(y=start2V, times=t, func=modN2, 
                         parms=c(K=fit$K_D0R0[1], r=fit$r_D0R0[1], 
                                 d=fit$d_D0R0[1], beta_vh=0, beta_vh=fit$beta_vh_R0[1],
                                 beta_hv=fit$beta_hv[1])))
sim3 <- data.frame(lsoda(y=start2W, times=t, func=modN2, 
                         parms=c(K=fit$K_D0R0[1], r=fit$r_D1R0[1], 
                                 d=fit$d_D1R0[1], beta_vh=fit$beta_vh_R0[1],
                                 beta_hv=fit$beta_hv[1])))
sim5 <- data.frame(lsoda(y=start2V, times=t, func=modN2, 
                         parms=c(K=fit$K_D0R1[1], r=fit$r_D0R1[1], 
                                 d=fit$d_D0R1[1], beta_vh=fit$beta_vh_R0[1],
                                 beta_hv=fit$beta_hv[1])))
sim6 <- data.frame(lsoda(y=start2W, times=t, func=modN2, 
                         parms=c(K=fit$K_D0R1[1], r=fit$r_D1R1[1], 
                                 d=fit$d_D1R1[1], beta_vh=fit$beta_vh_R0[1],
                                 beta_hv=fit$beta_hv[1])))


png("treatments.N2.png", width = 6, height = 6, res = 600, units='in')

par(mfrow=c(3,2),mar=c(0,0,0.5,0.5), oma=c(2.5,2.5,0,0))
# save 600 x 600

plot(jitter(data[data$ttmt==2,]$day), data[data$ttmt==2,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,255), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim2$time, sim2$V1+sim2$W1, lty=2, col="dark gray")
lines(sim2$time, sim2$V2+sim2$W2, lty=1, lwd=2, col="dark gray")
axis(side=2, at=c(0,100,200), cex.axis=1.2)

plot(jitter(data[data$ttmt==5,]$day), data[data$ttmt==5,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,255), xaxt="n", yaxt="n")
lines(sim5$time, sim5$V1+sim5$W1, lty=2, col="dark gray")
lines(sim5$time, sim5$V2+sim5$W2, lty=1, lwd=2, col="dark gray")

plot(jitter(data[data$ttmt==3,]$day), data[data$ttmt==3,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,255), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim3$time, sim3$V1+sim3$W1, lty=2, col="black")
lines(sim3$time, sim3$V2+sim3$W2, lty=1, lwd=2, col="black")
axis(side=2, at=c(0,100,200), cex.axis=1.2)

plot(jitter(data[data$ttmt==6,]$day), data[data$ttmt==6,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,255), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim6$time, sim6$V1+sim6$W1, lty=2, col="black")
lines(sim6$time, sim6$V2+sim6$W2, lty=1, lwd=2, col="black")

plot(jitter(twrdata[twrdata$ttmt==3,]$day), twrdata[twrdata$ttmt==3,]$prev, ylim=c(0,1.2), 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylab="infection prevalence", xaxt="n", yaxt="n")
lines(sim3$time, sim3$I1/(sim3$S1+sim3$I1), col="red", lty=2, ylim=c(0,1.2), type="l")
lines(sim3$time, sim3$I2/(sim3$S2+sim3$I2), col="red", lwd=2)
axis(side=2, at=c(0,0.5,1), cex.axis=1.2)
axis(side=1, at=c(0,14,28,42,56), cex.axis=1.2)

plot(jitter(twrdata[twrdata$ttmt==6,]$day), twrdata[twrdata$ttmt==6,]$prev, ylim=c(0,1.2), 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylab="infection prevalence", xaxt="n", yaxt="n")
lines(sim6$time, sim6$I1/(sim6$S1+sim6$I1), col="red", lty=2, ylim=c(0,1.2), type="l")
lines(sim6$time, sim6$I2/(sim6$S2+sim6$I2), col="red", lwd=2)
axis(side=1, at=c(0,14,28,42,56), cex.axis=1.2)

dev.off()


##########################################################################
###########   PLOT BOOTSTRAPPED PARAMETERS OF BEST MODEL   ###############
##########################################################################

png("params.png", width = 6, height = 4, res = 600, units='in')

m <- rbind(c(1,1,2,2),c(3,3,4,5))
layout(m)
par(mar=c(4,6,0,0), oma=c(0,0,0.5,0.5))


x4 <- c(1,2,3,4)
x2 <- c(1,2)
x1 <- 1


# all four r's:
plot(x4, x4, xlim=c(0.5,4.5), ylim=c(0.16,0.34), cex=0, xaxt="n", yaxt="n", ann=F)
plotCI(x4, c(fit$r_D0R0[1], fit$r_D1R0[1], fit$r_D0R1[1], fit$r_D1R1[1]), 
       li=c(fit$r_D0R0[2], fit$r_D1R0[2], fit$r_D0R1[2], fit$r_D1R1[2]),
       ui=c(fit$r_D0R0[3], fit$r_D1R0[3], fit$r_D0R1[3], fit$r_D1R1[3]),
       add=T, err='y', sfrac=0, gap=0, cex=0)
points(x4, c(fit$r_D0R0[1], fit$r_D1R0[1], fit$r_D0R1[1], fit$r_D1R1[1]), 
       pch=c(25,25,24,24), cex=2, bg=c("white", "black", "white", "black"))
axis(side=2, at=c(0.2,0.25,0.3), cex.axis=0.9)

# # r's on log scale: 
# # decided better to keep untransformed, becuase looks same on log scale. 
# # so might as well keep the y axis more interpretable. 
# plot(x4, x4, xlim=c(0.5,4.5), ylim=c(-2,-1), cex=0, xaxt="n", yaxt="n", ann=F)
# plotCI(x4, log(c(fit$r_D0R0[1], fit$r_D1R0[1], fit$r_D0R1[1], fit$r_D1R1[1])), 
#        li=log(c(fit$r_D0R0[2], fit$r_D1R0[2], fit$r_D0R1[2], fit$r_D1R1[2])),
#        ui=log(c(fit$r_D0R0[3], fit$r_D1R0[3], fit$r_D0R1[3], fit$r_D1R1[3])),
#        add=T, err='y', sfrac=0, gap=0, cex=0)
# points(x4, log(c(fit$r_D0R0[1], fit$r_D1R0[1], fit$r_D0R1[1], fit$r_D1R1[1])), 
#        pch=c(25,25,24,24), cex=2, bg=c("white", "black", "white", "black"))
# axis(side=2, at=c(-2,-1), cex.axis=0.9)

# both K's
plot(x2, x2, xlim=c(0.5,2.5), ylim=c(70,180), cex=0, xaxt="n", yaxt="n", ann=F)
plotCI(x2, c(fit$K_D0R0[1], fit$K_D0R1[1]), li=c(fit$K_D0R0[2], fit$K_D0R1[2]),
       ui=c(fit$K_D0R0[3], fit$K_D0R1[3]), add=T, err='y', sfrac=0, gap=0, cex=0)
points(x2, c(fit$K_D0R0[1], fit$K_D0R1[1]), pch=c(25,24), cex=2, bg="dark gray")
axis(side=2, at=c(80,120,160), cex.axis=0.9)

# # # # all four d's:
# plot(x4, x4, xlim=c(0.5,4.5), ylim=c(0,0.55), cex=0, xaxt="n", yaxt="n", ann=F)
# plotCI(x4, c(fit$d_D0R0[1], fit$d_D1R0[1], fit$d_D0R1[1], fit$d_D1R1[1]),
#        li=c(fit$d_D0R0[2], fit$d_D1R0[2], fit$d_D0R1[2],fit$d_D1R1[2]),
#        ui=c(fit$d_D0R0[3], fit$d_D1R0[3], fit$d_D0R1[3], fit$d_D1R1[3]),
#        add=T, err='y', sfrac=0, gap=0, cex=0)
# points(x4, c(fit$d_D0R0[1], fit$d_D1R0[1], fit$d_D0R1[1], fit$d_D1R1[1]),
#        pch=c(25,25,24,24), cex=2, bg=c("white", "black", "white", "black"))
# abline(h=0.5, col="gray", lty=2)
# axis(side=2, at=c(0,0.25,0.5), cex.axis=0.9)

# d's on a log scale:
plot(x4, x4, xlim=c(0.5,4.5), ylim=c(-6.5,2), cex=0, xaxt="n", yaxt="n", ann=F)
plotCI(x4, log(c(fit$d_D0R0[1], fit$d_D1R0[1], fit$d_D0R1[1], fit$d_D1R1[1])), 
       li=log(c(fit$d_D0R0[2], fit$d_D1R0[2], fit$d_D0R1[2], fit$d_D1R1[2])),
       ui=log(c(fit$d_D0R0[3], fit$d_D1R0[3], fit$d_D0R1[3], fit$d_D1R1[3])),
       add=T, err='y', sfrac=0, gap=0, cex=0)
points(x4, log(c(fit$d_D0R0[1], fit$d_D1R0[1], fit$d_D0R1[1], fit$d_D1R1[1])), 
       pch=c(25,25,24,24), cex=2, bg=c("white", "black", "white", "black"))
abline(h=log(0.5), col="gray", lty=2)
axis(side=2, at=c(-6,-3,0), cex.axis=0.9)

# beta
plot(x1, x1, xlim=c(0.5,1.5), ylim=c(0.003,0.011), cex=0, xaxt="n", yaxt="n", ann=F)
plotCI(x1, fit$beta_vh_R0[1], ui=fit$beta_vh_R0[3], li=fit$beta_vh_R0[2], 
       add=T, err='y', sfrac=0, gap=0, cex=0)
points(x1, fit$beta_vh_R0[1], pch=21, cex=2, bg="black")
axis(side=2, at=c(0.003,0.006,0.009), cex.axis=0.9)

# theta
plot(x1, x1, xlim=c(0.5,1.5), ylim=c(0.2,0.4), cex=0, xaxt="n", yaxt="n", ann=F)
plotCI(x1, fit$theta_D0R0[1], ui=fit$theta_D0R0[3], li=fit$theta_D0R0[2], 
       add=T, err='y', sfrac=0, gap=0, cex=0)
points(x1, fit$theta_D0R0[1], pch=c(21), cex=2, bg="dark gray")
axis(side=2, at=c(0.22,0.28,0.34), cex.axis=0.9)

dev.off()


##########################################################################
#################   PLOT EACH MODEL FITTED TO D1R0   #####################
##########################################################################



n1.fits <- read.csv("model.competition.phase.A.csv")
fitN1 <- n1.fits[n1.fits$X=="minus.unimportant",]
disp.fits <- read.csv("model.competition.phase.B.csv")
fitN2 <- disp.fits[disp.fits$X=="N2.4d",]
fitN5 <- disp.fits[disp.fits$X=="N5.4d",]

t <- seq(from=1, to=60, by=1) # for simulations

start1W <- c(V=0, W=0.1, S=100, I=0)
start2W <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)
start5W <- c(V1=0, V2=0, V3=0, V4=0, V5=0,
            W1=2.5, W2=0, W3=0, W4=0, W5=0, 
            S1=4, S2=12, S3=20, S4=28, S5=36, 
            I1=0, I2=0, I3=0, I4=0, I5=0)

sim2_N1 <- data.frame(lsoda(y=start1W, times=t, func=modN1, 
                            parms=c(K=fitN1$K_D0R0[1], r=fitN1$r_D1R0[1], 
                                    beta_vh=fitN2$beta_vh_R0[1],
                                    beta_hv=fitN2$beta_hv[1])))

sim2_N2 <- data.frame(lsoda(y=start2W, times=t, func=modN2, 
                            parms=c(K=fitN2$K_D0R0[1], r=fitN2$r_D1R0[1], 
                                    d=fitN2$d_D1R0[1], beta_vh=fitN2$beta_vh_R0[1],
                                    beta_hv=fitN2$beta_hv[1])))

sim2_N5 <- data.frame(lsoda(y=start5W, times=t, func=modN5, 
                            parms=c(K=fitN5$K_D0R0[1], r=fitN5$r_D1R0[1], 
                                    d=fitN5$d_D1R0[1], beta_vh=fitN5$beta_vh_R0[1],
                                    beta_hv=fitN5$beta_hv[1])))

setwd("C:/Users/straussa/Documents/Research/Seabloom-Borer Lab/Aphid Arenas/Fert/figures")
png("fits.compare.png", width = 6, height = 6, res = 600, units='in')

par(mfrow=c(3,2),mar=c(0,0,0.5,0.5), oma=c(2.5,2.5,0,2))
# save 600 x 600

plot(jitter(data[data$ttmt==3,]$day), data[data$ttmt==3,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,190), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim2_N1$time, sim2_N1$V+sim2_N1$W, lty=1, lwd=1, col="black")
axis(side=2, at=c(0,50,100,150), cex.axis=1.2)

plot(jitter(twrdata[twrdata$ttmt==3,]$day), twrdata[twrdata$ttmt==3,]$prev, ylim=c(0,1.2), 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylab="infection prevalence", xaxt="n", yaxt="n")
lines(sim3_N1$time, sim3_N1$I/(sim3_N1$S+sim3_N1$I), col="black", lty=1, ylim=c(0,1.2), type="l")
axis(side=4, at=c(0,0.5,1), cex.axis=1.2)

plot(jitter(data[data$ttmt==3,]$day), data[data$ttmt==3,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,190), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim2_N2$time, sim2_N2$V1+sim2_N2$W1, lty=1, lwd=1, col="dark gray")
lines(sim2_N2$time, sim2_N2$V2+sim2_N2$W2, lty=1, lwd=1, col="black")
axis(side=2, at=c(0,50,100,150), cex.axis=1.2)


plot(jitter(twrdata[twrdata$ttmt==3,]$day), twrdata[twrdata$ttmt==3,]$prev, ylim=c(0,1.2), 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylab="infection prevalence", xaxt="n", yaxt="n")
lines(sim3_N2$time, sim3_N2$I1/(sim3_N2$S1+sim3_N2$I1), col="dark gray", lty=1, ylim=c(0,1.2), type="l")
lines(sim3_N2$time, sim3_N2$I2/(sim3_N2$S2+sim3_N2$I2), col="black", lty=1, ylim=c(0,1.2), type="l")
axis(side=4, at=c(0,0.5,1), cex.axis=1.2)


plot(jitter(data[data$ttmt==3,]$day), data[data$ttmt==3,]$aph, 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylim=c(0,190), ylab="aphids per plant", xaxt="n", yaxt="n")
lines(sim2_N5$time, sim2_N5$V1+sim2_N5$W1, lty=1, lwd=1, col="dark gray")
lines(sim2_N5$time, sim2_N5$V2+sim2_N5$W2, lty=1, lwd=1, col="red")
lines(sim2_N5$time, sim2_N5$V3+sim2_N5$W3, lty=1, lwd=1, col="orange")
lines(sim2_N5$time, sim2_N5$V4+sim2_N5$W4, lty=1, lwd=1, col="blue")
lines(sim2_N5$time, sim2_N5$V5+sim2_N5$W5, lty=1, lwd=1, col="purple")
axis(side=2, at=c(0,50,100,150), cex.axis=1.2)
axis(side=1, at=c(0,14,28,42,56), cex.axis=1.2)


plot(jitter(twrdata[twrdata$ttmt==3,]$day), twrdata[twrdata$ttmt==3,]$prev, ylim=c(0,1.2), 
     pch=ifelse(data$ring==0, 21, 
                ifelse(data$ring==1, 22, 
                       ifelse(data$ring==2, 24, 25))),
     cex=ifelse(data$ring==0, 1, 0.8),
     bg=ifelse(data$ring==0, "red", 
               ifelse(data$ring==1, "orange", 
                      ifelse(data$ring==2, "blue", "purple"))),
     ylab="infection prevalence", xaxt="n", yaxt="n")
lines(sim3_N5$time, sim3_N5$I1/(sim3_N5$S1+sim3_N5$I1), col="dark gray", lty=1, ylim=c(0,1.2), type="l")
lines(sim3_N5$time, sim3_N5$I2/(sim3_N5$S2+sim3_N5$I2), col="red", lty=1, ylim=c(0,1.2), type="l")
lines(sim3_N5$time, sim3_N5$I3/(sim3_N5$S3+sim3_N5$I3), col="orange", lty=1, ylim=c(0,1.2), type="l")
lines(sim3_N5$time, sim3_N5$I4/(sim3_N5$S4+sim3_N5$I4), col="blue", lty=1, ylim=c(0,1.2), type="l")
lines(sim3_N5$time, sim3_N5$I5/(sim3_N5$S5+sim3_N5$I5), col="purple", lty=1, ylim=c(0,1.2), type="l")
axis(side=4, at=c(0,0.5,1), cex.axis=1.2)
axis(side=1, at=c(0,14,28,42,56), cex.axis=1.2)

dev.off()

