
#############################################################################
####  FITTING SPATIOTEMPORAL DISEASE MODELS TO THE APHID ARENA EXPERIMENT ####
##############################################################################

library(deSolve)
library(bbmle)
library(plyr)
library(gplots)


##########################################################################
###################   A FEW DIFFERENT COMPETING MODELS   #################
##########################################################################

# non-spatial model:
modN1 <- function(t,y,params){
  V <- y[1]; W <- y[2]; S <- y[3]; I <- y[4]
  with(as.list(params), {
    dV <- r*(V+W)*(1-((V+W)/K)) - (I*V*beta_hv)
    dW <- I*V*beta_hv
    dS <- -beta_vh*S*W
    dI <- beta_vh*S*W
    res <- c(dV, dW, dS, dI)
    list(res)})}


#  lagged dispersal model AKA global dispersal AKA N=2
modN2 <- function(t,y,params){
  V1 <- y[1]; V2 <- y[2]; W1 <- y[3]; W2 <- y[4];
  S1 <- y[5]; S2 <- y[6]; I1 <- y[7]; I2 <- y[8]
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I2*V2*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*W1) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2)))) 
    dW2 <- (I2*V2*beta_hv) - (d*W2) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    res <- c(dV1, dV2, dW1, dW2, dS1, dS2, dI1, dI2)
    list(res)})}


# traveling wave / local dispersal / N=5
modN5 <- function(t,y,params){
  V1<-y[1]; V2<-y[2]; V3<-y[3]; V4<-y[4]; V5<-y[5] 
  W1<-y[6]; W2<-y[7]; W3<-y[8]; W4<-y[9]; W5<-y[10] 
  S1<-y[11]; S2<-y[12]; S3<-y[13]; S4<-y[14]; S5<-y[15] 
  I1<-y[16]; I2<-y[17]; I3<-y[18]; I4<-y[19]; I5<-y[20] 
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                                        (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                                        (V1*(S2+I2)/(S1+I1+S2+I2))+
                                                        (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4)))) - (I2*V2*beta_hv)
    dV3 <- (r*(V3+W3)*(1-((V3+W3)/K))) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                                        (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                                        (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))- (I3*V3*beta_hv)
    dV4 <- (r*(V4+W4)*(1-((V4+W4)/K))) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                                        (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                                        V5*(S4+I4)/(S4+I4+S5+I5)))- (I4*V4*beta_hv)
    dV5 <- (r*(V5+W5)*(1-((V5+W5)/K))) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                                        (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5)))) - (I5*V5*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                            (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3))))
    dW2 <- (I2*V2*beta_hv) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                            (V1*(S2+I2)/(S1+I1+S2+I2))+
                                            (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4))))
    dW3 <- (I3*V3*beta_hv) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                            (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                            (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))
    dW4 <- (I4*V4*beta_hv) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                            (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                            V5*(S4+I4)/(S4+I4+S5+I5)))
    dW5 <- (I5*V5*beta_hv) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                            (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dS3 <- -beta_vh*S3*W3
    dS4 <- -beta_vh*S4*W4
    dS5 <- -beta_vh*S5*W5
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    dI3 <- beta_vh*S3*W3
    dI4 <- beta_vh*S4*W4
    dI5 <- beta_vh*S5*W5
    res <- c(dV1, dV2, dV3, dV4, dV5,
             dW1, dW2, dW3, dW4, dW5,
             dS1, dS2, dS3, dS4, dS5,
             dI1, dI2, dI3, dI4, dI5)
    list(res)})}


##############################################################
###################     READ IN THE DATA   ###################
##############################################################

setwd()
data <- read.csv("AA1.csv") 
data$juv[is.na(data$juv)] <- 0; data$adults[is.na(data$adults)] <- 0; data$wings[is.na(data$wings)] <- 0 # fill in 0s:
data$aph <- data$juv + data$adults + data$wings; data$logaph <- log(data$aph+1)
data$pwings <- data$wings / data$aph
data$day <- data$week*7-14 # convert weeks to days

# add treatment IDs
data$fert <- ifelse(data$ttmt==1 | data$ttmt==2 | data$ttmt==3, 0, 1) # keeping numeric so i can 'average' in ddply
data$inf <- ifelse(data$ttmt==1 | data$ttmt==4, -1, ifelse(data$ttmt==2 | data$ttmt==5, 0, 1))
data <- data[data$ring != "Lost aphids" ,] # aphids that were on tray but don't know which plant they came from

# identifiers for different ways to split and summarize data:
data$dID <- paste(data$block, data$ttmt) # dorm-ID 
data$dwID <- paste(data$block, data$ttmt, data$week) # dorm-week ID 
data$twrID <- paste(data$ttmt, data$week, data$ring) # treatment-week ID
inf.data <- read.csv("inf.data.csv") # read in infection data
data <- merge(data, inf.data, by=c("week", "block", "ttmt", "ring"), all.x=T) # merge
data$I[is.na(data$I)] <- 0 # fill in 0s:
data$ring <- 4-as.numeric(data$ring) # convert to 0,1,2,3, where 0 is the center
data$ttmt <- as.factor(data$ttmt) # don't want as numeric

# average by dorm by week (for plotting)
dwdata <- ddply(data, .(dwID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], blk=block[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                prev=sum(I)/4, pwings=mean(pwings)) 

dwdata$dorm <- paste(dwdata$blk, dwdata$ttmt) # dorm-ID 
dwdata$tdID <- paste(dwdata$ttmt, dwdata$day) # treatment-week ID

# average by treatment by week (for plotting)
twdata <- ddply(dwdata, .(tdID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), p.sd=sd(prev),
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), p=mean(prev),
                pwings=mean(pwings)) 
twdata <- twdata[order(twdata$day),] 

# average by treatment-week (i.e., to get prevalence at each ring)
twrdata <- ddply(data, .(twrID), summarize, # dorm-week data: for each dorm in each week, summarize across 4 plants
                day=day[1], ttmt=ttmt[1], fert=fert[1], inf=inf[1], ring=ring[1],
                aph.sd=sd(aph), juv.sd=sd(juv), adults.sd=sd(adults), 
                aph=mean(aph), juv=mean(juv), adults=mean(adults), wings=mean(wings), 
                prev=sum(I)/5) 

fitdata <- data[data$ttmt!=1 & data$ttmt!=4 , ] # omit the aphid free treatments
treats <- unique(fitdata$ttmt) # to loop through each treatment


########################################################################
######################   LIKELIHOOD FUNCTIONS   ########################
########################################################################

# notice the if/else statements in the parameter lists inside the function
# they let me set parameters to zero if i want to try a simpler model

#########################
### NON-SPATIAL MODEL ###
#########################


ll.func.N1 <- function(r_D0R0, r_D0R1, r_D1R0, r_D1R1, K_D0R0, K_D0R1, K_D1R0, K_D1R1,
                       beta_vh_R0, beta_vh_R1, theta_D0R0, theta_D0R1, theta_D1R0, theta_D1R1,
                       beta_hv){
  # ensure all positive:
  r_D0R0=exp(log(r_D0R0)); r_D1R0=exp(log(r_D1R0)); r_D0R1=exp(log(r_D0R1)); r_D1R1=exp(log(r_D1R1))
  K_D0R0=exp(log(K_D0R0)); K_D1R0=exp(log(K_D1R0)); K_D0R1=exp(log(K_D0R1)); K_D1R1=exp(log(K_D1R1))
  beta_vh_R0=exp(log(beta_vh_R0)); beta_vh_R1=exp(log(beta_vh_R1))
  theta_D0R0=exp(log(theta_D0R0)); theta_D1R0=exp(log(theta_D1R0)); theta_D0R1=exp(log(theta_D0R1)); theta_D1R1=exp(log(theta_D1R1))
  
  # rescale K's:
  K_D0R0=1000*K_D0R0; K_D0R1=1000*K_D0R1; K_D1R0=1000*K_D1R0; K_D1R1=1000*K_D1R1
  
  # which parameters used for each simulation
  # need a bunch of if/else statements for backwards model selection.
  par_D0R0 <- c(r=r_D0R0, K=K_D0R0, beta_vh=0, theta=theta_D0R0, beta_hv=beta_hv)
  par_D1R0 <- c(r=ifelse(r_D1R0>0, r_D1R0, r_D0R0), K=ifelse(K_D1R0>0, K_D1R0, K_D0R0), beta_hv=beta_hv, 
                beta_vh=beta_vh_R0, theta=ifelse(theta_D1R0>0, theta_D1R0, theta_D0R0))
  par_D0R1 <- c(r=ifelse(r_D0R1>0, r_D0R1, r_D0R0), K=ifelse(K_D0R1>0, K_D0R1, K_D0R0), beta_hv=beta_hv, 
                beta_vh=0, theta=ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))
  par_D1R1 <- c(r=ifelse(r_D1R1>0, r_D1R1, ifelse(r_D1R0>0, r_D1R0, ifelse(r_D0R1>0, r_D0R1, r_D0R0))),
                K=ifelse(K_D1R1>0, K_D1R1, ifelse(K_D1R0>0, K_D1R0, ifelse(K_D0R1>0, K_D0R1, K_D0R0))),
                beta_vh=ifelse(beta_vh_R1>0, beta_vh_R1, beta_vh_R0), beta_hv=beta_hv,
                theta=ifelse(theta_D1R1>0, theta_D1R1, ifelse(theta_D1R0>0, theta_D1R0, ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))))
  par_lists <- list(par_D0R0, par_D1R0, par_D0R1, par_D1R1) # order must match treatments
  
    # for all simulations:
  start_D0 <- c(V=0.1, W=0, S=100, I=0)
  start_D1 <- c(V=0, W=0.1, S=100, I=0)
  t <- seq(from=1, to=60, by=1)
  
  treat.lls <- data.frame(treats=treats) 
  for(i in 1:length(treats)){
    tdata <- data[data$ttmt==treats[i],]
    tdata <- tdata[!(is.na(tdata$aph)),] # omit cases with unknown aphid #
    if (max(tdata$inf)>0){start=start_D1} else {start=start_D0}
    sim <- data.frame(lsoda(y=start, times=t, func=modN1, parms=as.list(par_lists[[i]])))
    res.df <- merge(data.frame(time=tdata$day, aph=tdata$aph, ring=tdata$ring, inf=tdata$I), sim, by="time")
    res.df$ll.aph <- dnbinom(res.df$aph, mu=res.df$V+res.df$W, size=1/par_lists[[i]]["theta"] , log=T)
    res.df$ll.inf <- dbinom(res.df$inf, size=1, prob=pmin(res.df$I/(res.df$S+res.df$I), 1), log=T)
    res.df$ll.inf <- ifelse(res.df$ll.inf==-Inf, -100, res.df$ll.inf) # replace -inf (from rounding error) with big negative
    treat.lls$ll[i] <- -sum(res.df$ll.aph)+-sum(res.df$ll.inf)
  }
  sum(treat.lls$ll)
}

#########################
### GLOBAL DISPERSAL ####
#########################

ll.func.N2 <- function(r_D0R0, r_D0R1, r_D1R0, r_D1R1, K_D0R0, K_D0R1, K_D1R0, K_D1R1,
                       d_D0R0, d_D0R1, d_D1R0, d_D1R1, 
                       beta_vh_R0, beta_vh_R1, theta_D0R0, theta_D0R1, theta_D1R0, theta_D1R1,
                       beta_hv){
  # ensure all positive:
  r_D0R0=exp(log(r_D0R0)); r_D1R0=exp(log(r_D1R0)); r_D0R1=exp(log(r_D0R1)); r_D1R1=exp(log(r_D1R1))
  K_D0R0=exp(log(K_D0R0)); K_D1R0=exp(log(K_D1R0)); K_D0R1=exp(log(K_D0R1)); K_D1R1=exp(log(K_D1R1))
  d_D0R0=exp(log(d_D0R0)); d_D1R0=exp(log(d_D1R0)); d_D0R1=exp(log(d_D0R1)); d_D1R1=exp(log(d_D1R1))
  beta_vh_R0=exp(log(beta_vh_R0)); beta_vh_R1=exp(log(beta_vh_R1))
  theta_D0R0=exp(log(theta_D0R0)); theta_D1R0=exp(log(theta_D1R0)); theta_D0R1=exp(log(theta_D0R1)); theta_D1R1=exp(log(theta_D1R1))
  
  # rescale K's:
  K_D0R0=1000*K_D0R0; K_D0R1=1000*K_D0R1; K_D1R0=1000*K_D1R0; K_D1R1=1000*K_D1R1
  
  # which parameters used for each simulation
  # need a bunch of if/else statements for backwards model selection.
  par_D0R0 <- c(r=r_D0R0, K=K_D0R0, d=d_D0R0, beta_vh=0, theta=theta_D0R0, beta_hv=beta_hv)
  par_D1R0 <- c(r=ifelse(r_D1R0>0, r_D1R0, r_D0R0), K=ifelse(K_D1R0>0, K_D1R0, K_D0R0), 
                d=ifelse(d_D1R0>0, d_D1R0, d_D0R0), beta_hv=beta_hv,
                beta_vh=beta_vh_R0, theta=ifelse(theta_D1R0>0, theta_D1R0, theta_D0R0))
  par_D0R1 <- c(r=ifelse(r_D0R1>0, r_D0R1, r_D0R0), K=ifelse(K_D0R1>0, K_D0R1, K_D0R0),
                d=ifelse(d_D0R1>0, d_D0R1, d_D0R0), beta_hv=beta_hv, 
                beta_vh=0, theta=ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))
  par_D1R1 <- c(r=ifelse(r_D1R1>0, r_D1R1, ifelse(r_D1R0>0, r_D1R0, ifelse(r_D0R1>0, r_D0R1, r_D0R0))),
                K=ifelse(K_D1R1>0, K_D1R1, ifelse(K_D1R0>0, K_D1R0, ifelse(K_D0R1>0, K_D0R1, K_D0R0))),
                d=ifelse(d_D1R1>0, d_D1R1, ifelse(d_D1R0>0, d_D1R0, ifelse(d_D0R1>0, d_D0R1, d_D0R0))),
                beta_vh=ifelse(beta_vh_R1>0, beta_vh_R1, beta_vh_R0), beta_hv=beta_hv,
                theta=ifelse(theta_D1R1>0, theta_D1R1, ifelse(theta_D1R0>0, theta_D1R0, ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))))
  par_lists <- list(par_D0R0, par_D1R0, par_D0R1, par_D1R1) # order must match treatments
  
  # for all simulations:
  start_D0 <- c(V1=2.5, V2=0, W1=0, W2=0, S1=4, S2=96, I1=0, I2=0)
  start_D1 <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)
  t <- seq(from=1, to=60, by=1)
  
  treat.lls <- data.frame(treats=treats)
  for(i in 1:length(treats)){
    tdata <- data[data$ttmt==treats[i],]
    tdata <- tdata[!(is.na(tdata$aph)),] # omit cases with unknown aphid #
    if (max(tdata$inf)>0){start=start_D1} else {start=start_D0}  
    sim <- data.frame(lsoda(y=start, times=t, func=modN2, parms=as.list(par_lists[[i]])))
    res.df <- merge(data.frame(time=tdata$day, aph=tdata$aph, ring=tdata$ring, inf=tdata$I), sim, by="time")
    res.df$ll.aph <- dnbinom(res.df$aph, mu=res.df$V2+res.df$W2, size=1/par_lists[[i]]["theta"] , log=T)
    res.df$ll.inf <- dbinom(res.df$inf, size=1, prob=pmin(res.df$I2/(res.df$S2+res.df$I2),1), log=T)
    res.df$ll.inf <- ifelse(res.df$ll.inf==-Inf, -100, res.df$ll.inf) # replace -inf (from rounding error) with big negative
    treat.lls$ll[i] <- -sum(res.df$ll.aph)+-sum(res.df$ll.inf)
  }
  sum(treat.lls$ll)
}


#########################
#### LOCAL DISPERSAL ####
#########################


ll.func.N5 <- function(r_D0R0, r_D0R1, r_D1R0, r_D1R1, K_D0R0, K_D0R1, K_D1R0, K_D1R1,
                       d_D0R0, d_D0R1, d_D1R0, d_D1R1, 
                       beta_vh_R0, beta_vh_R1, theta_D0R0, theta_D0R1, theta_D1R0, theta_D1R1,
                       beta_hv){
  # ensure all positive:
  r_D0R0=exp(log(r_D0R0)); r_D1R0=exp(log(r_D1R0)); r_D0R1=exp(log(r_D0R1)); r_D1R1=exp(log(r_D1R1))
  K_D0R0=exp(log(K_D0R0)); K_D1R0=exp(log(K_D1R0)); K_D0R1=exp(log(K_D0R1)); K_D1R1=exp(log(K_D1R1))
  d_D0R0=exp(log(d_D0R0)); d_D1R0=exp(log(d_D1R0)); d_D0R1=exp(log(d_D0R1)); d_D1R1=exp(log(d_D1R1))
  beta_vh_R0=exp(log(beta_vh_R0)); beta_vh_R1=exp(log(beta_vh_R1))
  theta_D0R0=exp(log(theta_D0R0)); theta_D1R0=exp(log(theta_D1R0)); theta_D0R1=exp(log(theta_D0R1)); theta_D1R1=exp(log(theta_D1R1))
  
  # rescale K's:
  K_D0R0=1000*K_D0R0; K_D0R1=1000*K_D0R1; K_D1R0=1000*K_D1R0; K_D1R1=1000*K_D1R1
  
  # which parameters used for each simulation
  # need a bunch of if/else statements for backwards model selection.
  par_D0R0 <- c(r=r_D0R0, K=K_D0R0, d=d_D0R0, beta_vh=0, theta=theta_D0R0, beta_hv=beta_hv)
  par_D1R0 <- c(r=ifelse(r_D1R0>0, r_D1R0, r_D0R0), K=ifelse(K_D1R0>0, K_D1R0, K_D0R0), 
                d=ifelse(d_D1R0>0, d_D1R0, d_D0R0), beta_hv=beta_hv,
                beta_vh=beta_vh_R0, theta=ifelse(theta_D1R0>0, theta_D1R0, theta_D0R0))
  par_D0R1 <- c(r=ifelse(r_D0R1>0, r_D0R1, r_D0R0), K=ifelse(K_D0R1>0, K_D0R1, K_D0R0),
                d=ifelse(d_D0R1>0, d_D0R1, d_D0R0), beta_hv=beta_hv,
                beta_vh=0, theta=ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))
  par_D1R1 <- c(r=ifelse(r_D1R1>0, r_D1R1, ifelse(r_D1R0>0, r_D1R0, ifelse(r_D0R1>0, r_D0R1, r_D0R0))),
                K=ifelse(K_D1R1>0, K_D1R1, ifelse(K_D1R0>0, K_D1R0, ifelse(K_D0R1>0, K_D0R1, K_D0R0))),
                d=ifelse(d_D1R1>0, d_D1R1, ifelse(d_D1R0>0, d_D1R0, ifelse(d_D0R1>0, d_D0R1, d_D0R0))),
                beta_vh=ifelse(beta_vh_R1>0, beta_vh_R1, beta_vh_R0), beta_hv=beta_hv,
                theta=ifelse(theta_D1R1>0, theta_D1R1, ifelse(theta_D1R0>0, theta_D1R0, ifelse(theta_D0R1>0, theta_D0R1, theta_D0R0))))
  par_lists <- list(par_D0R0, par_D1R0, par_D0R1, par_D1R1) # order must match treatments
  
  #for all simulations:
  start_D0 <- c(V1=2.5, V2=0, V3=0, V4=0, V5=0,
                W1=0, W2=0, W3=0, W4=0, W5=0, 
                S1=4, S2=12, S3=20, S4=28, S5=36,
                I1=0, I2=0, I3=0, I4=0, I5=0)
  start_D1 <- c(V1=0, V2=0, V3=0, V4=0, V5=0,
                W1=2.5, W2=0, W3=0, W4=0, W5=0, 
                S1=4, S2=12, S3=20, S4=28, S5=36,
                I1=0, I2=0, I3=0, I4=0, I5=0)
  t <- seq(from=1, to=60, by=1)
  
  treat.lls <- data.frame(treats=treats)
  for(i in 1:length(treats)){
    tdata <-  data[data$ttmt==treats[i],]
    tdata <- tdata[!(is.na(tdata$aph)),] # omit cases with unknown aphid #
    if (max(tdata$inf)>0){start=start_D1} else {start=start_D0}  
    sim <- data.frame(lsoda(y=start, times=t, func=modN5, parms=as.list(par_lists[[i]])))
    
    res.df <- merge(data.frame(time=tdata$day, aph=tdata$aph, ring=tdata$ring, inf=tdata$I), sim, by="time")
    res.df$pred.aph <- ifelse(res.df$ring==0, res.df$V2+res.df$W2, ifelse(
      res.df$ring==1, res.df$V3+res.df$W3, ifelse(
        res.df$ring==2, res.df$V4+res.df$W4, ifelse(
          res.df$ring==3, res.df$V5+res.df$W5, NA))))
    res.df$pred.inf <- ifelse(res.df$ring==0, res.df$I2/(res.df$S2+res.df$I2), ifelse(
      res.df$ring==1, res.df$I3/(res.df$S3+res.df$I3), ifelse(
        res.df$ring==2, res.df$I4/(res.df$S4+res.df$I4), ifelse(
          res.df$ring==3, res.df$I5/(res.df$S5+res.df$I5), NA))))
    res.df$ll.aph <- dnbinom(res.df$aph, mu=res.df$pred.aph, size=1/par_lists[[i]]["theta"] , log=T)
    res.df$ll.inf <- dbinom(res.df$inf, size=1, prob=pmin(res.df$pred.inf, 1), log=T)
    res.df$ll.inf <- ifelse(res.df$ll.inf==-Inf, -100, res.df$ll.inf) # replace -inf (from rounding error) with big negative
    treat.lls$ll[i] <- -sum(res.df$ll.aph)+-sum(res.df$ll.inf)
  }
  sum(treat.lls$ll)
}



########################################################################
########################################################################
############   FIT MODELS, TWO PHASE MODEL COMPETITION   ###############
########################################################################
########################################################################


################################################################
####### PHASE 1:  BACKWARDS MODEL SIMPLIFICATION      ##########
#######           IN THE NON-SPATIAL MODEL            ##########
################################################################


#########################################
# null model: no treatment efects

start=list(r_D0R0=0.2, K_D0R0=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.null <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                 start=start, 
                 upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                 fixed=c(r_D0R1=0, r_D1R0=0, r_D1R1=0, K_D0R1=0, K_D1R0=0, K_D1R1=0,
                         beta_vh_R1=0, theta_D0R1=0, theta_D1R0=0, theta_D1R1=0,
                         beta_hv=0.68),
                 control=list(parscale=unlist(start)))
fit.null
summary(fit.null)


#########################################

# commented out code shows that the optimizer "Nelder-Mead" is not robust
# to different starting conditions. But L-BFGS-B works much better:

# 1A) full model: ALL treatment efects. takes around ~7 minutes to fit. 

# start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2,
#            K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
#            beta_vh_R0=0.005, beta_vh_R1=0.005,
#            theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
# # do the parameters in this model depend on starting conditions, or robust?
# # with standard starting values, ll = -2404
# start=as.list(unlist(start)/1.5) # if start at 50%... some dissimilarity. ll= -2431
# start=as.list(unlist(start)*1.5) # if start at 150%... big dissimilarity. ll= -3157
# # alarming that ll and params change so much with different starting conditions here. 
# 
# fit.full <- mle2(ll.func.N1, method="Nelder-Mead", skip.hessian=F, 
#                  start=start, control=list(parscale=unlist(start),maxit=1000000))
# fit.full


# trying L-BFGS-B instead:
start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2,
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005, beta_vh_R1=0.005,
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
# with default starting parameters, ll=-2403
#start=as.list(unlist(start)/1.5) # if start at 50%, consistent results! ll=-2403
#start=as.list(unlist(start)*1.5) # if start at 150%, consistent results! ll=-2403
# so using L-BFGS-B for phase 1 of model competition. 

fit.full <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                 start=start, 
                 upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                 control=list(parscale=unlist(start)),
                 fixed=c(beta_hv=0.68))
fit.full


#########################################
# 1B) remove effect or R on K
start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D1R0=0.12, 
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
fit.minus.RonK <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                       start=start, fixed=c(K_D0R1=0, K_D1R1=0,
                                            beta_hv=0.68),
                       upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                       control=list(parscale=unlist(start)))
fit.minus.RonK


#########################################
# 1C) remove effect or D on K

start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, 
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3) 
fit.minus.DonK <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                       start=start, fixed=c(K_D1R0=0, K_D1R1=0,
                                            beta_hv=0.68),
                       upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                       control=list(parscale=unlist(start)))
fit.minus.DonK


#########################################
# 1D) remove effect or R on r

start=list(r_D0R0=0.2, r_D1R0=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
fit.minus.Ronr <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                       start=start, fixed=c(r_D0R1=0, r_D1R1=0,
                                            beta_hv=0.68),
                       upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                       control=list(parscale=unlist(start)))
fit.minus.Ronr


#########################################
# 1E) remove effect or D on r

start=list(r_D0R0=0.2, r_D0R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
fit.minus.Donr <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                       start=start, fixed=c(r_D1R0=0, r_D1R1=0,
                                            beta_hv=0.68),
                       upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                       control=list(parscale=unlist(start)))
fit.minus.Donr


#########################################
# 1F) remove effect or R on beta

start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005,  
           theta_D0R0=0.3, theta_D0R1=0.3, theta_D1R0=0.3, theta_D1R1=0.3)
fit.minus.Ronbeta <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                          start=start, fixed=c(beta_vh_R1=0,
                                               beta_hv=0.68),
                          upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                          control=list(parscale=unlist(start)))
fit.minus.Ronbeta


#########################################
# 1G) remove effect or R on theta

start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D1R0=0.3) 
fit.minus.Rontheta <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                           start=start, fixed=c(theta_D0R1=0,theta_D1R1=0,
                                                beta_hv=0.68),
                           upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                           control=list(parscale=unlist(start)))
fit.minus.Rontheta


#########################################
# 1H) remove effect or D on theta

start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12,
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D0R1=0.3)
fit.minus.Dontheta <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                           start=start, fixed=c(theta_D1R0=0, theta_D1R1=0,
                                                beta_hv=0.68),
                           upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                           control=list(parscale=unlist(start)))
fit.minus.Dontheta


#########################################
# 2A) remove all unimportant effects (criteria: AIC within 6)

start=list(r_D0R0=0.2, r_D0R1=0.2, r_D1R0=0.2, r_D1R1=0.2, 
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.minus.unimportant <- mle2(ll.func.N1, method="L-BFGS-B", skip.hessian=F, 
                              start=start, fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, 
                                                   theta_D1R0=0, theta_D1R1=0, theta_D0R1=0,
                                                   beta_hv=0.68),
                              upper=as.list(unlist(start)*5), lower=as.list(unlist(start)/5),
                              control=list(parscale=unlist(start)))
fit.minus.unimportant


###########################################################
# save summary statistics for phase 1 of model competition

fit.summary <- data.frame(rbind(coef(fit.null), coef(fit.full), coef(fit.minus.DonK), coef(fit.minus.Donr),
                                coef(fit.minus.Dontheta), coef(fit.minus.Ronbeta), coef(fit.minus.RonK),
                                coef(fit.minus.Ronr), coef(fit.minus.Rontheta), coef(fit.minus.unimportant)))
# rescale K's back here
fit.summary$K_D0R0 <- fit.summary$K_D0R0*1000; fit.summary$K_D0R1 <- fit.summary$K_D0R1*1000
fit.summary$K_D1R0 <- fit.summary$K_D1R0*1000; fit.summary$K_D1R1 <- fit.summary$K_D1R1*1000

rownames(fit.summary) <- c("null","full","minus.DonK","minus.Donr","minus.Dontheta", 
                           "minus.Ronbeta","minus.RonK","minus.Ronr","minus.Rontheta",
                           "minus.unimportant")

fit.summary$ll <- c(summary(fit.null)@m2logL/-2, summary(fit.full)@m2logL/-2, summary(fit.minus.DonK)@m2logL/-2, 
                    summary(fit.minus.Donr)@m2logL/-2, summary(fit.minus.Dontheta)@m2logL/-2, 
                    summary(fit.minus.Ronbeta)@m2logL/-2, summary(fit.minus.RonK)@m2logL/-2,
                    summary(fit.minus.Ronr)@m2logL/-2, summary(fit.minus.Rontheta)@m2logL/-2, 
                    summary(fit.minus.unimportant)@m2logL/-2)
write.csv(fit.summary, file="model.competition.phase.A.csv")

###########################################################


################################################################
########## PHASE 2:  MODEL COMPETITION INCLUDING      ##########
##########           DISPERSAL SCENARIOS            ############
################################################################

# model 2A fit above


##################################
# 2B) global dispersal, single d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, d_D0R0=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.N2.1d <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F, 
                  start=start,  
                  fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                          theta_D1R1=0, theta_D0R1=0, d_D1R0=0, d_D0R1=0, d_D1R1=0,
                          beta_hv=0.68),
                  upper=as.list(unlist(start)*c(5,5,5,5,5,5,5,5,5)), # more flexibility for d 
                  lower=as.list(unlist(start)/c(5,5,5,5,100,5,5,5,5)), # more flexibility for d
                  control=list(parscale=unlist(start), maxit=500))
fit.N2.1d
summary(fit.N2.1d) # looks good

##################################
# 2C) global dispersal, R on d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, 
           d_D0R0=0.1, d_D0R1=0.1, 
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.N2.Rond <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F, 
                    start=start, 
                    fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                            theta_D1R1=0, theta_D0R1=0, d_D1R0=0, d_D1R1=0,
                            beta_hv=0.68),
                    upper=as.list(unlist(start)*c(5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
                    lower=as.list(unlist(start)/c(5,5,5,5,100,100,5,5,5,5)), # more flexibility for d's
                    control=list(parscale=unlist(start), maxit=500))
fit.N2.Rond
summary(fit.N2.Rond)
summary(fit.N2.Rond)@m2logL/-2
# d is on the boundary d>0.5, but meaningless. 

##################################
# 2D) global dispersal, D on d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22,
           d_D0R0=0.1, d_D1R0=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.N2.Dond <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F, 
                    start=start, 
                    fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                            theta_D1R1=0, theta_D0R1=0, d_D0R1=0, d_D1R1=0,
                            beta_hv=0.68),
                    upper=as.list(unlist(start)*c(5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
                    lower=as.list(unlist(start)/c(5,5,5,5,100,100,5,5,5,5)), # more flexibility for d's
                    control=list(parscale=unlist(start), maxit=500))
fit.N2.Dond
summary(fit.N2.Dond) 
summary(fit.N2.Rond)@m2logL/-2


##################################
# 2E) global dispersal, D on d and R on D

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22,
           d_D0R0=0.1, d_D1R0=0.1, d_D0R1=0.1, d_D1R1=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)

fit.N2.4d <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F, 
                  start=start, 
                  fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                          theta_D1R1=0, theta_D0R1=0,
                          beta_hv=0.68),
                  upper=as.list(unlist(start)*c(5,5,5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
                  lower=as.list(unlist(start)/c(5,5,5,5,100,100,100,100,5,5,5,5)), # more flexibility for d's
                  control=list(parscale=unlist(start), maxit=500))
fit.N2.4d
summary(fit.N2.4d) 
summary(fit.N2.4d)@m2logL/-2
# 2 d's on the boundary, but d>0.5, but meaningless. 


# # try repeating with start at the fits. check to is if parameters are consistent, and errors estimated
# run1 <- coef(fit.N2.4d)
# est2 <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F,
#              start=as.list(run1[run1>0]),
#              fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0,
#                      theta_D1R1=0, theta_D0R1=0),
#              upper=as.list(run1[run1>0]*c(5,5,5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
#              lower=as.list(run1[run1>0]/c(5,5,5,5,100,100,100,100,5,5,5,5)), # more flexibility for d's
#              control=list(maxit=500))
# run2 <- coef(est2)
# summary(est2)
# run1[run1>0]
# run2[run2>0]
# # yes! consistent!

# since this ends up being the best model, also want to look at the likelihood surface 
# for r and K.. are they correlated? 



##################################
# 2F) local dispersal, single d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, d_D0R0=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3) 
fit.N5.1d <- mle2(ll.func.N5, method="L-BFGS-B", skip.hessian=F, 
                  start=start, 
                  fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                          theta_D1R1=0, theta_D0R1=0, d_D1R0=0, d_D0R1=0, d_D1R1=0,
                          beta_hv=0.68),
                  upper=as.list(unlist(start)*c(5,5,5,5,10,5,5,5,5)), # more flexibility for d's
                  lower=as.list(unlist(start)/c(5,5,5,5,10,5,5,5,5)), # more flexibility for d's
                  control=list(parscale=unlist(start), maxit=500))
fit.N5.1d
summary(fit.N5.1d)

##################################
# 2G) local dispersal, R on d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, 
           d_D0R0=0.1, d_D0R1=0.1, 
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3) 
fit.N5.Rond <- mle2(ll.func.N5, method="L-BFGS-B", skip.hessian=F, 
                    start=start, 
                    fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                            theta_D1R1=0, theta_D0R1=0, d_D1R0=0, d_D1R1=0,
                            beta_hv=0.68),
                    upper=as.list(unlist(start)*c(5,5,5,5,10,10,5,5,5,5)), # more flexibility for d's
                    lower=as.list(unlist(start)/c(5,5,5,5,10,10,5,5,5,5)), # more flexibility for d's
                    control=list(parscale=unlist(start), maxit=500))
fit.N5.Rond
summary(fit.N5.Rond)

##################################
# 2H) local dispersal, D on d

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, 
           d_D0R0=0.1, d_D1R0=0.1, 
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3)
fit.N5.Dond <- mle2(ll.func.N5, method="L-BFGS-B", skip.hessian=F, 
                    start=start, 
                    fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                            theta_D1R1=0, theta_D0R1=0, d_D0R1=0, d_D1R1=0,
                            beta_hv=0.68),
                    upper=as.list(unlist(start)*c(5,5,5,5,10,10,5,5,5,5)), # more flexibility for d's
                    lower=as.list(unlist(start)/c(5,5,5,5,10,10,5,5,5,5)), # more flexibility for d's
                    control=list(parscale=unlist(start), maxit=500))
fit.N5.Dond
summary(fit.N5.Dond)

##################################
# 2I) local dispersal, D on d and R on D

# this one seems especially sensitive to the non finite initial error
# so trying with much more restrictive box constrains on optimizer

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22, 
           d_D0R0=0.1, d_D1R0=0.1, d_D0R1=0.1, d_D1R1=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, beta_vh_R0=0.005, theta_D0R0=0.3) 
fit.N5.4d <- mle2(ll.func.N5, method="L-BFGS-B", skip.hessian=F, 
                  start=start, 
                  fixed=c(K_D1R0=0, K_D1R1=0, beta_vh_R1=0, theta_D1R0=0, 
                          theta_D1R1=0, theta_D0R1=0,
                          beta_hv=0.68),
                  upper=as.list(unlist(start)*c(2,2,2,2,5,5,5,5,2,2,2,2)), # more flexibility for d's
                  lower=as.list(unlist(start)/c(2,2,2,2,5,5,5,5,2,2,2,2)), # more flexibility for d's
                  control=list(parscale=unlist(start), maxit=500))
fit.N5.4d
summary(fit.N5.4d)


################################################################
########## PHASE 3:  ADDING TERMS BACK TO BEST MODEL   #########
##########          CONFIRM THAT THEY ARE UNIMPORTANT  #########
################################################################


##################################
# 3A) global dispersal, full model with all non-spatial and spatial params

start=list(r_D0R0=0.22, r_D0R1=0.22, r_D1R0=0.22, r_D1R1=0.22,
           d_D0R0=0.1, d_D1R0=0.1, d_D0R1=0.1, d_D1R1=0.1,
           K_D0R0=0.12,  K_D0R1=0.12, K_D1R0=0.12, K_D1R1=0.12, 
           beta_vh_R0=0.005, beta_vh_R1=0.005, 
           theta_D0R0=0.3, theta_D1R0=0.3, theta_D1R1=0.3, theta_D0R1=0.3)

fit.N2.full <- mle2(ll.func.N2, method="L-BFGS-B", skip.hessian=F, 
                  start=start, 
                  fixed=c(beta_hv=0.68),
                  upper=as.list(unlist(start)*c(5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
                  lower=as.list(unlist(start)/c(5,5,5,5,100,100,100,100,5,5,5,5,5,5,5,5,5,5)), # more flexibility for d's
                  control=list(parscale=unlist(start), maxit=1000))
fit.N2.full
summary(fit.N2.full) 
summary(fit.N2.full)@m2logL/-2



################################################################
################################################################
# save summary statistics for phase 2 & 3 of model competition
################################################################
################################################################


fit.summary.B <- data.frame(rbind(coef(fit.N2.1d), coef(fit.N2.Rond), coef(fit.N2.Dond), coef(fit.N2.4d),
                                  coef(fit.N5.1d), coef(fit.N5.Rond), coef(fit.N5.Dond), coef(fit.N5.4d),
                                  coef(fit.N2.full)))
# rescale K's back here
fit.summary.B$K_D0R0 <- fit.summary.B$K_D0R0*1000; fit.summary.B$K_D0R1 <- fit.summary.B$K_D0R1*1000
fit.summary.B$K_D1R0 <- fit.summary.B$K_D1R0*1000; fit.summary.B$K_D1R1 <- fit.summary.B$K_D1R1*1000

rownames(fit.summary.B) <- c("N2.1d","N2.Rond","N2.Dond","N2.4d",
                             "N5.1d","N5.Rond","N5.Dond","N5.4d",
                             "N2.full")

fit.summary.B$ll <- c(summary(fit.N2.1d)@m2logL/-2, summary(fit.N2.Rond)@m2logL/-2, 
                      summary(fit.N2.Dond)@m2logL/-2, summary(fit.N2.4d)@m2logL/-2,
                      summary(fit.N5.1d)@m2logL/-2, summary(fit.N5.Rond)@m2logL/-2, 
                      summary(fit.N5.Dond)@m2logL/-2, summary(fit.N5.4d)@m2logL/-2,
                      summary(fit.N2.full)@m2logL/-2)
write.csv(fit.summary.B, file="model.competition.phase.B.csv")

