
##########################################################################
####  COMPETING LINKED LOGISTIC MODELS FOR APHID DISPERSAL AND GROWTH ####
##########################################################################

library(deSolve)
library(bbmle)
library(plyr)
library(gplots)

##########################################################################
###################   A FEW DIFFERENT COMPETING MODELS   #################
##########################################################################

# non-spatial model:
modN1 <- function(t,y,params){
  V <- y[1]; W <- y[2]; S <- y[3]; I <- y[4]
  with(as.list(params), {
    dV <- r*(V+W)*(1-((V+W)/K)) - (I*V*beta_hv)
    dW <- I*V*beta_hv
    dS <- -beta_vh*S*W
    dI <- beta_vh*S*W
    res <- c(dV, dW, dS, dI)
    list(res)})}


#  lagged dispersal model AKA global dispersal AKA N=2
modN2 <- function(t,y,params){
  V1 <- y[1]; V2 <- y[2]; W1 <- y[3]; W2 <- y[4];
  S1 <- y[5]; S2 <- y[6]; I1 <- y[7]; I2 <- y[8]
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*(V1*((S1+I1)/(S1+I1+S2+I2))+V2*((S2+I2)/(S1+I1+S2+I2)))) - (I2*V2*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*W1) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2)))) 
    dW2 <- (I2*V2*beta_hv) - (d*W2) + (d*(W1*((S1+I1)/(S1+I1+S2+I2))+W2*((S2+I2)/(S1+I1+S2+I2))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    res <- c(dV1, dV2, dW1, dW2, dS1, dS2, dI1, dI2)
    list(res)})}


# traveling wave / local dispersal / N=5
modN5 <- function(t,y,params){
  V1<-y[1]; V2<-y[2]; V3<-y[3]; V4<-y[4]; V5<-y[5] 
  W1<-y[6]; W2<-y[7]; W3<-y[8]; W4<-y[9]; W5<-y[10] 
  S1<-y[11]; S2<-y[12]; S3<-y[13]; S4<-y[14]; S5<-y[15] 
  I1<-y[16]; I2<-y[17]; I3<-y[18]; I4<-y[19]; I5<-y[20] 
  with(as.list(params), {
    dV1 <- (r*(V1+W1)*(1-((V1+W1)/K))) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                                        (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3)))) - (I1*V1*beta_hv)
    dV2 <- (r*(V2+W2)*(1-((V2+W2)/K))) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                                        (V1*(S2+I2)/(S1+I1+S2+I2))+
                                                        (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4)))) - (I2*V2*beta_hv)
    dV3 <- (r*(V3+W3)*(1-((V3+W3)/K))) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                                        (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                                        (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))- (I3*V3*beta_hv)
    dV4 <- (r*(V4+W4)*(1-((V4+W4)/K))) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                                        (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                                        V5*(S4+I4)/(S4+I4+S5+I5)))- (I4*V4*beta_hv)
    dV5 <- (r*(V5+W5)*(1-((V5+W5)/K))) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                                        (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5)))) - (I5*V5*beta_hv)
    dW1 <- (I1*V1*beta_hv) - (d*V1) + (d*((V1*(S1+I1)/(S1+I1+S2+I2))+
                                            (V2*(S1+I1)/(S1+I1+S2+I2+S3+I3))))
    dW2 <- (I2*V2*beta_hv) - (d*V2) + (d*((V2*(S2+I2)/(S1+I1+S2+I2+S3+I3))+
                                            (V1*(S2+I2)/(S1+I1+S2+I2))+
                                            (V3*(S2+I2)/(S2+I2+S3+I3+S4+I4))))
    dW3 <- (I3*V3*beta_hv) - (d*V3) + (d*((V3*(S3+I3)/(S2+I2+S3+I3+S4+I4))+
                                            (V2*(S3+I3)/(S1+I1+S2+I2+S3+I3))+
                                            (V4*(S3+I3)/(S3+I3+S4+I4+S5+I5))))
    dW4 <- (I4*V4*beta_hv) - (d*V4) + (d*((V4*(S4+I4)/(S3+I3+S4+I4+S5+I5))+
                                            (V3*(S4+I4)/(S2+I2+S3+I3+S4+I4))+
                                            V5*(S4+I4)/(S4+I4+S5+I5)))
    dW5 <- (I5*V5*beta_hv) - (d*V5) + (d*((V5*(S5+I5)/(S4+I4+S5+I5))+
                                            (V4*(S5+I5)/(S3+I3+S4+I4+S5+I5))))
    dS1 <- -beta_vh*S1*W1
    dS2 <- -beta_vh*S2*W2
    dS3 <- -beta_vh*S3*W3
    dS4 <- -beta_vh*S4*W4
    dS5 <- -beta_vh*S5*W5
    dI1 <- beta_vh*S1*W1
    dI2 <- beta_vh*S2*W2
    dI3 <- beta_vh*S3*W3
    dI4 <- beta_vh*S4*W4
    dI5 <- beta_vh*S5*W5
    res <- c(dV1, dV2, dV3, dV4, dV5,
             dW1, dW2, dW3, dW4, dW5,
             dS1, dS2, dS3, dS4, dS5,
             dI1, dI2, dI3, dI4, dI5)
    list(res)})}


###############################################
# look at shapes that these models can make: ##
###############################################

t <- seq(from=1, to=80, by=1)
par1 <- c(r=0.2, K=150, beta_vh=0.001, beta_hv=0.68) # make beta_hv super fast; should look similar to earlier model
par2 <- c(r=0.2, K=150, d=0.01, beta_vh=0.001, beta_hv=0.68)

start1 <- c(V=0, W=0.1, S=100, I=0)
start2 <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)
start5 <- c(V1=0, V2=0, V3=0, V4=0, V5=0,
            W1=2.5, W2=0, W3=0, W4=0, W5=0, 
            S1=4, S2=12, S3=20, S4=28, S5=36, 
            I1=0, I2=0, I3=0, I4=0, I5=0)

sim1 <- data.frame(lsoda(y=start1, times=t, func=modN1, parms=par2, rtol=1, atol=1))
sim2 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2, rtol=1, atol=1))
sim5 <- data.frame(lsoda(y=start5, times=t, func=modN5, parms=par2, rtol=1, atol=1))


png("3mod.cartoon.png", width = 12, height = 8, res = 600, units='in')

par(mfrow=c(3,2),mar=c(1,0,0,1), oma=c(2,1,1,2))
# save 1200 x 800?

plot(sim1$time, sim1$V, col="white", xaxt="n", yaxt="n")

plot(sim1$time, sim1$V+sim1$W, type="l", col="black", ylim=c(0,160), xlab="time",
     ylab="number of aphids", xaxt="n", yaxt="n", lwd=2)
lines(sim1$time, 100*sim1$I/(sim1$I+sim1$S), lty=1, lwd=2, col="red")
axis(side=4, at=c(0,50,100,150), cex.axis=2)

plot(sim1$time, sim1$V, col="white", xaxt="n", yaxt="n")

plot(sim2$time, sim2$V1+sim2$W1, type="l", col="black", ylim=c(0,160), xlab="time",
     ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim2$time, sim2$V2+sim2$W2, lwd=2, lty=1)
lines(sim2$time, 100*sim2$I1/(sim2$I1+sim2$S1), lty=2, col="red", lwd=2)
lines(sim2$time, 100*sim2$I2/(sim2$I2+sim2$S2), lty=1, col="red", lwd=2)
axis(side=4, at=c(0,50,100,150), cex.axis=2)

plot(sim1$time, sim1$A, col="white", xaxt="n", yaxt="n")

plot(sim5$time, sim5$V1+sim5$W1, type="l", col="black", ylim=c(0,160), xlab="time",
     ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim5$time, sim5$V2+sim5$W2, lwd=2, lty=1)
lines(sim5$time, sim5$V3+sim5$W3, lwd=2, lty=1)
lines(sim5$time, sim5$V4+sim5$W4, lwd=2, lty=1)
lines(sim5$time, sim5$V5+sim5$W5, lwd=2, lty=1)
lines(sim5$time, 100*sim5$I1/(sim5$I1+sim5$S1), lty=2, col="red", lwd=2)
lines(sim5$time, 100*sim5$I2/(sim5$I2+sim5$S2), lty=1, col="red", lwd=2)
lines(sim5$time, 100*sim5$I3/(sim5$I3+sim5$S3), lty=1, col="red", lwd=2)
lines(sim5$time, 100*sim5$I4/(sim5$I4+sim5$S4), lty=1, col="red", lwd=2)
lines(sim5$time, 100*sim5$I5/(sim5$I5+sim5$S5), lty=1, col="red", lwd=2)
axis(side=4, at=c(0,50,100,150), cex.axis=2)
axis(side=1, at=c(0,40,80), cex.axis=2)

dev.off()

###########################################
# explore each parameter in a N=2 model: ##
###########################################

t <- seq(from=1, to=100, by=1)
start2 <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)

par2 <- c(r=0.2, K=150, d=0.01, beta_vh=0.001, beta_hv=0.68)
sim2 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2, rtol=1, atol=1))

par.r1 <- c(r=0.6, K=150, d=0.01, beta_vh=0.001, beta_hv=0.68)
sim.r1 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par.r1, rtol=1, atol=1))
par.K1 <- c(r=0.2, K=250, d=0.01, beta_vh=0.001, beta_hv=0.68)
sim.K1 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par.K1, rtol=1, atol=1))
par.d1 <- c(r=0.2, K=150, d=0.1, beta_vh=0.001, beta_hv=0.68)
sim.d1 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par.d1, rtol=1, atol=1))
par.beta1 <- c(r=0.2, K=150, d=0.01, beta_vh=0.0001, beta_hv=0.68)
sim.beta1 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par.beta1, rtol=1, atol=1))

par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))


########################
# baseline:

png("baseline.png", width = 2.25, height = 2.25, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))
plot(sim2$time, sim2$V1+sim2$W1, type="l", col="black", ylim=c(0,250), xlim=c(0,60),
     xlab="time", ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim2$time, sim2$V2+sim2$W2, lty=1, lwd=2)
lines(sim2$time, 100*sim2$I1/(sim2$I1+sim2$S1), lty=2, col="red", lwd=2)
lines(sim2$time, 100*sim2$I2/(sim2$I2+sim2$S2), lty=1, col="red", lwd=2)
axis(side=1, at=c(0,30,60), cex.axis=0.8)
axis(side=2, at=c(0,100,200), cex.axis=0.8)
dev.off()

########################
# effect of changing r:

png("r.png", width = 2.25, height = 2.25, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))
plot(sim.r1$time, sim.r1$V1+sim.r1$W1, type="l", col="black", ylim=c(0,250), xlim=c(0,60),
     xlab="time", ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim.r1$time, sim.r1$V2+sim.r1$W2, lwd=2, lty=1)
lines(sim.r1$time, 100*sim.r1$I1/(sim.r1$I1+sim.r1$S1), lty=2, col="red", lwd=2)
lines(sim.r1$time, 100*sim.r1$I2/(sim.r1$I2+sim.r1$S2), lty=1, col="red", lwd=2)
dev.off()

########################
# effect of changing K:

png("K.png", width = 2.25, height = 2.25, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))
plot(sim.K1$time, sim.K1$V1+sim.K1$W1, type="l", col="black", ylim=c(0,250), xlim=c(0,60),
     ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim.K1$time, sim.K1$V2+sim.K1$W2, lwd=2, lty=1)
lines(sim.K1$time, 100*sim.K1$I1/(sim.K1$I1+sim.K1$S1), lty=2, col="red", lwd=2)
lines(sim.K1$time, 100*sim.K1$I2/(sim.K1$I2+sim.K1$S2), lty=1, col="red", lwd=2)
dev.off()

########################
# effect of changing d:

png("d.png", width = 2.25, height = 2.25, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))
plot(sim.d1$time, sim.d1$V1+sim.d1$W1, type="l", col="black", ylim=c(0,250), xlim=c(0,60),
     xlab="time", ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim.d1$time, sim.d1$V2+sim.d1$W2, lwd=2, lty=1)
lines(sim.d1$time, 100*sim.d1$I1/(sim.d1$I1+sim.d1$S1), lty=2, col="red", lwd=2)
lines(sim.d1$time, 100*sim.d1$I2/(sim.d1$I2+sim.d1$S2), lty=1, col="red", lwd=2)
dev.off()

##########################
# effect of changing beta:

png("beta.png", width = 2.25, height = 2.25, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))
plot(sim.beta1$time, sim.beta1$V1+sim.beta1$W1, type="l", col="black", ylim=c(0,250), xlim=c(0,60),
     xlab="time", ylab="number of aphids", xaxt="n", yaxt="n", lwd=2, lty=2)
lines(sim.beta1$time, sim.beta1$V2+sim.beta1$W2, lwd=2, lty=1)
lines(sim.beta1$time, 100*sim.beta1$I1/(sim.beta1$I1+sim.beta1$S1), lty=2, col="red", lwd=2)
lines(sim.beta1$time, 100*sim.beta1$I2/(sim.beta1$I2+sim.beta1$S2), lty=1, col="red", lwd=2)
dev.off()


#######################
# troubleshooting range of d parameters:


t <- seq(from=1, to=100, by=1)
start1 <- c(V=0, W=0.1, S=100, I=0)
start2 <- c(V1=0, V2=0, W1=2.5, W2=0, S1=4, S2=96, I1=0, I2=0)
par1 <- c(r=0.2, K=150, beta_vh=0.001, beta_hv=0.68)
par2_d1 <- c(r=0.2, K=150, d=0.001, beta_vh=0.001, beta_hv=0.68)
par2_d2 <- c(r=0.2, K=150, d=0.01, beta_vh=0.001, beta_hv=0.68)
par2_d3 <- c(r=0.2, K=150, d=0.1, beta_vh=0.001, beta_hv=0.68)
par2_d4 <- c(r=0.2, K=150, d=0.5, beta_vh=0.001, beta_hv=0.68)

sim1 <- data.frame(lsoda(y=start1, times=t, func=modN1, parms=par1, rtol=1, atol=1))

sim2_d1 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2_d1, rtol=1, atol=1))
sim2_d2 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2_d2, rtol=1, atol=1))
sim2_d3 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2_d3, rtol=1, atol=1))
sim2_d4 <- data.frame(lsoda(y=start2, times=t, func=modN2, parms=par2_d4, rtol=1, atol=1))


png("d.converge.png", width = 3, height = 3, res = 600, units='in')
par(mfrow=c(1,1),mar=c(2,2,0,0), oma=c(0,0,0.5,0.5))

plot(sim1$time, sim1$V+sim1$W, type="l", col="light gray", ylim=c(0,160), xlim=c(0,50), 
     xaxt="n", yaxt="n", lwd=8, lty=1)
lines(sim2_d1$time, sim2_d1$V2+sim2_d1$W2, lwd=1, lty=1, col="red")
lines(sim2_d1$time, sim2_d1$V1+sim2_d1$W1, lwd=1, lty=3, col="red")
lines(sim2_d2$time, sim2_d2$V2+sim2_d2$W2, lwd=1, lty=1, col="orange")
lines(sim2_d2$time, sim2_d2$V1+sim2_d2$W1, lwd=1, lty=3, col="orange")
lines(sim2_d3$time, sim2_d3$V2+sim2_d3$W2, lwd=1, lty=1, col="green3")
lines(sim2_d3$time, sim2_d3$V1+sim2_d3$W1, lwd=1, lty=3, col="green3")
lines(sim2_d4$time, sim2_d4$V2+sim2_d4$W2, lwd=1, lty=1, col="blue")
lines(sim2_d4$time, sim2_d4$V1+sim2_d4$W1, lwd=1, lty=3, col="blue")
axis(side=1, at=c(0,25,50), cex.axis=1)
axis(side=2, at=c(0,75,150), cex.axis=1)

dev.off()



# # local dispersal model:

#t <- seq(from=1, to=1000, by=1) # running out longer to confirm equilibria 
t <- seq(from=1, to=100, by=1)

par5_d1 <- c(r=0.2, K=150, d=0.01, beta_vh=0.001, beta_hv=0.68)
par5_d2 <- c(r=0.2, K=150, d=0.1, beta_vh=0.001, beta_hv=0.68)
par5_d3 <- c(r=0.2, K=150, d=0.5, beta_vh=0.001, beta_hv=0.68)
par5_d4 <- c(r=0.2, K=150, d=1, beta_vh=0.001, beta_hv=0.68)

sim5_d1 <- data.frame(lsoda(y=start5, times=t, func=modN5, parms=par5_d1, rtol=1, atol=1))
sim5_d2 <- data.frame(lsoda(y=start5, times=t, func=modN5, parms=par5_d2, rtol=1, atol=1))
sim5_d3 <- data.frame(lsoda(y=start5, times=t, func=modN5, parms=par5_d3, rtol=1, atol=1))
sim5_d4 <- data.frame(lsoda(y=start5, times=t, func=modN5, parms=par5_d4, rtol=1, atol=1))

#
png("d.waves.png", width = 5, height = 5, res = 600, units='in')

par(mfrow=c(4,2),mar=c(2,2,0,0), oma=c(0,0,0.5,2))

plot(sim5_d1$time, sim5_d1$V1+sim5_d1$W1, type="l", col="red", ylim=c(0,170), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d1$time, sim5_d1$V2+sim5_d1$W2, lwd=1, lty=1, col="orange")
lines(sim5_d1$time, sim5_d1$V3+sim5_d1$W3, lwd=1, lty=1, col="green")
lines(sim5_d1$time, sim5_d1$V4+sim5_d1$W4, lwd=1, lty=1, col="blue")
lines(sim5_d1$time, sim5_d1$V5+sim5_d1$W5, lwd=1, lty=1, col="purple")
abline(h=150, col="dark gray", lty=3, lwd=2)
axis(side=2, at=c(0,75,150), cex.axis=1)

plot(sim5_d1$time, sim5_d1$I1/(sim5_d1$S1+sim5_d1$I1), type="l", col="red", ylim=c(0,1.2), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d1$time, sim5_d1$I2/(sim5_d1$S2+sim5_d1$I2), lwd=1, lty=1, col="orange")
lines(sim5_d1$time, sim5_d1$I3/(sim5_d1$S3+sim5_d1$I3), lwd=1, lty=1, col="green")
lines(sim5_d1$time, sim5_d1$I4/(sim5_d1$S4+sim5_d1$I4), lwd=1, lty=1, col="blue")
lines(sim5_d1$time, sim5_d1$I5/(sim5_d1$S5+sim5_d1$I5), lwd=1, lty=1, col="purple")
abline(h=1, col="dark gray", lty=3, lwd=2)
axis(side=4, at=c(0,0.5,1), cex.axis=1)

plot(sim5_d2$time, sim5_d2$V1+sim5_d2$W1, type="l", col="red", ylim=c(0,170), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d2$time, sim5_d2$V2+sim5_d2$W2, lwd=1, lty=1, col="orange")
lines(sim5_d2$time, sim5_d2$V3+sim5_d2$W3, lwd=1, lty=1, col="green")
lines(sim5_d2$time, sim5_d2$V4+sim5_d2$W4, lwd=1, lty=1, col="blue")
lines(sim5_d2$time, sim5_d2$V5+sim5_d2$W5, lwd=1, lty=1, col="purple")
abline(h=150, col="dark gray", lty=3, lwd=2)
axis(side=2, at=c(0,75,150), cex.axis=1)

plot(sim5_d2$time, sim5_d2$I1/(sim5_d2$S1+sim5_d2$I1), type="l", col="red", ylim=c(0,1.2), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d2$time, sim5_d2$I2/(sim5_d2$S2+sim5_d2$I2), lwd=1, lty=1, col="orange")
lines(sim5_d2$time, sim5_d2$I3/(sim5_d2$S3+sim5_d2$I3), lwd=1, lty=1, col="green")
lines(sim5_d2$time, sim5_d2$I4/(sim5_d2$S4+sim5_d2$I4), lwd=1, lty=1, col="blue")
lines(sim5_d2$time, sim5_d2$I5/(sim5_d2$S5+sim5_d2$I5), lwd=1, lty=1, col="purple")
abline(h=1, col="dark gray", lty=3, lwd=2)
axis(side=4, at=c(0,0.5,1), cex.axis=1)

plot(sim5_d3$time, sim5_d3$V1+sim5_d3$W1, type="l", col="red", ylim=c(0,170), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d3$time, sim5_d3$V2+sim5_d3$W2, lwd=1, lty=1, col="orange")
lines(sim5_d3$time, sim5_d3$V3+sim5_d3$W3, lwd=1, lty=1, col="green")
lines(sim5_d3$time, sim5_d3$V4+sim5_d3$W4, lwd=1, lty=1, col="blue")
lines(sim5_d3$time, sim5_d3$V5+sim5_d3$W5, lwd=1, lty=1, col="purple")
abline(h=150, col="dark gray", lty=3, lwd=2)
axis(side=2, at=c(0,75,150), cex.axis=1)

plot(sim5_d3$time, sim5_d3$I1/(sim5_d3$S1+sim5_d3$I1), type="l", col="red", ylim=c(0,1.2), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d3$time, sim5_d3$I2/(sim5_d3$S2+sim5_d3$I2), lwd=1, lty=1, col="orange")
lines(sim5_d3$time, sim5_d3$I3/(sim5_d3$S3+sim5_d3$I3), lwd=1, lty=1, col="green")
lines(sim5_d3$time, sim5_d3$I4/(sim5_d3$S4+sim5_d3$I4), lwd=1, lty=1, col="blue")
lines(sim5_d3$time, sim5_d3$I5/(sim5_d3$S5+sim5_d3$I5), lwd=1, lty=1, col="purple")
abline(h=1, col="dark gray", lty=3, lwd=2)
axis(side=4, at=c(0,0.5,1), cex.axis=1)

plot(sim5_d4$time, sim5_d4$V1+sim5_d4$W1, type="l", col="red", ylim=c(0,170), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d4$time, sim5_d4$V2+sim5_d4$W2, lwd=1, lty=1, col="orange")
lines(sim5_d4$time, sim5_d4$V3+sim5_d4$W3, lwd=1, lty=1, col="green")
lines(sim5_d4$time, sim5_d4$V4+sim5_d4$W4, lwd=1, lty=1, col="blue")
lines(sim5_d4$time, sim5_d4$V5+sim5_d4$W5, lwd=1, lty=1, col="purple")
abline(h=150, col="dark gray", lty=3, lwd=2)
axis(side=1, at=c(0,40,80), cex.axis=1)
axis(side=2, at=c(0,75,150), cex.axis=1)

plot(sim5_d4$time, sim5_d4$I1/(sim5_d4$S1+sim5_d4$I1), type="l", col="red", ylim=c(0,1.2), xlim=c(0,80),
     xaxt="n", yaxt="n", lwd=1, lty=1)
lines(sim5_d4$time, sim5_d4$I2/(sim5_d4$S2+sim5_d4$I2), lwd=1, lty=1, col="orange")
lines(sim5_d4$time, sim5_d4$I3/(sim5_d4$S3+sim5_d4$I3), lwd=1, lty=1, col="green")
lines(sim5_d4$time, sim5_d4$I4/(sim5_d4$S4+sim5_d4$I4), lwd=1, lty=1, col="blue")
lines(sim5_d4$time, sim5_d4$I5/(sim5_d4$S5+sim5_d4$I5), lwd=1, lty=1, col="purple")
abline(h=1, col="dark gray", lty=3, lwd=2)
axis(side=4, at=c(0,0.5,1), cex.axis=1)
axis(side=1, at=c(0,40,80), cex.axis=1)

dev.off()
# 

