
"""
    In order to find the quantities of interest from our data, we do a
    cubic spline interpolation of the data and then find the desired
    data points from this spline.

    Based on the solution provided at https://stackoverflow.com/
    questions/50371298/find-maximum-minimum-of-a-1d-interpolated-function
"""

# ------------------------------ LOAD LIBRARIES -------------------------------
import pandas as pd
import numpy as np
from scipy.interpolate import InterpolatedUnivariateSpline as InterpSpline

# ------------------------------- DEFINITIONS ---------------------------------
def quadratic_spline_roots(spline):
    roots = []
    knots = spline.get_knots()
    for a, b in zip(knots[:-1], knots[1:]):
        u, v, w = spline(a), spline((a + b) / 2), spline(b)
        t = np.roots([u + w - 2 * v, w - u, 2 * v])
        t = t[np.isreal(t) & (np.abs(t) <= 1)]
        roots.extend(t * (b - a) / 2 + (b + a) / 2)
    return np.array(roots)


# -------------------------------- PARAMETERS ---------------------------------
case = 'Rubber22'
diam = int(''.join(filter(str.isdigit, case))) * 1e-3

H_max = [0.008, 0.004, 0.002]
dt_inv = 2000

if case == 'PTFE6':
    h0 = 0.1616616
    inner_fac  = 7
    dist0 = 2e-4
    gamma_c = 0.38
    t_max = 0.69
elif case == 'Rubber22':
    h0 = 0.1461203
    inner_fac = 4
    dist0 = 2e-4
    gamma_c = 12
    t_max = 0.88
else:
    raise Exception('Unknown case')

for h_max in H_max:
    # --------------------------------- DATA ----------------------------------
    file_in = case + '/' + case + \
        '-iso-RotSym_hmax{}inner{}bottom1dtinv{}BDF2_dist{}gc{}.txt'.format(
        h_max, inner_fac, dt_inv, dist0, gamma_c)

    data = pd.read_csv(file_in, sep="\t")
    data = data[(data["time"] > 0.01) & (data["time"] < t_max)]

    # ------------------------------ PROCESSING -------------------------------
    # Determine t*, v* and f*
    spline_h_03 = InterpSpline(x=data['time'], y=data['height'] - diam, k=3)
    spline_vel = InterpSpline(x=data['time'], y=data['vel_ball'], k=3)
    spline_drag = InterpSpline(x=data['time'], y=data['drag_y'], k=3)
    t_star = spline_h_03.roots()[0]
    v_star = spline_vel(t_star)
    f_star = spline_drag(t_star)

    # Determine t_cont
    spline_t0 = InterpSpline(x=data['time'], 
                             y=data['height'] - (h0 - (diam / 2)), k=3)
    t0 = spline_t0.roots()[0]
    t_star = t_star - t0

    spline_h = InterpSpline(x=data['time'], y=data['height'], k=3)
    cr_pnt = quadratic_spline_roots(spline_h.derivative())
    cr_vals = spline_h(cr_pnt)
    t_cont = cr_pnt[0] - t0

    # Determine t_jump and d_jump
    i_max = np.argmax(cr_vals)
    t_jump, d_jump = cr_pnt[i_max] - cr_pnt[0], cr_vals[i_max]


    # -------------------------------- OUTPUT ---------------------------------
    str_out = "& $[{:7.5f},{:5.3f}]$ & \\nf{{1}}{{{:}}} &  &  & ".format(
        h_max / inner_fac, h_max, dt_inv)
    str_out += "{:8.6f} & ${: 9.7f}$ & \\num{{{:9.7e}}} & ".format(
        t_star, v_star, f_star)
    str_out += "{:8.6f} & ".format(t_cont)
    str_out += "{:8.6f} & \\num{{{:9.7e}}}".format(t_jump, d_jump)
    str_out += "\\\\"
    print(str_out)