/*----------------------------   problems.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __problems_H
#define __problems_H
/*----------------------------   problems.h     ---------------------------*/


/**
 *
 * Copyright (C) 2020 by the Gascoigne 3D authors
 *
 * This file is part of Gascoigne 3D
 *
 * Gascoigne 3D is free software: you can redistribute it
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later
 * version.
 *
 * Gascoigne 3D is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * Please refer to the file LICENSE.TXT for further information
 * on this license.
 *
 **/



#include "navierstokescylinder.h"
#include "navierstokes3d-ale.h"
#include "aleproblem.h"
#include "problemdescriptorbase.h"

/* ----------------------------------------- */

namespace Gascoigne
{
  
  /**
   * Defines the DirichletData of the problem.
   * Parabolic inflow on the upper boundary
   **/
  class CylinderDD : public DirichletData
  {
  protected:
    
  public:
    CylinderDD(const ParamFile& pf) : DirichletData(pf) { }
    
    void operator()(DoubleVector &b, const Vertex2d &v, int color) const
    {
      b.zero();
      if (color==8)
	b[2] = -(1.0-v.x()*v.x()/0.055/0.055)*0.01;
    }
  };

  /**
   * Defines the DirichletData for the moving sphere
   * Dirichlet zero on top / bottom / right
   * Dirichlet (0,1)*vel on the sphere
   * symmetry on left
   **/
  class MovingSphereDD : public DirichletData
  {
  protected:
    const ProblemData* data;
    
  public:
    MovingSphereDD(const ParamFile& pf, const ProblemData& dat) : DirichletData(pf), data(&dat) { }
    
    void operator()(DoubleVector &b, const Vertex2d &v, int color) const
    {
      b.zero();
      if (color==5)
      	b[2] = data->vel;
    }
    void operator()(DoubleVector &b, const Vertex3d &v, int color) const
    {
      b.zero();
      if (color==5)
	{ 
	  // motion + Rotation
	  b[1] = data->vel3d[0] + (data->rot3d[1]*(v.z()-0.05) - data->rot3d[2]*(v.y()     ));
	  b[2] = data->vel3d[1] + (data->rot3d[2]*(v.x()     ) - data->rot3d[0]*(v.z()-0.05));
	  b[3] = data->vel3d[2] + (data->rot3d[0]*(v.y()     ) - data->rot3d[1]*(v.x()     ));
	}
    }
  };


  /// Problemdescriptor for stationary Navier-Stokes in cylindrical ALE-cooridnates

  class FlowProblem : public ProblemDescriptorBase
  {
    ProblemData data;

  public:

    const ProblemData& GetData() const
    {
      return data;
    }
  
    void BasicInit(const ParamFile &pf)
    {
      data.BasicInit(pf);
    
      GetParamFile()     = pf;
      GetEquationPointer()      = new NavierStokesCylinder(data);
      GetDirichletDataPointer() = new CylinderDD(pf);
      ProblemDescriptorBase::BasicInit(pf);
    }
  };



  /// Problemdescriptor for non-stationary Navier-Stokes in cylindrical ALE-cooridnates
  class MovingRubberProblem : public ProblemDescriptorBase
  {
    ProblemData data;

  public:

    const ProblemData& GetData() const
    {
      return data;
    }
  
    void BasicInit(const ParamFile &pf)
    {
      data.BasicInit(pf);
    
      GetParamFile()     = pf;
      GetEquationPointer()      = new NavierStokesCylinderBDF2(data);
      GetDirichletDataPointer() = new MovingSphereDD(pf,data);
      ProblemDescriptorBase::BasicInit(pf);
    }
  };


  /// Problemdescriptor for non-stationary Navier-Stokes in cylindrical ALE-cooridnates
  class NonstationaryFlowProblem3d : public ProblemDescriptorBase
  {
    ProblemData data;

  public:

    const ProblemData& GetData() const
    {
      return data;
    }
  
    void BasicInit(const ParamFile &pf)
    {
      data.BasicInit(pf);
      GetParamFile()     = pf;
      GetEquationPointer()      = new NavierStokesBDF2(data);
      GetDirichletDataPointer() = new MovingSphereDD(pf,data);
      ProblemDescriptorBase::BasicInit(pf);
    }
  };


}


/*----------------------------   problems.h     ---------------------------*/
/* end of #ifndef __problems_H */
#endif
/*----------------------------   problems.h     ---------------------------*/
