/*----------------------------   navierstokescylinder.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __navierstokescylinder_H
#define __navierstokescylinder_H
/*----------------------------   navierstokescylinder.h     ---------------------------*/

/**
 *
 * Copyright (C) 2020 by the Gascoigne 3D authors
 *
 * This file is part of Gascoigne 3D
 *
 * Gascoigne 3D is free software: you can redistribute it
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later
 * version.
 *
 * Gascoigne 3D is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * Please refer to the file LICENSE.TXT for further information
 * on this license.
 *
 **/


#include "equation.h"
#include "dirichletdata.h"
#include "boundaryequation.h"
#include "lpsequation.h"
#include "paramfile.h"

#include "aleproblem.h"
#include "problemdata.h"




/*-----------------------------------------*/


namespace Gascoigne
{


  /**
   * \brief Navier-Stokes in cylindrical coordinates (r,z) in ALE-mapping
   *
   * Implementation of the Navier-Stokes equations in cylindrical coordinates
   * in an arbitrary coordinate system which is deformed in z-direction by the
   * displacement (id + *ALE) to be specified as external fem function
   *
   * (R,Z) = (r,z + *ALE[0])
   *
   * V(r,Z) = v(r,z)   and   P(r,Z) = p(r,z)
   *
   */
  class NavierStokesCylinder : public virtual Equation, public LpsEquation
  {
  protected:
    const ProblemData* data;

    mutable double _alphax,_alphay;  // stabilization parameter
    mutable double _r;    // radius of the present quadrature point
    mutable double J;     // transformation of the domain
    mutable double dt_T;  // time derivative of deformation     
  public:
    NavierStokesCylinder(const ProblemData &PD) : data(&PD) { }

    NavierStokesCylinder* createNew() const
    {
      return new NavierStokesCylinder(*data);
    }

    int GetNcomp() const { return 3; } 

    // Form and Matrix
    void point(double h, const FemFunction &U, const Vertex2d &v) const
    {
      _r = v.x();
      J = ALE_J(*data,v.y());
      dt_T = ALE_V(*data,v.y());
    }
    
    void Form(VectorIterator b, const FemFunction &U, const TestFunction &N) const;
    void Matrix(EntryMatrix &A,
                const FemFunction &U,
                const TestFunction &M,
                const TestFunction &N) const;

    // LPS Stabilization
    void lpspoint(double h, const FemFunction &U, const Vertex2d &v) const;
    void StabForm(VectorIterator b,
                  const FemFunction &U,
                  const FemFunction &UP,
                  const TestFunction &NP) const;
    void StabMatrix(EntryMatrix &A,
                    const FemFunction &U,
                    const TestFunction &Np,
                    const TestFunction &Mp) const;

  };


  /**
   *
   * same but with BDF2 time stepping
   *
   * 1.5 / dt u_n +  A(u_n) = 2 / dt u_[n-1] - 0.5 / dt u_[n-2]
   */
  class NavierStokesCylinderBDF2 : public virtual NavierStokesCylinder
  {
    // navier stokes solution at last and second last time step
    mutable FemFunction *OLD, *OLDOLD;

  protected:

  public:
    NavierStokesCylinderBDF2(const ProblemData &PD) : NavierStokesCylinder(PD) { assert (data->dt>0); }

    NavierStokesCylinderBDF2* createNew() const
    {
      return new NavierStokesCylinderBDF2(*data);
    }

    // Retrieves all vectos displacement field
    void SetFemData(FemData& q) const
    {
      assert(q.find("OLD")!=q.end());
      OLD = &q["OLD"];

      assert(q.find("OLDOLD")!=q.end());
      OLDOLD = &q["OLDOLD"];
    }
    // Form and Matrix
    void point(double h, const FemFunction &U, const Vertex2d &v) const
    {
      _r = v.x();
      J        = ALE_J(*data,v.y());
      dt_T = ALE_V(*data,v.y());
      assert(J>0.01);
    }
    
    void Form(VectorIterator b, const FemFunction &U, const TestFunction &N) const;
    void Matrix(EntryMatrix &A,
                const FemFunction &U,
                const TestFunction &M,
                const TestFunction &N) const;
    
    // LPS Stabilization
    void lpspoint(double h, const FemFunction &U, const Vertex2d &v) const
    {
      J = ALE_J(*data,v.y());
      _alphax = data->alpha/(data->mu_f/data->rho_f/h/h     + 1. / data->dt); // + velocity?
      _alphay = data->alpha/(data->mu_f/data->rho_f/h/h/J/J + 1. / data->dt); // + velocity?
      _r = v.x();
    }

  };

}

/*----------------------------   navierstokescylinder.h     ---------------------------*/
/* end of #ifndef __navierstokescylinder_H */
#endif
/*----------------------------   navierstokescylinder.h     ---------------------------*/
