#include "navierstokescylinder.h"


namespace Gascoigne
{

  /*
   * Since we move the ball in z-direction only, simplified ALE
   *
   *
   *     (1    0   )
   * F = (         )   J = 1+uz
   *     (ur   1+uz)
   *         (1+uz 0   )      
   * F^{-1} =(         )  / J 
   *	     (-ur  1+uz)      
   *
   * p(r,z) = P(r,Z)  p_r = P_r - u_r/J P_Z
   *                  p_z = 1/J P_Z
   * same for xi
   *
   * v(r,z) = V(r,Z)
   *
   * vr_r = Vr_r - Vr_Z u_r/J
   * vz_r = Vz_r - VZ_Z u_r/J
   *
   * vr_z = Vr_Z/J
   * vz_z = VZ_Z/J
   *  
   * same for phi
   */
  
  void NavierStokesCylinder::Form(VectorIterator b,
				  const FemFunction &U,
				  const TestFunction &N) const
  {
    // Divergence
    b[0] += J * data->rho_f * (U[1].m() + _r * ( U[1].x() + U[2].y() /J )) * N.m();
    // Pressure
    b[1] -= J * U[0].m() * (N.m() + _r * N.x());
    b[2] -= J * U[0].m() *  _r * N.y()/J;
    
    // Laplace
    b[1] += J * data->mu_f * (_r * (U[1].x()*N.x()+U[1].y()*N.y()/J/J) + 1.0/_r * U[1].m()*N.m());
    b[2] += J * data->mu_f * (_r * (U[2].x()*N.x()+U[2].y()*N.y()/J/J) );

    // Convection
    b[1] += J * data->rho_f  * _r * (U[1].m() * U[1].x() + U[2].m() * U[1].y()/J) * N.m();
    b[2] += J * data->rho_f  * _r * (U[1].m() * U[2].x() + U[2].m() * U[2].y()/J) * N.m();
  }
  
  void NavierStokesCylinder::Matrix(EntryMatrix &A,
				    const FemFunction &U,
				    const TestFunction &M,
				    const TestFunction &N) const
  {
    // Divergence
    //    b[0] += J * (U[1].m() + _r * ( (U[1].x() - U[1].y() * (*ALE)[0].x()/J) + U[2].y() /J )) * N.m();
    A(0,1) += J * data->rho_f * (M.m() + _r * M.x()) * N.m();
    A(0,2) += J * data->rho_f * ( _r * ( M.y() /J )) * N.m();
    
    // Pressure
    //b[1] -= J * U[0].m() * (N.m() + _r * (N.x() - N.y() * (*ALE)[0].x()/J));
    A(1,0) -= J * M.m() * (N.m() + _r * N.x());
    //b[2] -= J * U[0].m() *  _r * N.y()/J;
    A(2,0) -= J * M.m() *  _r * N.y()/J;
    
    // Laplace
    //    b[1] += J * data->mu_f * (_r * ((U[1].x()-U[1].y() * (*ALE)[0].x()/J)*(N.x()-N.y()*(*ALE)[0].x()/J)+U[1].y()*N.y()/J/J) + 1.0/_r * U[1].m()*N.m());
    A(1,1) += J * data->mu_f  * (_r * ((M.x())*(N.x())+M.y()*N.y()/J/J) + 1.0/_r * M.m()*N.m());
    //b[2] += J * data->mu_f * (_r * ((U[2].x()-U[2].y() * (*ALE)[0].x()/J)*(N.x()-N.y()*(*ALE)[0].x()/J)+U[2].y()*N.y()/J/J) );
    A(2,2) += J * data->mu_f  * (_r * ((M.x())*(N.x())+M.y()*N.y()/J/J) );

    // Convection
    //    b[1] += J * data->rho_f * _r * (U[1].m() * (U[1].x()-U[1].y()*(*ALE)[0].x()/J) + U[2].m() * U[1].y()/J) * N.m();
    A(1,1) += J  * data->rho_f * _r * (M.m() * (U[1].x()) + U[2].m() * M.y()/J) * N.m();
    A(1,1) += J  * data->rho_f * _r * (U[1].m() * (M.x())) * N.m();
    A(1,2) += J  * data->rho_f* _r * (M.m() * U[1].y()/J) * N.m();
    //b[2] += J  * data->rho_f* _r * (U[1].m() * (U[2].x()-U[2].y()*(*ALE)[0].x()/J) + U[2].m() * U[2].y()/J) * N.m();
    A(2,1) += J  * data->rho_f* _r * (M.m() * (U[2].x())) * N.m();
    A(2,2) += J  * data->rho_f* _r * (U[1].m() * (M.x()) + M.m() * U[2].y()/J) * N.m();
    A(2,2) += J  * data->rho_f* _r * (U[2].m() * M.y()/J) * N.m();
  }

  void NavierStokesCylinder::lpspoint(double h, const FemFunction &U, const Vertex2d &v) const
  {
    J = ALE_J(*data,v.y());
    _alphax = data->alpha/(data->mu_f/data->rho_f/h/h     ); // + velocity?
    _alphay = data->alpha/(data->mu_f/data->rho_f/h/h/J/J ); // + velocity?
    _r = v.x();
  }
  
  void NavierStokesCylinder::StabForm(VectorIterator b,
				 const FemFunction &U,
				 const FemFunction &UP,
				 const TestFunction &NP) const
  {
    b[0] += J * _alphax * ( _r * (UP[0].x() * NP.x()) + 1.0/_r * UP[0].m() * NP.m());
    b[0] += J * _alphay * ( _r * (UP[0].y() * NP.y()/J/J));
  }
  
  void NavierStokesCylinder::StabMatrix(EntryMatrix &A,
				   const FemFunction &U,
				   const TestFunction &Np,
				   const TestFunction &Mp) const
  {
    A(0,0) += J * _alphax * ( _r * (Mp.x() * Np.x()) + 1.0/_r * Mp.m() * Np.m());
    A(0,0) += J * _alphay * ( _r * (Mp.y() * Np.y()/J/J));
  }



  
  void NavierStokesCylinderBDF2::Form(VectorIterator b, const FemFunction &U, const TestFunction &N) const
  {
    NavierStokesCylinder::Form(b,U,N);
    if (data->bdf1)
      {
	b[1] += _r * N.m()/data->dt * data->rho_f * J * (U[1].m() - (*OLD)[1].m());
	b[2] += _r * N.m()/data->dt * data->rho_f * J * (U[2].m() - (*OLD)[2].m());
      }
    else
      {
	b[1] += _r * N.m()/data->dt * data->rho_f * J * (1.5 * U[1].m() - 2.0 * (*OLD)[1].m() + 0.5 * (*OLDOLD)[1].m());
	b[2] += _r * N.m()/data->dt * data->rho_f * J * (1.5 * U[2].m() - 2.0 * (*OLD)[2].m() + 0.5 * (*OLDOLD)[2].m());
      }
    
    // ALE deformation
    b[1] += - _r  * dt_T * data->rho_f * U[1].y() * N.m();
    b[2] += - _r  * dt_T * data->rho_f * U[2].y() * N.m();
  }
  
  void NavierStokesCylinderBDF2::Matrix(EntryMatrix &A,
					const FemFunction &U,
					const TestFunction &M,
					const TestFunction &N) const
  {
    NavierStokesCylinder::Matrix(A,U,M,N);
    if (data->bdf1)
      {
	A(1,1) += _r * N.m()/data->dt * data->rho_f * (J * M.m());
	A(2,2) += _r * N.m()/data->dt * data->rho_f * (J * M.m());
      }
    else
      {
	A(1,1) += _r * N.m()/data->dt * data->rho_f * (1.5 * J * M.m());
	A(2,2) += _r * N.m()/data->dt * data->rho_f * (1.5 * J * M.m());
      }
    
    // ALE deformation
    A(1,1) += - _r * dt_T * data->rho_f * M.y() * N.m();
    A(2,2) += - _r * dt_T * data->rho_f * M.y() * N.m();
  }


} // namespace Gascoigne
