/*----------------------------   navierstokes3d-ale.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __navierstokes3d_ale_H
#define __navierstokes3d_ale_H
/*----------------------------   navierstokes3d-ale.h     ---------------------------*/


#include "aleproblem.h"
#include "problemdata.h"

#include  "eigen3/Eigen/Dense"




namespace Gascoigne
{
  
  // full 3d Navier-Stokes
  

  class NavierStokesBDF2 : public virtual Equation, public LpsEquation
  {
  protected:
    const ProblemData* data;

    mutable double _alphax,_alphaz;  // stabilization parameter
    mutable FemFunction *OLD, *OLDOLD;

    typedef Eigen::Matrix<double, 3, 3> MATRIX;
    typedef Eigen::Matrix<double, 3, 1>   VECTOR;

    mutable MATRIX F, F_inv, SIGMAf, NV, psi_matrix;
    
    mutable VECTOR dt_U, phi,psi,V,X;
    mutable double J;
    

    mutable MATRIX CONV_dV1, PRESSURE_P;
    
    mutable double CONV_dV2, DOMAIN_V;
    
    mutable std::array<double,3> DIVERGENCE_V;
    mutable MATRIX TENSOR_dV[3];

    
    
    
    
  public:
    NavierStokesBDF2(const ProblemData &PD) : data(&PD)
    { assert (data->radius>0); assert(data->cyl_radius>0); }

    NavierStokesBDF2* createNew() const
    {
      return new NavierStokesBDF2(*data);
    }

    // Retrieves all vectos displacement field
    void SetFemData(FemData& q) const
    {
      assert(q.find("OLD")!=q.end());
      OLD = &q["OLD"];

      assert(q.find("OLDOLD")!=q.end());
      OLDOLD = &q["OLDOLD"];
    }


    int GetNcomp() const { return 4; } 

    // Form and Matrix
    void point(double h, const FemFunction &U, const Vertex3d &v) const
    {
      F << 
	ALE3d_dx_Tx(*data,v), ALE3d_dy_Tx(*data,v), 0,
	ALE3d_dx_Ty(*data,v), ALE3d_dy_Ty(*data,v), 0,
	0                   , 0                   , ALE3d_dz_Tz(*data,v);
      
      F_inv = F.inverse();

      NV <<
	U[1].x(), U[1].y(), U[1].z(),
	U[2].x(), U[2].y(), U[2].z(),
	U[3].x(), U[3].y(), U[3].z();

      V << U[1].m(), U[2].m(), U[3].m();
      J = F.determinant();

      assert(J>0);
      
      dt_U << ALE3d_Vx(*data,v), ALE3d_Vy(*data,v), ALE3d_Vz(*data,v);
      
      
      SIGMAf = (data->mu_f * (NV*F_inv + F_inv.transpose()*NV.transpose()) - U[0].m() * MATRIX::Identity()) * F_inv.transpose();
    }


    void point_M(const FemFunction& U, const TestFunction& M) const
    {
      psi << M.x(), M.y(), M.z();
      
      CONV_dV1 = data->rho_f * M.m() * J * NV * F_inv;

      for (int j=0;j<3;++j)
	{
	  psi_matrix=MATRIX::Zero();
	  psi_matrix.block(j,0,1,3)=psi.transpose();

       	  TENSOR_dV[j] 
	    = data->mu_f * J * (psi_matrix*F_inv + F_inv.transpose()*psi_matrix.transpose()) * F_inv.transpose();
	  
	  DIVERGENCE_V[j] =  J * data->rho_f * (psi.transpose() * F_inv.block(0,j,3,1))(0,0);
	}
      
      CONV_dV2 =   J * data->rho_f * (psi.transpose()*F_inv * V   )(0,0);
      DOMAIN_V = - J * data->rho_f * (psi.transpose()*F_inv * dt_U)(0,0);
      
      PRESSURE_P = - M.m() * J * F_inv.transpose();
    }
    
    void Form(VectorIterator b, const FemFunction &U, const TestFunction &N) const
    {
       // pressure zero
      b[0] += 1.e-16 * data->dt * U[0].m() * N.m();
      // time derivative
      if (data->bdf1)
	{
	  for (int i=0;i<3;++i)
	    b[i+1] += J/data->dt * data->rho_f * (U[i+1].m() - (*OLD)[i+1].m()) * N.m();
	}
      else
	{
	  for (int i=0;i<3;++i)
	    b[i+1] += J/data->dt * data->rho_f *
	      (1.5 * U[i+1].m() - 2.0 * (*OLD)[i+1].m() + 0.5 * (*OLDOLD)[i+1].m()) * N.m();
	}

      // divergence
      b[0] += J * data->rho_f * N.m() * (F_inv.transpose().array() * NV.array()).sum();
      

      // Tensor
      phi << N.x(), N.y(), N.z();

      for (int i=0;i<3;++i)
	b[i+1] += J * (SIGMAf*phi)(i,0);

      // convection
      for (int i=0;i<3;++i)
      	b[i+1] += J * data->rho_f * ( NV * F_inv * (V-dt_U) )(i,0) * N.m();
    }

    void MatrixBlock(EntryMatrix& A, const FemFunction& U_dummy, const FemFunction& N) const
    {
      for (int j=0; j<N.size();++j)  // trial
	{
#define M N[j]
	  point_M(U_dummy,M);
	  	  
	  for (int i=0; i<N.size();++i) // test
	    {
	      A.SetDofIndex(i,j);
	      Matrix(A,U_dummy,M , N[i]);
	    }
#undef M
	}
    }
    
    void Matrix(EntryMatrix &A,
                const FemFunction &U,
                const TestFunction &M,
                const TestFunction &N) const
    {
       // pressure zero
      A(0,0) += 1.e-16 * data->dt * M.m() * N.m();

      // time derivative
      if (data->bdf1)
	{
	  for (int i=0;i<3;++i)
	    A(i+1,i+1) += J/data->dt * data->rho_f * M.m() * N.m();
	}
      else
	{
	  for (int i=0;i<3;++i)
	    A(i+1,i+1) += J/data->dt * data->rho_f *  1.5 * M.m() * N.m();
	}

      // divergence
      for (int j=0;j<3;++j)
	A(0,j+1) += DIVERGENCE_V[j] * N.m();

      phi << N.x(), N.y(), N.z();
      
      // Tensor
      for (int j=0;j<3;++j)
	{
	  X = TENSOR_dV[j]*phi;
	  for (int i=0;i<3;++i)
	    A(i+1,j+1) += X(i,0);
	}
      
      for (int i=0;i<3;++i)
      	for (int j=0;j<3;++j)
      	  A(i+1,j+1) +=  CONV_dV1(i,j)  * N.m();
      for (int j=0;j<3;++j)
      	A(j+1,j+1) +=   CONV_dV2 * N.m();
	
      // wrt V
      for (int j=0;j<3;++j)
      	A(j+1,j+1) += DOMAIN_V * N.m();

		
      /////////////// pressure
      X = PRESSURE_P*phi;
      for (int i=0;i<3;++i)
	A(i+1,0)+=X(i,0);
    }
    

    // LPS Stabilization
    void lpspoint(double h, const FemFunction &U, const Vertex3d &v) const
    {
      F << 
	ALE3d_dx_Tx(*data,v), ALE3d_dy_Tx(*data,v), 0,
	ALE3d_dx_Ty(*data,v), ALE3d_dy_Ty(*data,v), 0,
	0                   , 0                   ,  ALE3d_dz_Tz(*data,v);
      /* F << */
      /* 	1,0,0, */
      /* 	0,1,0, */
      /* 	0,0,ALE3d_dz_Tz(*data,v); */
      J = F.determinant();
      
      _alphax = data->alpha/(data->mu_f/h/h/data->rho_f     + 1.0/data->dt); // + velocity?
      _alphaz = data->alpha/(data->mu_f/h/h/J/J/data->rho_f + 1.0/data->dt); // + velocity?
    }
    
    void StabForm(VectorIterator b,
                  const FemFunction &U,
                  const FemFunction &UP,
                  const TestFunction &NP) const
    {
      b[0] += J * _alphax * (UP[0].x()*NP.x() + UP[0].y()*NP.y());
      b[0] += J * _alphaz * (UP[0].z()*NP.z()/J/J);
    }
    
    void StabMatrix(EntryMatrix &A,
                    const FemFunction &U,
                    const TestFunction &Np,
                    const TestFunction &Mp) const
    {
      A(0,0) += J * _alphax * (Mp.x()*Np.x() + Mp.y()*Np.y());
      A(0,0) += J * _alphaz * (Mp.z()*Np.z()/J/J);
    }

  };

}




/*----------------------------   navierstokes3d-ale.h     ---------------------------*/
/* end of #ifndef __navierstokes3d-ale_H */
#endif
/*----------------------------   navierstokes3d-ale.h     ---------------------------*/
