/*----------------------------   loop.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __loop_H
#define __loop_H
/*----------------------------   loop.h     ---------------------------*/


/**
 *
 * Copyright (C) 2020 by the Gascoigne 3D authors
 *
 * This file is part of Gascoigne 3D
 *
 * Gascoigne 3D is free software: you can redistribute it
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later
 * version.
 *
 * Gascoigne 3D is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * Please refer to the file LICENSE.TXT for further information
 * on this license.
 *
 **/

#include "meshagent.h"
#include "stdloop.h"
#include "boundaryfunction.h"

#include "stdmultilevelsolver.h"
#include "stdsolver.h"
#include "aleproblem.h"
#include "filescanner.h"

#include "problems.h"

#include "dofhandler.h"


/* ----------------------------------------- */




// defines the surface of the disc as zero contour
class Sphere : public Gascoigne::BoundaryFunction<2>
{
  double _r;
  
public:
  double operator()(const Gascoigne::Vertex2d &c) const
  {
    return _r*_r-c.x()*c.x()-(c.y()-0.05)*(c.y()-0.05);
  }
  void set_radius(double radius)
  {
    _r = radius;
  }
};
// defines the surface of the disc as zero contour
class Ball : public Gascoigne::BoundaryFunction<3>
{
  double _r;
  
public:
  double operator()(const Gascoigne::Vertex3d &c) const
  {
    return _r*_r-c.x()*c.x()-c.y()*c.y()-(c.z()-0.05)*(c.z()-0.05);
  }
  void set_radius(double radius)
  {
    _r = radius;
  }
};

// defines the surface of the disc as zero contour
class CylinderZ : public Gascoigne::BoundaryFunction<3>
{
  double _r;
  
public:
  double operator()(const Gascoigne::Vertex3d &c) const
  {
    return _r*_r-c.x()*c.x()-c.y()*c.y();
  }
  void set_radius(double radius)
  {
    _r = radius;
  }
};


/**
 * We overwrite the Multilevelsolver to define a new Solver
 * This is required to overwrite the output-Function, where we
 * want to add the deformation 
 */
class ALESolver : public Gascoigne::StdSolver
{
public:
  void AleVisu(const Gascoigne::ProblemData& data, const std::string& name, const Gascoigne::VectorInterface &gu,int i) const
  {
    const Gascoigne::GlobalVector& u = GetGV(gu);
    
    const Gascoigne::DofHandler<2>* cDH2 = dynamic_cast<const Gascoigne::DofHandler<2>* > (GetMesh());
    const Gascoigne::DofHandler<3>* cDH3 = dynamic_cast<const Gascoigne::DofHandler<3>* > (GetMesh());
    assert(cDH2 || cDH3);
    Gascoigne::DofHandler<2>* DH2 = const_cast<Gascoigne::DofHandler<2>* > (cDH2);
    Gascoigne::DofHandler<3>* DH3 = const_cast<Gascoigne::DofHandler<3>* > (cDH3);
    assert(DH2 || DH3);

    if (DH2)
      {
	std::vector<Gascoigne::Vertex<2> >& vv = DH2->GetVertexVector();
	std::vector<Gascoigne::Vertex<2> >  vvsave = vv;

	for (int i=0;i<DH2->nnodes();++i)
	  vv[i][1] += Gascoigne::ALE_T(data,vv[i][1]);
	GetDiscretization()->VisuVtk(GetProblemDescriptor()->GetComponentInformation(), _paramfile, name, u, i);
	vv = vvsave;
      }
    else if (DH3)
      {
	std::vector<Gascoigne::Vertex<3> >& vv = DH3->GetVertexVector();
	std::vector<Gascoigne::Vertex<3> >  vvsave = vv;
	
	for (int i=0;i<DH3->nnodes();++i)
	  {
	    vv[i][0] += Gascoigne::ALE3d_Tx(data,vv[i]);
	    vv[i][1] += Gascoigne::ALE3d_Ty(data,vv[i]);
	    vv[i][2] += Gascoigne::ALE3d_Tz(data,vv[i]);
	  }
	
	GetDiscretization()->VisuVtk(GetProblemDescriptor()->GetComponentInformation(), _paramfile, name, u, i);
	vv = vvsave;
      }
    else
      {std::cerr << "XXX" << std::endl; abort();}



  }
};

class ALEMultiLevelSolver : public Gascoigne::StdMultiLevelSolver
{
public:
  virtual Gascoigne::StdSolver* NewSolver(int solverlevel)
  { return new ALESolver; }
};



/**
 * New Loop to add a curved shape. The Boundary segment with color
 * is pulled onto a half-circle
 **/
class Loop : public Gascoigne::StdLoop
{
  Sphere sphere;
  Ball   ball;
  CylinderZ cylz;

public:
  void BasicInit(const Gascoigne::ProblemData& data,
		 const Gascoigne::ParamFile& paramfile,
		 const Gascoigne::ProblemContainer* PC,
                 const Gascoigne::FunctionalContainer* FC)
  {
    assert(data.radius>0);
    sphere.set_radius(data.radius);
    ball.set_radius(data.radius);
    cylz.set_radius(0.055);
    
    assert (GetMeshAgentPointer() == NULL);

    GetMeshAgentPointer() = new Gascoigne::MeshAgent;
    GetMeshAgent()->AddShape(5,&sphere);
    GetMeshAgent()->AddShape(5,&ball);
    GetMeshAgent()->AddShape(4,&cylz);

    GetMultiLevelSolverPointer() = new ALEMultiLevelSolver();
    assert(GetMultiLevelSolver());
    
    StdLoop::BasicInit(paramfile,PC,FC);
  }

  
  /// Get the degrees of freedom and the non-zero matrix entries for statistics
  std::array<double,2> matrix_entries(const Gascoigne::Matrix& A) const;

  /// performs 'ir' initial refinements around the ball
  void initialrefine(int ir, std::set<int> colors);

  // The loops to represent the different examples
  void run_stationary();
  void run_nonstationary();
  void run_fallingball();
  void run_fallingball3d();

  
};




/*----------------------------   loop.h     ---------------------------*/
/* end of #ifndef __loop_H */
#endif
/*----------------------------   loop.h     ---------------------------*/
