#include <random>

#include "sparseblockmatrix.h"
#include "fmatrixblock.h"

#include "loop.h"
#include "problems.h"



/// Get the degrees of freedom and the non-zero matrix entries for statistics
std::array<double,2> Loop::matrix_entries(const Gascoigne::Matrix& A) const
{
  const Gascoigne::MatrixInterface& mA = GetMultiLevelSolver()->GetSolver()->GetMatrix(A);
  
  std::array<double,2> NN;
  
  typedef const Gascoigne::SparseBlockMatrix<Gascoigne::FMatrixBlock<3> > MATRIX2d;
  typedef const Gascoigne::SparseBlockMatrix<Gascoigne::FMatrixBlock<4> > MATRIX3d;
  
  if (dynamic_cast< MATRIX2d* >(&mA))
    {
      MATRIX2d &MA = dynamic_cast<MATRIX2d&>(mA);
      NN[0] = MA.n()*3./1000.;
      NN[1] = MA.nentries()*3.*3./1000000.;
    }
  else if (dynamic_cast< MATRIX3d* >(&mA))
    {
      MATRIX3d &MA = dynamic_cast<MATRIX3d&>(mA);
      NN[0] = MA.n()*4./1000.;
      NN[1] = MA.nentries()*4.*4./1000000.;
    }
  else
    {
      std::cerr << "can't determine type of matrix" << std::endl;
      abort();
    }
  return NN;
}

//////////////////////////////////////////////////

void Loop::initialrefine(int ir, std::set<int> colors)
{
  for (int pref=0;pref<ir;++pref)
    {
      Gascoigne::IntVector vv;
      for (auto col : colors)
	{
	  Gascoigne::IntVector vc = *(GetMeshAgent()->GetMesh()->VertexOnBoundary(col));
	  for (auto it : vc)
	    vv.push_back(it);
	}
      GetMeshAgent()->refine_nodes(vv);
    }
}


//////////////////////////////////////////////////

void Loop::run_stationary()
{
  // vectors and matrix to solve the problem
  Gascoigne::VectorInterface u("u"), f("f");
  Gascoigne::Matrix A("A");

  // do 2 steps of initial mesh grading around the obstacle (with color 5)
  initialrefine(2,std::set<int>{5});



  // the adaptive loop; _niter steps on refined meshes
  for (_iter = 1; _iter <= _niter; _iter++)
    {
      GetMultiLevelSolver()->ReInit();
      GetMultiLevelSolver()->SetProblem("stationaryflow");

        // get access to the problem data
      const Gascoigne::ProblemData& data = dynamic_cast<const Gascoigne::FlowProblem*>(GetMultiLevelSolver()->GetProblemDescriptor())->GetData();
      
      // get the position of the ball
      Gascoigne::DataFormatHandler DFH;
      DFH.insert("deformation", &data.def, 0.);
      Gascoigne::FileScanner FS(DFH, _paramfile, "Equation");
      
      std::cout << "\n================== " << _iter << " ================\t nodes: "
		<< GetMultiLevelSolver()->GetSolver()->GetMesh()->nnodes() << std::endl;

  
      // Navier-Stokes Problem
      GetMultiLevelSolver()->ReInitMatrix(A);
      GetMultiLevelSolver()->ReInitVector(u);
      GetMultiLevelSolver()->ReInitVector(f);
      InitSolution(u);
      
      GetSolverInfos()->GetNLInfo().control().matrixmustbebuild() = 1;
      assert(Solve(A, u, f) == "converged");
      
      const ALESolver* S = dynamic_cast<const ALESolver*> (GetMultiLevelSolver()->GetSolver());
      assert(S);
      S->AleVisu(data,"Results/u",u,_iter);

      
      std::array<double,2> NN = matrix_entries(A);
      std::cout << "NDOFS / NNZ:\t" << NN[0] << "\t" << NN[1] << std::endl;      
      Gascoigne::DoubleVector juh = Functionals(u, f);

      if (_iter < _niter)
	{
	  Gascoigne::DoubleVector eta;
	  AdaptMesh(eta);
	}
    }
}


//////////////////////////////////////////////////

void Loop::run_nonstationary()
{
  // read some information on the time stepping scheme and the problem setup
  std::string material;  
  Gascoigne::DataFormatHandler DFH;
  DFH.insert("material",    &material, "none");
  Gascoigne::FileScanner FS(DFH, _paramfile, "Equation");

  // vectors & matrix to describe the problem
  Gascoigne::VectorInterface u("u"), old("old"), oldold("oldold"), f("f");
  Gascoigne::Matrix A("A");

  // refine on surface of ball and along symmetry boundary
  initialrefine(3,std::set<int>{4,5}); 

  // reinit the solver & the discretization
  GetMultiLevelSolver()->ReInit();     
  GetMultiLevelSolver()->SetProblem("movingrubber");
  GetMultiLevelSolver()->ReInitMatrix(A);
  GetMultiLevelSolver()->ReInitVector(u);
  GetMultiLevelSolver()->ReInitVector(old);
  GetMultiLevelSolver()->ReInitVector(oldold);
  GetMultiLevelSolver()->ReInitVector(f);
  
  GetSolverInfos()->GetNLInfo().control().matrixmustbebuild() = 1;

  // get access to the data describing the problem and controlling the time-stepping
  const Gascoigne::ProblemData& data =
    dynamic_cast<const Gascoigne::MovingRubberProblem*>(GetMultiLevelSolver()->GetProblemDescriptor())->GetData();
 
  // center of ball is given at d(t) = 0.1 + 0.05 cos( 0.1 Pi t)
  // we must subtract the radius of the sphere since 'def' is storing
  // the distance of the lower boundary from the bottom of the sphere
  data.def       = 0.1 + 0.05-data.radius;
  data.defold    = data.def;
  data.vel       = 0.00;
  data.velold    = data.vel;

  // direct access to the solver object. For easier access to Visualization function
  const ALESolver* S = dynamic_cast<const ALESolver*> (GetMultiLevelSolver()->GetSolver());
  assert(S);

  // set the name for the log-file and encode some information on the discretization into it.
  std::array<double,2> NN = matrix_entries(A);  
  char s[80];
  sprintf(s,"%s-%6.6f-%2.2f-%2.2f.txt",
	  material.c_str(),
	  data.dt,NN[0],NN[1]);
  std::ofstream OUT(s);

  double time = 0;  // 1st output
  OUT << time << "\t" << data.def << "\t" << data.vel << "\t0\t0" << std::endl;
  
  std::cout << std::endl << "================================================== Start Time Loop" << std::endl;

  _niter = static_cast<int>(1.e-10 + 20.0/data.dt); // set the number of iterations
  for (_iter=1;_iter<=_niter;++_iter)
    {
      // we use the BDF2 time-stepping scheme. To start it, we use BDF1 in 1st step
      if (_iter==1) data.bdf1 = true;
      else          data.bdf1 = false;
      time += data.dt;

      std::cout << std::endl << "==================== step " << _iter
		<< "\ttime = " << time 
		<< "\tdef  = " << data.def 
		<< "\tvel  = " << data.vel << std::endl;

      // save the last and second last solution for time stepping
      GetMultiLevelSolver()->Equ(oldold,1.,old);
      GetMultiLevelSolver()->Equ(old,1.,u);
      data.defold = data.def;
      data.velold = data.vel;

      // set the current location of the ball
      data.def       =  0.1-data.radius +   0.05      * cos(0.1*M_PI*time);
      data.vel       =                - 0.1*0.05*M_PI * sin(0.1*M_PI*time);

      // pass the old solutions to the discretization
      GetMultiLevelSolver()->AddNodeVector("OLD",old);
      GetMultiLevelSolver()->AddNodeVector("OLDOLD",oldold);

      // solve the problem
      if (Solve(A, u, f) != "converged")
	abort();
      
      Gascoigne::DoubleVector juh = Functionals(u, f);

      GetMultiLevelSolver()->DeleteNodeVector("OLD");
      GetMultiLevelSolver()->DeleteNodeVector("OLDOLD");

      
      S->AleVisu(data,"Results/u",u,_iter);
      OUT << time << "\t"
	  << data.def << "\t"
	  << data.vel << "\t"
	  << juh << std::endl;
      
    }
  OUT.close();
}


/**
 * Loop for a ball that is falling by gravity
 **/
void Loop::run_fallingball()
{
  
  // prerefine the mesh 3 times along the ball
  initialrefine(2, std::set<int>{5});
  

  double dt;
  double initialdef;
  std::string material;
  Gascoigne::DataFormatHandler DFH;
  DFH.insert("dt", &dt, 0.);
  DFH.insert("deformation", &initialdef, 0.);
  DFH.insert("material",    &material, "none");
  Gascoigne::FileScanner FS(DFH, _paramfile, "Equation");
  
  Gascoigne::VectorInterface u("u"), old("old"), oldold("oldold"), f("f");
  Gascoigne::Matrix A("A");
  
  
  GetMultiLevelSolver()->ReInit();
  
  std::cout << "\n================== " << _iter << " ================\t nodes: "
	    << GetMultiLevelSolver()->GetSolver()->GetMesh()->nnodes() << std::endl;

  GetMultiLevelSolver()->SetProblem("nonstationaryflow");
  GetMultiLevelSolver()->ReInitMatrix(A);
  GetMultiLevelSolver()->ReInitVector(u);
  GetMultiLevelSolver()->ReInitVector(old);
  GetMultiLevelSolver()->ReInitVector(oldold);
  GetMultiLevelSolver()->ReInitVector(f);

  
  //  InitSolution(u);
  InitSolution(u);
  GetMultiLevelSolver()->Equ(old,1.0,u);
  GetMultiLevelSolver()->Equ(oldold,1.0,u);
  GetSolverInfos()->GetNLInfo().control().matrixmustbebuild() = 1;
  
  // at t=0 the sphere is located at (0,0.16) with velocity v=0
  const Gascoigne::ProblemData& data = dynamic_cast<const Gascoigne::MovingRubberProblem*>(GetMultiLevelSolver()->GetProblemDescriptor())->GetData();
  data.def       = initialdef;
  data.defold    = initialdef;
  data.defoldold = initialdef;
  data.vel       = 0.00;
  data.velold    = 0.00;
  data.veloldold = 0.00;

  const ALESolver* S = dynamic_cast<const ALESolver*> (GetMultiLevelSolver()->GetSolver());
  assert(S);

  double time = 0.0;

  // Code some information into the name of the log-file
  std::array<double,2> NN = matrix_entries(A);  
  char s[80];
  sprintf(s,"%s-%6.6f-%2.2f-%2.2f.txt",
	  material.c_str(),
	  dt,NN[0],NN[1]);
  std::ofstream OUT(s);

  
  Gascoigne::DoubleVector juh(2,0.0);
  double force = juh[1]; //
  
  std::cout << std::endl << "================================================== Start Time Loop" << std::endl;

  double g  = -(data.rho_s-data.rho_f)/data.rho_s * 9.81;        // effective gravity
  double iM =  1.0 / data.rho_s / (4.0/3.0*M_PI*pow(data.radius,3.0));  // inverse mass of ball
  
  
  for (_iter=1;_iter<=_niter;++_iter)
    {
      if (_iter==1)
	data.bdf1 = true;
      else
	data.bdf1 = false;
      
      time += dt;
      std::cout << std::endl << "==================== step " << _iter
		<< "\ttime = " << time 
		<< "\tdef  = " << data.def 
		<< "\tvel  = " << data.vel << std::endl;

      // Save old solution
      GetMultiLevelSolver()->Equ(oldold,1.,old);
      GetMultiLevelSolver()->Equ(old,1.,u);
      data.defoldold = data.defold;
      data.veloldold = data.velold;
      data.defold = data.def;
      data.velold = data.vel;
      
      GetMultiLevelSolver()->AddNodeVector("OLD",old);
      GetMultiLevelSolver()->AddNodeVector("OLDOLD",oldold);

      double omega = 0.8;
      int    iit   = 0;
      double error = 0;
      do
	{
	  ++iit;
	  double vel,def;
	  if (_iter == 1) // Implicit Euler
	    {
	      vel  = data.velold + data.dt * (g + iM * force);
	      def  = data.defold + data.dt * data.vel;
	    }
	  else // BDF2
	    {
	      vel  = 4.0/3.0 * data.velold - 1.0/3.0 * data.veloldold  + 2.0/3.0 * data.dt * (g + iM * force);
	      def  = 4.0/3.0 * data.defold - 1.0/3.0 * data.defoldold  + 2.0/3.0 * data.dt * data.vel;
	    }
	  error = fabs(data.def - def) + fabs(data.vel - vel);
	  std::cout << "==================== Iteration " << _iter << "/" << iit
		    << "\t error " << fabs(data.def - def) << "\t" << fabs(data.vel - vel) << "\t" << error << std::endl;
	  
	  data.def = omega * def + (1.0-omega)*data.def;
	  data.vel = omega * vel + (1.0-omega)*data.vel;
	  
	  assert(Solve(A, u, f) == "converged");
	  
	  
	  
	  juh = Functionals(u, f);
	  assert(juh.size()==2);
	  force = juh[1];
	}
      while (error>1.e-7);
      std::cerr << iit << std::endl;
      S->AleVisu(data,"Results/u",u,_iter);
      
      OUT << time << "\t"
	  << data.def << "\t"
	  << data.vel << "\t"
	  << juh << std::endl;
      
      GetMultiLevelSolver()->DeleteNodeVector("OLD");
      GetMultiLevelSolver()->DeleteNodeVector("OLDOLD");
    }
  OUT.close();
}



/**
 * Loop for a ball that is falling by gravity
 **/
void Loop::run_fallingball3d()
{
  // read a couple of problem paramters
  double initialdef;
  std::string material;  
  Gascoigne::DataFormatHandler DFH;
  DFH.insert("deformation", &initialdef, 0.);
  DFH.insert("material",    &material, "none");
  Gascoigne::FileScanner FS(DFH, _paramfile, "Equation");
  
  Gascoigne::VectorInterface u("u"), old("old"), oldold("oldold"), f("f");
  Gascoigne::Matrix A("A");

  // prerefine the mesh 2 times around the vall
  initialrefine(2,std::set<int>{5});
  
  // initialize Gascoigne
  GetMultiLevelSolver()->ReInit();
  GetMultiLevelSolver()->SetProblem("nonstationaryflow");  
  GetMultiLevelSolver()->ReInitMatrix(A);
  GetMultiLevelSolver()->ReInitVector(u);
  GetMultiLevelSolver()->ReInitVector(old);
  GetMultiLevelSolver()->ReInitVector(oldold);
  GetMultiLevelSolver()->ReInitVector(f);
  
  InitSolution(u);
  GetMultiLevelSolver()->Equ(old,1.,u);
  GetMultiLevelSolver()->Equ(oldold,1.,u);
  GetSolverInfos()->GetNLInfo().control().matrixmustbebuild() = 1;

  std::cout << "\n================== " << _iter << " ================\t nodes: "
	    << GetMultiLevelSolver()->GetSolver()->GetMesh()->nnodes() << std::endl;
  
  assert(dynamic_cast<const Gascoigne::NonstationaryFlowProblem3d*>(GetMultiLevelSolver()->GetProblemDescriptor()));
  const Gascoigne::ProblemData& data = dynamic_cast<const Gascoigne::NonstationaryFlowProblem3d*>(GetMultiLevelSolver()->GetProblemDescriptor())->GetData();

  // set initial position and velocity of the ball. 
  data.def3d[0]    = 0.0; 
  data.def3d[1]    = 0.0;
  data.def3d[2]    = initialdef;
  
  data.vel3d[0]    = 0.0;
  data.vel3d[1]    = 0.0;
  data.vel3d[2]    = 0.0;

  data.rot3d[0]    = 0.0;
  data.rot3d[1]    = 0.0;
  data.rot3d[2]    = 0.0;

  const ALESolver* S = dynamic_cast<const ALESolver*> (GetMultiLevelSolver()->GetSolver());
  assert(S);
  
  // Code some information into the name of the log-file
  std::array<double,2> NN = matrix_entries(A);
  char s[80];
  sprintf(s,"%s-%6.6f-%2.2f-%2.2f.txt",
	  material.c_str(),
	  data.dt,NN[0],NN[1]);
  std::ofstream OUT(s);

  // store the Forces
  Gascoigne::DoubleVector juh(6,0.0);
  Gascoigne::numfixarray<3,double> force, torque, gravity;
  
  force[0]=juh[0];   force[1]=juh[1];   force[2]=juh[2];
  torque[0]=juh[3];  torque[1]=juh[4];  torque[2]=juh[5];
  gravity[0] = 0;    gravity[1] = 0;    gravity[2] = -(data.rho_s-data.rho_f)/data.rho_s * 9.81; 

  // mass and moment of inertia (and inverse)
  double mass = data.rho_s * 4.0/3.0 * M_PI * pow(data.radius,3.0);
  double iM   =  1.0 / mass;
  double momentinertia  = mass * 2.0/5.0 * pow(data.radius,2.0); 
  double iJ             = 1.0/momentinertia;
  
     
  std::cout << std::endl << "================================================== Start Time Loop" << std::endl;  
  double time = 0.0;
  for (_iter=1;_iter<=_niter;++_iter)
    {
      if (_iter==1) data.bdf1 = true;
      else          data.bdf1 = false;

      time += data.dt;
      std::cout << std::endl << "==================== step " << _iter
		<< "\ttime = " << time  << std::endl
		<< "\t\tdef  = " << data.def3d << std::endl
		<< "\t\tvel  = " << data.vel3d << std::endl
		<< "\t\trot  = " << data.rot3d << std::endl;

      // Save old solution
      GetMultiLevelSolver()->Equ(oldold,1.,old);
      GetMultiLevelSolver()->Equ(old,1.,u);
      data.defoldold3d = data.defold3d;
      data.veloldold3d = data.velold3d;
      data.rotoldold3d = data.rotold3d;
      data.defold3d = data.def3d;
      data.velold3d = data.vel3d;
      data.rotold3d = data.rot3d;
      
      GetMultiLevelSolver()->AddNodeVector("OLD",old);
      GetMultiLevelSolver()->AddNodeVector("OLDOLD",oldold);

      // inner loop to find implicit solution
      double omega = 0.9;
      int    iit   = 0;
      double error = 0;
      do
	{
	  ++iit;
	  Gascoigne::numfixarray<3,double>  vel,def,rot;
	  
	  if (_iter == 1) // Implicit Euler
	    {
	      vel.equ(1.0,data.velold3d, data.dt, gravity, data.dt*iM   ,force);
	      def.equ(1.0,data.defold3d,                   data.dt      ,data.vel3d);
	      rot.equ(1.0,data.rotold3d,                   data.dt*iJ   ,torque);
	    }
	  else // BDF2
	    {
	      vel.equ( 4.0/3.0, data.velold3d, - 1.0/3.0, data.veloldold3d, + 2.0/3.0 * data.dt     , gravity    , 2.0/3.0 * data.dt* iM, force);
	      def.equ( 4.0/3.0, data.defold3d, - 1.0/3.0, data.defoldold3d, + 2.0/3.0 * data.dt     , data.vel3d);
	      rot.equ( 4.0/3.0, data.rotold3d, - 1.0/3.0, data.rotoldold3d, + 2.0/3.0 * data.dt * iJ, torque);
	    }
	  
	  Gascoigne::numfixarray<3,double> ev = vel-data.vel3d;
	  Gascoigne::numfixarray<3,double> ed = def-data.def3d;
	  Gascoigne::numfixarray<3,double> er = rot-data.rot3d;
	  std::cout << "--------------------------------------------------" << std::endl;
	  std::cout << ev << std::endl;
	  std::cout << ed << std::endl;
	  std::cout << er << std::endl;
	  std::cout << "--------------------------------------------------" << std::endl;
	  error = ev.norm() + ed.norm() + er.norm();
	  
	  std::cout << "==================== Iteration " << _iter << "/" << iit
	  	    << "\t deferror " << ev.norm()
		    << "\t velerror " << ed.norm()
		    << "\t roterror " << er.norm()
		    << "\t error " << error << std::endl;
	  
	  data.def3d.equ(omega, def, (1.0-omega), data.def3d);
	  data.vel3d.equ(omega, vel, (1.0-omega), data.vel3d);
	  data.rot3d.equ(omega, rot, (1.0-omega), data.rot3d);
	  
	  std::cout <<   "\t def " << data.def3d
		    << "\n\t vel " << data.vel3d 
		    << "\n\t rot " << data.rot3d << std::endl;
	  
	  if (Solve(A, u, f) != "converged")
	    abort();

	  juh = Functionals(u, f);

	  force[0]=juh[0];  force[1]=juh[1];  force[2]=juh[2];
	  torque[0]=juh[3]; torque[1]=juh[4]; torque[2]=juh[5];

	  assert(juh.size()==6);
	}
      while (error>1.e-7);
      
      //      S->AleVisu(data,"Results/u3d",u,_iter);
	  
      std::cerr << iit << std::endl;
      
      OUT << time << "\t"
	  << data.def3d << "\t"
	  << data.vel3d << "\t"
	  << data.rot3d << "\t"
	  << juh << std::endl;
      
      GetMultiLevelSolver()->DeleteNodeVector("OLD");
      GetMultiLevelSolver()->DeleteNodeVector("OLDOLD");
    }
  OUT.close();
}
