/*----------------------------   functionals.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __functionals_H
#define __functionals_H
/*----------------------------   functionals.h     ---------------------------*/

/**
*
* Copyright (C) 2020 by the Gascoigne 3D authors
*
* This file is part of Gascoigne 3D
*
* Gascoigne 3D is free software: you can redistribute it
* and/or modify it under the terms of the GNU General Public
* License as published by the Free Software Foundation, either
* version 3 of the License, or (at your option) any later
* version.
*
* Gascoigne 3D is distributed in the hope that it will be
* useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
* PURPOSE.  See the GNU General Public License for more
* details.
*
* Please refer to the file LICENSE.TXT for further information
* on this license.
*
**/

#include <string>

#include "dirichletdatabycolor.h"
#include "domainfunctional.h"
#include "residualfunctional.h"

#include "problemdata.h"


namespace Gascoigne
{


// Defines the drag, evaluated as boundary integral
class BoundaryDrag : public Gascoigne::BoundaryFunctional
{
  const Gascoigne::ProblemData& data;

public:

  // copy the problem data
  BoundaryDrag(const Gascoigne::ProblemData& dat) : BoundaryFunctional(), data(dat)
  {
  }

  std::string GetName() const
  {
    return "Boundary-Drag";
  }


  // evaluates:
  // [ nu * (nabla u + nabla u^T) n - pn ] * ex   where ex = (1,0) (2d) or ex = (1,0,0) (3d)
  double J(const Gascoigne::FemFunction& U, const Gascoigne::Vertex2d& v,
           const Gascoigne::Vertex2d& n, int color) const
  {
    double S11 = data.mu_f * (U[1].x()-1.0/v.x()*U[1].m())- U[0].m();
    double S12 = data.mu_f * U[1].y();
    // double S21 = data.mu_f * U[2].x();
    // double S22 = data.mu_f * U[2].y()- U[0].m();
    if (color == 5)
      return - 2.0*M_PI * v.x() * (S11*n.x()+S12*n.y());
    else
      return 0.0;
  }
};

  class TorqueDD : public DirichletData
  {
    std::string _ft;
  public:
    TorqueDD(const std::string ft) : _ft(ft)
    {
      colors.insert(5);
      comp_on_color[5].push_back(1);
      comp_on_color[5].push_back(2);
      comp_on_color[5].push_back(3);
    }
    void operator()(DoubleVector& b, const Vertex3d& v, int col) const
    {
      b.zero();

      if (_ft=="t1")
	{
	  b[2] = (-(v.z()-0.05));
	  b[3] = (  v.y());
	}
      else if (_ft=="t2")
	{
	  b[1] = ((v.z()-0.05) );
	  b[3] = (-v.x() );
	}
      else if (_ft=="t3")
	{
	  b[1] = (-v.y());
	  b[2] = ( v.x());
	}
      else abort();
    }
  };
  
  class ForceDD : public DirichletData
  {
    std::string _ft;
  public:
    ForceDD(const std::string ft) : _ft(ft)
    {
      if ( (ft == "2dx") || (ft == "2dy") )
	{
	  colors.insert(5);
	  comp_on_color[5].push_back(1);
	  comp_on_color[5].push_back(2);
	  
	  colors.insert(4);
	  comp_on_color[4].push_back(1);
	  comp_on_color[4].push_back(2);
	  
	  colors.insert(8);
	  comp_on_color[8].push_back(1);
	  comp_on_color[8].push_back(2);
	  
	  colors.insert(0);
	  comp_on_color[0].push_back(1);
	  comp_on_color[0].push_back(2);

	  colors.insert(6);
	  comp_on_color[6].push_back(1);
	  comp_on_color[6].push_back(2);
	}
      else if ( (ft == "3dx") || (ft == "3dy") || (ft == "3dz") )
	{
	  colors.insert(5);
	  comp_on_color[5].push_back(1);
	  comp_on_color[5].push_back(2);
	  comp_on_color[5].push_back(3);
	  
	  colors.insert(4);
	  comp_on_color[4].push_back(1);
	  comp_on_color[4].push_back(2);
	  comp_on_color[4].push_back(3);
	  
	  colors.insert(8);
	  comp_on_color[8].push_back(1);
	  comp_on_color[8].push_back(2);
	  comp_on_color[8].push_back(3);
	  
	  colors.insert(0);
	  comp_on_color[0].push_back(1);
	  comp_on_color[0].push_back(2);
	  comp_on_color[0].push_back(3);
	}
      else abort();
      
    }
    void operator()(DoubleVector& b, const Vertex2d& v, int col) const
    {
      b.zero();

      if (_ft=="2dx")
	{
	  if (col == 5)
	    {
	      b[1] = 2.0*M_PI;
	      b[2] = 0.0;
	    }
	  
	}
      else if (_ft=="2dy")
	{
	  if (col==5)
	    {
	      b[1] = 0.0;
	      b[2] = 2.0*M_PI;
	    }
	}
      else abort();
    }
    void operator()(DoubleVector& b, const Vertex3d& v, int col) const
    {
      b.zero();

      if      (_ft=="3dx")
	{
	  if (col == 5)
	    b[1] = 1.0;
	}
      else if (_ft=="3dy")
	{
	  if (col == 5)
	    b[2] = 1.0;
	}
      else if (_ft=="3dz")
	{
	  if (col == 5)
	    b[3] = 1.0;
	}
      else abort();
    }
};

  class ForceFunctional
    : public virtual ResidualFunctional
  {
  public:
    std::string _s_force_type;
    ForceFunctional(std::string s_force_type)
      : ResidualFunctional()
    {
      _s_force_type = s_force_type;
      if ( (s_force_type=="2dx") ||
	   (s_force_type=="2dy") ||
	   (s_force_type=="3dx") ||
	   (s_force_type=="3dy") ||
	   (s_force_type=="3dz") )
	__DD = new ForceDD(s_force_type);

      
      // torque
      //
      // (x-x0) x sn = ( 
      //
      
      if (s_force_type == "t1")
	{
	  __DD = new TorqueDD(s_force_type);
	}
      if (s_force_type == "t2")
	{
	  __DD = new TorqueDD(s_force_type);
	}
      if (s_force_type == "t3")
	{
	  __DD = new TorqueDD(s_force_type);
	}

      assert(__DD);
      
      ExactValue() = 0.;



    }
    
    std::string GetName() const
    {
      return _s_force_type;
    }
  };
}
  



/*----------------------------   functionals.h     ---------------------------*/
/* end of #ifndef __functionals_H */
#endif
/*----------------------------   functionals.h     ---------------------------*/
