/*----------------------------   aleproblem.h     ---------------------------*/
/*      $Id:$                 */
#ifndef __aleproblem_H
#define __aleproblem_H
/*----------------------------   aleproblem.h     ---------------------------*/

/**
 *
 * Copyright (C) 2020 by the Gascoigne 3D authors
 *
 * This file is part of Gascoigne 3D
 *
 * Gascoigne 3D is free software: you can redistribute it
 * and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later
 * version.
 *
 * Gascoigne 3D is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * Please refer to the file LICENSE.TXT for further information
 * on this license.
 *
 **/


#include "equation.h"
#include "dirichletdata.h"
#include "boundaryequation.h"
#include "lpsequation.h"
#include "paramfile.h"
#include "problemdata.h"


/*-----------------------------------------*/

namespace Gascoigne
{
  ////////////////////////////////////////////////// 2d. y-direction only
  inline double ALE_T(const ProblemData& data, double y)
  {
    if (y<0.05-2.*data.radius)  return (data.def-0.05+data.radius) * y       / (0.05-2.*data.radius);
    if (y<0.05+2.*data.radius)  return (data.def-0.05+data.radius);
    return                             (data.def-0.05+data.radius) * (0.2-y) / (0.2-(0.05+2.*data.radius));
  }
  inline double ALE_V(const ProblemData& data, double y)
  {
    if (y<0.05-2.*data.radius)  return data.vel * y       / (0.05-2.*data.radius);
    if (y<0.05+2.*data.radius)  return data.vel;
    return                             data.vel * (0.2-y) / (0.2-(0.05+2.*data.radius));
  }
  inline double ALE_dy_T(const ProblemData& data, double y)
  {
    if (y<0.05-2.*data.radius)  return (data.def-0.05+data.radius) /(0.05-2.*data.radius);
    if (y<0.05+2.*data.radius)  return 0.0;
    return                            -(data.def-0.05+data.radius)/ (0.2-(0.05+2.*data.radius));
  }
  inline double ALE_J(const ProblemData& data, double y)
  {
    return ALE_dy_T(data,y) + 1.0;
  }


  

  ////////////////////////////////////////////////// 3d. z-direction like 2d y
  // smooth transition from [0,1] -> [1,0]
  inline double trans0(double x)
  {
    if (x<=0)
      return 1.0;
    if (x>=1.0)
      return 0.0;
    return 1.0-1.0/(1.0+exp( (1.0-2.0*x)/(x-x*x) ));
  }
  inline double Dtrans0(double x)
  {
    if (x<=0)
      return 0.0;
    if (x>=1.0)
      return 0.0;
    if (x>=0.5) // for stability
      {
	double g = exp((1.0-2.*x)/(x-x*x));
	return g/(1.0+g)/(1.0+g)*(-2.0*x*x+2.0*x-1.0) * pow(x*(x-1.0),-2.0);
      }
    else
      {
	x = 1.0-x;
	double g = exp((1.0-2.*x)/(x-x*x));
	return g/(1.0+g)/(1.0+g)*(-2.0*x*x+2.0*x-1.0) * pow(x*(x-1.0),-2.0);
      }
  }
  
  // smooth transition [r0,r1] -> [1,0]
  inline double trans(double r, double r0, double r1)
  {
    return trans0( (r-r0)/(r1-r0) );
  }
  inline double Dtrans(double r, double r0, double r1)
  {
    return Dtrans0( (r-r0)/(r1-r0) ) / (r1-r0);
  }


  
  // x+y 
  inline double ALE3d_Tz(const ProblemData& data, const Vertex3d& v)
  {
    if (v.z()<0.05-2.*data.radius)  return (data.def3d[2]-0.05+data.radius) * v.z()       / (0.05-2.*data.radius);
    if (v.z()<0.05+2.*data.radius)  return (data.def3d[2]-0.05+data.radius);
    return                                 (data.def3d[2]-0.05+data.radius) * (0.2-v.z()) / (0.2-(0.05+2.*data.radius));
  }
  inline double ALE3d_Tx(const ProblemData& data,  const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<=data.radius)
      return data.def3d[0];
    return trans(r,data.radius,data.cyl_radius) * data.def3d[0];
  }
  inline double ALE3d_Ty(const ProblemData& data,  const Vertex3d& v)
  { 
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<=data.radius)
      return data.def3d[1];
    return trans(r,data.radius,data.cyl_radius) * data.def3d[1];
  }

  inline double ALE3d_dz_Tz(const ProblemData& data, const Vertex3d& v)
  {
    if (v.z()<0.05-2.*data.radius)  return 1.0+(data.def3d[2]-0.05+data.radius) /(0.05-2.*data.radius);
    if (v.z()<0.05+2.*data.radius)  return 1.0;
    return                                 1.0-(data.def3d[2]-0.05+data.radius)/ (0.2-(0.05+2.*data.radius));
  }
  
  inline double ALE3d_dx_Tx(const ProblemData& data, const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return 1.0;
    return 1.0 + Dtrans(r,data.radius,data.cyl_radius) * data.def3d[0] / r * v.x();
  }
  inline double ALE3d_dy_Tx(const ProblemData& data, const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return 0.0;
    return 0.0 + Dtrans(r,data.radius,data.cyl_radius) * data.def3d[0] / r * v.y();
  }
  
  inline double ALE3d_dx_Ty(const ProblemData& data, const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return 0.0;
    return       Dtrans(r,data.radius,data.cyl_radius) * data.def3d[1] / r * v.x();
  }
  inline double ALE3d_dy_Ty(const ProblemData& data, const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return 1.0;
    return 1.0 + Dtrans(r,data.radius,data.cyl_radius) * data.def3d[1] / r * v.y();
  }


  
  ////////////////////////////////////////////////// 3d. z-direction like 2d y
  // x+y 
  inline double ALE3d_Vz(const ProblemData& data, const Vertex3d& v)
  {
    if (v.z()<0.05-2.*data.radius)  return data.vel3d[2] * v.z()       / (0.05-2.*data.radius);
    if (v.z()<0.05+2.*data.radius)  return data.vel3d[2];
    return                                 data.vel3d[2] * (0.2-v.z()) / (0.2-(0.05+2.*data.radius));
  }
  inline double ALE3d_Vx(const ProblemData& data,  const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return data.vel3d[0];
    return trans(r,data.radius,data.cyl_radius) * data.vel3d[0];
  }
  inline double ALE3d_Vy(const ProblemData& data,  const Vertex3d& v)
  {
    double r = sqrt(v.x()*v.x()+v.y()*v.y());
    if (r<data.radius)
      return data.vel3d[1];
    return trans(r,data.radius,data.cyl_radius) * data.vel3d[1];
  }
  
  inline double ALE3d_J(const ProblemData& data, const Vertex3d& v)
  {
    abort();
    return ALE3d_dz_Tz(data,v) * (ALE3d_dx_Tx(data,v)*ALE3d_dy_Ty(data,v) - ALE3d_dy_Tx(data,v)*ALE3d_dx_Ty(data,v));
  }
}


/*----------------------------   aleproblem.h     ---------------------------*/
/* end of #ifndef __aleproblem_H */
#endif
/*----------------------------   aleproblem.h     ---------------------------*/
