from math import pi, e
import numpy as np  # This makes tuples behave likes vectors using np.array
import plotly.graph_objs as go  # This is used for plotting
from datetime import datetime
import os
import cv2
# You also need to install Orca if you haven't already..the error prompts will guide you well on how to do so

start = datetime.now()
print(datetime.now() - start, " - ", datetime.now().replace(microsecond=0))
i = (-1) ** 0.5  # imaginary unit
wl = 1  # wavelength
FoV = 100
c = 3 * 10 ** 8  # speed of light
k = 2 * pi / wl  # wave number
omega = c * k  # angular frequency
period = 2 * pi / omega


# Select colors for elements of plot:
grid_color = (180, 180, 180)
sphere_color = (169, 169, 169)
horizontal_sphere_color = (159, 159, 159)
wall_projection_color = (50, 50, 50)
darker_ellipse_color = (201, 33, 14)
lighter_ellipse_color = (245, 84, 66)


def R(position):  # This returns magnitude of position vector
    return np.linalg.norm(position, axis=0)


def n(position):  # This returns unitary direction vector
    return position / R(position)


def find_nearest(array, value):  # This return nearest element in array to given value
    idx = (np.abs(array - value)).argmin()
    return array[idx]


def calculate_electric_dipole(polarizability):
    # This calculates the excited dipoles by multiplying by (scalar) amplitude of projection to the excitation field.

    return polarizability * np.dot(polarizability, incoming_polarization) / np.linalg.norm(polarizability)


def calculate_magnetic_dipole(polarizability):
    # This calculates the excited dipoles by multiplying by (scalar) amplitude of
    #  projection to the normal of excitation field (B-field).
    B_field = np.dot(rotation_matrix(pi / 2), incoming_polarization)

    return polarizability * np.dot(polarizability, B_field) / np.linalg.norm(polarizability)


def calculate_electric_quadrupole(polarizability, position):
    # Makes polarizability a valid matrix
    polarizability[:, 0] = polarizability[0, :]
    polarizability[2, 1] = polarizability[1, 2]
    # Returns quadrupole vector calculated based on quadrupole matrix (is a function of n(r))
    quadrupole_vector = np.zeros(np.array(position).shape)
    for element in range(3):
        quadrupole_vector[element] = (
                polarizability[element][(element + 0) % 3] * n(position)[(element + 0) % 3] +
                polarizability[element][(element + 1) % 3] * n(position)[(element + 1) % 3] +
                polarizability[element][(element + 2) % 3] * n(position)[(element + 2) % 3])
    return quadrupole_vector * np.dot(polarizability[2], incoming_polarization) / np.linalg.norm(polarizability[2])


def calculate_magnetic_quadrupole(polarizability, position):
    # Makes polarizability a valid matrix
    polarizability[:, 0] = polarizability[0, :]
    polarizability[2, 1] = polarizability[1, 2]
    # Returns quadrupole vector calculated based on quadrupole matrix (is a function of n(r))
    quadrupole_vector = np.zeros(np.array(position).shape)
    for element in range(3):
        quadrupole_vector[element] = (
                polarizability[element][(element + 0) % 3] * n(position)[(element + 0) % 3] +
                polarizability[element][(element + 1) % 3] * n(position)[(element + 1) % 3] +
                polarizability[element][(element + 2) % 3] * n(position)[(element + 2) % 3])
    return quadrupole_vector * np.dot(polarizability[2],
                                      np.dot(rotation_matrix(pi / 2), incoming_polarization)) / np.linalg.norm(
        polarizability[2])


def get_E_p(position, dipole_p, phase_shift):  # This calculates E-field of electric dipoles
    get_H_p = k ** 2 * e ** (i * k * R(position) + i * phase_shift) * np.cross(n(position), dipole_p, axis=0) / (
            4 * pi * R(position))
    return np.array(377 * np.cross(get_H_p, n(position), axis=0))


def get_E_m(position, dipole_m, phase_shift):  # This calculates E-field of a magnetic dipole
    return -377 * k ** 2 * e ** (i * k * R(position) + i * phase_shift) * np.cross(n(position), dipole_m, axis=0) / (
            4 * pi * R(position))


def get_E_eq(position, my_quadrupole_vector, phase_shift):
    get_H_q = (-i * k ** 3 * e ** (i * k * R(position) + i * phase_shift)
               * np.cross(n(position), my_quadrupole_vector, axis=0) / (24 * pi * R(position)))
    return 377 * np.cross(get_H_q, n(position), axis=0)


def get_E_mq(position, my_quadrupole_vector, phase_shift):
    return 377 * (-i * k ** 3 * e ** (i * k * R(position) + i * phase_shift)
                  * np.cross(n(position), my_quadrupole_vector, axis=0) / (24 * pi * R(position)))


def get_E2(position):  # This returns E times E*
    E = get_E(position)
    return np.sum(E * np.conj(E), axis=0).real


def add_sphere():
    radius = 85
    angles = np.array([pi / 4, pi / 2, 3 * pi / 4])
    # circles connecting sphere
    point_angles = np.linspace(0, 2 * pi, 200)
    circle = radius * np.array([np.cos(point_angles), point_angles * 0, np.sin(point_angles)]).T

    for circle_angle in np.linspace(-pi, pi, 8, endpoint=False):
        rotated_circle = np.empty((0, 3))
        for point in circle:
            rotated_point = np.dot(rotation_matrix(circle_angle), point)
            rotated_circle = np.vstack([rotated_circle, rotated_point])

        fig.add_trace(go.Scatter3d(x=rotated_circle[:, 0], y=rotated_circle[:, 1], z=rotated_circle[:, 2],
                                   mode="lines",
                                   line=dict(color=f'rgb{sphere_color}', width=5.5),
                                   showlegend=False,
                                   name="Sphere indicator",
                                   legendgroup="Polarization indicators"))

    point_angles = np.linspace(0, 2 * pi, 200)
    circle = radius * np.array([np.cos(point_angles), np.sin(point_angles), point_angles * 0]).T

    # tropic of cancer and capricorn (or others?)
    for angle in angles:
        fig.add_trace(go.Scatter3d(x=np.sin(angle) * circle[:, 0], y=np.sin(angle) * circle[:, 1],
                                   z=circle[:, 2] * 0 + radius * np.cos(angle),
                                   mode="lines",
                                   line=dict(color=f'rgb{horizontal_sphere_color}', width=5.7),
                                   showlegend=False,
                                   name="Sphere indicator",
                                   legendgroup="Polarization indicators"))

    # adds intersections of circles
    phies, thetas = np.meshgrid(np.linspace(-pi, pi, 8, endpoint=False), angles)
    fig.add_trace(go.Scatter3d(x=radius * (np.sin(thetas) * np.cos(phies)).flatten(),
                               y=radius * (np.sin(thetas) * np.sin(phies)).flatten(),
                               z=radius * (np.cos(thetas)).flatten(),
                               mode="markers",
                               marker=dict(
                                   size=2,
                                   color=f'rgb(100,100,100)'),
                               showlegend=False,
                               name="Sphere indicator",
                               legendgroup="Polarization indicators"))


def add_nanopillar(x_span, y_span, z_span):
    # Adds cuboid centered in [0,0,0] with sides of x_span, y_span and z_span
    spans = [[-x_span / 2, x_span / 2], [-y_span / 2, y_span / 2], [-z_span / 2, z_span / 2]]

    # This plots the faces of nanopillar
    corners = np.array([list(corner) for corner in np.array(np.meshgrid(*spans)).T.reshape(-1, len(spans))])
    rotated_corners = np.empty((0, 3))
    for corner in corners:
        rotated_corners = np.vstack([rotated_corners, np.dot(rotation_matrix(phi), corner)])
    fig.add_trace(go.Mesh3d(x=rotated_corners[:, 0], y=rotated_corners[:, 1], z=rotated_corners[:, 2],
                            contour=dict(color="rgb(0,0,0)"),
                            lighting=dict(specular=0.5, roughness=0.9),
                            lightposition=dict(x=100, y=100, z=100),
                            alphahull=0,
                            opacity=1,
                            color="rgb(69, 46, 89)",
                            name="Nano-pillar"))

    # This highlights the edges
    edges = np.empty((0, 3))
    edges = np.vstack([edges, rotated_corners[[0, 1, 3, 2, 0, 4, 5, 1, 5, 7, 3, 7, 6, 2, 6, 4]]])
    fig.add_trace(go.Scatter3d(x=edges[:, 0], y=edges[:, 1], z=edges[:, 2],
                               mode="lines",
                               line=dict(color='rgb(0,0,0)', width=0.5),
                               hoverinfo="name",
                               name="Nano-pillar",
                               showlegend=False))


def add_center_indicator():
    # Adds three lines parallel to x,y,z-axis intersecting at [0,0,0]
    for dimension in range(3):

        show_in_legend = False
        if dimension == 0:
            show_in_legend = True

        line = [[0, 0], [0, 0], [0, 0]]
        line[dimension] = [-FoV - 20, FoV + 20]

        fig.add_trace(go.Scatter3d(x=line[0], y=line[1], z=line[2],
                                   mode="lines",
                                   line=dict(color='rgb(255, 230, 0)', width=7),
                                   name="Center indicator",
                                   hoverinfo="name",
                                   legendgroup="Center indicator",
                                   showlegend=show_in_legend,
                                   visible='legendonly'))


def add_zero_order_state_indicators():
    # this is for LOWER indicator
    # adds the body of an arrow indicator
    fig.add_trace(go.Scatter3d(x=[0, 0], y=[0, 0], z=[-95 + 5, -95 + 14],
                               mode="lines",
                               line=dict(color=f"rgb{darker_ellipse_color}", width=20),
                               name="Incoming polarization",
                               hoverinfo="name",
                               legendgroup="Incoming",
                               showlegend=True))

    # adds head of the arrow indicator
    fig.add_trace(go.Cone(x=[0], y=[0], z=[-95 + 25], u=[0], v=[0], w=[1],
                          sizemode="absolute",
                          sizeref=12,
                          anchor="tip",
                          colorscale=[[0, f"rgb{darker_ellipse_color}"], [1, f"rgb{darker_ellipse_color}"]],
                          showscale=False,
                          name="Incoming polarization",
                          hoverinfo="name",
                          legendgroup="Incoming",
                          showlegend=False))

    # adds the ellipse representing polarization
    time_moment = np.linspace(time, time + period, 100).reshape((-1, 1))
    points_in_period = 30 * (incoming_polarization_state / (np.linalg.norm(np.abs(incoming_polarization_state)))
                             * e ** (-i * omega * time_moment)).real
    fig.add_trace(go.Scatter3d(x=points_in_period[:, 0], y=points_in_period[:, 1],
                               z=points_in_period[:, 2] + -95,
                               mode="lines",
                               line=dict(color=f"rgb{darker_ellipse_color}", width=12),
                               name="Incoming polarization",
                               legendgroup="Incoming",
                               showlegend=False))

    # adds RHP vs LHP indicator arrow
    fig.add_trace(
        go.Cone(x=[points_in_period[5, 0]], y=[points_in_period[5, 1]], z=[points_in_period[5, 2] + -95],
                u=[points_in_period[5, 0] - points_in_period[0, 0]],
                v=[points_in_period[5, 1] - points_in_period[0, 1]],
                w=[points_in_period[5, 2] - points_in_period[0, 2]],
                sizemode="absolute",
                sizeref=8,
                anchor="tip",
                colorscale=[[0, f"rgb{darker_ellipse_color}"], [1, f"rgb{darker_ellipse_color}"]],
                showscale=False,
                name="Incoming polarization",
                hoverinfo="name",
                legendgroup="Incoming",
                showlegend=False))

    # This is for TOP indicator
    # adds the body of an arrow indicator
    fig.add_trace(go.Scatter3d(x=[0, 0], y=[0, 0], z=[95 + 5, 95 + 14],
                               mode="lines",
                               line=dict(color=f"rgb{darker_ellipse_color}", width=20),
                               name="Outgoing polarization",
                               hoverinfo="name",
                               legendgroup="Outgoing",
                               showlegend=True))

    # adds head of the arrow indicator
    fig.add_trace(go.Cone(x=[0], y=[0], z=[95 + 25], u=[0], v=[0], w=[1],
                          sizemode="absolute",
                          sizeref=12,
                          anchor="tip",
                          colorscale=[[0, f"rgb{darker_ellipse_color}"], [1, f"rgb{darker_ellipse_color}"]],
                          showscale=False,
                          name="Outgoing polarization",
                          hoverinfo="name",
                          legendgroup="Outgoing",
                          showlegend=False))

    # adds the ellipse representing the outgoing polarization
    E = get_E(np.array([0, 0, 95]))
    time_moment = np.linspace(time, time + period, 100).reshape((-1, 1))
    points_in_period = 30 * ((E * e ** (-i * omega * time_moment)).real / (np.linalg.norm(np.abs(E))))
    fig.add_trace(go.Scatter3d(x=points_in_period[:, 0], y=points_in_period[:, 1], z=points_in_period[:, 2] + 95,
                               mode="lines",
                               line=dict(color=f"rgb{darker_ellipse_color}", width=12),
                               name="Outgoing polarization",
                               legendgroup="Outgoing",
                               showlegend=False))

    # adds RHP vs LHP indicator arrow
    fig.add_trace(
        go.Cone(x=[points_in_period[5, 0]], y=[points_in_period[5, 1]], z=[points_in_period[5, 2] + 95],
                u=[points_in_period[5, 0] - points_in_period[0, 0]],
                v=[points_in_period[5, 1] - points_in_period[0, 1]],
                w=[points_in_period[5, 2] - points_in_period[0, 2]],
                sizemode="absolute",
                sizeref=8,
                anchor="tip",
                colorscale=[[0, f"rgb{darker_ellipse_color}"], [1, f"rgb{darker_ellipse_color}"]],
                showscale=False,
                name="Outgoing polarization",
                hoverinfo="name",
                legendgroup="Outgoing",
                showlegend=False))


def add_ellipses_to_isosurface():
    samples = np.empty((0, 3))
    rr = 1000000
    # For a single element, draw ellipses in +-45°
    if array_count == 1:
        # this finds points ("samples") on isosurface closest (in angle) to several directions from origin and stacks them in shape = (-1, 3)
        for z_angle in np.linspace(pi / 4, 3 * pi / 4, 3):
            for x_y_angle in np.linspace(-pi, pi, 8, endpoint=False):
                # and the ellipse will then be drawn at that distance and given angles
                my_sample = [rr * np.sin(z_angle) * np.cos(x_y_angle), rr * np.sin(z_angle) * np.sin(x_y_angle),
                             rr * np.cos(z_angle)]
                samples = np.vstack([samples, my_sample])

    # For an array, match ellipses with diffraction orders
    else:
        for dif_order_phi in np.linspace(-pi+phi, pi+phi, 8, endpoint=False):
            dif_order_theta = np.linspace(0, pi, n_of_points)
            dif_locs = np.array(
                [np.sin(dif_order_theta) * np.cos(dif_order_phi), np.sin(dif_order_theta) * np.sin(dif_order_phi),
                 np.cos(dif_order_theta)])
            radiation_pattern = np.array([get_E2(rr * dif_locs)])
            radiation_pattern = 60 * radiation_pattern / np.amax(radiation_pattern)
            dif_order_found = dif_order_theta.flatten()[radiation_pattern.flatten() > 30]
            dif_order_found = np.split(dif_order_found, (np.arange(dif_order_found.size - 1) + 1)[
                np.abs(np.diff(dif_order_found) > 0.1)])
            # Ignore poles
            dif_order_found = dif_order_found[1:-1]
            for each_dif_order in dif_order_found:
                my_sample = [rr * np.sin(np.average(each_dif_order)) * np.cos(dif_order_phi),
                             rr * np.sin(np.average(each_dif_order)) * np.sin(dif_order_phi),
                             rr * np.cos(np.average(each_dif_order))]
                samples = np.vstack([samples, my_sample])

    # Calculate field on none-meshgrid shaped coordinates using apply_along_axis()
    if samples.size > 0:
        result_field = 12 * (np.apply_along_axis(get_E, 1, samples) /
                             (np.linalg.norm(abs(np.apply_along_axis(get_E, 1, samples)), axis=1))[:, np.newaxis])

        for sample in range(samples.shape[0]):
            # this makes only first entry to be displayed in legend
            show_in_legend = True if sample == 0 else False

            # Stacks points of ellipse sample by sample
            E = result_field[sample, :]
            time_moment = np.linspace(time, time + period, 100).reshape((-1, 1))
            points_in_ellipse = (E * e ** (-i * omega * time_moment)).real

            # plots one of the ellipses
            fig.add_trace(
                go.Scatter3d(x=points_in_ellipse[:, 0] + 0.000085 * samples[sample, 0],
                             y=points_in_ellipse[:, 1] + 0.000085 * samples[sample, 1],
                             z=points_in_ellipse[:, 2] + 0.000085 * samples[sample, 2],
                             mode="lines",
                             line=dict(color=f'rgb{lighter_ellipse_color}', width=8),
                             name="Polarization indicators",
                             legendgroup="Polarization indicators",
                             showlegend=show_in_legend))

            # adds RHP vs LHP indicator arrow
            fig.add_trace(
                go.Cone(x=points_in_ellipse[:, 0] + 0.000085 * samples[sample, 0],
                        y=points_in_ellipse[:, 1] + 0.000085 * samples[sample, 1],
                        z=points_in_ellipse[:, 2] + 0.000085 * samples[sample, 2],
                        u=[points_in_ellipse[1, 0] - points_in_ellipse[0, 0]],
                        v=[points_in_ellipse[1, 1] - points_in_ellipse[0, 1]],
                        w=[points_in_ellipse[1, 2] - points_in_ellipse[0, 2]],
                        sizemode="absolute",
                        sizeref=4,
                        anchor="tip",
                        colorscale=[[0, f'rgb{darker_ellipse_color}'], [1, f'rgb{darker_ellipse_color}']],
                        showscale=False,
                        name="Polarization indicators",
                        hoverinfo="name",
                        legendgroup="Polarization indicators",
                        showlegend=False))


def plot_isosurface():
    phi_angles, theta_angles = np.meshgrid(np.linspace(0, 2 * pi, n_of_points),
                                           np.linspace(0, pi, n_of_points))
    # Equidistant points on sphere
    loc = np.array([np.sin(theta_angles) * np.cos(phi_angles), np.sin(theta_angles) * np.sin(phi_angles),
                    np.cos(theta_angles)])
    # Triangles tell which points should connect to form nice surface
    ind = np.arange(n_of_points ** 2 - n_of_points).reshape((-1, 1))
    triangles = np.vstack(
        [np.hstack([ind, ind + 1, ind + n_of_points]), np.hstack([ind, ind + n_of_points, ind + n_of_points - 1])])

    # Calculates field very far away and the intensity at given point results in surface distance from center at that point
    radiation_pattern = np.array([get_E2(1000000 * loc)])
    radiation_pattern = 60 * radiation_pattern / np.amax(radiation_pattern)
    points = radiation_pattern * loc

    fig.add_trace(go.Mesh3d(x=points[0, :].flatten(),
                            y=points[1, :].flatten(),
                            z=points[2, :].flatten(),
                            i=triangles[:, 0].flatten(),
                            j=triangles[:, 1].flatten(),
                            k=triangles[:, 2].flatten(),
                            name="radiation pattern",
                            legendgroup="pattern",
                            colorscale="Viridis",
                            cmin=0,
                            cmax=65,
                            colorbar=dict(title=dict(text="Distance from center [λ]", font=dict(size=36)),
                                          titleside="right", xpad=50, ypad=100, tickfont=dict(size=28)),
                            showscale=False,
                            intensity=R(points).flatten()))


def rotation_matrix(angle_around_z=0, angle_around_y=0, angle_around_x=0):
    rotation_matrix_z = np.array([[np.cos(angle_around_z), -np.sin(angle_around_z), 0],
                                  [np.sin(angle_around_z), np.cos(angle_around_z), 0],
                                  [0, 0, 1]])

    rotation_matrix_y = np.array([[np.cos(angle_around_y), 0, np.sin(angle_around_y)],
                                  [0, 1, 0],
                                  [-np.sin(angle_around_y), 0, np.cos(angle_around_y)]])

    rotation_matrix_x = np.array([[1, 0, 0],
                                  [0, np.cos(angle_around_x), -np.sin(angle_around_x)],
                                  [0, np.sin(angle_around_x), np.cos(angle_around_x)]])

    return np.dot(np.dot(rotation_matrix_y, rotation_matrix_z), rotation_matrix_x)


def rotation_around_arbitrary_axis(angle_of_axis_from_z, angle_of_axis_from_x, angle_around_axis):
    # first two arguments specify angle of axis around which to rotate by third argument
    rotation_around_z = rotation_matrix(angle_of_axis_from_x, 0, 0)
    rotation_around_y = rotation_matrix(0, angle_of_axis_from_z, 0)
    rotation_around_z_by_angle = rotation_matrix(angle_around_axis, 0, 0)
    rotation_around_y_back = rotation_matrix(0, -angle_of_axis_from_z, 0)
    rotation_around_z_back = rotation_matrix(-angle_of_axis_from_x, 0, 0)

    return np.dot(np.dot(np.dot(np.dot(rotation_around_z, rotation_around_y), rotation_around_z_by_angle),
                         rotation_around_y_back), rotation_around_z_back)


def rotate(multipole):
    return np.dot(rotation_matrix(phi), np.dot(rotation_matrix(0, theta), np.dot(rotation_matrix(alpha), multipole)))


def add_radiation_pattern():
    angles = np.linspace(0, 2 * pi, 10000)
    loc = np.array([np.cos(angles), angles * 0, np.sin(angles)])
    radiation_pattern = np.array([get_E2(10000 * loc)])
    radiation_pattern = 90 * radiation_pattern / np.amax(radiation_pattern)
    fig.add_trace(
        go.Scatter3d(x=(radiation_pattern * loc)[0, :],
                     y=-FoV * np.ones((loc.shape[1])),
                     z=(radiation_pattern * loc)[2, :],
                     mode="lines",
                     line=dict(color=f"rgb{wall_projection_color}", width=9),
                     hoverinfo="name",
                     name="Projections on walls",
                     legendgroup="projection"))

    angles = np.linspace(0, 2 * pi, 10000)
    loc = np.array([angles * 0, np.cos(angles), np.sin(angles)])
    radiation_pattern = np.array([get_E2(10000 * loc)])
    radiation_pattern = 90 * radiation_pattern / np.amax(radiation_pattern)
    fig.add_trace(
        go.Scatter3d(x=-FoV * np.ones((loc.shape[1])),
                     y=(radiation_pattern * loc)[1, :],
                     z=(radiation_pattern * loc)[2, :],
                     mode="lines",
                     line=dict(color=f"rgb{wall_projection_color}", width=9),
                     hoverinfo="name",
                     name="Projections on walls",
                     legendgroup="projection",
                     showlegend=False))


def plot_in_2D(path_to_file):
    # This makes the parallel lines seem parallel
    fig.layout.scene.camera.projection.type = "orthographic"

    # This sets axes to be of the same scale
    fig.update_layout(scene=dict(
        xaxis=dict( showspikes=False, tickmode='array', title=dict(text=""), tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="outside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        yaxis=dict(showspikes=False, tickmode='array', title=dict(text=""), tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="inside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        zaxis=dict(showspikes=False, tickmode='array', title=dict(text=""), tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="outside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        aspectmode='cube'))

    # This sets the default view angle
    fig.update_layout(scene_camera=dict(
        eye=dict(x=1.35, y=1.20, z=1.25)))

    # This adds axis titles
    fig.update_layout(
        scene=dict(
            annotations=[
                dict(showarrow=False,
                     x=0,
                     y=100,
                     z=-100,
                     text="← x",
                     xanchor="center",
                     xshift=0,
                     yshift=-40,
                     textangle=-33,
                     font=dict(
                         color="black",
                         size=70)),

                dict(showarrow=False,
                     x=100,
                     y=0,
                     z=-100,
                     text="y →",
                     xanchor="center",
                     xshift=0,
                     yshift=-50,
                     textangle=33,
                     font=dict(
                         color="black",
                         size=70)),

                dict(showarrow=False,
                     x=100,
                     y=-100,
                     z=0,
                     text="z →",
                     xanchor="center",
                     xshift=-60,
                     yshift=20,
                     textangle=-90,
                     font=dict(
                         color="black",
                         size=70)),

                dict(showarrow=False,
                     x=69,
                     y=-100,
                     z=100,
                     text=f"({letter})",
                     xanchor="center",
                     yshift=280,
                     textangle=0,
                     font=dict(
                         color="black",
                         size=70))
            ]))

    fig.update_layout(showlegend=False,
                      plot_bgcolor="#FFF",
                      # title=dict(
                      #     text="Rotation angle of " + "{:.2f}".format(phi / pi) + " pi",
                      #     font=dict(size=30),
                      #     x=0.44,
                      #     y=0.9,
                      #     xanchor='center',
                      #     yanchor='top'),
                      font=dict(size=18))

    fig.update_layout(scene=dict(
        xaxis=dict(title=dict(text="")),
        yaxis=dict(title=dict(text="")),
        zaxis=dict(title=dict(text=""))))

    # This sets resolution for image saves
    fig.update_layout(
        autosize=False,
        width=2000,
        height=2000,
        margin=dict(t=0, r=0, l=0, b=0))

    parent_folder = os.path.dirname(path_to_file)
    if not os.path.exists(parent_folder):
        os.makedirs(parent_folder)
    fig.write_image(path_to_file)
    img = cv2.imread(path_to_file)
    img = img[257:-160, 200:-220]
    cv2.imwrite(path_to_file, img)
    print(datetime.now() - start, " ...and also saved!")


def plot_in_3D():
    # This puts toggles to the bottom
    fig.update_layout(legend_orientation="h")

    fig.update_layout(scene=dict(
        xaxis=dict(title=dict(text="x")),
        yaxis=dict(title=dict(text="y")),
        zaxis=dict(title=dict(text="z"))))

    # This makes it look 3D
    fig.layout.scene.camera.projection.type = "perspective"

    fig.show()


def formatting():
    # This sets axes to be of the same scale
    fig.update_layout(scene=dict(
        xaxis=dict(range=[-FoV, FoV], showspikes=False, tickmode='array', tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="outside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        yaxis=dict(range=[-FoV, FoV], showspikes=False, tickmode='array', tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="inside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        zaxis=dict(range=[-FoV, FoV+20], showspikes=False, tickmode='array', tickvals=[-100, -50, 0, 50, 100],  ticktext=["", "", "", "", ""],
                   ticks="outside", tickcolor='white', ticklen=0, backgroundcolor="white",
                   gridcolor=f"rgb{grid_color}"),
        aspectmode='cube'))


def get_E(position):
    total_E = np.zeros(position.shape)
    if array_count == 1 or 0:
        array_element = np.array([[0, 0, 0]])
    else:
        array_element = np.array(
            [np.meshgrid(np.linspace(-area_length / 2, area_length / 2, array_count),
                         np.linspace(-area_length / 2, area_length / 2, array_count), 0)]).T.reshape(-1, 3)

    for element_location in array_element:
        element_position = np.subtract(position.T, np.dot(rotation_matrix(0), element_location)).T

        total_E = total_E + get_E_p(element_position, dipole_px, 0) + get_E_p(element_position, dipole_py, Phase_shift)
        # total_E = total_E + get_E_m(element_position, dipole_mx, 0) + get_E_m(element_position, dipole_my, Phase_shift)
        # total_E = total_E + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_xz, element_position), 0) + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_yz, element_position), Phase_shift)
        # total_E = total_E + get_E_m(element_position, dipole_mx, Phase_shift) + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_xz, element_position), 0)

        # total_E = total_E + get_E_p(element_position, dipole_px, 0) + get_E_m(element_position, dipole_my, Phase_shift)
        # total_E = total_E + get_E_p(element_position, dipole_px, 0) + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_xz, element_position), Phase_shift)
        # total_E = total_E + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_xz, element_position), 0) + get_E_mq(element_position, calculate_magnetic_quadrupole(quadrupole_m_yz, element_position), Phase_shift)
        #
        # total_E = total_E + get_E_p(element_position, dipole_px, 0) + get_E_p(element_position, dipole_py, Phase_shift) + get_E_m(element_position, dipole_my, 0) + get_E_m(element_position, dipole_mx, Phase_shift)
        # total_E = total_E + get_E_p(element_position, dipole_px, 0) + get_E_p(element_position, dipole_py, Phase_shift) + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_xz, element_position), pi / 2) + get_E_eq(element_position, calculate_electric_quadrupole(quadrupole_e_yz, element_position), pi / 2 + Phase_shift)

    return total_E


# TODO ======= PARAMETERS ARE SET BELOW =======
incoming_polarization_state = np.array([1, i, 0])
array_count = 4  # Number of elements in sidelength area, if 1 -> single element in center
area_length = 5  # Side length of the array in wavelengths
letter = "a"
Phase_shift = 1 * pi  # This is the phase shift between the radiation of multipoles
time = 0 * period
theta = 0  # Rotation around y, not used
phi = 0  # Rotation around z, this is used to rotate the element
alpha = 0  # Rotation around "axis" but in fact it is around z, not used

n_of_points = 400   # Sets the resolution, 100 enough for single multipole, 500 is good for arrays
# TODO ======= PARAMETERS ARE SET ABOVE =======

incoming_polarization = (incoming_polarization_state * e ** (-i * omega * time))
# Calculate the excited dipoles according to polarizabilities and incoming polarization
dipole_px = calculate_electric_dipole(rotate(np.array([1, 0, 0])))
dipole_py = calculate_electric_dipole(rotate(np.array([0, 1, 0])))
dipole_mx = calculate_magnetic_dipole(rotate(np.array([1, 0, 0])))
dipole_my = calculate_magnetic_dipole(rotate(np.array([0, -1, 0])))

quadrupole_e_xz = rotate(np.array([[0, 0, 1],
                                   [0, 0, 0],
                                   [1, 0, 0]]))

quadrupole_e_yz = rotate(np.array([[0, 0, 0],
                                   [0, 0, 1],
                                   [0, 1, 0]]))

quadrupole_m_yz = rotate(np.array([[0, 0, 0],
                                   [0, 0, 1],
                                   [0, 1, 0]]))

fig = go.Figure()
# add_nanopillar(35, 20, 60)
plot_isosurface()
add_ellipses_to_isosurface()
add_zero_order_state_indicators()
add_radiation_pattern()
add_sphere()
add_center_indicator()
formatting()
print(datetime.now() - start, " All is calculated")
plot_in_3D()
# plot_in_2D(f"Python pictures/For article/Fig1(6)_{letter}.png")
