## ANALYSES AND FIGURES FOR THE MANUSCRIPT ENTITLED 'STATUS, RIVALRY AND ADMIRATION-SEEKING IN NARCISSISM AND DEPRESSION', Anna Szucs, Katalin Szanto, Jade Adalbert, Aidan G. C. Wright, Luke Clark, Alexandre Y. Dombrovski 
## The script follows the organization of the manuscript
## Last updated Augsut 10th 2020


#########################
####### GLOSSARY ########
#########################

## subject-level variables 
##########################
# subject = participant's ID
# sample = categorical variable indicating to which sample the participant's belongs (sample 1 = Vancouver - psychology undergrads, sample 2 = Pittsburgh - majoritarily depressed elderly)
# age = participants' age when playing the task
# sex = participants' sex, categorical variable with levels 'male', 'female'
# education = years of education (assumed to be 13 for all undergrads in Sample 1)
# household_income = ranked variable (see Supplemental Table S1 for ranks), used in Sample 1 and the pooled analysis
# household_income_s2, household_income_s2_log = household income in dollars in the Pittsburgh sample; the log-transformed variable was used in analysis
# gameExp = experience with video games on computers (question built in the task's introduction section), smart phones and tablets (continuous variable with range 1 = 'played < 5 times/lifetime' to 5 = 'plays weekly')
# pointSteal_taskmean = task-level mean point stealing
# rankBuy_taskmean = task-level mean rank buying
# race = categorical variable, with categories 'Asian', 'African-Canadian or African-American', 'Caucasian', 'Other'
# stereotypical_resp = true/false variable indicating whether responses were stereotypical (i.e. unvaried) on both point stealing and rank buying 
# qMot1 to qMot8 = end-game motivation questions (see Supplemental Figure S7 for more details)
# qA1control, qA2fair, qA3enjoyed, qA4satisfied = end-game appreciation questions (see Supplemental Figure S7 for more details)
# dass21_depression = DASS21 depression subscale (collected in Sample 1)
# HRSD = Hamilton Rating Scale for Depression (collected in Sample 2)
# depression_percentile = percentile norms based on dass21_depression and HRSD scores in resp. Sample 1 and Sample 2 (used in pooled analysis)
# group_dep = participants' group attribution to healthy or depressed study groups in Sample 2, upon recruitment for a larger study (attribution to the clinical group is based on a baseline HRSD >=14 and does not necessary match participants' HRSD score at the time of the task)
# ffni_total and ffni_... = Five-factor Narcissism Inventory (FFNI) total scores and subscales
# bpni_total and bpni_... = Brief Pathological Narcissism Inventory (BPNI) total scores and subscales
# ipip_total = trait dominance measured by the International Personality Item Pool - Dominance Subscale (this scale was only collected in Sample 2)

## trial-level variables
########################
#trial = measures time on task (total task = 24 trials)
#scoreDiff = absolute score difference between the participant and their opponent on a given trial (predefined, pseudorandom variable)
#closerun = based on scoreDiff : 1 = tight competition (close-run; if scoreDiff =< 2), 0 = large score difference between participant and opponent (displayed to players during the snake arcade game for the last 10 seconds of the game).
#outcome = the trial's outcome. 0 = defeat, 1 = victory (predefined, pseudorandom variable comprising 2/3 of defeats)
#oppScore = the opponent's score on a given trial (computed on each trial as snake_score +/- scoreDiff based on the predefined outcome)
#oppRank = opponent's rank ranging from 200 = lowest to 1 = highest rank
#oppRank_inv = opponent's rank inverted, with 1 = lowest and 200 = highest rank (the inverted ranks are used in the analysis)
#rankStart = participant's rank at after learning the outcome of a given trial but before buying rank
#rankEnd = participant's rank after buying rank at the end of the trial
#rankEnd.lag1 = participant's rank when entering a given trial
#snake_score = participant's score on the snake arcade game (number of 'apples' eaten by the snake in 20 seconds)
#pointSteal = point stealing ranging from 1 = no point stealing to 5 = stealing ten points
#rankBuy = rank buying ranging from 1 = no rank buying to 5 = buying 5 extra ranks
#..._pmcentered = person-mean centered variables
#....lag = lagged variable (displays its value from the previous trial)


##############################################################################################################################################################################################################################################################

# libraries required:
library(readxl)
library(readr)
library(lme4)
library(ggplot2)
library(dplyr)
library(psych)
library(gdata)
library(R.matlab)
library(xtable)
library(Hmisc)
library(foreign)
library(MASS)
library("lsmeans")
library(effect)
library(arm)
library(emmeans)
library(corrplot)
library(data.table)
library(plyr)
library(car)
library(sjPlot)
library(stargazer)
library(grid)
library(gridExtra)
library(tableone)



#########################
####### DATASETS ########
#########################

#clear environment
rm(list=ls())


#wide-format
ds_pooled_wide <- read.csv("ds_pooled_wide.csv")

ds_pooled_wide <- transform(ds_pooled_wide, subject = as.factor(subject), sample = as.factor(sample), sex = as.factor(sex), race = as.factor(race),
                            stereotypical_resp = as.factor(stereotypical_resp))

ds_pooled_wide$race <- relevel(ds_pooled_wide$race, ref = 'caucas')
ds_pooled_wide$stereotypical_resp <- relevel(ds_pooled_wide$stereotypical_resp, ref = 'TRUE')
ds_pooled_wide$sex <- relevel(ds_pooled_wide$sex, ref = 'male')

ds_sample1_wide <- ds_pooled_wide[ds_pooled_wide$sample == "sample 1",]
ds_sample2_wide <- ds_pooled_wide[ds_pooled_wide$sample == "sample 2",]


#long-format
ds_pooled <- read.csv("ds_pooled.csv")

ds_pooled <- transform(ds_pooled, subject = as.factor(subject), sample = as.factor(sample), sex = as.factor(sex), race = as.factor(race),
                       outcome = as.factor(outcome), outcome.lag1 = as.factor(outcome.lag1), closerun = as.factor(closerun), closerun.lag1 = as.factor(closerun.lag1), group_dep = as.factor(group_dep))

ds_pooled$race <- relevel(ds_pooled$race, ref = 'caucas')
ds_pooled$group_dep <- relevel(ds_pooled$group_dep, ref = 'healthy')


ds_sample1 <- ds_pooled[ds_pooled$sample == "sample 1",]
ds_sample2 <- ds_pooled[ds_pooled$sample == "sample 2",]


#########################
##### MAIN ANALYSES #####
#########################

#### SELECTED MODELS WITH DESIGN VARIABLES
### POINT STEALING

#Sample1
mPoint_design_s1 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_design_s1)
car::Anova(mPoint_design_s1, type = 'III')
car::vif(mPoint_design_s1)

#Sample2
mPoint_design_s2 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_design_s2)
car::Anova(mPoint_design_s2, type = 'III')
car::vif(mPoint_design_s2)

#Pooled
mPoint_design_pooled <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_design_pooled)
car::Anova(mPoint_design_pooled, type = 'III')
car::vif(mPoint_design_pooled)


### RANK BUYING

#Sample1
mRank_design_s1 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_design_s1)
car::Anova(mRank_design_s1, type = 'III')
car::vif(mRank_design_s1)

#Sample2
mRank_design_s2 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(rankStart_inv) + scale(snake_score) + scale(rankBuy_pmcentered.lag1)  + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_design_s2)
car::Anova(mRank_design_s2, type = 'III')
car::vif(mRank_design_s2)

#Pooled
mRank_design_pooled <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_design_pooled)
car::Anova(mRank_design_pooled, type = 'III')
car::vif(mRank_design_pooled)


#### INTERACTIONS WITH PSYCHOMETRIC VARIABLES: SELECTED MODELS

### POINT STEALING
## TRAIT DOMINANCE (IPIP-DS)
#Sample2
mPoint_dom_s2 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ipip_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(ipip_total) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_dom_s2)
car::Anova(mPoint_dom_s2, type = 'III')
car::vif(mPoint_dom_s2)

## NARCISSISM (FFNI)
#Sample1
mPoint_narc_s1 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income) +  race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_narc_s1)
car::Anova(mPoint_narc_s1, type = 'III')
car::vif(mPoint_narc_s1)

#Sample2
mPoint_narc_s2 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_narc_s2)
car::Anova(mPoint_narc_s2, type = 'III')
car::vif(mPoint_narc_s2)

#Pooled
mPoint_narc_pooled <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_narc_pooled)
car::Anova(mPoint_narc_pooled, type = 'III')
car::vif(mPoint_narc_pooled)

## DEPRESSION
#Sample1
mPoint_dep_s1_noCovar <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(dass21_depression)*scale(oppRank_inv) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_dep_s1_noCovar)
car::Anova(mPoint_dep_s1_noCovar, type = 'III')
car::vif(mPoint_dep_s1_noCovar)

mPoint_dep_s1 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(dass21_depression)*scale(oppRank_inv) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income) + race + sex + scale(gameExp) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_dep_s1)
car::Anova(mPoint_dep_s1, type = 'III')
car::vif(mPoint_dep_s1)

#Sample2
mPoint_dep_s2 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(HRSD)*scale(oppRank_inv) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_dep_s2)
car::Anova(mPoint_dep_s2, type = 'III')
car::vif(mPoint_dep_s2)

#with depression as categorical variable in Sample 2 (Supplemental figure 3, Panel C)
mPoint_dep_s2cat <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + group_dep*scale(oppRank_inv) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_dep_s2cat)
car::Anova(mPoint_dep_s2cat, type = 'III')
car::vif(mPoint_dep_s2cat)

#Pooled
mPoint_dep_pooled <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(depression_percentile) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_dep_pooled)
car::Anova(mPoint_dep_pooled, type = 'III')
car::vif(mPoint_dep_pooled)


### RANK BUYING
## TRAIT DOMINANCE (IPIP-DS)
#Sample2
mRank_dom_s2 <- lmer(rankBuy_pmcentered ~ scale(ipip_total)*scale(trial) + outcome + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_dom_s2)
car::Anova(mRank_dom_s2, type = 'III')
car::vif(mRank_dom_s2)


## NARCISSISM (FFNI)
#Sample1
mRank_narc_s1 <- lmer(rankBuy_pmcentered ~ scale(ffni_total)*scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) +  race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_narc_s1)
car::Anova(mRank_narc_s1, type = 'III')
car::vif(mRank_narc_s1)

#Sample2
mRank_narc_s2 <- lmer(rankBuy_pmcentered ~ scale(ffni_total)*scale(trial) + scale(trial)*scale(oppRank_inv) + scale(rankStart_inv) + scale(snake_score) + outcome + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_narc_s2)
car::Anova(mRank_narc_s2, type = 'III')
car::vif(mRank_narc_s2)

#Pooled
mRank_narc_pooled <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_total) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_narc_pooled)
car::Anova(mRank_narc_pooled, type = 'III')
car::vif(mRank_narc_pooled)



##########################
###### MAIN FIGURES ######
##########################

## Figure 2
pSample1_design_mPoint <- plot_model(mPoint_design_s1, type = "est", show.p = TRUE, show.values = TRUE, title = "Sample 1 (N = 70)",
                                    rm.terms = c("scale(age)", "scale(dass21_depression)", "scale(gameExp)", "scale(household_income)", "raceblack", "raceother","raceasian", "sexmale"),
                                    order.terms = c(1,4,5,7,3,2,6),
                                    colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                    axis.labels = rev(c("trial (scaled)", "opponent's rank (scaled)\nhigh = positive", "previous score (scaled)", "opponent's rank*previous score", "victory on previous trial", "closerun-run on previous trial", "current rank (scaled)"))) +
                                    ylim(-.2, .2) + theme_bw()

pSample1_design_mRank <- plot_model(mRank_design_s1, type = "est", show.p = TRUE, show.values = TRUE, title = "Sample 1 (N = 70)",
                                   rm.terms = c("scale(age)", "scale(dass21_depression)", "scale(gameExp)", "scale(household_income)", "raceblack", "raceother","raceasian", "sexmale"),
                                   order.terms = c(1,2,7,3,4,5,6), colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                   axis.labels = rev(c("trial (scaled)", "opponent's rank (scaled)\nhigh = positive", "opponent's rank*trial", "victory on previous trial", "score on previous trial", "current rank (scaled)", "mean of previous rank\nbuying choices (scaled)"))) +
                                    ylim(-.2, .2) + theme_bw()

pSample2_design_mPoint <- plot_model(mPoint_design_s2, show.p = TRUE, show.values = TRUE, title = "Sample 2 (N = 85)",
                                    rm.terms = c("scale(age)", "scale(HRSD)", "scale(gameExp)", "scale(household_income_s2_log)", "raceasian", "raceblack","raceother", "sexmale", "scale(education)"),
                                    order.terms = c(1,4,5,7,3,2,6), colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                    axis.labels = rev(c(" ", " ", " ", " ", " ", " ", " "))) +
                                    ylim(-.2, .2) + theme_bw()

pSample2_design_mRank <- plot_model(mRank_design_s2, type = "est", show.p = TRUE, show.values = TRUE, title = "Sample 2 (N = 85)",
                                   rm.terms = c("scale(age)", "scale(HRSD)", "scale(gameExp)", "scale(household_income_s2_log)", "raceasian", "raceother","raceblack", "sexmale", "scale(education)"),
                                   order.terms = c(1,2,7,3,5,4,6), colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                   axis.labels = rev(c(" ", " ", " ", " ", " ", " ", " "))) +
                                    ylim(-.2, .2) + theme_bw()

pPooled_design_mPoint <- plot_model(mPoint_design_pooled, show.p = TRUE, show.values = TRUE, title = "POOLED ANALYSIS (N = 155)",
                                   rm.terms = c("samplesample 2", "scale(age)", "scale(depression_percentile)", "scale(gameExp)", "scale(household_income)", "raceasian", "raceblack","raceother", "sexmale", "scale(education)"),
                                   order.terms = c(1,4,5,7,3,2,6), colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                   axis.labels = rev(c(" ", " ", " ", " ", " ", " ", " "))) +
                                    ylim(-.2, .2) + theme_bw()

pPooled_design_mRank <- plot_model(mRank_design_pooled, type = "est", show.p = TRUE, show.values = TRUE, title = "POOLED ANALYSIS (N = 155)",
                                  rm.terms = c("samplesample 2", "scale(age)", "scale(depression_percentile)", "scale(gameExp)", "scale(household_income)", "raceasian", "raceblack","raceother", "sexmale", "scale(education)"),
                                  order.terms = c(1,2,7,3,4,5,6), colors = c("#008600"), show.legend = FALSE, value.offset = .3, value.size = 4, vline.color = 'grey80', grid.breaks = .1,
                                  axis.labels = rev(c(" ", " ", " ", " ", " ", " ", " "))) +
                                  ylim(-.2, .2) + theme_bw()

#overall figure 2
p2 <- grid.arrange(pSample2_design_mPoint,pPooled_design_mPoint,pSample2_design_mRank,pPooled_design_mRank,
             layout_matrix = matrix(c(1,2,3,4), nrow = 2, byrow=TRUE))

p1 <- grid.arrange(pSample1_design_mPoint,pSample1_design_mRank,
                   layout_matrix = matrix(c(1,2), nrow = 2, byrow=TRUE))

pfig2 <- grid.arrange(p1,p2,
                   layout_matrix = matrix(cbind(c(1), c(2,2)), nrow = 1, byrow=TRUE))


## Figure 3
domSD <- sd(ds_sample2_wide$ipip_total, na.rm = TRUE)
domMean <- mean(ds_sample2_wide$ipip_total, na.rm = TRUE)
dom1 <- round(domMean - domSD)
dom2 <- round(domMean + domSD)

ipipfig1 <- emmip(mPoint_dom_s2, ipip_total ~ trial, at = list(ipip_total = c(dom1,dom2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme_bw() +
  theme(legend.position = "right")+
  xlab("Trial") + ylab("Point stealing (mean-centered)")

ipipfig2 <- emmip(mRank_dom_s2, ipip_total ~ trial, at = list(ipip_total = c(dom1,dom2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme_bw() +
  theme(legend.position = "right") +
  xlab("Trial") + ylab("Rank buying (mean-centered)")


ffniSD <- sd(ds_pooled_wide$ffni_total, na.rm = TRUE)
ffniMean <- mean(ds_pooled_wide$ffni_total, na.rm = TRUE)
ffni1 <- round(ffniMean - ffniSD)
ffni2 <- round(ffniMean + ffniSD)

ffniPooledfig1 <- emmip(mPoint_narc_pooled, ffni_total ~ trial, at = list(ffni_total = c(ffni1,ffni2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme_bw() +
  theme(legend.position = "right")+
  xlab("Trial") + ylab("Point stealing (mean-centered)")

ffniPooledfig2 <- emmip(mRank_narc_pooled, ffni_total ~ trial, at = list(ffni_total = c(ffni1,ffni2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme_bw() +
  theme(legend.position = "right") +
  xlab("Trial") + ylab("Rank buying (mean-centered)")

# overall figure 3
grid.arrange(ipipfig1, ipipfig2, ffniPooledfig1, ffniPooledfig2, ncol=2)


## Figure 4
ipipfig3 <- emmip(mPoint_dom_s2, ipip_total ~ oppRank_inv, at = list(ipip_total = c(dom1,dom2), oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  scale_x_discrete(breaks=c("1","100","200"), labels=c("200th", "100th", "1st")) +
  theme(legend.position = "bottom")+
  xlab("Opponent's rank") + ylab("Point stealing (mean-centered)")


depSD <- sd(ds_pooled$dep, na.rm = TRUE)
depMean <- mean(ds_pooled$dep, na.rm = TRUE)
dep1 <- round(depMean - depSD)
dep2 <- round(depMean + depSD)

depPooledfig1 <- emmip(mPoint_dep_pooled, depression_percentile~ oppRank_inv, at = list(depression_percentile = c(dep1,dep2), oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  scale_x_discrete(breaks=c("1","100","200"), labels=c("200th", "100th", "1st")) +
  theme(legend.position = "bottom")+
  xlab("Opponent's rank") + ylab("Point stealing (mean-centered)")

#overall figure 4
grid.arrange(ipipfig3, depPooledfig1, ncol=2)



###########################################
###### SUPPLEMENTAL TABLES & FIGURES ######
###########################################

### DEMOGRAPHIC TABLE (TABLE S1)

## Vancouver (Sample 1)
listVars <- c("age","sex","race","household_income","gameExp","dass21_depression", "depression_percentile","ffni_total")
catVars <- c("sex", "race", "household_income")
table1_s1 <- CreateTableOne(vars = listVars, data = ds_sample1_wide, factorVars = catVars)
table1_s1

## Pittsburgh (Sample 2)
listVars <- c("age","sex","race","household_income","gameExp","HRSD","depression_percentile","ffni_total","ipip_total")
catVars <- c("sex", "race")
table1_s2 <- CreateTableOne(vars = listVars, data = ds_sample2_wide, factorVars = catVars)
table1_s2

#by study groups (depressed vs. healthy)
listVars <- c("age","sex","race","household_income","gameExp","HRSD","depression_percentile","ffni_total","ipip_total")
catVars <- c("sex", "race")
table1_s2_group <- CreateTableOne(vars = listVars, data = ds_sample2_wide, factorVars = catVars, strata = "group_dep")
table1_s2_group


### HISTOGRAMS (FIGURE S1)
h1 <- ggplot(ds_sample1_wide, aes(x = ffni_total)) + geom_histogram(binwidth = 10) + geom_rug() + theme_bw() + scale_y_continuous(breaks=c(1,2,3,4,5,6,7,8,9,10))
h2 <- ggplot(ds_sample2_wide, aes(x = ffni_total)) + geom_histogram(binwidth = 10) + geom_rug() + theme_bw() + scale_y_continuous(breaks=c(1,2,3,4,5,6,7,8,9,10))

#overall Supplemental Figure S1
grid.arrange(h1,h2, ncol=2)


### CORRELATION FIGURES (FIGURE S2)

#Vancouver (Sample 1)
chars1 <- ds_sample1[,c('ffni_total','dass21_depression','trial','oppRank_inv','outcome.lag1','scoreDiff','rankEnd_inv.lag1', 'pointSteal', 'rankBuy', 'snake_score')]
chars1$outcome.lag1 <- as.numeric(chars1$outcome.lag1)
res1 <- cor.mtest(chars1, conf.level = .95, method = "spearman")
corrplot(cor(chars1, method = "spearman", use = "na.or.complete"), method = "color", type = "full", addCoef.col = "black", tl.col = "black", number.cex = 0.9, tl.pos = 'lt', p.mat = res1$p, insig = "pch", diag = TRUE)

#Pittsburgh (sample 2)
chars2 <- ds_sample2[,c('ffni_total','ipip_total','HRSD','trial','oppRank_inv','outcome.lag1','scoreDiff','rankEnd_inv.lag1', 'pointSteal', 'rankBuy', 'snake_score')]
chars2$outcome.lag1 <- as.numeric(chars2$outcome.lag1)
res2 <- cor.mtest(chars2, conf.level = .95, method = "spearman")
corrplot(cor(chars2, method = "spearman", use = "na.or.complete"), method = "color", type = "full", addCoef.col = "black", tl.col = "black", number.cex = .8, tl.pos = 'lt', p.mat = res2$p, insig = "pch", diag = TRUE)

### SUPPLEMENTAL SAMPLEWISE PLOTS (FIGURE S3 & FIGURE S4)

ffniSD_s1 <- sd(ds_sample1_wide$ffni_total, na.rm = TRUE)
ffniMean_s1 <- mean(ds_sample1_wide$ffni_total, na.rm = TRUE)
ffni1_s1 <- round(ffniMean_s1 - ffniSD_s1)
ffni2_s1 <- round(ffniMean_s1 + ffniSD_s1)

ffniSample1fig1 <- emmip(mRank_narc_s1, ffni_total ~ trial, at = list(ffni_total = c(ffni1_s1,ffni2_s1), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  theme_bw() +
  theme(legend.position = "bottom")+
  xlab("Trial") + ylab("Point stealing (mean-centered)") +
  ggtitle("Sample1")
  
#supplemental figure S3, panel A
ffniSD_s2 <- sd(ds_sample2_wide$ffni_total, na.rm = TRUE)
ffniMean_s2 <- mean(ds_sample2_wide$ffni_total, na.rm = TRUE)
ffni1_s2 <- round(ffniMean_s2 - ffniSD_s2)
ffni2_s2 <- round(ffniMean_s2 + ffniSD_s2)

ffniSample2fig1 <- emmip(mPoint_narc_s2, ffni_total ~ trial, at = list(ffni_total = c(ffni1_s2,ffni2_s2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  theme_bw() +
  theme(legend.position = "bottom")+
  xlab("Trial") + ylab("Point stealing (mean-centered)") +
  ggtitle("Sample2")

ffniSample2fig2 <- emmip(mRank_narc_s2, ffni_total ~ trial, at = list(ffni_total = c(ffni1_s2,ffni2_s2), trial = c(1,12,24)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  theme_bw() +
  theme(legend.position = "bottom") +
  xlab("Trial") + ylab("Rank buying (mean-centered)") +
  ggtitle("    ")

grid.arrange(ffniSample1fig1, ffniSample2fig1, ffniSample2fig2, ncol=3)

#supplemental figure S3, panel B
depSD_s1 <- sd(ds_sample1_wide$dass21_depression, na.rm = TRUE)
depMean_s1 <- mean(ds_sample1_wide$dass21_depression, na.rm = TRUE)
dep1_s1 <- round(depMean_s1 - depSD_s1)
dep2_s1 <- round(depMean_s1 + depSD_s1)

depSample1fig <- emmip(mPoint_dep_s1, dass21_depression ~ oppRank_inv, at = list(dass21_depression = c(dep1_s1,dep2_s1), oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  scale_x_discrete(breaks=c("1","100","200"), labels=c("200th", "100th", "1st")) +
  theme(legend.position = "bottom")+
  xlab("Opponent's rank") + ylab("Point stealing (mean-centered)")


depSD_s2 <- sd(ds_sample2_wide$HRSD, na.rm = TRUE)
depMean_s2 <- mean(ds_sample2_wide$HRSD, na.rm = TRUE)
dep1_s2 <- round(depMean_s2 - depSD_s2)
dep2_s2 <- round(depMean_s2 + depSD_s2)

depSample2fig <- emmip(mPoint_dep_s2, HRSD ~ oppRank_inv, at = list(HRSD = c(dep1_s2,dep2_s2), oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  scale_x_discrete(breaks=c("1","100","200"), labels=c("200th", "100th", "1st")) +
  theme(legend.position = "bottom")+
  xlab("Opponent's rank") + ylab("Point stealing (mean-centered)")

grid.arrange(depSample1fig, depSample2fig, ncol=2)

#supplemental figure S3, panel C
depSample2fig_cat <- emmip(mPoint_dep_s2cat, group_dep ~ oppRank_inv, at = list(oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#c55a11","#008600"), style = "factor") +
  theme_grey() +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  scale_x_discrete(breaks=c("1","100","200"), labels=c("200th", "100th", "1st")) +
  theme(legend.position = "bottom")+
  xlab("Opponent's rank") + ylab("Point stealing (mean-centered)")

#supplemental figure S4
narcSample1_rank_fig1 <- emmip(mRank_narc_s1, ffni_total ~ oppRank_inv, at = list(ffni_total = c(ffni1_s1, ffni2_s1), oppRank_inv = c(1,200)), CIs = TRUE, col = c("#008600", "#c55a11"), style = "factor") +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) +
  scale_x_discrete(breaks=c("1","200"), labels=c("200th", "1st"))+
  theme_bw() +
  xlab("Opponent's rank") + ylab("Rank buying (mean-centered)") +
  theme(legend.position = "right")

oppRank_names <- c('1' = "Opponent's rank = 200th", '200' = "Opponent's rank = 1st")

narcSample1_rank_fig2 <- emmip(mRank_narc_s1, ffni_total*oppRank_inv ~ trial, at = list(ffni_total = c(ffni1_s1, ffni2_s1), trial = c(1,12,24), oppRank_inv = c(1,200)), CIs = TRUE, col = c("#008600", "#c55a11","#008600", "#c55a11"), style = "factor") +
  scale_fill_manual(values = c("#008600", "#c55a11","#008600", "#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11", "#008600", "#c55a11")) +
  xlab("Trial") + ylab("Rank buying (mean-centered)") +
  facet_wrap(~ oppRank_inv, labeller = as_labeller(oppRank_names)) +
  theme_bw() +
  theme(legend.position = "bottom")

grid.arrange(narcSample1_rank_fig1, narcSample1_rank_fig2, ncol=2)


### RELIABILITY COEFFICIENTS (TABLE S2)

## Sample 1

ffni_df_sample1 <- read.csv("ds_items_s1_ffni.csv")
ffni_df_S1 <- ffni_df_sample1[,c(1:18,61,20:26,63,28:37,62,39:60)]
names(ffni_df_S1)
psych::alpha(ffni_df_S1, use = "pairwise.complete.obs")

bpni_df_sample1 <- read.csv("ds_items_s1_bpni.csv")
psych::alpha(bpni_df_sample1, use = "pairwise.complete.obs")

dass21_df_sample1 <- read.csv("ds_items_s1_dass21.csv")
dass21_df_dep <- dass21_df_sample1[,c(3,5,10,13,16,17,21)]*2 #scores were multiplied by 2 in the original scoring
names(dass21_df_dep)
psych::alpha(dass21_df_dep, use = "pairwise.complete.obs")

#narcissistic subscales
ffniG_df_sample1 <- ffni_df_sample1[,c("ffni_9", "ffni_24", "ffni_39", "ffni_54", "ffni_6", "ffni_21", "ffni_36", "ffni_51", "ffni_3", "ffni_18", "ffni_33", "ffni_48", "ffni_8", "ffni_23", "ffni_38r", "ffni_53", 
                                       "ffni_11", "ffni_26", "ffni_41", "ffni_56", "ffni_7", "ffni_22", "ffni_37", "ffni_52", "ffni_5", "ffni_20", "ffni_35", "ffni_50", "ffni_10", "ffni_25", "ffni_40", "ffni_55", 
                                       "ffni_2", "ffni_17", "ffni_32", "ffni_47", "ffni_1", "ffni_16", "ffni_31", "ffni_46", "ffni_15", "ffni_30", "ffni_45", "ffni_60")]
ffniV_df_sample1 <- ffni_df_sample1[,c("ffni_13", "ffni_28", "ffni_43", "ffni_58", "ffni_14", "ffni_29", "ffni_44", "ffni_59", "ffni_12", "ffni_27r", "ffni_42", "ffni_57", "ffni_4", "ffni_19r", "ffni_34", "ffni_49")]
ffniE_df_sample1 <- ffni_df_sample1[,c("ffni_1", "ffni_16", "ffni_31", "ffni_46", "ffni_3", "ffni_18", "ffni_33", "ffni_48", "ffni_8", "ffni_23", "ffni_38r", "ffni_53","ffni_6", "ffni_21", "ffni_36", "ffni_51")]
ffniA_df_sample1 <- ffni_df_sample1[,c("ffni_11", "ffni_26", "ffni_41", "ffni_56", "ffni_7", "ffni_22", "ffni_37", "ffni_52", "ffni_5", "ffni_20", "ffni_35", "ffni_50",
                                       "ffni_10", "ffni_25", "ffni_40", "ffni_55", "ffni_2", "ffni_17", "ffni_32", "ffni_47", "ffni_13", "ffni_28", "ffni_43", "ffni_58",
                                       "ffni_4", "ffni_19r", "ffni_34", "ffni_49", "ffni_15", "ffni_30", "ffni_45", "ffni_60")]
ffniN_df_sample1 <- ffni_df_sample1[,c("ffni_14", "ffni_29", "ffni_44", "ffni_59", "ffni_9", "ffni_24", "ffni_39", "ffni_54", "ffni_12", "ffni_27r", "ffni_42", "ffni_57")]

psych::alpha(ffniG_df_sample1, use = "pairwise.complete.obs")
psych::alpha(ffniV_df_sample1, use = "pairwise.complete.obs")
psych::alpha(ffniE_df_sample1, use = "pairwise.complete.obs")
psych::alpha(ffniA_df_sample1, use = "pairwise.complete.obs")
psych::alpha(ffniN_df_sample1, use = "pairwise.complete.obs", keys = c(1,1,1,1,-1,-1,-1,-1,1,1,1,1))

bpniG_df_sample1 <- bpni_df_sample1[,c("bpni_1", "bpni_4", "bpni_6", "bpni_11", "bpni_10", "bpni_12", "bpni_19", "bpni_24", "bpni_13", "bpni_17", "bpni_25", "bpni_26")]
bpniV_df_sample1 <- bpni_df_sample1[,c("bpni_2", "bpni_16", "bpni_18", "bpni_21", "bpni_3", "bpni_15", "bpni_27", "bpni_28", "bpni_7", "bpni_9", "bpni_14", "bpni_20", "bpni_5", "bpni_8", "bpni_22", "bpni_23")]

psych::alpha(bpniG_df_sample1, use = "pairwise.complete.obs")
psych::alpha(bpniV_df_sample1, use = "pairwise.complete.obs")


## Sample 2

ds_items_s2 <- read.csv("ds_items_s2.csv")

ffni_df <- ds_items_s2[,c(2:19,101,21:27,102,29:38,103,40:61)]
names(ffni_df)
psych::alpha(ffni_df, use = "pairwise.complete.obs")

bpni_df <- ds_items_s2[,c(73:100)]
names(bpni_df)
psych::alpha(bpni_df, use = "pairwise.complete.obs")

ipip_df <- ds_items_s2[,c(62,104,64:72)]
names(ipip_df)
psych::alpha(ipip_df, use = "pairwise.complete.obs")

hrsd_df <- ds_items_s2[,c(105:121)]
names(hrsd_df)
psych::alpha(hrsd_df, use = "pairwise.complete.obs")


#subscales:
ffniG_df <- ds_items_s2[,c("ffni_9", "ffni_24", "ffni_39", "ffni_54", "ffni_6", "ffni_21", "ffni_36", "ffni_51", "ffni_3", "ffni_18", "ffni_33", "ffni_48", "ffni_8", "ffni_23", "ffni_38r", "ffni_53", 
                           "ffni_11", "ffni_26", "ffni_41", "ffni_56", "ffni_7", "ffni_22", "ffni_37", "ffni_52", "ffni_5", "ffni_20", "ffni_35", "ffni_50", "ffni_10", "ffni_25", "ffni_40", "ffni_55", 
                           "ffni_2", "ffni_17", "ffni_32", "ffni_47", "ffni_1", "ffni_16", "ffni_31", "ffni_46", "ffni_15", "ffni_30", "ffni_45", "ffni_60")]
ffniV_df <- ds_items_s2[,c("ffni_13", "ffni_28", "ffni_43", "ffni_58", "ffni_14", "ffni_29", "ffni_44", "ffni_59", "ffni_12", "ffni_27r", "ffni_42", "ffni_57", "ffni_4", "ffni_19r", "ffni_34", "ffni_49")]
ffniE_df <- ds_items_s2[,c("ffni_1", "ffni_16", "ffni_31", "ffni_46", "ffni_3", "ffni_18", "ffni_33", "ffni_48", "ffni_8", "ffni_23", "ffni_38r", "ffni_53","ffni_6", "ffni_21", "ffni_36", "ffni_51")]
ffniA_df <- ds_items_s2[,c("ffni_11", "ffni_26", "ffni_41", "ffni_56", "ffni_7", "ffni_22", "ffni_37", "ffni_52", "ffni_5", "ffni_20", "ffni_35", "ffni_50",
                           "ffni_10", "ffni_25", "ffni_40", "ffni_55", "ffni_2", "ffni_17", "ffni_32", "ffni_47", "ffni_13", "ffni_28", "ffni_43", "ffni_58",
                           "ffni_4", "ffni_19r", "ffni_34", "ffni_49", "ffni_15", "ffni_30", "ffni_45", "ffni_60")]
ffniN_df <- ds_items_s2[,c("ffni_14", "ffni_29", "ffni_44", "ffni_59", "ffni_9", "ffni_24", "ffni_39", "ffni_54", "ffni_12", "ffni_27r", "ffni_42", "ffni_57")]

psych::alpha(ffniG_df, use = "pairwise.complete.obs")
psych::alpha(ffniV_df, use = "pairwise.complete.obs")
psych::alpha(ffniE_df, use = "pairwise.complete.obs")
psych::alpha(ffniA_df, use = "pairwise.complete.obs")
psych::alpha(ffniN_df, use = "pairwise.complete.obs", keys = c(1,1,1,1,-1,-1,-1,-1,1,1,1,1))

bpniG_df <- ds_items_s2[,c("bpni_1", "bpni_4", "bpni_6", "bpni_11", "bpni_10", "bpni_12", "bpni_19", "bpni_24", "bpni_13", "bpni_17", "bpni_25", "bpni_26")]
bpniV_df <- ds_items_s2[,c("bpni_2", "bpni_16", "bpni_18", "bpni_21", "bpni_3", "bpni_15", "bpni_27", "bpni_28", "bpni_7", "bpni_9", "bpni_14", "bpni_20", "bpni_5", "bpni_8", "bpni_22", "bpni_23")]

psych::alpha(bpniG_df, use = "pairwise.complete.obs")
psych::alpha(bpniV_df, use = "pairwise.complete.obs")


##################################
###### SENSITIVITY ANALYSES ######
##################################

#########################################################################
## sensitivity a) stereotypical responding

#demographics
listVars_s1 <- c("age", "sex", "race", "household_income", "gameExp", "dass21_depression", "ffni_total")
listVars_s2 <- c("age", "sex", "race", "household_income_s2", "gameExp", "education", "HRSD", "ffni_total", "ipip_total")

catVars_s1 <- c("sex", "race", "household_income")
catVars_s2 <- c("sex", "race")

table1_stereo_s1 <- CreateTableOne(vars = listVars_s1, data = ds_sample1_wide, strata = c("stereotypical_resp"), factorVars = catVars_s1)
table1_stereo_s1

table1_stereo_s2 <- CreateTableOne(vars = listVars_s2, data = ds_sample2_wide, strata = c("stereotypical_resp"))
table1_stereo_s2

#main findings tested without sterotypical responders
ds_sample1_no_stereo <- ds_sample1[ds_sample1$stereotypical_resp == 'FALSE',]
ds_sample2_no_stereo <- ds_sample2[ds_sample2$stereotypical_resp == 'FALSE',]
ds_pooled_no_stereo <- ds_pooled[ds_pooled$stereotypical_resp == 'FALSE',]

## Sample 1
mPoint_s1_no_stereo_design <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mPoint_s1_no_stereo_design)
car::Anova(mPoint_s1_no_stereo_design, type = 'III')

mRank_s1_no_stereo_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mRank_s1_no_stereo_design)
car::Anova(mRank_s1_no_stereo_design, type = 'III')

mPoint_s1_no_stereo_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mPoint_s1_no_stereo_narc)
car::Anova(mPoint_s1_no_stereo_narc, type = 'III')

mRank2_s1_no_stereo_narc <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*scale(ffni_total) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mRank2_s1_no_stereo_narc)
car::Anova(mRank2_s1_no_stereo_narc, type = 'III')

mPoint_s1_no_stereo_dep_noCovar <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(dass21_depression) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mPoint_s1_no_stereo_dep_noCovar)
car::Anova(mPoint_s1_no_stereo_dep_noCovar, type = 'III')

mPoint_s1_no_stereo_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(dass21_depression) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + sex + scale(gameExp) + (1|subject),  data = ds_sample1_no_stereo, na.action = na.omit)
summary(mPoint_s1_no_stereo_dep)
car::Anova(mPoint_s1_no_stereo_dep, type = 'III')

## Sample 2
mPoint_s2_no_stereo_design <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mPoint_s2_no_stereo_design)
car::Anova(mPoint_s2_no_stereo_design, type = 'III')

mRank_s2_no_stereo_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(rankStart_inv) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mRank_s2_no_stereo_design)
car::Anova(mRank_s2_no_stereo_design, type = 'III')

mPoint_s2_no_stereo_dom <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ipip_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(ipip_total) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mPoint_s2_no_stereo_dom)
car::Anova(mPoint_s2_no_stereo_dom, type = 'III')

mRank_s2_no_stereo_dom <- lmer(rankBuy_pmcentered ~ scale(ipip_total)*scale(trial) + outcome + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mRank_s2_no_stereo_dom)
car::Anova(mRank_s2_no_stereo_dom, type = 'III')

mPoint_s2_no_stereo_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mPoint_s2_no_stereo_narc)
car::Anova(mPoint_s2_no_stereo_narc, type = 'III')

mRank_s2_no_stereo_narc <- lmer(rankBuy_pmcentered ~ scale(ffni_total)*scale(trial) + outcome + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + sex + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mRank_s2_no_stereo_narc)
car::Anova(mRank_s2_no_stereo_narc, type = 'III')

mPoint_s2_no_stereo_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(HRSD) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + (1|subject),  data = ds_sample2_no_stereo, na.action = na.omit)
summary(mPoint_s2_no_stereo_dep)
car::Anova(mPoint_s2_no_stereo_dep, type = 'III')

##pooled
mPoint_pooled_no_stereo_design <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled_no_stereo, na.action = na.omit)
summary(mPoint_pooled_no_stereo_design)
car::Anova(mPoint_pooled_no_stereo_design, type = 'III')

mRank_pooled_no_stereo_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled_no_stereo, na.action = na.omit)
summary(mRank_pooled_no_stereo_design)
car::Anova(mRank_pooled_no_stereo_design, type = 'III')

mPoint_pooled_no_stereo_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled_no_stereo, na.action = na.omit)
summary(mPoint_pooled_no_stereo_narc)
car::Anova(mPoint_pooled_no_stereo_narc, type = 'III')

mRank_pooled_no_stereo_narc <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + scale(trial)*scale(ffni_total) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled_no_stereo, na.action = na.omit)
summary(mRank_pooled_no_stereo_narc)
car::Anova(mRank_pooled_no_stereo_narc, type = 'III')

mPoint_pooled_no_stereo_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(depression_percentile) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + (1|subject),  data = ds_pooled_no_stereo, na.action = na.omit)
summary(mPoint_pooled_no_stereo_dep)
car::Anova(mPoint_pooled_no_stereo_dep, type = 'III')


##########################################################
#### sensitivity b) main findings tested for confounding and moderation by sex

### sex*outcome, sex*trial and sex*oppRank_inv added to main models as covariates

## Sample 1
mPoint_s1_sex_design <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sex_design)
car::Anova(mPoint_s1_sex_design, type = 'III')
car::vif(mPoint_s1_sex_design)

mRank_s1_sex_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + scale(trial)*sex + outcome*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sex_design)
car::Anova(mRank_s1_sex_design, type = 'III')
car::vif(mRank_s1_sex_design)

mPoint_s1_sex_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sex_narc)
car::Anova(mPoint_s1_sex_narc, type = 'III')
car::vif(mPoint_s1_sex_narc)

mRank_s1_sex_narc <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*scale(ffni_total) + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + scale(trial)*sex + outcome*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sex_narc)
car::Anova(mRank_s1_sex_narc, type = 'III')
car::vif(mRank_s1_sex_narc)

mPoint_s1_sex_dep_noCovar <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(dass21_depression) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sex_dep_noCovar)
car::Anova(mPoint_s1_sex_dep_noCovar, type = 'III')

mPoint_s1_sex_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(dass21_depression) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sex_dep)
car::Anova(mPoint_s1_sex_dep, type = 'III')
car::vif(mPoint_s1_sex_dep)

## Sample 2
mPoint_s2_sex_design <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sex_design)
car::Anova(mPoint_s2_sex_design, type = 'III')
car::vif(mPoint_s2_sex_design)

mRank_s2_sex_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + scale(rankStart_inv) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + scale(trial)*sex + outcome*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sex_design)
car::Anova(mRank_s2_sex_design, type = 'III')
car::vif(mRank_s2_sex_design)

mPoint_s2_sex_dom <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ipip_total) + closerun.lag1 + scale(oppRank_inv)*scale(ipip_total) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sex_dom)
car::Anova(mPoint_s2_sex_dom, type = 'III')
car::vif(mPoint_s2_sex_dom)

mRank_s2_sex_dom <- lmer(rankBuy_pmcentered ~ scale(ipip_total)*scale(trial) + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + scale(trial)*sex + outcome*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sex_dom)
car::Anova(mRank_s2_sex_dom, type = 'III')
car::vif(mRank_s2_sex_dom)

mPoint_s2_sex_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + closerun.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sex_narc)
car::Anova(mPoint_s2_sex_narc, type = 'III')
car::vif(mPoint_s2_sex_narc)

mRank_s2_sex_narc <- lmer(rankBuy_pmcentered ~ scale(ffni_total)*scale(trial) + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + scale(trial)*sex + outcome*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sex_narc)
car::Anova(mRank_s2_sex_narc, type = 'III')
car::vif(mRank_s2_sex_narc)

mPoint_s2_sex_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(HRSD) + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(trial)*sex + outcome.lag1*sex + scale(oppRank_inv)*sex + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sex_dep)
car::Anova(mPoint_s2_sex_dep, type = 'III')
car::vif(mPoint_s2_sex_dep)

## pooled
mPoint_pooled_sex_design <- lmer(pointSteal_pmcentered ~ scale(trial)*sex + scale(oppRank_inv)*sex + outcome.lag1*sex + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sex_design)
car::Anova(mPoint_pooled_sex_design, type = 'III')
car::vif(mPoint_pooled_sex_design)

mRank_pooled_sex_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv) + scale(trial)*sex + scale(oppRank_inv)*sex + outcome*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sex_design)
car::Anova(mRank_pooled_sex_design, type = 'III')
car::vif(mRank_pooled_sex_design)

mPoint_pooled_sex_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total) + scale(trial)*sex + scale(oppRank_inv)*sex + outcome.lag1*sex + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(oppRank_inv) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sex_narc)
car::Anova(mPoint_pooled_sex_narc, type = 'III')
car::vif(mPoint_pooled_sex_narc)

mRank_pooled_sex_narc <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_total) + scale(trial)*sex + scale(oppRank_inv)*sex + scale(trial)*scale(oppRank_inv) + outcome*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sex_narc)
car::Anova(mRank_pooled_sex_narc, type = 'III')
car::vif(mRank_pooled_sex_narc)

mPoint_pooled_sex_dep <- lmer(pointSteal_pmcentered ~ scale(oppRank_inv)*scale(depression_percentile) + scale(trial)*sex + scale(oppRank_inv)*sex + outcome.lag1*sex  + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sex_dep)
car::Anova(mPoint_pooled_sex_dep, type = 'III')
car::vif(mPoint_pooled_sex_dep)


### each main finding tested for interactions with sex

## Sample 1
mPoint_s1_sexInt_design1 <- lmer(pointSteal_pmcentered ~ scale(trial)*sex + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sexInt_design1)
car::Anova(mPoint_s1_sexInt_design1, type = 'III')
car::vif(mPoint_s1_sexInt_design1)

mPoint_s1_sexInt_design2 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1)*sex + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sexInt_design2)
car::Anova(mPoint_s1_sexInt_design2, type = 'III')
car::vif(mPoint_s1_sexInt_design2)

mPoint_s1_sexInt_design3 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(oppRank_inv)*sex + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sexInt_design3)
car::Anova(mPoint_s1_sexInt_design3, type = 'III')
car::vif(mPoint_s1_sexInt_design3)

mRank_s1_sexInt_design1 <- lmer(rankBuy_pmcentered ~ scale(trial)*sex + scale(trial)*scale(oppRank_inv) + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sexInt_design1)
car::Anova(mRank_s1_sexInt_design1, type = 'III')
car::vif(mRank_s1_sexInt_design1)

mRank_s1_sexInt_design2 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sexInt_design2)
car::Anova(mRank_s1_sexInt_design2, type = 'III')
car::vif(mRank_s1_sexInt_design2)

mPoint_s1_sexInt_narc1 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total)*sex + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_sexInt_narc1)
car::Anova(mPoint_s1_sexInt_narc1, type = 'III')
car::vif(mPoint_s1_sexInt_narc1)

mRank_s1_sexInt_narc1 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*scale(ffni_total) + scale(trial)*scale(ffni_total)*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sexInt_narc1)
car::Anova(mRank_s1_sexInt_narc1, type = 'III')
car::vif(mRank_s1_sexInt_narc1)

mRank_s1_sexInt_dep1 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(dass21_depression)*sex + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_sexInt_dep1)
car::Anova(mRank_s1_sexInt_dep1, type = 'III')
car::vif(mRank_s1_sexInt_dep1)


## Sample 2
mPoint_s2_sexInt_design1 <- lmer(pointSteal_pmcentered ~ scale(trial)*sex + closerun.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_design1)
car::Anova(mPoint_s2_sexInt_design1, type = 'III')
car::vif(mPoint_s2_sexInt_design1)

mPoint_s2_sexInt_design2 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 +  scale(oppRank_inv)*sex + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_design2)
car::Anova(mPoint_s2_sexInt_design2, type = 'III')
car::vif(mPoint_s2_sexInt_design2)

mPoint_s2_sexInt_design3 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1)*sex + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_design3)
car::Anova(mPoint_s2_sexInt_design3, type = 'III')
car::vif(mPoint_s2_sexInt_design3)

mRank_s2_sexInt_design1 <- lmer(rankBuy_pmcentered ~ scale(trial)*sex + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sexInt_design1)
car::Anova(mRank_s2_sexInt_design1, type = 'III')
car::vif(mRank_s2_sexInt_design1)

mRank_s2_sexInt_design2 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*sex + scale(rankStart_inv) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sexInt_design2)
car::Anova(mRank_s2_sexInt_design2, type = 'III')
car::vif(mRank_s2_sexInt_design2)

mPoint_s2_sexInt_dom1 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ipip_total)*sex + closerun.lag1 + scale(oppRank_inv)*scale(ipip_total) +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_dom1)
car::Anova(mPoint_s2_sexInt_dom1, type = 'III')
car::vif(mPoint_s2_sexInt_dom1)

mPoint_s2_sexInt_dom2 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ipip_total) + closerun.lag1 + scale(oppRank_inv)*scale(ipip_total)*sex +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_dom2)
car::Anova(mPoint_s2_sexInt_dom2, type = 'III')
car::vif(mPoint_s2_sexInt_dom2)

mRank_s2_sexInt_dom1 <- lmer(rankBuy_pmcentered ~ scale(ipip_total)*scale(trial)*sex + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sexInt_dom1)
car::Anova(mRank_s2_sexInt_dom1, type = 'III')
car::vif(mRank_s2_sexInt_dom1)

mPoint_s2_sexInt_narc1 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total)*sex + closerun.lag1 +  scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_narc1)
car::Anova(mPoint_s2_sexInt_narc1, type = 'III')
car::vif(mPoint_s2_sexInt_narc1)

mRank_s2_sexInt_narc1 <- lmer(rankBuy_pmcentered ~ scale(ffni_total)*scale(trial)*sex + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_sexInt_narc1)
car::Anova(mRank_s2_sexInt_narc1, type = 'III')
car::vif(mRank_s2_sexInt_narc1)

mPoint_s2_sexInt_dep1 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(HRSD)*sex + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(education) + race + scale(gameExp) + scale(household_income_s2_log) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_sexInt_dep1)
car::Anova(mPoint_s2_sexInt_dep1, type = 'III')
car::vif(mPoint_s2_sexInt_dep1)

## pooled
mPoint_pooled_sexInt_design1 <- lmer(pointSteal_pmcentered ~ scale(trial)*sex + outcome.lag1 + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sexInt_design1)
car::Anova(mPoint_pooled_sexInt_design1, type = 'III')
car::vif(mPoint_pooled_sexInt_design1)

mPoint_pooled_sexInt_design2 <- lmer(pointSteal_pmcentered ~ scale(trial) + outcome.lag1 + closerun.lag1 + scale(oppRank_inv)*sex + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sexInt_design2)
car::Anova(mPoint_pooled_sexInt_design2, type = 'III')
car::vif(mPoint_pooled_sexInt_design2)

mPoint_pooled_sexInt_design3 <- lmer(pointSteal_pmcentered ~ scale(trial) + outcome.lag1 + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1)*sex + scale(rankEnd_inv.lag1)  + sample +  scale(age) + scale(household_income) + race + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sexInt_design3)
car::Anova(mPoint_pooled_sexInt_design3, type = 'III')
car::vif(mPoint_pooled_sexInt_design3)

mRank_pooled_sexInt_design1 <- lmer(rankBuy_pmcentered ~ scale(trial)*sex + scale(trial)*scale(oppRank_inv) + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sexInt_design1)
car::Anova(mRank_pooled_sexInt_design1, type = 'III')
car::vif(mRank_pooled_sexInt_design1)

mRank_pooled_sexInt_design2 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sexInt_design2)
car::Anova(mRank_pooled_sexInt_design2, type = 'III')
car::vif(mRank_pooled_sexInt_design2)

mPoint_pooled_sexInt_narc1 <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total)*sex + closerun.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_sexInt_narc1)
car::Anova(mPoint_pooled_sexInt_narc1, type = 'III')
car::vif(mPoint_pooled_sexInt_narc1)

mRank_pooled_sexInt_narc1 <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*scale(ffni_total) + scale(trial)*scale(ffni_total)*sex + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sexInt_narc1)
car::Anova(mRank_pooled_sexInt_narc1, type = 'III')
car::vif(mRank_pooled_sexInt_narc1)

mRank_pooled_sexInt_dep1 <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + scale(oppRank_inv)*scale(depression_percentile)*sex + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1)  + scale(age) + scale(household_income) + race + scale(gameExp) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_sexInt_dep1)
car::Anova(mRank_pooled_sexInt_dep1, type = 'III')
car::vif(mRank_pooled_sexInt_dep1)


#########################################################################
## sensitivity c) interactions with sample in the pooled analysis

# design variables
mPoint_pooled_samplewiseDiff_design <- lmer(pointSteal_pmcentered ~ scale(trial)*sample + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1)*sample + scale(oppRank_inv)*sample + scale(rankEnd_inv.lag1) +  scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_samplewiseDiff_design)
car::Anova(mPoint_pooled_samplewiseDiff_design, type = 'III')
car::vif(mPoint_pooled_samplewiseDiff_design)

emmip(mPoint_pooled_samplewiseDiff_design, sample ~ oppRank_inv, at = list(oppRank_inv = c(1,100,200)), CIs = TRUE, col = c("#008600", "#c55a11")) +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) + theme_bw()

mRank_pooled_samplewiseDiff_design <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(oppRank_inv)*sample + scale(trial)*sample + scale(oppRank_inv)*sample + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_samplewiseDiff_design)
car::Anova(mRank_pooled_samplewiseDiff_design, type = 'III')
car::vif(mRank_pooled_samplewiseDiff_design)

emmip(mRank_pooled_samplewiseDiff_design, sample ~ trial, at = list(trial = c(1,12,24)), CIs = TRUE, col = c("#008600", "#c55a11")) +
  scale_fill_manual(values = c("#008600","#c55a11")) +
  scale_color_manual(values = c("#008600","#c55a11")) + theme_bw()

# narcissism
mPoint_pooled_samplewiseDiff_narc <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_total)*sample + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_samplewiseDiff_narc)
car::Anova(mPoint_pooled_samplewiseDiff_narc, type = 'III')
car::vif(mPoint_pooled_samplewiseDiff_narc)

mRank_pooled_samplewiseDiff_narc <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_total)*sample + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_samplewiseDiff_narc)
car::Anova(mRank_pooled_samplewiseDiff_narc, type = 'III')
car::vif(mRank_pooled_samplewiseDiff_narc)

# depression
mPoint_pooled_samplewiseDiff_dep <- lmer(pointSteal_pmcentered ~ scale(trial) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(depression_percentile)*sample + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_samplewiseDiff_dep)
car::Anova(mPoint_pooled_samplewiseDiff_dep, type = 'III')
car::vif(mPoint_pooled_samplewiseDiff_dep)


#########################################################################
## sensitivity d) main findings found with FFNI narcissism tested with the BPNI (total scores)

## Sample 1
mPoint_s1_narc_bpni <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(bpni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income) + race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mPoint_s1_narc_bpni)
car::Anova(mPoint_s1_narc_bpni, type = 'III')
car::vif(mPoint_s1_narc_bpni)

mRank_s1_narc_bpni <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(bpni_total) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income) +  race + sex + scale(gameExp) + scale(dass21_depression) + (1|subject),  data = ds_sample1, na.action = na.omit)
summary(mRank_s1_narc_bpni)
car::Anova(mRank_s1_narc_bpni, type = 'III')
car::vif(mRank_s1_narc_bpni)

## Sample 2
mPoint_s2_narc_bpni <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(bpni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mPoint_s2_narc_bpni)
car::Anova(mPoint_s2_narc_bpni, type = 'III')
car::vif(mPoint_s2_narc_bpni)

mRank_s2_narc_bpni <- lmer(rankBuy_pmcentered ~ scale(bpni_total)*scale(trial) + scale(rankStart_inv) + scale(oppRank_inv)*scale(trial) + scale(snake_score) + outcome + scale(rankBuy_pmcentered.lag1) + scale(age) + scale(household_income_s2_log) + sex + scale(education) + race + scale(gameExp) + scale(HRSD) + (1|subject),  data = ds_sample2, na.action = na.omit)
summary(mRank_s2_narc_bpni)
car::Anova(mRank_s2_narc_bpni, type = 'III')
car::vif(mRank_s2_narc_bpni)

## Pooled
mPoint_pooled_narc_bpni <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(bpni_total) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_bpni)
car::Anova(mPoint_pooled_narc_bpni, type = 'III')
car::vif(mPoint_pooled_narc_bpni)

mRank_pooled_narc_bpni <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(bpni_total) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_bpni)
car::Anova(mRank_pooled_narc_bpni, type = 'III')
car::vif(mRank_pooled_narc_bpni)


##################################
###### EXPLORATORY ANALYSES ######
##################################

#########################################################################
## exploratory a) main findings for FFNI narcissism tested with FFNI and BPNI subscales in the pooled analysis (Table 2)

# FFNI subscales
#point stealing
mPoint_pooled_narc_ffniG <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_grandiosity) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_ffniG)
car::Anova(mPoint_pooled_narc_ffniG, type = 'III')
car::vif(mPoint_pooled_narc_ffniG)

mPoint_pooled_narc_ffniV <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_vulnerability) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_ffniV)
car::Anova(mPoint_pooled_narc_ffniV, type = 'III')
car::vif(mPoint_pooled_narc_ffniV)

mPoint_pooled_narc_ffniAE <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_agentic_extraversion) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_ffniAE)
car::Anova(mPoint_pooled_narc_ffniAE, type = 'III')
car::vif(mPoint_pooled_narc_ffniAE)

mPoint_pooled_narc_ffniA <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_antagonism) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_ffniA)
car::Anova(mPoint_pooled_narc_ffniA, type = 'III')
car::vif(mPoint_pooled_narc_ffniA)

mPoint_pooled_narc_ffniN <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(ffni_neuroticism) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_ffniN)
car::Anova(mPoint_pooled_narc_ffniN, type = 'III')
car::vif(mPoint_pooled_narc_ffniN)

#rank buying
mRank_pooled_narc_ffniG <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_grandiosity) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_ffniG)
car::Anova(mRank_pooled_narc_ffniG, type = 'III')
car::vif(mRank_pooled_narc_ffniG)

mRank_pooled_narc_ffniV <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_vulnerability) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_ffniV)
car::Anova(mRank_pooled_narc_ffniV, type = 'III')
car::vif(mRank_pooled_narc_ffniV)

mRank_pooled_narc_ffniAE <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_agentic_extraversion) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_ffniAE)
car::Anova(mRank_pooled_narc_ffniAE, type = 'III')
car::vif(mRank_pooled_narc_ffniAE)

mRank_pooled_narc_ffniA <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_antagonism) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_ffniA)
car::Anova(mRank_pooled_narc_ffniA, type = 'III')
car::vif(mRank_pooled_narc_ffniA)

mRank_pooled_narc_ffniN <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(ffni_neuroticism) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_ffniN)
car::Anova(mRank_pooled_narc_ffniN, type = 'III')
car::vif(mRank_pooled_narc_ffniN)


# BPNI subscales
#point stealing
mPoint_pooled_narc_bpniG <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(bpni_grandiosity) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_bpniG)
car::Anova(mPoint_pooled_narc_bpniG, type = 'III')
car::vif(mPoint_pooled_narc_bpniG)

mPoint_pooled_narc_bpniV <- lmer(pointSteal_pmcentered ~ scale(trial)*scale(bpni_vulnerability) + closerun.lag1 + outcome.lag1 + scale(oppRank_inv)*scale(snake_score.lag1) + scale(rankEnd_inv.lag1) + sample + scale(age) + scale(household_income) + race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mPoint_pooled_narc_bpniV)
car::Anova(mPoint_pooled_narc_bpniV, type = 'III')
car::vif(mPoint_pooled_narc_bpniV)

#rank buying
mRank_pooled_narc_bpniG <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(bpni_grandiosity) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_bpniG)
car::Anova(mRank_pooled_narc_bpniG, type = 'III')
car::vif(mRank_pooled_narc_bpniG)

mRank_pooled_narc_bpniV <- lmer(rankBuy_pmcentered ~ scale(trial)*scale(bpni_vulnerability) + scale(trial)*scale(oppRank_inv) + outcome + scale(snake_score) + scale(rankStart_inv) + scale(rankBuy_pmcentered.lag1) + sample + scale(age) + scale(household_income) +  race + sex + scale(education) + scale(gameExp) + scale(depression_percentile) + (1|subject),  data = ds_pooled, na.action = na.omit)
summary(mRank_pooled_narc_bpniV)
car::Anova(mRank_pooled_narc_bpniV, type = 'III')
car::vif(mRank_pooled_narc_bpniV)


#########################################################################
## exploratory b) analysis of main findings in linear models predicting raw scores on the snake arcade game (performance as a marker of task engagement)

# Main selected models (Supplemental Figure S5)

#Sample 1
mScore_s1_main <- lmer(snake_score ~ scale(trial) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_sample1) 
summary(mScore_s1_main)
car::Anova(mScore_s1_main,'III')
car::vif(mScore_s1_main)

#Sample 2
mScore_s2_main <- lmer(snake_score ~ scale(trial) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_sample2)
summary(mScore_s2_main)
car::Anova(mScore_s2_main,'III')
car::vif(mScore_s2_main)

#pooled
mScore_pooled_main <- lmer(snake_score ~ scale(trial) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_main)
car::Anova(mScore_pooled_main,'III')
car::vif(mScore_pooled_main)

ls_mScore_pooled_main <- lsmeans(mScore_pooled_main,"race")
pairs(ls_mScore_pooled_main, adjust = 'none')

#Figure S5
pScore_s1_main <- plot_model(mScore_s1_main, type = "est", show.p = TRUE, show.values = TRUE, title = "Sample 1", colors = c("#c55a11","#008600"), order.terms = c(1,4,5,3,2,6,8,7,9), value.offset = .3, value.size = 4, vline.color = 'grey80',
                                 axis.labels = rev(c("trial (scaled)", "opponent's rank (scaled)\n high = positive", "victory on previous trial", "age", "male sex", "game experience", "African-American\nAfrican-Canadian", "Asian", "other race"))) +
                                  theme_bw() + ylab("  ")

pScore_s2_main <- plot_model(mScore_s2_main, type = "est", show.p = TRUE, show.values = TRUE, title = "Sample 2", colors = c("#c55a11","#008600"),
                                 order.terms = c(1,4,5,3,2,6,7,8,9), value.offset = .3, value.size = 4, vline.color = 'grey80',
                                 axis.labels = c(" ", " ", " ", " ", " ", " ", " ", " ", " ")) +
                                  theme_bw() + ylab("lower <=   Score   => higher")


pScore_pooled_main <- plot_model(mScore_pooled_main, type = "est", show.p = TRUE, show.values = TRUE, title = "Pooled analysis", colors = c("#c55a11","#008600"),
                                  order.terms = c(1,4,5,3,2,6,8,7,9), value.offset = .3, value.size = 4, vline.color = 'grey80',
                                  axis.labels = c(" ", " ", " ", " ", " ", " ", " ", " ", " ")) +
                                  theme_bw() + ylab("  ")


pScore_comb2 <- grid.arrange(pScore_s2_main, pScore_pooled_main,
             layout_matrix = matrix(c(1,2), ncol=2, byrow=TRUE))

grid.arrange(pScore_s1_main, pScore_comb2,
             layout_matrix = matrix(c(1,2), ncol=2, byrow=TRUE))


## Samplewise differences in the pooled analysis predicting scores on the snake arcade game (Supplemental Figure S6, panel A)

mScore_pooled_samplewise <- lmer(snake_score ~ scale(trial)*sample + sex*sample + scale(oppRank_inv)*sample + outcome.lag1*sample + scale(gameExp)*sample + race*sample + (1|subject), data = ds_pooled) 
summary(mScore_pooled_samplewise)
car::Anova(mScore_pooled_samplewise,'III')
car::vif(mScore_pooled_samplewise)

#Figure S6, panel A
FigureS6A_a <- emmip(mScore_pooled_samplewise, sample ~ trial, at = list(trial = c(1,12,24)), CIs = TRUE, col = c("#008600", "#c55a11","#008600", "#c55a11"), style = "factor") +
  theme_bw() +
  #  scale_y_reverse () +
  scale_fill_manual(values = c("#008600", "#c55a11","#008600", "#c55a11","#008600", "#c55a11")) +
  scale_color_manual(values = c("#008600", "#c55a11","#008600", "#c55a11","#008600", "#c55a11")) +
  theme(legend.position = "left")+
  xlab("Trial") + ylab("Score")

FigureS6A_b <- emmip(mScore_pooled_samplewise, sample ~ gameExp, at = list(gameExp = c(1,5)), CIs = TRUE, col = c("#008600", "#c55a11","#008600", "#c55a11"), style = "factor") +
  theme_bw() +
  #  scale_y_reverse () +
  scale_fill_manual(values = c("#008600", "#c55a11")) +
  scale_color_manual(values = c("#008600", "#c55a11")) +
  theme(legend.position = "left")+
  xlab("Game experience") + ylab("Score")

FigureS6A_c <- emmip(mScore_pooled_samplewise, sample ~ sex, CIs = TRUE, col = c("#008600", "#c55a11","#008600", "#c55a11"), style = "factor") +
  theme_bw() +
  #  scale_y_reverse () +
  scale_fill_manual(values = c("#008600", "#c55a11")) +
  scale_color_manual(values = c("#008600", "#c55a11")) +
  theme(legend.position = "left")+
  xlab("Sex") + ylab("Score")

grid.arrange(FigureS6A_a, FigureS6A_b, FigureS6A_c,
             layout_matrix = matrix(c(1,2,3), ncol=3, byrow=TRUE))


# Interactions with psychometric measures predicting scores in the pooled analysis (and in Sample 2 in the case of trait dominance; supplemental table S8)

#trait dominance (in Sample 2)
mScore_s2_dom <- lmer(snake_score ~ scale(trial)*scale(ipip_total) + scale(oppRank_inv) + outcome.lag1 + sex + scale(age) + scale(gameExp) + race + (1|subject), data = ds_sample2)
summary(mScore_s2_dom)
car::Anova(mScore_s2_dom,'III')
car::vif(mScore_s2_dom)

#narcissism (FFNI; pooled analysis)
mScore_pooled_ffni <- lmer(snake_score ~ scale(trial)*scale(ffni_total) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffni)
car::Anova(mScore_pooled_ffni,'III')
car::vif(mScore_pooled_ffni)

#depression (pooled analysis)
mScore_pooled_dep <- lmer(snake_score ~ scale(trial)*scale(depression_percentile) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_dep)
car::Anova(mScore_pooled_dep,'III')
car::vif(mScore_pooled_dep)

#Figure S6, panle B
FigureS6B1 <- emmip(mScore_s2_dom, ipip_total ~ trial, at = list(ipip_total = c(dom1,dom2), trial = c(1,12,24)), CIs = TRUE, col = c("#008600", "#c55a11"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme(legend.position = "left")+
  xlab("Trial") + ylab("Score")

FigureS6B2 <- emmip(mScore_pooled_ffni, ffni_total ~ trial, at = list(ffni_total = c(ffni1,ffni2), trial = c(1,12,24)), CIs = TRUE, col = c("#008600", "#c55a11"), style = "factor") +
  theme_bw() +
  scale_fill_manual(values = c("#c55a11","#008600")) +
  scale_color_manual(values = c("#c55a11","#008600")) +
  theme(legend.position = "left")+
  xlab("Trial") + ylab("Score")

grid.arrange(FigureS6B1, FigureS6B2,
             layout_matrix = matrix(c(1,2), ncol=2, byrow=TRUE))

# Narcissistic subscales (pooled analysis; Supplemental Table S8)

#FFNI subscales
mScore_pooled_ffniG <- lmer(snake_score ~ scale(trial)*scale(ffni_grandiosity) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffniG)
car::Anova(mScore_pooled_ffniG,'III')
car::vif(mScore_pooled_ffniG)

mScore_pooled_ffniV <- lmer(snake_score ~ scale(trial)*scale(ffni_vulnerability) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffniV)
car::Anova(mScore_pooled_ffniV,'III')
car::vif(mScore_pooled_ffniV)

mScore_pooled_ffniAE <- lmer(snake_score ~ scale(trial)*scale(ffni_agentic_extraversion) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffniAE)
car::Anova(mScore_pooled_ffniAE,'III')
car::vif(mScore_pooled_ffniAE)

mScore_pooled_ffniA <- lmer(snake_score ~ scale(trial)*scale(ffni_antagonism) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffniA)
car::Anova(mScore_pooled_ffniA,'III')
car::vif(mScore_pooled_ffniA)

mScore_pooled_ffniN <- lmer(snake_score ~ scale(trial)*scale(ffni_neuroticism) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_ffniN)
car::Anova(mScore_pooled_ffniN,'III')
car::vif(mScore_pooled_ffniN)

#BPNI total score and subscales
mScore_pooled_bpni <- lmer(snake_score ~ scale(trial)*scale(bpni_total) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_bpni)
car::Anova(mScore_pooled_bpni,'III')
car::vif(mScore_pooled_bpni)

mScore_pooled_bpniG <- lmer(snake_score ~ scale(trial)*scale(bpni_grandiosity) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_bpniG)
car::Anova(mScore_pooled_bpniG,'III')
car::vif(mScore_pooled_bpniG)

mScore_pooled_bpniV <- lmer(snake_score ~ scale(trial)*scale(bpni_vulnerability) + sex + scale(age) + scale(oppRank_inv) + outcome.lag1 + scale(gameExp) + race + (1|subject), data = ds_pooled) 
summary(mScore_pooled_bpniV)
car::Anova(mScore_pooled_bpniV,'3')
car::vif(mScore_pooled_bpniV)

##################################################################################
## exploratory c) correlations with end-game questions

chars_mot <- ds_pooled_wide[, c("pointSteal_taskmean", "rankBuy_taskmean","depression_percentile","ffni_total","bpni_total","ffni_grandiosity", "bpni_grandiosity",  "ffni_vulnerability", "bpni_vulnerability", "ffni_agentic_extraversion", "ffni_antagonism", "ffni_neuroticism","qA1control", "qA2fair", "qA3enjoyed", "qA4satisfied", "qMot1", "qMot2", "qMot3", "qMot4", "qMot5", "qMot6", "qMot7", "qMot8")]
res_mot <- cor.mtest(chars_mot, conf.level = .95, method = "spearman")

corrplot(cor(chars_mot, method = "spearman", use = "na.or.complete"), method = "color", type = "lower", addCoef.col = "black", tl.col = "black", tl.srt = 45, number.cex = 0.5, tl.pos = 'lt', upper = 'circle', lower = 'number', p.mat = res_mot$p, insig = "pch", diag = FALSE)
