library(diagram)
library(lubridate)
library(seraphim)
library(sf)
library(treeio)

# 1. Analysing the outputs of the preliminary discrete phylogeographic analysis 
# 2. Preparing the continuous phylogeographic analysis (RRW, Cauchy model)
# 3. Running BEAST and building the maximum clade consensus (MCC) tree
# 4. Extracting spatio-temporal information embedded in MCC and posterior trees
# 5. Generating a dispersal history graph (mapped MCC trees, 80% HPD polygons)
# 6. Estimating and plotting dispersal statistics associated with lineages
# 7. Comparing intra- and inter-states lineage migration events through time

analysis = "MCC_tree_from_DTA"
writingFiles = FALSE; showingPlots = FALSE

stateIDs = read.csv("All_Brazilian_states.csv", head=T)
brazil = raster::getData("GADM", country="BRA", level=0)
states = raster::getData("GADM", country="BRA", level=1)
brazil_light = gSimplify(brazil, 0.05); states_light = gSimplify(states, 0.05)
municipalities = raster::getData("GADM", country="BRA", level=2)
districts = raster::getData("GADM", country="BRA", level=3)

# 1. Analysing the outputs of the preliminary discrete phylogeographic analysis 

tree = readAnnotatedNexus(paste0(analysis,".tree"))
metadata = read.csv(paste0(analysis,".csv"), head=T); txt = c(); tab = c()
for (i in 1:length(tree$tip.label))
	{
		index = which(metadata[,"id"]==gsub("'","",tree$tip.label[i]))
		date = as.character(metadata[index,"date"])
		txt = c(txt, paste0(">",gsub("'","",tree$tip.label[i])),"NNNN")
		if (!grepl("Brazil",tree$tip.label[i])) location = "other"
		if (grepl("Brazil",tree$tip.label[i])) location = "Brazil"
		tab = rbind(tab, cbind(gsub("'","",tree$tip.label[i]),location,date))
	}
colnames(tab) = c("trait","location","collection_date")
if (showingPlots)
	{
		samplingDates = decimal_date(ymd(gsub("\\/","-",tab[,"collection_date"]))); mostRecentSamplingYear = max(samplingDates)
		selectedDates = decimal_date(ymd(c("2019-12-01","2019-12-15","2020-01-01","2020-01-15","2020-02-01","2020-02-15",
										   "2020-03-01","2020-03-15","2020-04-01","2020-04-15","2020-05-01")))
		rootHeight = max(nodeHeights(tree)); root_time = mostRecentSamplingYear-rootHeight
		selectedLabels = c("01-12","15-12","01-01","15-01","01-02","15-02","01-03","15-03","01-04","15-04","01-05")
		cols = rep("gray30",dim(tree$edge)[1]); lwds = rep(0.1,dim(tree$edge)[1])
		for (i in 1:dim(tree$edge)[1])
			{
				if (tree$edge[i,1]%in%tree$edge[,2])
					{
						index = which(tree$edge[,2]==tree$edge[i,1])
						if ((tree$annotations[[index]]$BR_not=="Brazil") & (tree$annotations[[i]]$BR_not=="Brazil"))
							{
								cols[i] = "chartreuse3"; lwds[i] = 0.4
							}
					}
			}
		dev.new(width=7, height=7); par(oma=c(0,0,0,0), mar=c(0,0,0,0.0), lwd=0.1)
		plot(tree, type="fan", show.tip.label=F, show.node.label=F, edge.width=lwds, cex=0.6, align.tip.label=3, col="gray30", edge.color=cols)
		for (i in 1:dim(tree$edge)[1])
			{
				if ((!tree$edge[i,2]%in%tree$edge[,1]) & (grepl("Brazil",tree$tip.label[tree$edge[i,2]])))
					{
						nodelabels(node=tree$edge[i,2], pch=16, cex=0.3, col="chartreuse3")
						nodelabels(node=tree$edge[i,2], pch=1, cex=0.3, col="gray30", lwd=0.5)
					}
				if (tree$annotations[[i]]$BR_not == "Brazil")
					{
						index = which(tree$edge[,2]==tree$edge[i,1])
						if (tree$annotations[[index]]$BR_not != "Brazil")
							{
								nodelabels(node=tree$edge[i,2], pch=16, cex=0.6, col="chartreuse3")
								nodelabels(node=tree$edge[i,2], pch=1, cex=0.6, col="gray30", lwd=0.5)
							}
					}
			}
		add.scale.bar(x=0.0, y=-0.01, length=NULL, ask=F, lwd=0.5 , lcol ="gray30", cex=0.7)
	}

municipalities@data[,"NAME_2"] = gsub(" da "," Da ",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub(" de "," De ",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub(" do "," Do ",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub(" del "," Del ",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub(" dos "," Dos ",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub(" ","",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("d'O","Do",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("'","",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("á","a",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("Á","A",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("ã","a",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("â","a",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("ç","c",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("é","e",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("í","i",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("ó","o",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("ô","o",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("ú","u",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("\\(capital\\)","",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("AguasDeLindoia","AguasdeLindoia",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("BalnearioCamboriu","BalnearioCamboriu",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("BalnearioCamboriu","BalnearioCamboriu",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("Bomdespacho","BomDespacho",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("Embu","EmbuDasArtes",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("FeiraDeSantana","FeiradeSantana",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("Joinvile","Joinville",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("JuizDeFora","JuizdeFora",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("Lajedao","Lajeado",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("PocosDeCaldas","PocosdeCaldas",municipalities@data[,"NAME_2"])
municipalities@data[,"NAME_2"] = gsub("RioDeJaneiro","RiodeJaneiro",municipalities@data[,"NAME_2"])
metadata[,"Municipality__autocolor"] = gsub(" ","",metadata[,"Municipality__autocolor"])
metadata[which(metadata[,"Municipality__autocolor"]=="BalnearioCamboriu"),"State__autocolor"] = "SC"
metadata[,"Municipality__autocolor"] = gsub("SãoJoaoDelRei","SaoJoaoDelRei",metadata[,"Municipality__autocolor"])
if (!file.exists("WorldPop_pop_log10.asc"))
	{
		pop1 = raster("WorldPop_pop_raster.tif"); pop2 = aggregate(pop1, fact=10, fun=sum)
		pop3 = pop2; pop3[] = log10(pop2[]+1); writeRaster(pop3, "WorldPop_pop_log10.asc", overwrite=T)
	}
pop = raster("WorldPop_pop_log10.asc")
brazilBranches = c(); brazilIntroductions = c()
brazilTipBranches = c(); sampledSequences = c()
for (i in 1:dim(tree$edge)[1])
	{
		if (tree$annotations[[i]]$Brazil_or_not == "Brazil")
			{
				brazilBranches = c(brazilBranches,i)
				index = which(tree$edge[,2]==tree$edge[i,1])
				if (tree$annotations[[index]]$Brazil_or_not != "Brazil")
					{
						brazilIntroductions = c(brazilIntroductions, i)
					}
				if (!tree$edge[i,2]%in%tree$edge[,1])
					{
						brazilTipBranches = c(brazilTipBranches, i)
						sampledSequences = c(sampledSequences, gsub("'","",tree$tip.label[tree$edge[i,2]]))
					}
			}
	}
for (i in 1:length(brazilIntroductions))
	{
		if (i == 1) clusters1 = list()
		if (tree$edge[brazilIntroductions[i],2]%in%tree$edge[,1])
			{
				subtree = tree_subset(tree, tree$edge[brazilIntroductions[i],2], levels_back=0)
				clusters1[[i]] = gsub("'","",subtree$tip.label)
			}	else		{
				clusters1[[i]] = gsub("'","",tree$tip.label[tree$edge[brazilIntroductions[i],2]]	)
			}
	}
if (!file.exists(paste0("Sampling_in_Brazil.csv")))
	{
		samplingData = matrix(nrow=length(sampledSequences), ncol=6)
		colnames(samplingData) = c("sequenceID","collectionDate","state","municipality","longitude","latitude")
		samplingData[,"sequenceID"] = sampledSequences
		for (i in 1:dim(samplingData)[1])
			{
				index1 = which(metadata[,"id"]==samplingData[i,"sequenceID"])
				date = dmy(gsub("\\/","-",metadata[index1,"date"]))
				samplingData[i,"collectionDate"] = decimal_date(date)
				state = stateIDs[which(stateIDs[,"ID"]==as.character(metadata[index1,"State__autocolor"])),"state"]
				samplingData[i,"state"] = as.character(state)		
				index2 = which((municipalities@data[,"NAME_1"]==as.character(state))&
							   (municipalities@data[,"NAME_2"]==as.character(metadata[index1,"Municipality__autocolor"])))
				if (length(index2) != 1)
					{
						index3 = which((districts@data[,"NAME_1"]==as.character(state))&
							   		   (districts@data[,"NAME_3"]==as.character(metadata[index1,"Municipality__autocolor"])))
						if (length(index3) != 1)
							{
								print(c(i,as.character(state),as.character(metadata[index1,"Municipality__autocolor"])))
								samplingData[i,"longitude"] = metadata[index1,"longitude"]
								samplingData[i,"latitude"] = metadata[index1,"latitude"]
							}	else	{
								maxArea = 0; polIndex = 0
								for (j in 1:length(districts@polygons[[index3]]@Polygons))
									{
										if (maxArea < districts@polygons[[index3]]@Polygons[[j]]@area)
											{
												maxArea = districts@polygons[[index3]]@Polygons[[j]]@area; polIndex = j
											}
									}
								pol = districts@polygons[[index3]]@Polygons[[polIndex]]
								p = Polygon(pol@coords); ps = Polygons(list(p),1); sps = SpatialPolygons(list(ps))
								pol = sps; proj4string(pol) = districts@proj4string
								samplingData[i,c("longitude","latitude")] = coordinates(pol) # to avoid a jitter:
								samplingData[i,c("longitude","latitude")] = spsample(pol, 1, type="random")@coords
							}
					}	else	{
						samplingData[i,"municipality"] = metadata[index1,"Municipality__autocolor"]
						maxArea = 0; polIndex = 0
						for (j in 1:length(municipalities@polygons[[index2]]@Polygons))
							{
								if (maxArea < municipalities@polygons[[index2]]@Polygons[[j]]@area)
									{
										maxArea = municipalities@polygons[[index2]]@Polygons[[j]]@area; polIndex = j
									}
							}
						pol = municipalities@polygons[[index2]]@Polygons[[polIndex]]
						p = Polygon(pol@coords); ps = Polygons(list(p),1); sps = SpatialPolygons(list(ps))
						pol = sps; proj4string(pol) = municipalities@proj4string
						samplingData[i,c("longitude","latitude")] = coordinates(pol) # to avoid a jitter:
						samplingData[i,c("longitude","latitude")] = spsample(pol, 1, type="random")@coords
					}
			}
		write.csv(samplingData, "Sampling_in_Brazil.csv", quote=F, row.names=F)
	}	
samplingData = read.csv("Sampling_in_Brazil.csv", head=T)
for (i in 1:length(brazilIntroductions))
	{
		tab = c()
		if (i == 1)
			{
				clusters2 = list(); centroids = list()
			}
		for (j in 1:length(clusters1[[i]]))
			{
				index = which(samplingData[,"sequenceID"]==clusters1[[i]][j])
				if (length(index) == 1)
					{
						line = cbind(as.numeric(samplingData[index,"collectionDate"]),as.numeric(samplingData[index,"longitude"]),as.numeric(samplingData[index,"latitude"]))
						row.names(line) = clusters1[[i]][j]; tab = rbind(tab, line)
					}
			}
		colnames(tab) = c("collectionDate","longitude","latitude"); clusters2[[i]] = tab
		centroids[[i]] = cbind(mean(tab[!is.na(tab[,"longitude"]),"longitude"]), mean(tab[!is.na(tab[,"latitude"]),"latitude"]))
	}
clusterSizes = rep(NA, length(clusters1))
collectionDates = c()
for (i in 1:length(clusters1))
	{
		clusterSizes[i] = length(clusters1[[i]])
		collectionDates = c(collectionDates, clusters2[[i]][,"collectionDate"])
	}
if (showingPlots)
	{
		dev.new(width=3.3, height=8); par(mfrow=c(2,1), oma=c(0,0,0,0), mar=c(2,2,1,1), lwd=0.2, col="gray30")
		hist(clusterSizes, breaks=35, axes=F, ann=F, title=NULL, col="#66CD0099", border="gray30")
		axis(side=2, lwd.tick=0.2, cex.axis=0.65, mgp=c(0,0.20,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30")
		axis(side=1, lwd.tick=0.2, cex.axis=0.65, mgp=c(0,0.00,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30")
		hist(collectionDates, breaks=50, axes=F, ann=F, title=NULL, col="#66CD0099", border="gray30")
		axis(side=2, lwd.tick=0.2, cex.axis=0.65, mgp=c(0,0.20,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30")
		axis(side=1, lwd.tick=0.2, cex.axis=0.65, mgp=c(0,0.00,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30",
			 at=decimal_date(ymd(c("2020-02-01","2020-02-15","2020-03-01","2020-03-15","2020-04-01","2020-04-15","2020-05-01"))),
			 labels=c("01-02-2020","15-02-2020","01-03-2020","15-03-2020","01-04-2020","15-04-2020","01-05-2020"))
	}
if (showingPlots)
	{
		dev.new(width=7, height=6); par(oma=c(0,0,0,0), mar=c(2,6,2,0), lwd=0.2, col="gray30")
		cols = c(colorRampPalette(brewer.pal(9,"BuPu"))(161)[1:101])
		cols = c(colorRampPalette(brewer.pal(9,"YlGnBu"))(161)[1:101])
		cols = rev(colorRampPalette(brewer.pal(9,"RdYlBu"))(151)[1:101])
		plot(pop, col=cols, axes=F, ann=F, box=F, legend=F)
		plot(brazil_light, border="gray40", lwd=0.4, add=T)
		for (i in 1:length(clusters1))
			{
				if (!is.na(centroids[[i]][,1]))
					{
						if (length(clusters1[[i]]) > 1)
							{
								for (j in 1:dim(clusters2[[i]])[1])
									{
										if (!is.na(clusters2[[i]][j,1]))
											{
												segments(centroids[[i]][,1],centroids[[i]][,2],clusters2[[i]][j,"longitude"],clusters2[[i]][j,"latitude"], lwd=0.5, col="gray30")	
											}
									}
							}
					}
			}
		for (i in 1:length(clusters1))
			{
				if (!is.na(centroids[[i]][,1]))
					{
						for (j in 1:dim(clusters2[[i]])[1])
							{
								points(clusters2[[i]][,"longitude"], clusters2[[i]][,"latitude"], pch=16, cex=0.6, col="chartreuse3")
								points(clusters2[[i]][,"longitude"], clusters2[[i]][,"latitude"], pch=1, cex=0.6, col="gray30", lwd=0.2)
							}
					}
			}
		for (i in 1:length(clusters1))
			{
				if (!is.na(centroids[[i]][,1]))
					{
						if (length(clusters1[[i]]) > 1)
							{
								points(centroids[[i]][,1], centroids[[i]][,2], pch=16, cex=0.4, col="red")
								points(centroids[[i]][,1], centroids[[i]][,2], pch=1, cex=0.4, col="gray30", lwd=0.2)
							}
					}
			}
		legendRast = raster(as.matrix(c(min(pop[],na.rm=T),max(pop[],na.rm=T))))
		mtext("Human population", col="gray30", cex=0.7, line=-23.4, at=-67.4)
		mtext("(log10-transformed)", col="gray30", cex=0.7, line=-24.15, at=-67.4)
		plot(legendRast, legend.only=T, col=cols, legend.width=0.5, legend.shrink=0.3, smallplot=c(0.268,0.277,0.2,0.45),
			 alpha=1, horizontal=F, legend.args=list(text="", cex=0.7, line=0.5, col="gray30"), axis.args=list(cex.axis=0.55, lwd=0,
			 lwd.tick=0.2, tck=-0.7, col.axis="gray30", line=0, mgp=c(0,0.4,0)))
	}

# 2. Preparing the continuous phylogeographic analysis (RRW, Cauchy model)

analyses = c(); template = scan("TreeTime_RRW_temp.xml", what="", sep="\n", quiet=T, blank.lines.skip=F)
for (i in 1:length(clusters2))
	{
		if ((dim(clusters2[[i]])[1] >= 3)&(sum(!is.na(clusters2[[i]][,"longitude"])) >= 3))
			{
				analyses = c(analyses, paste0("Clade_",i))
				if (!file.exists(paste0("Phylogeographic_runs/Clade_",i,".xml")))
					{
						xml = gsub("TEMPLATE", paste0("Clade_",i), template)
						tre = tree_subset(tree, tree$edge[brazilIntroductions[i],2], levels_back=0)
						tips_to_drop = tre$tip.label[which(!gsub("'","",tre$tip.label)%in%row.names(clusters2[[i]]))]
						if (length(tips_to_drop) > 0) tre = ape::drop.tip(tre, tips_to_drop)
						write.tree(tre, paste0("Phylogeographic_runs/Clade_",i,".tre"))
						tre = scan(paste0("Phylogeographic_runs/Clade_",i,".tre"), what="", sep="\n", quiet=T)
						sink(file=paste0("Phylogeographic_runs/Clade_",i,".xml"))
						for (j in 1:length(xml))
							{
								cat(xml[j]); cat("\n")
								if (xml[j]=="\t<taxa id=\"taxa\">")
									{
										for (k in 1:dim(clusters2[[i]])[1])
											{
												cat(paste0("\t\t<taxon id=\"",row.names(clusters2[[i]])[k],"\">","\n"))
												cat(paste0("\t\t\t<date value=\"",clusters2[[i]][k,"collectionDate"],"\" direction=\"forwards\" units=\"years\"/>","\n"))
												cat("\t\t\t<attr name=\"latitude\">\n")
												cat(paste0("\t\t\t\t",clusters2[[i]][k,"latitude"],"\n"))
												cat("\t\t\t</attr>\n")
												cat("\t\t\t<attr name=\"longitude\">\n")
												cat(paste0("\t\t\t\t",clusters2[[i]][k,"longitude"],"\n"))
												cat("\t\t\t</attr>\n")
												cat("\t\t\t<attr name=\"coordinates\">\n")
												cat(paste0("\t\t\t\t",clusters2[[i]][k,"latitude"]," ",clusters2[[i]][k,"longitude"],"\n"))
												cat("\t\t\t</attr>\n")
												cat("\t\t</taxon>\n")
											}
									}
								if (xml[j]=="\t<alignment id=\"alignment\" dataType=\"nucleotide\">")
									{
										for (k in 1:dim(clusters2[[i]])[1])
											{
												cat("\t\t<sequence>\n")
												cat(paste0("\t\t\t<taxon idref=\"",row.names(clusters2[[i]])[k],"\"/>","\n"))
												cat("\t\t\tNNNN\n")
												cat("\t\t</sequence>\n")
											}
									}
								if (xml[j]=="\t<newick id=\"startingTree\">")
									{
										cat(paste0("\t\t",tre,"\n"))
									}
							}
						sink(NULL)
					}
			}
	}

# 3. Running BEAST and building the maximum clade consensus (MCC) tree

source("MCC_tree_extraction.r")
sink(file=paste0("Phylogeographic_runs/Analyses.sh"))
for (i in 1:length(analyses))
	{
		cat(paste0("java -jar beast_1104.jar -overwrite ",analyses[i],".xml\n"))
	}
sink(NULL)
runningNewAnalyses = FALSE
wd = getwd(); setwd(paste0(wd,"/Phylogeographic_runs/"))
if (runningNewAnalyses)
	{
		system("bash Analyses.sh", ignore.stdout=T, ignore.stderr=F)
	}
burnIns = rep(501, length(analyses)); burnIns[12] = 2001
if (runningNewAnalyses)
	{
		for (i in 1:length(analyses))
			{
system(paste0("BEAST_1104/bin/treeannotator -burninTrees ",burnIns[i]," -heights keep ",analyses[i],".trees ",analyses[i],".tree"), ignore.stdout=F, ignore.stderr=F)
			}
	}
setwd(wd)

# 4. Extracting spatio-temporal information embedded in MCC and posterior trees

wd = getwd(); setwd(paste0(wd,"/Phylogeographic_runs/"))
for (i in 1:length(analyses))
	{
		if (!file.exists(paste0(analyses[i],".csv")))
			{
				index = as.numeric(unlist(strsplit(analyses[i],"_"))[2])
				mostRecentSamplingDatum = max(clusters2[[index]][which(!is.na(clusters2[[index]][,"longitude"])),"collectionDate"])
				mcc_tre = readAnnotatedNexus(paste0(analyses[i],".tree"))
				mcc_tab = MCC_tree_extraction(mcc_tre, mostRecentSamplingDatum)
				write.csv(mcc_tab, paste0(analyses[i],".csv"), row.names=F, quote=F)
			}
	}
nberOfTreesToSample = 1000; randomSampling = FALSE; coordinateAttributeName = "coordinates"; nberOfCores = 5
for (i in 1:length(analyses))
	{
		localTreesDirectory = paste0(analyses[i],"_ext")
		if (!file.exists(paste0(localTreesDirectory,"/TreeExtractions_1.csv")))
			{
				index = as.numeric(unlist(strsplit(analyses[i],"_"))[2]); burnIn = burnIns[i]
				mostRecentSamplingDatum = max(clusters2[[index]][which(!is.na(clusters2[[index]][,"longitude"])),"collectionDate"])
				allTrees = scan(file=paste0(analyses[i],".trees"), what="", sep="\n", quiet=T, blank.lines.skip=F)
				treeExtractions(localTreesDirectory, allTrees, burnIn, randomSampling, nberOfTreesToSample, mostRecentSamplingDatum, coordinateAttributeName, nberOfCores)
			}
	}
for (i in 1:length(analyses))
	{
		tab = read.csv(paste0(analyses[i],".csv"), head=T)
		if (i == 1)
			{
				all = tab
			}	else	{
				maxNodeID = max(all[,c("node1","node2")])
				tab[,c("node1","node2")] = tab[,c("node1","node2")]+maxNodeID
				all = rbind(all, tab)
			}
	}
write.csv(all, "All_clades.csv", row.names=F, quote=F)
dir.create(file.path("All_clades_ext"), showWarnings=F)
nberOfExtractionFiles = nberOfTreesToSample
for (i in 1:nberOfExtractionFiles)
	{
		for (j in 1:length(analyses))
			{
				tab = read.csv(paste0(analyses[j],"_ext/TreeExtractions_",i,".csv"), head=T)
				if (j == 1)
					{
						all = tab
					}	else	{
						maxNodeID = max(all[,c("node1","node2")])
						tab[,c("node1","node2")] = tab[,c("node1","node2")]+maxNodeID
						all = rbind(all, tab)
					}
			}
		write.csv(all, paste0("All_clades_ext/TreeExtractions_",i,".csv"), row.names=F, quote=F)
	}
setwd(wd)

# 5. Generating a dispersal history graph (mapped MCC trees, 80% HPD polygons)

localTreesDirectory = paste0("Phylogeographic_runs/All_clades_ext"); nberOfExtractionFiles = 1000
percentage = 80; prob = percentage/100; precision = 1/(365/3.5); croppingPolygons = TRUE; colourBranches = FALSE
mcc = read.csv("Phylogeographic_runs/All_clades.csv", head=T); startDatum = min(mcc[,"startYear"])
polygons = suppressWarnings(spreadGraphic2(localTreesDirectory, nberOfExtractionFiles, prob, startDatum, precision))
if (showingPlots)
	{
		colourScale = colorRampPalette(brewer.pal(11,"RdBu"))(121)[24:121]
		minYear = min(mcc[,"startYear"]); maxYear = max(mcc[,"endYear"])
		startYears_indices = (((mcc[,"startYear"]-minYear)/(maxYear-minYear))*100)+1
		endYears_indices = (((mcc[,"endYear"]-minYear)/(maxYear-minYear))*100)+1
		startYears_colours = colourScale[startYears_indices]
		endYears_colours = colourScale[endYears_indices]
		polygons_colours = rep(NA, length(polygons))
		cexNode = 0.8; LWD = 1.0
		for (i in 1:length(polygons))
			{
				date = as.numeric(names(polygons[[i]]))
				polygon_index = round((((date-minYear)/(maxYear-minYear))*100)+1)
				polygons_colours[i] = paste0(colourScale[polygon_index],"40")
			}
		firstTimePeriod = TRUE; secondTimePeriod = FALSE
		firstTimePeriod = FALSE; secondTimePeriod = TRUE
		firstTimePeriod = FALSE; secondTimePeriod = FALSE
		pdf("Figure_A_NEW.pdf", width=7.3, height=6) # dev.new(width=7.3, height=6)
		par(oma=c(0,0,0,0), mar=c(1,1,1,1), lwd=0.2, col="gray30")
		cols = c(colorRampPalette(brewer.pal(9,"YlGnBu"))(161)[1:101])
		cols = c(colorRampPalette(brewer.pal(9,"Greys"))(201)[1:101])
		plot(brazil_light, border=NA, col="gray95", lwd=LWD)
		if (colourBranches == FALSE)
			{
				if ((firstTimePeriod == FALSE)&(secondTimePeriod == FALSE))
					{
						for (i in 1:length(polygons))
							{
								pol = polygons[[i]]; crs(pol) = crs(brazil)
								for (j in 1:length(pol@polygons))
									{
										pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
									}
								if (croppingPolygons == TRUE) pol = crop(pol, brazil)
								plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
							}
					}
				if (firstTimePeriod == TRUE)
					{
						for (i in 1:length(polygons))
							{
								if (as.numeric(names(polygons[[i]])) <= decimal_date(ymd("2020-03-21")))
									{
										pol = polygons[[i]]; crs(pol) = crs(brazil)
										for (j in 1:length(pol@polygons))
											{
												pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
											}
										if (croppingPolygons == TRUE) pol = crop(pol, brazil)
										plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
									}
							}
					}
				if (secondTimePeriod == TRUE)
					{
						for (i in 1:length(polygons))
							{
								if (as.numeric(names(polygons[[i]])) > decimal_date(ymd("2020-03-21")))
									{
										pol = polygons[[i]]; crs(pol) = crs(brazil)
										for (j in 1:length(pol@polygons))
											{
												pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
											}
										if (croppingPolygons == TRUE) pol = crop(pol, brazil)
										plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
									}
							}
					}
			}
		plot(states_light, border="white", col=NA, add=T, lwd=LWD)
		plot(brazil_light, border="gray30", col=NA, add=T, lwd=0.4); croppingPolygons = TRUE
		selectedBranches = 1:dim(mcc)[1]
		if (firstTimePeriod) selectedBranches = which(mcc[,"endYear"]<=decimal_date(ymd("2020-03-21")))
		if (secondTimePeriod) selectedBranches = which(mcc[,"startYear"]>decimal_date(ymd("2020-03-21")))
		for (i in selectedBranches)
			{
				if (colourBranches == FALSE)
					{
						colBranch = "gray10"
					}	else	{
						if (mcc[i,"startState"] == mcc[i,"endState"]) colBranch = "red"
						if (mcc[i,"startState"] != mcc[i,"endState"]) colBranch = "gray30"
					}
				curvedarrow(cbind(mcc[i,"startLon"],mcc[i,"startLat"]), cbind(mcc[i,"endLon"],mcc[i,"endLat"]), arr.length=0,
							arr.width=0, lwd=0.2, lty=1, lcol=colBranch, arr.col=NA, arr.pos=F, curve=0.1, dr=NA, endhead=F)
			}
		if ((firstTimePeriod == FALSE)&(secondTimePeriod == FALSE))
			{
				for (i in 1:length(clusters2))
					{
						if (sum(!is.na(clusters2[[i]][,"longitude"])) < 3)
							{
								if (sum(!is.na(clusters2[[i]][,"longitude"])) == 2)
									{
										indices = which(!is.na(clusters2[[i]][,"longitude"]))
										if (firstTimePeriod)
											{
												indices = which((!is.na(clusters2[[i]][,"longitude"]))&(clusters2[[i]][,"collectionDate"]<decimal_date(ymd("2020-03-21"))))
											}
										if (secondTimePeriod)
											{
												indices = which((!is.na(clusters2[[i]][,"longitude"]))&(clusters2[[i]][,"collectionDate"]>decimal_date(ymd("2020-03-21"))))
											}
										if (length(indices) == 2)
											{
												if (colourBranches == FALSE)
													{
														curvedarrow(cbind(clusters2[[i]][indices[1],"longitude"],clusters2[[i]][indices[1],"latitude"]),
																	cbind(clusters2[[i]][indices[2],"longitude"],clusters2[[i]][indices[2],"latitude"]),
																	arr.length=0, arr.width=0, lwd=0.2, lty=2, lcol=colBranch, arr.col=NA, arr.pos=F,
																	curve=0.1, dr=NA, endhead=F)
													}
											}
									}
							}
					}
			}
		if (colourBranches == FALSE)
			{
				for (i in 1:length(clusters2))
					{
						if (sum(!is.na(clusters2[[i]][,"longitude"])) < 3)
							{
								for (j in 1:dim(clusters2[[i]])[1])
									{
										if (!is.na(clusters2[[i]][j,"longitude"]))
											{
												plotTheNode = TRUE
												if ((firstTimePeriod==TRUE)&(clusters2[[i]][j,"collectionDate"]>decimal_date(ymd("2020-03-21")))) plotTheNode = FALSE
												if ((secondTimePeriod==TRUE)&(clusters2[[i]][j,"collectionDate"]<decimal_date(ymd("2020-03-21")))) plotTheNode = FALSE
												if (plotTheNode)
													{
														index = (((clusters2[[i]][j,"collectionDate"]-minYear)/(maxYear-minYear))*100)+1
														points(clusters2[[i]][j,"longitude"], clusters2[[i]][j,"latitude"], pch=16, col=colourScale[index], cex=cexNode)
														points(clusters2[[i]][j,"longitude"], clusters2[[i]][j,"latitude"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
													}
											}
									}
							}
					}
				for (i in rev(selectedBranches))
					{
						if (!mcc[i,"node1"]%in%mcc[selectedBranches,"node2"])
							{
								points(mcc[i,"startLon"], mcc[i,"startLat"], pch=16, col=startYears_colours[i], cex=cexNode)
								points(mcc[i,"startLon"], mcc[i,"startLat"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
							}
						points(mcc[i,"endLon"], mcc[i,"endLat"], pch=16, col=endYears_colours[i], cex=cexNode)
						points(mcc[i,"endLon"], mcc[i,"endLat"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
					}
			}
		selectedDates = decimal_date(ymd(c("2020-02-15","2020-03-01","2020-03-21","2020-04-01","2020-04-15")))
		selectedLabels = c("15-02-2020","01-03-2020","21-03-2020","01-04-2020","15-04-2020")
		rast = raster(matrix(nrow=1, ncol=2)); rast[1] = min(mcc[,"startYear"]); rast[2] = max(mcc[,"endYear"])
		plot(rast, legend.only=T, col=colourScale, legend.width=0.5, legend.shrink=0.3, smallplot=c(0.268,0.277,0.2,0.45),
			 alpha=1, horizontal=F, legend.args=list(text="", cex=0.7, line=0.5, col="gray30"), axis.args=list(cex.axis=0.55, lwd=0,
			 lwd.tick=0.2, tck=-0.7, col.axis="gray30", line=0, mgp=c(0,0.4,0), at=selectedDates, labels=selectedLabels))
		dev.off()
	}
if (showingPlots)
	{
		selectedStates = c("São Paulo","Minas Gerais","Espírito Santo","Rio de Janeiro")
		selectedStates = states[states@data[,"NAME_1"]%in%selectedStates,]
		selectedStates_merged = maptools::unionSpatialPolygons(selectedStates, IDs=selectedStates@data[,"NAME_0"])
		selectedStates = gSimplify(crop(selectedStates, extent(-57,-34,-28,-12)), 0.01)
		selectedStates_merged = gSimplify(crop(selectedStates_merged, extent(-57,-34,-28,-12)), 0.01)
		colourScale = colorRampPalette(brewer.pal(11,"RdBu"))(121)[24:121]
		minYear = min(mcc[,"startYear"]); maxYear = max(mcc[,"endYear"])
		startYears_indices = (((mcc[,"startYear"]-minYear)/(maxYear-minYear))*100)+1
		endYears_indices = (((mcc[,"endYear"]-minYear)/(maxYear-minYear))*100)+1
		startYears_colours = colourScale[startYears_indices]
		endYears_colours = colourScale[endYears_indices]
		polygons_colours = rep(NA, length(polygons))
		cexNode = 0.8; LWD = 1.0
		for (i in 1:length(polygons))
			{
				date = as.numeric(names(polygons[[i]]))
				polygon_index = round((((date-minYear)/(maxYear-minYear))*100)+1)
				polygons_colours[i] = paste0(colourScale[polygon_index],"40")
			}
		firstTimePeriod = TRUE; secondTimePeriod = FALSE
		firstTimePeriod = FALSE; secondTimePeriod = TRUE
		firstTimePeriod = FALSE; secondTimePeriod = FALSE
		pdf("Figure_B_NEW.pdf", width=7.3, height=6) # dev.new(width=7.3, height=6)
		par(oma=c(0,0,0,0), mar=c(1,1,1,1), lwd=0.2, col="gray30")
		cols = c(colorRampPalette(brewer.pal(9,"YlGnBu"))(161)[1:101])
		cols = c(colorRampPalette(brewer.pal(9,"Greys"))(201)[1:101])
		plot(selectedStates, border=NA, col="gray95", lwd=LWD)
		if ((firstTimePeriod == FALSE)&(secondTimePeriod == FALSE))
			{
				for (i in 1:length(polygons))
					{
						pol = polygons[[i]]; crs(pol) = crs(brazil)
						for (j in 1:length(pol@polygons))
							{
								pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
							}
						if (croppingPolygons == TRUE) pol = crop(pol, brazil)
						plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
					}
			}
		if (firstTimePeriod == TRUE)
			{
				for (i in 1:length(polygons))
					{
						if (as.numeric(names(polygons[[i]])) <= decimal_date(ymd("2020-03-21")))
							{
								pol = polygons[[i]]; crs(pol) = crs(brazil)
								for (j in 1:length(pol@polygons))
									{
										pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
									}
								if (croppingPolygons == TRUE) pol = crop(pol, brazil)
								plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
							}
					}
			}
		if (secondTimePeriod == TRUE)
			{
				for (i in 1:length(polygons))
					{
						if (as.numeric(names(polygons[[i]])) > decimal_date(ymd("2020-03-21")))
							{
								pol = polygons[[i]]; crs(pol) = crs(brazil)
								for (j in 1:length(pol@polygons))
									{
										pol@polygons[[j]] = maptools::checkPolygonsHoles(pol@polygons[[j]])
									}
								if (croppingPolygons == TRUE) pol = crop(pol, brazil)
								plot(pol, axes=F, col=polygons_colours[i], add=T, border=NA)
							}
					}
			}
		plot(selectedStates, border="white", col=NA, add=T, lwd=LWD)
		plot(selectedStates_merged, border="gray30", col=NA, add=T, lwd=0.4); croppingPolygons = TRUE
		selectedBranches = 1:dim(mcc)[1]
		if (firstTimePeriod) selectedBranches = which(mcc[,"endYear"]<=decimal_date(ymd("2020-03-21")))
		if (secondTimePeriod) selectedBranches = which(mcc[,"startYear"]>decimal_date(ymd("2020-03-21")))
		for (i in selectedBranches)
			{
				curvedarrow(cbind(mcc[i,"startLon"],mcc[i,"startLat"]), cbind(mcc[i,"endLon"],mcc[i,"endLat"]), arr.length=0,
						    arr.width=0, lwd=0.2, lty=1, lcol="gray10", arr.col=NA, arr.pos=F, curve=0.1, dr=NA, endhead=F)
			}
		if ((firstTimePeriod == FALSE)&(secondTimePeriod == FALSE))
			{
				for (i in 1:length(clusters2))
					{
						if (sum(!is.na(clusters2[[i]][,"longitude"])) < 3)
							{
								if (sum(!is.na(clusters2[[i]][,"longitude"])) == 2)
									{
										indices = which(!is.na(clusters2[[i]][,"longitude"]))
										if (firstTimePeriod)
											{
												indices = which((!is.na(clusters2[[i]][,"longitude"]))&(clusters2[[i]][,"collectionDate"]<decimal_date(ymd("2020-03-21"))))
											}
										if (secondTimePeriod)
											{
												indices = which((!is.na(clusters2[[i]][,"longitude"]))&(clusters2[[i]][,"collectionDate"]>decimal_date(ymd("2020-03-21"))))
											}
										if (length(indices) == 2)
											{
												curvedarrow(cbind(clusters2[[i]][indices[1],"longitude"],clusters2[[i]][indices[1],"latitude"]),
															cbind(clusters2[[i]][indices[2],"longitude"],clusters2[[i]][indices[2],"latitude"]),
															arr.length=0, arr.width=0, lwd=0.2, lty=2, lcol="gray10", arr.col=NA, arr.pos=F,
															curve=0.1, dr=NA, endhead=F)
											}
									}
							}
					}
			}
		for (i in 1:length(clusters2))
			{
				if (sum(!is.na(clusters2[[i]][,"longitude"])) < 3)
					{
						for (j in 1:dim(clusters2[[i]])[1])
							{
								if (!is.na(clusters2[[i]][j,"longitude"]))
									{
										plotTheNode = TRUE
										if ((firstTimePeriod==TRUE)&(clusters2[[i]][j,"collectionDate"]>decimal_date(ymd("2020-03-21")))) plotTheNode = FALSE
										if ((secondTimePeriod==TRUE)&(clusters2[[i]][j,"collectionDate"]<decimal_date(ymd("2020-03-21")))) plotTheNode = FALSE
										if (plotTheNode)
											{
												index = (((clusters2[[i]][j,"collectionDate"]-minYear)/(maxYear-minYear))*100)+1
												points(clusters2[[i]][j,"longitude"], clusters2[[i]][j,"latitude"], pch=16, col=colourScale[index], cex=cexNode)
												points(clusters2[[i]][j,"longitude"], clusters2[[i]][j,"latitude"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
											}
									}
							}
					}
			}
		for (i in rev(selectedBranches))
			{
				if (!mcc[i,"node1"]%in%mcc[selectedBranches,"node2"])
					{
						points(mcc[i,"startLon"], mcc[i,"startLat"], pch=16, col=startYears_colours[i], cex=cexNode)
						points(mcc[i,"startLon"], mcc[i,"startLat"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
					}
				points(mcc[i,"endLon"], mcc[i,"endLat"], pch=16, col=endYears_colours[i], cex=cexNode)
				points(mcc[i,"endLon"], mcc[i,"endLat"], pch=1, col="gray10", cex=cexNode, lwd=0.4)
			}
		dev.off()
	}

# 6. Estimating and plotting dispersal statistics associated with lineages
	
localTreesDirectory = paste0("Phylogeographic_runs/All_clades_ext")
timSlices = 50; onlyTipBranches = FALSE; showingPlots = FALSE; nberOfCores = 5; slidingWindow = 1/(365/7)
nberOfExtractionFiles = 1000; outputName = paste0("All_dispersal_statistics/50_time_slices_7days_sliding_window/",analysis)
spreadStatistics(localTreesDirectory, nberOfExtractionFiles, timSlices, onlyTipBranches, showingPlots, outputName, nberOfCores, slidingWindow) 
timSlices = 100; onlyTipBranches = FALSE; showingPlots = FALSE; nberOfCores = 5; slidingWindow = 1/(365/14)
nberOfExtractionFiles = 1000; outputName = paste0("All_dispersal_statistics/100_time_slices_14days_sliding_window/",analysis)
spreadStatistics(localTreesDirectory, nberOfExtractionFiles, timSlices, onlyTipBranches, showingPlots, outputName, nberOfCores, slidingWindow) 
tab = read.table(paste0("All_dispersal_statistics/50_time_slices_7days_sliding_window/",analysis,"_estimated_dispersal_statistics.txt"), header=T)
vWs = round(tab[,"weighted_branch_dispersal_velocity"]/366,1); cat(median(vWs)," km/day (95% HPD interval = [",quantile(vWs,0.025),"-",quantile(vWs,0.975),"])",sep="")
if (showingPlots)
	{
		dev.new(width=2.5, height=3); par(mgp=c(0,0,0), oma=c(1,1,0.5,0.5), mar=c(1.5,1.5,1,1))
		col1 = rgb(100, 100, 100, 255, maxColorValue=255); col2 = rgb(100, 100, 100, 100, maxColorValue=255)
		directory = "All_dispersal_statistics/100_time_slices_14days_sliding_window/"
		directory = "All_dispersal_statistics/50_time_slices_7days_sliding_window/"
		tab1 = read.table(paste0(directory,"/",analysis,"_median_weighted_branch_dispersal_velocity.txt"), header=T)
		tab2 = read.table(paste0(directory,"/",analysis,"_95%HPD_weighted_branch_dispersal_velocity.txt"), header=T)
		tab1[,2] = tab1[,2]/366; tab2[,2:3] = tab2[,2:3]/366 # to have the lineage dispersal velocity in km/day
		mcc = read.csv("Phylogeographic_runs/All_clades.csv", head=T)
		minYear = min(mcc[,"startYear"]); maxYear = max(mcc[,"endYear"])
		plot(tab1[,1], tab1[,2], type="l", axes=F, ann=F, ylim=c(0,50000/366), xlim=c(minYear,maxYear), col=NA)
		slicedTimes = tab1[,1]; branchDispersalVelocityMeanValue = tab1[,2]; lower_l_1 = tab2[,2]; upper_l_1 = tab2[,3]
		xx_l = c(slicedTimes,rev(slicedTimes)); yy_l = c(lower_l_1,rev(upper_l_1))
		getOption("scipen"); opt = options("scipen"=20); polygon(xx_l, yy_l, col=col2, border=0)
		lines(slicedTimes, branchDispersalVelocityMeanValue, lwd=1, col=col1)
		ats = c(min(mcc[,"startYear"]),decimal_date(ymd(c("2020-02-15","2020-03-01","2020-03-15","2020-04-01","2020-04-15"))),max(mcc[,"endYear"]))
		axis(side=2, lwd.tick=0.2, cex.axis=0.6, mgp=c(0,0.20,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30")
		axis(side=1, lwd.tick=0.2, cex.axis=0.6, mgp=c(0,0.00,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30",
			 at=ats, labels=c(" ","15-02","01-03","15-03","01-04","15-04"," "))
	}

# 7. Comparing intra- and inter-states lineage migration events through time

states_sf = sf::st_as_sf(states); nberOfExtractionFiles = 1000
localTreesDirectory = paste0("Phylogeographic_runs/All_clades_ext")
for (i in 0:nberOfExtractionFiles)
	{
		if (i == 0) tab = read.csv("Phylogeographic_runs/All_clades.csv", head=T)
		if (i >= 1) tab = read.csv(paste0(localTreesDirectory,"/TreeExtractions_",i,".csv"), head=T)
		startEndStates = matrix(nrow=dim(tab)[1], ncol=2); colnames(startEndStates) = c("startState","endState")
		pts = sf::st_as_sf(tab[,c("startLon","startLat")], coords=c("startLon","startLat"), crs=crs(states))
		indices1 = st_intersects(pts, states_sf); indices2 = rep(NA, length(indices1))
		for (j in 1:length(indices1))
			{
				if (length(indices1[[j]]) > 0) indices2[j] = indices1[[j]]
			}
		startEndStates[,"startState"] = gsub(" ","",states@data[indices2,"NAME_1"])
		pts = sf::st_as_sf(tab[,c("endLon","endLat")], coords=c("endLon","endLat"), crs=crs(states))
		indices1 = st_intersects(pts, states_sf); indices2 = rep(NA, length(indices1))
		for (j in 1:length(indices1))
			{
				if (length(indices1[[j]]) > 0) indices2[j] = indices1[[j]]
			}
		startEndStates[,"endState"] = gsub(" ","",states@data[indices2,"NAME_1"])
		tab = cbind(tab, startEndStates)
		if (i == 0) write.csv(tab, "Phylogeographic_runs/All_clades.csv", row.names=F, quote=F)
		if (i >= 1) write.csv(tab, paste0(localTreesDirectory,"/TreeExtractions_",i,".csv"), row.names=F, quote=F)
	}
timeSlices = 100; slidingWindow = 7/366
minStartYear = 2021; maxEndYear = 0
for (i in 1:nberOfExtractionFiles)
	{
		tab = read.csv(paste0(localTreesDirectory,"/TreeExtractions_",i,".csv"), head=T)
		if (minStartYear > min(tab[,"startYear"])) minStartYear = min(tab[,"startYear"])
		if (maxEndYear < max(tab[,"endYear"])) maxEndYear = max(tab[,"endYear"])
	}
timeInterval = (maxEndYear-minStartYear)/timeSlices
startEndTimes = matrix(nrow=timeSlices, ncol=3)
for (i in 1:timeSlices)
	{
		time = minStartYear+((i-1)*timeInterval)+(timeInterval/2)
		startTime = time - (slidingWindow/2)
		endTime = time + (slidingWindow/2)
		startEndTimes[i,1:3] = cbind(time, startTime, endTime)
	}
if (!file.exist("All_dispersal_statistics/All_within_state_transition_events.csv"))
	{
		withinStateTransitions = matrix(0, nrow=timeSlices, ncol=nberOfExtractionFiles)
		amongStatesTransitions = matrix(0, nrow=timeSlices, ncol=nberOfExtractionFiles)		
		for (t in 1:nberOfExtractionFiles)
			{
				tab = read.csv(paste0(localTreesDirectory,"/TreeExtractions_",t,".csv"), head=T)
				tab = tab[with(tab,order(endYear, startYear)),]; tab = tab[order(tab[,"endYear"]),]
				for (i in 1:timeSlices)
					{
						time = startEndTimes[i,1]
						startTime = startEndTimes[i,2]
						endTime = startEndTimes[i,3]
						for (j in 1:dim(tab)[1])
							{
								branchInTimeInterval = FALSE
								if ((tab[j,"startYear"]<startTime)&(tab[j,"endYear"]>endTime)) branchInTimeInterval = TRUE
								if ((tab[j,"startYear"]>startTime)&(tab[j,"startYear"]<endTime)) branchInTimeInterval = TRUE
								if ((tab[j,"endYear"]>startTime)&(tab[j,"endYear"]<endTime)) branchInTimeInterval = TRUE
								if (branchInTimeInterval == TRUE)
									{
										if ((!is.na(tab[j,"startState"]))&(!is.na(tab[j,"endState"])))
											{
										if (as.character(tab[j,"startState"]) == as.character(tab[j,"endState"])) withinStateTransitions[i,t] = withinStateTransitions[i,t]+1
										if (as.character(tab[j,"startState"]) != as.character(tab[j,"endState"])) amongStatesTransitions[i,t] = amongStatesTransitions[i,t]+1
											}
									}
							}
					}
			}
		row.names(withinStateTransitions) = startEndTimes[,1]; row.names(amongStatesTransitions) = startEndTimes[,1]
		write.csv(withinStateTransitions, "All_dispersal_statistics/All_within_state_transition_events.csv", quote=F)
		write.csv(amongStatesTransitions, "All_dispersal_statistics/All_among_states_transition_events.csv", quote=F)
	}	else	{
		withinStateTransitions = as.matrix(read.csv("All_dispersal_statistics/All_within_state_transition_events.csv", header=T))
		amongStatesTransitions = as.matrix(read.csv("All_dispersal_statistics/All_among_states_transition_events.csv", header=T))
	}
withinStateTransitions_median_95HPD = matrix(nrow=dim(withinStateTransitions)[1], ncol=3)
amongStatesTransitions_median_95HPD = matrix(nrow=dim(amongStatesTransitions)[1], ncol=3)
for (i in 1:dim(withinStateTransitions)[1])
	{
		withinStateTransitions_median_95HPD[i,1] = median(withinStateTransitions[i,])
		withinStateTransitions_median_95HPD[i,2] = quantile(withinStateTransitions[i,], probs=c(0.025,0.975), na.rm=T)[1]
		withinStateTransitions_median_95HPD[i,3] = quantile(withinStateTransitions[i,], probs=c(0.025,0.975), na.rm=T)[2]
		amongStatesTransitions_median_95HPD[i,1] = median(amongStatesTransitions[i,])
		amongStatesTransitions_median_95HPD[i,2] = quantile(amongStatesTransitions[i,], probs=c(0.025,0.975), na.rm=T)[1]
		amongStatesTransitions_median_95HPD[i,3] = quantile(amongStatesTransitions[i,], probs=c(0.025,0.975), na.rm=T)[2]
	}
if (writingFiles)
	{
		tab = cbind(startEndTimes[,1], withinStateTransitions_median_95HPD[,1], amongStatesTransitions_median_95HPD[,1])
		colnames(tab) = c("time", "within_state_transitions", "among_state_transitions")
		write.csv(tab, "All_dispersal_statistics/Within_vs_among_states_transitions.csv", row.names=F, quote=F)
	}
if (showingPlots)
	{
		dev.new(width=4.5, height=3); par(mgp=c(0,0,0), oma=c(1,1,0.5,0.5), mar=c(1.5,1.5,1,1))
		col1a = rgb(255, 0, 0, 255, maxColorValue=255); col1b = rgb(255, 0, 0, 100, maxColorValue=255)
		col2a = rgb(100, 100, 100, 255, maxColorValue=255); col2b = rgb(100, 100, 100, 100, maxColorValue=255)
		mcc = read.csv("Phylogeographic_runs/All_clades.csv", head=T)
		minYear = min(mcc[,"startYear"]); maxYear = max(mcc[,"endYear"])
		plot(startEndTimes[,1], withinStateTransitions_median_95HPD[,1], type="l", axes=F, ann=F, xlim=c(minYear,maxYear), col=NA)
		slicedTimes = startEndTimes[,1]; lower_l_1 = withinStateTransitions_median_95HPD[,2]; upper_l_1 = withinStateTransitions_median_95HPD[,3]
		xx_l = c(slicedTimes,rev(slicedTimes)); yy_l = c(lower_l_1,rev(upper_l_1))
		getOption("scipen"); opt = options("scipen"=20); polygon(xx_l, yy_l, col=col1b, border=0)
		slicedTimes = startEndTimes[,1]; lower_l_1 = amongStatesTransitions_median_95HPD[,2]; upper_l_1 = amongStatesTransitions_median_95HPD[,3]
		xx_l = c(slicedTimes,rev(slicedTimes)); yy_l = c(lower_l_1,rev(upper_l_1))
		getOption("scipen"); opt = options("scipen"=20); polygon(xx_l, yy_l, col=col2b, border=0)
		lines(startEndTimes[,1], withinStateTransitions_median_95HPD[,1], lwd=1, col=col1a, lty=1)
		lines(startEndTimes[,1], amongStatesTransitions_median_95HPD[,1], lwd=1, col=col2a, lty=1)
		ats = c(decimal_date(ymd(c("2020-03-01","2020-03-21","2020-04-15"))))
		axis(side=2, lwd.tick=0.2, cex.axis=0.6, mgp=c(0,0.20,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30")
		axis(side=1, lwd.tick=0.2, cex.axis=0.6, mgp=c(0,0.00,0), lwd=0.2, tck=-0.015, col.tick="gray30", col.axis="gray30", col="gray30",
			 at=ats, labels=c("01-03","21-03","15-04"))
	}

