%% This script edits the wind and temperature profiles of restart files (NetCDF format) for CM1 simulations.

%% Read restart file.
fname=('cm1out_reduce7K5ms_rst_000240.nc');

ncid = netcdf.open(fname,'WRITE');
varid = netcdf.inqVarID(ncid,'zh');  %get the variable id for zh
zh = netcdf.getVar(ncid,varid);   %read the data
varid = netcdf.inqVarID(ncid,'ua');  %get the variable id for ua
ua = netcdf.getVar(ncid,varid);   %read the data
varid = netcdf.inqVarID(ncid,'tha');  %get the variable id for tha (perturbation potential temp)
tha = netcdf.getVar(ncid,varid);   %read the data
varid = netcdf.inqVarID(ncid,'th0');  %get the variable id for base-state theta
th0 = netcdf.getVar(ncid,varid);   %read the data


%% Increment amount per time, separate by variable.
%% Updates happen over 61 min (3 -> 4 h) and represent
%% amount by which to *decrease* over that interval.
incT = 7/61;
incU = 10/61;


%% Variable for the current time step (1->61, updated
%% by the driving csh script).
time = 61;


%% Because we may have already reduced tha in an earlier
%% BSS application, our thanom check needs to account for
%% the amount reduced (increment amount per time * times).
thanom = (tha+(incT*time*1.0));


%% Compute the percentage of the potential temperature decrease to account for at each height.
%% Maximum change at the lowest model level, zero change above level 9 (946.97 m).
for k=1:9
    percT(k,:) = 1.0-(zh(k,:)-zh(1,:))/(zh(9,:)-zh(1,:));
end

%% Define 1-D column of potential temperature adjustments in a time step.
for k = 1:9
    thdif(k) = percT(k,:) * incT;
    for k = 10:100
        thdif(k) = 0.0;
    end
end
thdif = thdif';


%% Compute the percentage of the wind decrease to account for at each height.
%% Maximum change above level 20, no change at the lowest model level.
for k=1:20
    percU(k,:) = zh(k,:)/zh(20,:);
end

%% Define 1-D column of u-wind adjustments in a time step.
for k = 1:20
    udif(k) = percU(k,:) * incU;
    for k = 21:100
        udif(k) = incU;
    end
end
udif = udif';


%% Change the values (1-800 in x, 1-680 in y, all 100 levels in z).
%% Only act if theta' before BSS is >= 0 at lowest model level.
for i=1:800
    for j=1:680
        for k=1:100
            if thanom(i,j,1) < 0
                tha(i,j,k) = tha(i,j,k);
                ua(i,j,k) = ua(i,j,k);
            else
                tha(i,j,k) = tha(i,j,k) - thdif(k);
                ua(i,j,k) = ua(i,j,k) - udif(k);
            end
        end
    end
end

% RE-write file to same filename with new u-wind and theta anomaly values
ncwrite('cm1out_reduce7K5ms_rst_000240.nc','tha',tha);
ncwrite('cm1out_reduce7K5ms_rst_000240.nc','ua',ua);
exit

