/*
 Stage structured
 
 Header file specifying the life history an invertebrate species feeding on 2, unstructured resources (simple size-structured biomass model)
 
 Last modification: Htb - Feb 14 2020
 
 
 */

/*
 *===========================================================================
 * 		DEFINITION OF PROBLEM DIMENSIONS AND NUMERICAL SETTINGS
 *===========================================================================
 */
// Dimension settings: Required
#define POPULATION_NR		2
#define STAGES              4
#define	I_STATE_DIM         2
#define	ENVIRON_DIM         2
#define INTERACT_DIM		12
#define	PARAMETER_NR		23

// Numerical settings: Optional (default values adopted otherwise)
#define MIN_SURVIVAL		1.0E-9		// Survival at which individual is considered dead
#define MAX_AGE             20000		// Give some absolute maximum for individual age

#define DYTOL               1.0E-7		// Variable tolerance
#define RHSTOL              1.0E-7		// Function tolerance
#define ESSTOL              1.0E-9
#define COHORT_NR           0
/*
 *===========================================================================
 * 		DEFINITION OF ALIASES
 *===========================================================================
 */
// Define aliases for the istate variables
#define AGE(n)         istate[n][0]
#define WEIGHT(n)      istate[n][1]            // Mass


// Define aliases for the environmental variables
#define R1           E[0]
#define R2           E[1]

// These are other variables or parameters occurring in the equations
#define DELTA       parameter[ 0]	// Resource turnover rate, default: 0.1
#define R1MAX		parameter[ 1]	// Max density of R1, default: Variable
#define R2MAX		parameter[ 2]	// Max density of R2, default: ?

#define SIGMA       parameter[ 3]	// Assimilation efficiency, default: 0.5
#define Q           parameter[ 4]	// Competitive difference J/A, default: 1
#define Tr          parameter[ 5]	// Maintenance costs, default: 0.1
#define AMAX        parameter[ 6]	// Maximum attack rate, default: 0.6
#define AMIN        parameter[ 7]   // Minimum attack rate, default: 0.06
#define H           parameter[ 8]	// Handing time, default: 1
#define SIG         parameter[ 9]   // Steepness optimal foraging curve, default: 30

#define SM          parameter[10]   // Size at maturation: 0.1
#define MUR         parameter[11]   // Background mortality rate,default: 0.02
#define XMIN        parameter[12]   // Size at which secondary food source becomes available, default, 0.01

#define QS          parameter[13]   // Proportion of weight lost during metamorphosis, default: 0.3
#define RHO         parameter[14]   // Probability to die during metamorphosis, default: 0.5


///EVOL PARAMETERS//////
#define SB_S1          parameter[15]    // Size at birth, default: 0.001
#define SJ_S1          parameter[16]   // Size at metamorphosis, default: 0.01
#define PSI_S1         parameter[17]    // Degree of specialization R2, default:0
#define META_S1        parameter[18]   // Degree of metamorphosis, default: 0
#define SB_S2          parameter[19]    // Size at birth, default: 0.001
#define SJ_S2          parameter[20]   // Size at metamorphosis, default: 0.01
#define PSI_S2         parameter[21]    // Degree of specialization R2, default:0
#define META_S2        parameter[22]   // Degree of metamorphosis, default: 0


/*
 *===========================================================================
 * 		DEFINITION OF NAMES AND DEFAULT VALUES OF THE PARAMETERS
 *===========================================================================
 */
// At least two parameters should be specified in this array
char  *parameternames[PARAMETER_NR] =
{"Delta","R1max","R2max","Sigma","Q","T","Amax","Amin","H","Sig","SM","MUR","XMIN","QS","RHO","SB_1","SJ_1",
    "PSI_1","META_1","SB_2","SJ_2","PSI_2","META_2"
};

// These are the default parameters values
double	parameter[PARAMETER_NR] =
{0.1, 5,5,0.5,1,0.1,0.6,0.06,1,10,0.000001,0.00001,0.0001,0.02,0.00001,0,0,0.2,0.5,0.0001,0.0001,0,1};

/*
 *===========================================================================
 * 		DEFINITION OF THE LIFE HISTORY MODELS FOLLOWS BELOW
 *===========================================================================
 * Specify the number of states at birth for the individuals in all structured
 * populations in the problem in the vector BirthStates[].
 *===========================================================================
 */

void SetBirthStates(int BirthStates[POPULATION_NR], double E[])
{
    int n;
    for (n=0; n<POPULATION_NR; n++){
        BirthStates[n] = 1;
    }
    
    return;
}


/*
 *===========================================================================
 * Specify all the possible states at birth for all individuals in all
 * structured populations in the problem. BirthStateNr represents the index of
 * the state of birth to be specified. Each state at birth should be a single,
 * constant value for each i-state variable.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */
void StateAtBirth(double *istate[POPULATION_NR], int BirthStateNr, double E[])
{
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB;
        if (n==0){
            SB = SB_S1;
        }
        else {
            SB = SB_S2;
        }
        AGE(n) = 0.0;
        WEIGHT(n) = SB;
    }
    
    
    return;
}


/*
 *===========================================================================
 * Specify the threshold determining the end point of each discrete life
 * stage in individual life history as function of the i-state variables and
 * the individual's state at birth for all populations in every life stage.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void IntervalLimit(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                   double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
                   double limit[POPULATION_NR])
{
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB, SJ;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
        }
        switch (lifestage[n])
        {
            case 0:
                limit[n] = WEIGHT(n) - XMIN;       //Secondary resource available
                break;
            case 1:
                limit[n] = WEIGHT(n) - XMIN - SJ;  //Metamorphosis (SJ+XMIN = size at meta)
                break;
            case 2:
                limit[n] = WEIGHT(n) - SM;        //Maturation
                break;
            case 3:
                limit[n] = AGE(n) - MAX_AGE;     //DEAD
                break;
        }
    }
    return;
}




/*
 *===========================================================================
 * Specify the individual development of individuals as function of i-state
 * and environment for all individuals of all populations in every life stage
 *
 * Notice that the first index of the variables 'istate[][]' and 'growth[][]'
 * refers to the number of the structured population, the second index refers
 * to the number of the individual state variable. The interpretation of the
 * latter is up to the user.
 *===========================================================================
 */

void Development(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                 double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
                 double development[POPULATION_NR][I_STATE_DIM])
{
    const double	  MAX_EXP = 50.0;
    double          PSI_A,a1,a2,pw,PHI,W1,W2,a_L,attack1,attack2,nurl,nurj;
    double          PSI1, PSI_A1;
    double Intake1_L1, Intake1_L,Intake2_L,Intake1_A,Intake2_A,nurl_1,nurl_2, nura;
    
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB, SJ, PSI, META;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
            PSI = PSI_S1;
            META = META_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
            PSI = PSI_S2;
            META = META_S2;
        }
        
        PSI_A = min(1, PSI+META);
        PSI1 = 1 - PSI;
        PSI_A1 = 1- PSI_A; 
        
        if (lifestage[n] == 0)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = 0;
        }
        else if (lifestage[n] == 1)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = PSI*(AMAX-AMIN)+AMIN;
        }
        else
        {
            a1 = PSI_A1*(AMAX-AMIN)+AMIN;
            a2 = PSI_A*(AMAX-AMIN)+AMIN;
        }
        
        pw = -SIG * (a1  * R1 - a2  * R2);
        pw = max(pw, -MAX_EXP);
        pw = min(pw,  MAX_EXP);
        pw = exp(pw);
        
        PHI = 1 / (1 + pw);
        W1 = PHI;
        W2 = (1-PHI);
        
        
        //Mass-specific intake rates//
        Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
        Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
        Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
        Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
        Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
        
        
        ///Check if net-energy production is positive or not 
        
        //Small larvae
        if ((SIGMA*Intake1_L1-Tr)>0){
            nurl_1    =   SIGMA*Intake1_L1-Tr;
        }
        else {
            nurl_1 = 0 ;
        }
        //Big larvae
        if ((SIGMA*(Intake1_L+Intake2_L)-Tr) >0) {
            nurl_2    =   SIGMA*(Intake1_L+Intake2_L)-Tr;
        }
        else {
            nurl_2 = 0;
        }
        //Everyone else
        if ((SIGMA*(Intake1_A+Intake2_A)-Tr) >0) {
            nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
        }
        else {
            nura = 0;
        }
        
        
        development[n][0] = 1;
        
        if (lifestage[n] == 0)  //Larvae, No second resource
        {
            development[n][1]= nurl_1*WEIGHT(n);
        }
        else if (lifestage[n] == 1)  //Larvae, secondary resource
        {
            development[n][1] = nurl_2*WEIGHT(n);
        }
        else if (lifestage[n] == 2)  // Metamorphosed juveniles
        {
            development[n][1] = nura*WEIGHT(n);
        }
        else   //metamorphosed adults
        {
            development[n][1] = 0;
        }
        
        /*if (WEIGHT(n) >= SM) {
            development[n][1] = 0;
        }*/
        
    }
    return;
}


/*
 *===========================================================================
 * Specify the possible discrete changes (jumps) in the individual state
 * variables when ENTERING the stage specified by 'lifestage[]'.
 *
 * Notice that the first index of the variables 'istate[][]' and 'growth[][]'
 * refers to the number of the structured population, the second index refers
 * to the number of the individual state variable. The interpretation of the
 * latter is up to the user.
 *===========================================================================
 */

void DiscreteChanges(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                     double *birthstate[POPULATION_NR], int BirthStateNr, double E[])
{
    
    int n;
    for (n=0; n<POPULATION_NR; n++){
        if (lifestage[n] == 2) {
        double SB, SJ, PSI, META;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
            PSI = PSI_S1;
            META = META_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
            PSI = PSI_S2;
            META = META_S2;
        }
        {
            //WEIGHT(n) = WEIGHT(n)-WEIGHT(n)*META*QS; //The higher Qs, the more expensive (Qs is 1-qs!!)
            WEIGHT(n) = WEIGHT(n) - META*QS * max(0,(WEIGHT(n) - SB));
            SetSurvival(n,(1.0 - META*RHO)*exp(istate[n][I_STATE_DIM]));
            
        }
    }
    }
    return;
}


/*
 *===========================================================================
 * Specify the fecundity of individuals as a function of the i-state
 * variables and the individual's state at birth for all populations in every
 * life stage.
 *
 * The number of offspring produced has to be specified for every possible
 * state at birth in the variable 'fecundity[][]'. The first index of this
 * variable refers to the number of the structured population, the second
 * index refers to the number of the birth state.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void Fecundity(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
               double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
                                                                            double *fecundity[POPULATION_NR])
{
    
    double          PSI_A,a1,a2,pw,PHI,W1,W2,a_L,attack1,attack2,nurj;
    const double	  MAX_EXP = 50.0;
    double          PSI1, PSI_A1;
    double Intake1_A, Intake2_A;
    double surv, nura;
    
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB, SJ, PSI, META;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
            PSI = PSI_S1;
            META = META_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
            PSI = PSI_S2;
            META = META_S2;
        }
        
        //specialization on the different resources//
        PSI_A = min(1, PSI+META);
        PSI1 = 1 - PSI;
        PSI_A1 = 1- PSI_A; 
        
        //attack rates depending on specialization
        if (lifestage[n] == 0)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = 0;
        }
        else if (lifestage[n] == 1)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = PSI*(AMAX-AMIN)+AMIN;
        }
        else
        {
            a1 = PSI_A1*(AMAX-AMIN)+AMIN;
            a2 = PSI_A*(AMAX-AMIN)+AMIN;
        }
        
        //proportion of time on R1 & R2
        pw = -SIG * (a1  * R1 - a2  * R2);
        pw = max(pw, -MAX_EXP);
        pw = min(pw,  MAX_EXP);
        pw = exp(pw);
        
        PHI = 1 / (1 + pw);
        W1 = PHI;
        W2 = (1-PHI);
        
        
        //Intake rate of adults
        Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
        Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
        
        
        //Egg survival
        if (SB > XMIN+SJ){
            surv = min(1.0,(1.0 - META*RHO));
        } else {
            surv = 1;
        }
        
        if ((SIGMA*(Intake1_A+Intake2_A)-Tr) >0) {
            nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
        }
        else {
            nura = 0;
        }
        
        
        if (lifestage[n] == 3)
        {
            fecundity[n][0] = surv*(nura*WEIGHT(n))/SB;
        }
        else
        {
            fecundity[n][0] = 0.0;
        }
    }
    
    return;
}



/*
 *===========================================================================
 * Specify the mortality of individuals as a function of the i-state
 * variables and the individual's state at birth for all populations in every
 * life stage.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void Mortality(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
               double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
               double mortality[POPULATION_NR])
{
    double murtotl_1,murtotl_2, murtotj;
    const double	  MAX_EXP = 50.0;
    double          PSI_A,a1,a2,pw,PHI,W1,W2,a_L,attack1,attack2,nurl,nurj;
    double          PSI1, PSI_A1;
    double          Intake1_L1, Intake1_L,Intake2_L,Intake1_A,Intake2_A,nurl_1,nurl_2, nura;
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB, SJ, PSI, META;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
            PSI = PSI_S1;
            META = META_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
            PSI = PSI_S2;
            META = META_S2;
        }
        
        PSI_A = min(1, PSI+META);
        PSI1 = 1 - PSI;
        PSI_A1 = 1- PSI_A;
        
        
        if (lifestage[n] == 0)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = 0;
        }
        else if (lifestage[n] == 1)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = PSI*(AMAX-AMIN)+AMIN;
        }
        else
        {
            a1 = PSI_A1*(AMAX-AMIN)+AMIN;
            a2 = PSI_A*(AMAX-AMIN)+AMIN;
            
        }
        
        pw = -SIG * (a1  * R1 - a2  * R2);
        pw = max(pw, -MAX_EXP);
        pw = min(pw,  MAX_EXP);
        pw = exp(pw);
        
        PHI = 1 / (1 + pw);
        W1 = PHI;
        W2=(1-PHI);
        
        
        ///Mass specific intake rates
        Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
        Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
        Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
        Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
        Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
        
        ///Check if net-biomass production is posiitve
        //Small larvae
        if ((SIGMA*Intake1_L1-Tr)<0){
            nurl_1    =   SIGMA*Intake1_L1-Tr;
        }
        else {
            nurl_1 = 0 ;
        }
        //Big larvae
        if ((SIGMA*(Intake1_L+Intake2_L)-Tr)<0) {
            nurl_2    =   SIGMA*(Intake1_L+Intake2_L)-Tr;
        }
        else {
            nurl_2 = 0;
        }
        //Everyone else
        if ((SIGMA*(Intake1_A+Intake2_A)-Tr)<0) {
            nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
        }
        else {
            nura = 0;
        }
        
        ///mortality rates
        murtotl_1 =   MUR-nurl_1;
        murtotl_2 =   MUR-nurl_2;
        murtotj   =   MUR-nura;
        
        
        switch(lifestage[n])
        {
            case 0:
                mortality[n] = murtotl_1;
                break;
            case 1:
                mortality[n] = murtotl_2;
                break;
            case 2:
                mortality[n] = murtotj;
                break;
            case 3:
                mortality[n] = murtotj;
                break;
        }
    }
    
    
    return;
}


/*
 *===========================================================================
 * For all the integrals (measures) that occur in interactions of the
 * structured populations with their environments and for all the integrals
 * that should be computed for output purposes (e.g. total juvenile or adult
 * biomass), specify appropriate weighing function dependent on the i-state
 * variables, the environment variables and the current life stage of the
 * individuals. These weighing functions should be specified for all
 * structured populations in the problem. The number of weighing functions
 * is the same for all of them.
 *
 * Notice that the first index of the variables 'istate[][]' and 'impact[][]'
 * refers to the number of the structured population, the second index of the
 * variable 'istate[][]' refers to the number of the individual state variable,
 * while the second index of the variable 'impact[][]' refers to the number of
 * the interaction variable. The interpretation of these second indices is up
 * to the user.
 *===========================================================================
 */

void Impact(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
            double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
            double impact[POPULATION_NR][INTERACT_DIM])
{
    
    double          PSI_A,a1,a2,pw,PHI,W1,W2,a_L,attack1,attack2, In1,In2;
    const double	  MAX_EXP = 50.0;
    double          PSI1, PSI_A1;
    double          Intake1_L1, Intake1_L, Intake2_L,Intake1_A, Intake2_A;;
    
    int n;
    for (n=0; n<POPULATION_NR; n++){
        double SB, SJ, PSI, META;
        if (n==0){
            SB = SB_S1;
            SJ = SJ_S1;
            PSI = PSI_S1;
            META = META_S1;
        }
        else {
            SB = SB_S2;
            SJ = SJ_S2;
            PSI = PSI_S2;
            META = META_S2;
        }
        
        //specialization on the different resources//
        PSI_A = min(1, PSI+META);
        PSI1 = 1 - PSI;
        PSI_A1 = 1- PSI_A; 
        
        
        if (lifestage[n] == 0)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = 0;
        }
        else if (lifestage[n] == 1)
        {
            a1 = PSI1*(AMAX-AMIN)+AMIN;
            a2 = PSI*(AMAX-AMIN)+AMIN;
        }
        else
        {
            a1 = PSI_A1*(AMAX-AMIN)+AMIN;
            a2 = PSI_A*(AMAX-AMIN)+AMIN;
            
        }
        
        pw = -SIG * (a1  * R1 - a2  * R2);
        pw = max(pw, -MAX_EXP);
        pw = min(pw,  MAX_EXP);
        pw = exp(pw);
        
        PHI = 1 / (1 + pw);
        W1 = PHI;
        W2=(1-PHI);
        
        
        ///Mass specific intake rates
        Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
        Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
        Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
        Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
        Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
        
        
        //Intake rates of an individual/////
        if (lifestage[n] == 0)
        {
            In1 = Intake1_L1*WEIGHT(n);
            In2 = 0;
        }
        else if (lifestage[n] == 1)
        {
            In1 = Intake1_L*WEIGHT(n);
            In2 = Intake2_L*WEIGHT(n);
        }
        else 
        {
            In1 = Intake1_A*WEIGHT(n);
            In2 = Intake2_A*WEIGHT(n);
        }
        
        
        
        
        switch (lifestage[n])
        {
            case 0:
                impact[n][0] = In1;                                             // Ingestion R1
                impact[n][1] = 0.0;                                             // Ingestion R2
                impact[n][2] = WEIGHT(n);                                          // Larvae  mass
                impact[n][3] = 0;                                               // Larvae2 mass
                impact[n][4] = 0.0;                                             // Juvenile mass
                impact[n][5] = 0.0;                                             // Adult mass
                impact[n][6] = 1.0;                                             // Larvae
                impact[n][7] = 0.0;                                             // Larvae 2
                impact[n][8] = 0.0;                                             // Juveniles
                impact[n][9] = 0.0;                                             // Adults
                impact[n][10] = 0.0;
                impact[n][11] = 0.0;
                break;
                
            case 1:
                impact[n][0] = In1;                                             // Ingestion R1
                impact[n][1] = In2;                                             // Ingestion R2
                impact[n][2] = 0;                                               // Larvae  mass
                impact[n][3] = WEIGHT(n);                                          // Larvae2 mass
                impact[n][4] = 0.0;                                             // Juvenile mass
                impact[n][5] = 0.0;                                             // Adult mass
                impact[n][6] = 0.0;                                             // Larvae
                impact[n][7] = 1.0;                                             // Larvae 2
                impact[n][8] = 0.0;                                             // Juveniles
                impact[n][9] = 0.0;                                             // Adults
                impact[n][10] = In1;
                impact[n][11] = In2;
                break;
                
            case 2:
                impact[n][0] = In1;                                             // Ingestion R1
                impact[n][1] = In2;                                             // Ingestion R2
                impact[n][2] = 0.0;                                             // Larvae  mass
                impact[n][3] = 0.0;                                             // Larvae2 mass
                impact[n][4] = WEIGHT(n);                                          // Juvenile mass
                impact[n][5] = 0.0;                                             // Adult mass
                impact[n][6] = 0.0;                                             // Larvae
                impact[n][7] = 0.0;                                             // Larvae 2
                impact[n][8] = 1.0;                                             // Juveniles
                impact[n][9] = 0.0;                                             // Adults
                impact[n][10] = In1;
                impact[n][11] = In2;
                break;
                
            case 3:
                impact[n][0] = In1;                                             // Ingestion R1
                impact[n][1] = In2;                                             // Ingestion R2
                impact[n][2] = 0.0;                                             // Larvae  mass
                impact[n][3] = 0.0;                                             // Larvae2 mass
                impact[n][4] = 0.0;                                             // Juvenile mass
                impact[n][5] = WEIGHT(n);                                          // Adult mass
                impact[n][6] = 0.0;                                             // Larvae
                impact[n][7] = 0.0;                                             // Larvae 2
                impact[n][8] = 0.0;                                             // Juveniles
                impact[n][9] = 1.0;                                             // Adults
                impact[n][10] = In1;
                impact[n][11] = In2;
                break;
        }
    }
    return;
}


/*
 *===========================================================================
 * Specify the type of each of the environment variables by setting
 * the entries in EnvironmentType[ENVIRON_DIM] to PERCAPITARATE, GENERALODE
 * or POPULATIONINTEGRAL based on the classification below:
 *
 * Set an entry to PERCAPITARATE if the dynamics of E[j] follow an ODE and 0
 * is a possible equilibrium state of E[j]. The ODE is then of the form
 * dE[j]/dt = P(E,I)*E[j], with P(E,I) the per capita growth rate of E[j].
 * Specify the equilibrium condition as condition[j] = P(E,I), do not include
 * the multiplication with E[j] to allow for detecting and continuing the
 * transcritical bifurcation between the trivial and non-trivial equilibrium.
 *
 * Set an entry to GENERALODE if the dynamics of E[j] follow an ODE and 0 is
 * NOT an equilibrium state of E. The ODE then has a form dE[j]/dt = G(E,I).
 * Specify the equilibrium condition as condition[j] = G(E,I).
 *
 * Set an entry to POPULATIONINTEGRAL if E[j] is a (weighted) integral of the
 * population distribution, representing for example the total population
 * biomass. E[j] then can be expressed as E[j] = I[p][i]. Specify the
 * equilibrium condition in this case as condition[j] = I[p][i].
 *
 * Notice that the first index of the variable 'I[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the interaction variable. The interpretation of the latter
 * is up to the user. Also notice that the variable 'condition[j]' should
 * specify the equilibrium condition of environment variable 'E[j]'.
 *===========================================================================
 */

const int EnvironmentType[ENVIRON_DIM] = {GENERALODE, GENERALODE};

void EnvEqui(double E[], double I[POPULATION_NR][INTERACT_DIM],
             double condition[ENVIRON_DIM])
{
    condition[0] = DELTA*(R1MAX - R1) - I[0][0] - I[1][0];
    condition[1] = DELTA*(R2MAX - R2) - I[0][1] - I[1][1];
    return;
}

/*==============================================================================*/

