/*
 
 
 Header file specifying the life history an invertebrate species feeding on 2, unstructured resources (simple size-structured biomass model)
 
 Last modification: Htb - Feb 24 2020
 

 */

/*
 *===========================================================================
 * 		DEFINITION OF PROBLEM DIMENSIONS AND NUMERICAL SETTINGS
 *===========================================================================
 */
// Dimension settings: Required
#define POPULATION_NR		1
#define STAGES              4
#define	I_STATE_DIM         2
#define	ENVIRON_DIM         2
#define INTERACT_DIM		10
#define	PARAMETER_NR		19

// Numerical settings: Optional (default values adopted otherwise)
#define MIN_SURVIVAL		1.0E-9		// Survival at which individual is considered dead
#define MAX_AGE             20000		  // Give some absolute maximum for individual age

#define DYTOL               1.0E-7		// Variable tolerance
#define RHSTOL              1.0E-7		// Function tolerance
#define ESSTOL              1.0E-7
#define COHORT_NR           20000
/*
 *===========================================================================
 * 		DEFINITION OF ALIASES
 *===========================================================================
 */
// Define aliases for the istate variables
#define AGE         istate[0][0]
#define WEIGHT      istate[0][1]            

// Define aliases for the environmental variables
#define R1           E[0]
#define R2           E[1]

// These are other variables or parameters occurring in the equations
#define DELTA       parameter[ 0]	// Resource turnover rate, default: 0.1
#define R1MAX		    parameter[ 1]	// Max density of R1, default: Variable
#define R2MAX		    parameter[ 2]	// Max density of R2, default: Variable

#define SIGMA       parameter[ 3]	// Assimilation efficiency, default: 0.5
#define Q           parameter[ 4]	// Competitive difference between Pre & postmetamorphs, default:1
#define Tr          parameter[ 5]	// Maintenance costs, default: 0.1
#define AMAX        parameter[ 6]	// Maximum attack rate, default: 0.6
#define AMIN        parameter[ 7] // Minimum attack rate, default: 0.06
#define H           parameter[ 8]	// Handing time, default: 1
#define SIG         parameter[ 9] // Steepness optimal foraging curve, default: 100

#define SB          parameter[10]	// Size at birth, default: 0.001
#define SJ          parameter[11] // Size at metamorphosis, default: 0.01
#define SM          parameter[12] // Size at maturation: 0.1

#define MUR         parameter[13] // Background mortality rate,default: 0.02

#define XMIN        parameter[14] // Size at which secondary food source becomes available, default, 0.01
#define PSI         parameter[15]	// Degree of specialization R2, default:0
#define META        parameter[16] // Degree of metamorphosis, default: 1
#define QS          parameter[17] // Proportion of weight lost during metamorphosis, default: 0.4
#define RHO         parameter[18] // Probability to die during metamorphosis, default: 0.5




/*
 *===========================================================================
 * 		DEFINITION OF NAMES AND DEFAULT VALUES OF THE PARAMETERS
 *===========================================================================
 */
// At least two parameters should be specified in this array
char  *parameternames[PARAMETER_NR] =
{"Delta","R1max","R2max","Sigma","Q","T","Amax","Amin","H","Sig","SB","SJ","SM","MUR","XMIN","PSI","META","QS","RHO"};

// These are the default parameters values
double	parameter[PARAMETER_NR] =
{0.1, 5,5,0.5,1,0.1,0.6,0.06,1,10,0.000001,0.00001,0.0001,0.02,0.00001,0,0,0.2,0.5};

/*
 *===========================================================================
 * 		DEFINITION OF THE LIFE HISTORY MODELS FOLLOWS BELOW
 *===========================================================================
 * Specify the number of states at birth for the individuals in all structured
 * populations in the problem in the vector BirthStates[].
 *===========================================================================
 */

void SetBirthStates(int BirthStates[POPULATION_NR], double E[])
{

    BirthStates[0] = 1;
    
    return;
}


/*
 *===========================================================================
 * Specify all the possible states at birth for all individuals in all
 * structured populations in the problem. BirthStateNr represents the index of
 * the state of birth to be specified. Each state at birth should be a single,
 * constant value for each i-state variable.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */
void StateAtBirth(double *istate[POPULATION_NR], int BirthStateNr, double E[])
{
    
    AGE = 0.0;
    WEIGHT = SB;
    
    
    return;
}


/*
 *===========================================================================
 * Specify the threshold determining the end point of each discrete life
 * stage in individual life history as function of the i-state variables and
 * the individual's state at birth for all populations in every life stage.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void IntervalLimit(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                   double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
                   double limit[POPULATION_NR])
{ 
  //Larvae that cannot feed upon second resource (L1), Larvae that can feed upon second resource (L2) [both premetamorphosis]
  //Juveniles [J], and Adults [A] [Both postmetamorphosis]
  
    switch (lifestage[0])
    {
        case 0:
            limit[0] = WEIGHT - XMIN;       //Secondary resource available 
            break;
        case 1:
            limit[0] = WEIGHT - XMIN - SJ;  //Metamorphosis (SJ+XMIN = size at meta)
            break;
        case 2:
            limit[0] = WEIGHT - SM;        //Maturation
            break;
        case 3:
            limit[0] = AGE - MAX_AGE;     //DEAD 
            break;

    }
  
    
    return;
}




/*
 *===========================================================================
 * Specify the individual development of individuals as function of i-state
 * and environment for all individuals of all populations in every life stage
 *
 * Notice that the first index of the variables 'istate[][]' and 'growth[][]'
 * refers to the number of the structured population, the second index refers
 * to the number of the individual state variable. The interpretation of the
 * latter is up to the user.
 *===========================================================================
 */

void Development(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                 double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
                                                                              double development[POPULATION_NR][I_STATE_DIM])
{
  
  const double	  MAX_EXP = 50.0;
  double          PSI_A,a1,a2,pw,PHI,W1,W2;
  double          PSI1, PSI_A1;
  double Intake1_L1, Intake1_L,Intake2_L,Intake1_A,Intake2_A,nurl_1,nurl_2, nura;
  
  //Specialization on the two different resources///
  PSI_A = min(1.0, PSI+META);
  PSI1 = 1 - PSI;
  PSI_A1 = 1 - PSI_A; 
  
  //Attack rates, depending on specialization//
  if (lifestage[0] == 0) //Small larvae
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = 0;
  }
  else if (lifestage[0] == 1)  //Large larvae
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = PSI*(AMAX-AMIN)+AMIN;
  }
  else  //postmetamorphs
  {
    a1 = PSI_A1*(AMAX-AMIN)+AMIN;
    a2 = PSI_A*(AMAX-AMIN)+AMIN;
  }
  
  //Time spend foraging on R1 & R2
  pw = -SIG * (a1  * R1 - a2  * R2);
  pw = max(pw, -MAX_EXP);
  pw = min(pw,  MAX_EXP);
  pw = exp(pw);
  
  PHI = 1 / (1 + pw);
  W1 = PHI;
  W2 = (1-PHI);
  
  //Mass-specific intake rates//
  Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
  Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
  Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
  Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
  Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
  
  
  ///Check if net-energy production is positive or not 
  
  //Small larvae
  if ((SIGMA*Intake1_L1-Tr)>0){
    nurl_1    =   SIGMA*Intake1_L1-Tr;
  }
  else {
    nurl_1 = 0 ;
  }
  //Big larvae
  if ((SIGMA*(Intake1_L+Intake2_L)-Tr) >0) {
    nurl_2    =   SIGMA*(Intake1_L+Intake2_L)-Tr;
  }
  else {
    nurl_2 = 0;
  }
  //Everyone else
  if ((SIGMA*(Intake1_A+Intake2_A)-Tr) >0) {
    nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
  }
  else {
    nura = 0;
  }
  
  //Increase in age///
  development[0][0] = 1;
  
  //Increase in weight//
  if (lifestage[0] == 0)  //Small Larvae, no access to second resource
  {
    development[0][1]= nurl_1*WEIGHT;
  }
  else if (lifestage[0] == 1)  //Big Larvae, access to secondary resource
  {
    development[0][1] = nurl_2*WEIGHT; 
  }
  else if (lifestage[0] == 2)  // Metamorphosed juveniles
  {
    development[0][1] = nura*WEIGHT;
  }
  else   //metamorphosed adults
  {
    development[0][1] = 0;
  }
  
  /*if (WEIGHT > SM) {
    development[0][1] = 0;
  }*/
  
  
  return;
}


/*
 *===========================================================================
 * Specify the possible discrete changes (jumps) in the individual state
 * variables when ENTERING the stage specified by 'lifestage[]'.
 *
 * Notice that the first index of the variables 'istate[][]' and 'growth[][]'
 * refers to the number of the structured population, the second index refers
 * to the number of the individual state variable. The interpretation of the
 * latter is up to the user.
 *===========================================================================
 */

void DiscreteChanges(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
                     double *birthstate[POPULATION_NR], int BirthStateNr, double E[])
{
   if ((lifestage[0] == 2) && ((SB - (XMIN+SJ)) <= 1E-9) )//Metamorphosis
    {
     WEIGHT = WEIGHT - META * QS * max(0, (WEIGHT - SB)); //Decrease in weight during metamorphosis
     SetSurvival(0,(1.0 - META*RHO)*exp(istate[0][I_STATE_DIM])); //the higher RHO, the riskier metamorphosis
    }
    return;
}

//Old ones
//WEIGHT = WEIGHT - META * WEIGHT * QS; //The higher Qs, the more expensive (Qs is 1-qs!!)
//SetSurvival(0, (( RHO - 1) * META + 1)*exp(istate[0][I_STATE_DIM])); //The higher Rho, the more risky? <- Seems wrong and otherway around?


/*
 *===========================================================================
 * Specify the fecundity of individuals as a function of the i-state
 * variables and the individual's state at birth for all populations in every
 * life stage.
 *
 * The number of offspring produced has to be specified for every possible
 * state at birth in the variable 'fecundity[][]'. The first index of this
 * variable refers to the number of the structured population, the second
 * index refers to the number of the birth state.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void Fecundity(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
               double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
               double *fecundity[POPULATION_NR])
{
  
  double          PSI_A,a1,a2,pw,PHI,W1,W2;
  const double	  MAX_EXP = 50.0;
  double          PSI1, PSI_A1;
  double          Intake1_A,Intake2_A,nura, Intake1_L, Intake2_L;
  double          surv, nurl_2;
  
  //specialization on the different resources//
  PSI_A = min(1.0, PSI+META);
  PSI1 = 1 - PSI;
  PSI_A1 = 1 - PSI_A; 
  
  //Attack rates, depending on specialization//
  if (lifestage[0] == 0)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = 0;
  }
  else if (lifestage[0] == 1)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = PSI*(AMAX-AMIN)+AMIN;
  }
  else
  {
    a1 = PSI_A1*(AMAX-AMIN)+AMIN;
    a2 = PSI_A*(AMAX-AMIN)+AMIN;
  }
  
  //proportion of time spend foraging on R1 & R2///  
  pw = -SIG * (a1  * R1 - a2  * R2);
  pw = max(pw, -MAX_EXP);
  pw = min(pw,  MAX_EXP);
  pw = exp(pw);
  
  PHI = 1 / (1 + pw);
  W1 = PHI;
  W2 = (1-PHI);
  
  //Intake rate of adults
  Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
  Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
  Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
  Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
  
  
  //Net-biomass production///
  if ((SIGMA*(Intake1_L+Intake2_L)-Tr) >0) {
    nurl_2    =   SIGMA*(Intake1_L+Intake2_L)-Tr;
  }
  else {
    nurl_2 = 0;
  }
  if ((SIGMA*(Intake1_A+Intake2_A)-Tr) >0) {
    nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
  }
  else {
    nura = 0;
  }
  
  //Egg survival
  if ((SB - (XMIN+SJ)) <= 1E-9){
    surv = 1;
  } else {
    surv = min(1.0,(1.0 - META*RHO));
  }
  
  //fecundity
  if (lifestage[0] ==3)
  {
    fecundity[0][0] = surv * (nura*WEIGHT)/SB;
  }
  
  else 
  {
    fecundity[0][0] = 0.0;
  }
  
  return;
}



/*
 *===========================================================================
 * Specify the mortality of individuals as a function of the i-state
 * variables and the individual's state at birth for all populations in every
 * life stage.
 *
 * Notice that the first index of the variable 'istate[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the individual state variable. The interpretation of the latter
 * is up to the user.
 *===========================================================================
 */

void Mortality(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
               double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
               double mortality[POPULATION_NR])
{
  
  double          murtotl_1,murtotl_2,murtotj;
  const double	  MAX_EXP = 50.0;
  double          PSI_A,a1,a2,pw,PHI,W1,W2;
  double          PSI1, PSI_A1;
  double          Intake1_L1, Intake1_L,Intake2_L,Intake1_A,Intake2_A,nurl_1,nurl_2, nura;
  
  //specialization on the different resources//
  PSI_A = min(1.0, PSI+META);
  PSI1 = 1 - PSI;
  PSI_A1 = 1- PSI_A; 
  
  //Attack rates, depending on specialization//
  if (lifestage[0] == 0)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = 0;
  }
  else if (lifestage[0] == 1)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = PSI*(AMAX-AMIN)+AMIN;
  }
  else
  {
    a1 = PSI_A1*(AMAX-AMIN)+AMIN;
    a2 = PSI_A*(AMAX-AMIN)+AMIN;
  }
  
  //proportion of time spend foraging on R1 & R2///  
  pw = -SIG * (a1  * R1 - a2  * R2);
  pw = max(pw, -MAX_EXP);
  pw = min(pw,  MAX_EXP);
  pw = exp(pw);
  
  PHI = 1 / (1 + pw);
  W1 = PHI;
  W2 = ( 1 - PHI);
  
  ///Mass specific intake rates
  Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
  Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
  Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
  Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
  Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
  
  ///Check if net-biomass production is posiitve
  //Small larvae
  if ((SIGMA*Intake1_L1-Tr)<0){
    nurl_1    =   SIGMA*Intake1_L1-Tr;
  }
  else {
    nurl_1 = 0 ;
  }
  //Big larvae
  if ((SIGMA*(Intake1_L+Intake2_L)-Tr)<0) {
    nurl_2    =   SIGMA*(Intake1_L+Intake2_L)-Tr;
  }
  else {
    nurl_2 = 0;
  }
  //Everyone else
  if ((SIGMA*(Intake1_A+Intake2_A)-Tr)<0) {
    nura    =   SIGMA*(Intake1_A+Intake2_A)-Tr;
  }
  else {
    nura = 0;
  }
  
  ///mortality rates
  murtotl_1 =   MUR-nurl_1;
  murtotl_2 =   MUR-nurl_2;
  murtotj   =   MUR-nura;
    
  
  switch(lifestage[0])
  {
  case 0:
    mortality[0] = murtotl_1;
    break;
  case 1:
    mortality[0] = murtotl_2;
    break;
  case 2:
    mortality[0] = murtotj;
    break;
  case 3:
    mortality[0] = murtotj;
    break;
  }
  
  
  return;
}


/*
 *===========================================================================
 * For all the integrals (measures) that occur in interactions of the
 * structured populations with their environments and for all the integrals
 * that should be computed for output purposes (e.g. total juvenile or adult
 * biomass), specify appropriate weighing function dependent on the i-state
 * variables, the environment variables and the current life stage of the
 * individuals. These weighing functions should be specified for all
 * structured populations in the problem. The number of weighing functions
 * is the same for all of them.
 *
 * Notice that the first index of the variables 'istate[][]' and 'impact[][]'
 * refers to the number of the structured population, the second index of the
 * variable 'istate[][]' refers to the number of the individual state variable,
 * while the second index of the variable 'impact[][]' refers to the number of
 * the interaction variable. The interpretation of these second indices is up
 * to the user.
 *===========================================================================
 */

void Impact(int lifestage[POPULATION_NR], double *istate[POPULATION_NR],
            double *birthstate[POPULATION_NR], int BirthStateNr, double E[],
            double impact[POPULATION_NR][INTERACT_DIM])
{
  
  double          PSI_A,a1,a2,pw,PHI,W1,W2, In1, In2;
  const double	  MAX_EXP = 50.0;
  double          PSI1, PSI_A1;
  double          Intake1_L1, Intake1_L,Intake2_L,Intake1_A,Intake2_A;
  
  //specialization on the different resources//
  PSI_A = min(1, PSI+META);
  PSI1 = 1 - PSI;
  PSI_A1 = 1- PSI_A; 
  
  //Attack rates, depending on specialization//
  if (lifestage[0] == 0)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = 0;
  }
  else if (lifestage[0] == 1)
  {
    a1 = PSI1*(AMAX-AMIN)+AMIN;
    a2 = PSI*(AMAX-AMIN)+AMIN;
  }
  else
  {
    a1 = PSI_A1*(AMAX-AMIN)+AMIN;
    a2 = PSI_A*(AMAX-AMIN)+AMIN;
  }
  
  //proportion of time spend foraging on R1 & R2///  
  pw = -SIG * (a1  * R1 - a2  * R2);
  pw = max(pw, -MAX_EXP);
  pw = min(pw,  MAX_EXP);
  pw = exp(pw);
  
  PHI = 1 / (1 + pw);
  W1 = PHI;
  W2 = ( 1 - PHI);
  
  ///Mass specific intake rates
  Intake1_L1    =   Q*a1*R1/(1+Q*(a1*H*R1));                             //Intake rate R1 Larvae_1
  Intake1_L     =   Q*W1*a1*R1/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R1 Larvae_2
  Intake2_L     =   Q*W2*a2*R2/(1+Q*(W1*a1*H*R1+W2*a2*H*R2));            //Intake rate R2 Larvae_2
  Intake1_A     =   W1*a1*R1/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X1 for Juveniles and adults
  Intake2_A     =   W2*a2*R2/(1+W1*a1*H*R1+W2*a2*H*R2);                  //Intake rate on X2 for Juveniles and adults
  
  
  //Intake rates of an individual/////
  if (lifestage[0] == 0)
  {
    In1 = Intake1_L1*WEIGHT;
    In2 = 0;
  }
  else if (lifestage[0] == 1)
  {
    In1 = Intake1_L*WEIGHT;
    In2 = Intake2_L*WEIGHT;
  }
  else 
  {
    In1 = Intake1_A*WEIGHT;
    In2 = Intake2_A*WEIGHT;
  }
  
  
  
  
  switch (lifestage[0])
  {
  case 0:
    impact[0][0] = In1;                                             // Ingestion R1
    impact[0][1] = 0.0;                                             // Ingestion R2
    impact[0][2] = WEIGHT;                                          // Larvae  mass
    impact[0][3] = 0;                                               // Larvae2 mass
    impact[0][4] = 0.0;                                             // Juvenile mass
    impact[0][5] = 0.0;                                             // Adult mass
    impact[0][6] = 1.0;                                             // Larvae
    impact[0][7] = 0.0;                                             // Larvae 2
    impact[0][8] = 0.0;                                             // Juveniles
    impact[0][9] = 0.0;                                             // Adults
    break;
    
  case 1:
    impact[0][0] = In1;                                             // Ingestion R1
    impact[0][1] = In2;                                             // Ingestion R2
    impact[0][2] = 0;                                               // Larvae  mass
    impact[0][3] = WEIGHT;                                          // Larvae2 mass
    impact[0][4] = 0.0;                                             // Juvenile mass
    impact[0][5] = 0.0;                                             // Adult mass
    impact[0][6] = 0.0;                                             // Larvae
    impact[0][7] = 1.0;                                             // Larvae 2
    impact[0][8] = 0.0;                                             // Juveniles
    impact[0][9] = 0.0;                                             // Adults
    break;
    
  case 2:
    impact[0][0] = In1;                                             // Ingestion R1
    impact[0][1] = In2;                                             // Ingestion R2
    impact[0][2] = 0.0;                                             // Larvae  mass
    impact[0][3] = 0.0;                                             // Larvae2 mass
    impact[0][4] = WEIGHT;                                          // Juvenile mass
    impact[0][5] = 0.0;                                             // Adult mass
    impact[0][6] = 0.0;                                             // Larvae
    impact[0][7] = 0.0;                                             // Larvae 2
    impact[0][8] = 1.0;                                             // Juveniles
    impact[0][9] = 0.0;                                             // Adults
    break;
    
  case 3:
    impact[0][0] = In1;                                             // Ingestion R1
    impact[0][1] = In2;                                             // Ingestion R2
    impact[0][2] = 0.0;                                             // Larvae  mass
    impact[0][3] = 0.0;                                             // Larvae2 mass
    impact[0][4] = 0.0;                                             // Juvenile mass
    impact[0][5] = WEIGHT;                                          // Adult mass
    impact[0][6] = 0.0;                                             // Larvae
    impact[0][7] = 0.0;                                             // Larvae 2
    impact[0][8] = 0.0;                                             // Juveniles
    impact[0][9] = 1.0;                                             // Adults
    break;
  }
  return;
}


/*
 *===========================================================================
 * Specify the type of each of the environment variables by setting
 * the entries in EnvironmentType[ENVIRON_DIM] to PERCAPITARATE, GENERALODE
 * or POPULATIONINTEGRAL based on the classification below:
 *
 * Set an entry to PERCAPITARATE if the dynamics of E[j] follow an ODE and 0
 * is a possible equilibrium state of E[j]. The ODE is then of the form
 * dE[j]/dt = P(E,I)*E[j], with P(E,I) the per capita growth rate of E[j].
 * Specify the equilibrium condition as condition[j] = P(E,I), do not include
 * the multiplication with E[j] to allow for detecting and continuing the
 * transcritical bifurcation between the trivial and non-trivial equilibrium.
 *
 * Set an entry to GENERALODE if the dynamics of E[j] follow an ODE and 0 is
 * NOT an equilibrium state of E. The ODE then has a form dE[j]/dt = G(E,I).
 * Specify the equilibrium condition as condition[j] = G(E,I).
 *
 * Set an entry to POPULATIONINTEGRAL if E[j] is a (weighted) integral of the
 * population distribution, representing for example the total population
 * biomass. E[j] then can be expressed as E[j] = I[p][i]. Specify the
 * equilibrium condition in this case as condition[j] = I[p][i].
 *
 * Notice that the first index of the variable 'I[][]' refers to the
 * number of the structured population, the second index refers to the
 * number of the interaction variable. The interpretation of the latter
 * is up to the user. Also notice that the variable 'condition[j]' should
 * specify the equilibrium condition of environment variable 'E[j]'.
 *===========================================================================
 */

const int EnvironmentType[ENVIRON_DIM] = {GENERALODE, GENERALODE};

void EnvEqui(double E[], double I[POPULATION_NR][INTERACT_DIM],
             double condition[ENVIRON_DIM])
{
    condition[0] = DELTA*(R1MAX - R1) - I[0][0];
    condition[1] = DELTA*(R2MAX - R2) - I[0][1];
    return;
}

/*==============================================================================*/

