
%% References: 

%%          Dolcetti, G., and Krynkin, A. (2020), Matlab Codes for Two-Dimensional
%%      Scattering Surface Reconstruction using Broadband Data, Zenodo

%%          Dolcetti, G., Alkmim, M., Cuenca, J., De Ryck, L., Krynkin, A. (2020), 
%%      Robust Surface Shape Inversion with a Linear Microphones Array, submitted to 
%%      Journal of Sound and Vibrations

%% Author: Giulio Dolcetti g.dolcetti@sheffield.ac.uk
%% Date: 17-07-2020
%% Distribution: Creative Commons Attribution 4.0 International

%%-------------------------------------------------------------------------

%% SurfaceInversion_MultiFrequency_2D:
%% inverts surface shape, using the method indicated by 'TypeExp':
%% 'SA':  Short Array method;
%% 'SA0': De-biased Short Array method;
%% 'SP':  Small Perturbation method.

%% If length(f0) > 1, applies the multiple-frequency extension
%% If Nk x Nm < Nr: under-determined problem. Solved with Singular Value Decomposition
%% If Nk x Nm > Nr: over-determined problem. Solved with Least Squares ('lsqr')


%% INPUT:

%% f0:                  array of frequencies (Hz)  (1 x Nk)

%% CoordMic_dimensional:        matrix of microphones co-ordinates (Nm x 3)
%%     column1: (x1, x2,..., xM ..., xNm)T   x-co-ordinate  (m)
%%     column2: (y1, y2,..., yM ..., yNm)T   y-co-ordinate  (m)
%%     column3: (z1, z2,..., zM ..., zNm)T   z-co-ordinate  (m)

%% CoordSource_dimensional:     matrix of source co-ordinates (1 x 5)
%%     column1: xS       x-co-ordinate               (m)
%%     column2: yS       y-co-ordinate               (m)
%%     column3: zS       z-co-ordinate               (m)
%%     column4: psi      angle from horizontal       (rad)
%%     column5: aS       equivalent source radius    (m)


%% xRec_dimensional:     array of reconstruction grid locations (Nr x 1)
%%     column1: xRec       x-co-ordinate               (m)

%% TypeExp:             reconstruction method
        %% admitted values: 'SA','SA0','SP'
        
%% U2D_Kir:         scattered acoustic potential (non-dimensional)  (Nm x Nk)
%%          U2D_Kir(i,j) is the complex potential measured (calculated) at
%%          the i-th microphone, at the j-th frequency

%% U02D_Kir:        acoustic potential reflected from a flat surface (non-dimensional)
%%          U02D_Kir(i,j) is the complex potential measured (calculated) at
%%          the i-th microphone, at the j-th frequency

%% OUTPUT

%% zRec:               surface shape recosntruction (m)   (Nr x 1)


function zRec = SurfaceInversion_MultiFrequency_2D(f0,CoordMic_dimensional,...
    CoordSource_dimensional,xRec_dimensional,TypeExp,...
    U2D_Kir,U02D_Kir)

C = zeros(size(CoordMic_dimensional,1)*length(f0),size(xRec_dimensional,1));
K = C;
qz0 = zeros(length(f0),size(xRec_dimensional,1));
    

k = 2*pi;
lambda = 340./f0;
    
%% identify average microphone location to expand qz:
zqz0 = mean(CoordMic_dimensional(:,3));
yqz0 = 0;
xqz0 = mean(CoordMic_dimensional(:,1));
Coordqz0_dimensional = [xqz0,yqz0,zqz0];  


%% Transfer matrix calculation:

for il = 1:length(lambda)
    
    %% non-dimensionalisation
CoordMic = CoordMic_dimensional/lambda(il);
CoordSource = CoordSource_dimensional/lambda(il);
CoordSource(4) = CoordSource_dimensional(4);
Coordqz0 = Coordqz0_dimensional/lambda(il);
xRec = xRec_dimensional/lambda(il);
    
xM=repmat(CoordMic(:,1),[1,size(xRec,1)]);
yM=repmat(CoordMic(:,2),[1,size(xRec,1)]);
zM=repmat(CoordMic(:,3),[1,size(xRec,1)]);

xM0=repmat(Coordqz0(:,1),[1,size(xRec,1)]);
yM0=repmat(Coordqz0(:,2),[1,size(xRec,1)]);
zM0=repmat(Coordqz0(:,3),[1,size(xRec,1)]);

xS=repmat(CoordSource(:,1),[size(CoordMic,1),size(xRec,1)]);
yS=repmat(CoordSource(:,2),[size(CoordMic,1),size(xRec,1)]);
zS=repmat(CoordSource(:,3),[size(CoordMic,1),size(xRec,1)]);
psi = CoordSource(:,4);
aS = CoordSource(:,5);

xx=repmat(xRec',[size(CoordMic,1),1]);
yy=repmat(xRec'*0,[size(CoordMic,1),1]);
    
Dx = mean(diff(xx(1,:)));
    
    
% components of distance vectors R1 (source-surface), R2 (surface
% receiver), and R0 (surface average receiver)
R1vx=xx-xS; R1vy=yy-yS; R1vz=-zS;
R2vx=xM-xx; R2vy=yM-yy; R2vz=zM;
R2vx0=xM0-xRec'; R2vy0=yM0; R2vz0=zM0;

% modulus of R1, R2, R0
R1m=sqrt(R1vx.^2+R1vy.^2+R1vz.^2);
R2m=sqrt(R2vx.^2+R2vy.^2+R2vz.^2);
R2m0=sqrt(R2vx0.^2+R2vy0.^2+R2vz0.^2);
            
%% Directivity
% vector of source axis
SourceAxis=[-cos(psi) 0 -sin(psi)];

% projections of the distance vectors with respect to source axis
cos1x=R1vx.*SourceAxis(:,1);
cos1y=R1vy.*SourceAxis(:,2);
cos1z=R1vz.*SourceAxis(:,3);
    
% angle relative to the source axis
theta1=acos(((cos1x+cos1y+cos1z)./R1m).*((cos1x+cos1y+cos1z)./R1m<1)+(1).*((cos1x+cos1y+cos1z)./R1m>=1));

% Directivity (piston with infinite baffle - Morse & Ingard (1986), Theoretical Acoustics, p.381)
Dir=2*besselj(1,k*aS*sin(theta1))./(k*aS*sin(theta1)); Dir(theta1==0)=1;
                 
    
%% Integrand                
XX = exp(1i*k*(R1m + R2m));        
            
qz = k*(R2vz./R2m - R1vz./R1m);
            
qz0_l = k*(R2vz0./R2m0 - R1vz(1,:)./R1m(1,:))/lambda(il);
            
A = -1i/(4*pi) *Dir ./(R1m.*R2m) .*qz/k;

%% Stationary phase expansion
C_l = exp(1i*pi/4) .*sqrt(2*pi/k) .*sqrt(R1m.*R2m./(R1m + R2m)) .*A .*XX;

%% Small perturbation
K_l = -1i*qz.*C_l/lambda(il);
            
C(1+(il-1)*size(CoordMic,1):il*size(CoordMic,1),:) = C_l*Dx;
K(1+(il-1)*size(CoordMic,1):il*size(CoordMic,1),:) = K_l*Dx;
qz0(il,:) = qz0_l;
            
            
end
    
U2D_Kir = reshape(U2D_Kir,[],1);
U02D_Kir = reshape(U02D_Kir,[],1);
    
qz0 = mean(qz0,1);
        
    
    InversionMethod = 'SVD';
    if size(C,1)>=size(C,2)
        InversionMethod = 'lsqr';
        lsqrTol = 1e-02;
        fprintf('\n Overdetermined problem. Finding a least squares solution... \n');
    end
            
    
            switch TypeExp
                case 'SA'
               
                    switch InversionMethod
                        case 'SVD'
                            EX4 =svd_AK(C,U2D_Kir);
                        case 'lsqr'
                            EX4 = lsqr(C,U2D_Kir,lsqrTol,200);
                    end
                    EX4Unwr = CentreUnwrap(EX4,'centre');
                    zRec = -EX4Unwr./((qz0).');
                    
                case 'SA0'
               
                    switch InversionMethod
                        case 'SVD'
                            EX4 =svd_AK(C,U2D_Kir-U02D_Kir);
                        case 'lsqr'
                            EX4 = lsqr(C,U2D_Kir-U02D_Kir,lsqrTol,200); 
                    end
                    EX4Unwr = CentreUnwrap(1+EX4,'centre');
                    zRec = -EX4Unwr./((qz0).');
                    
                 case 'SP'
               
                    switch InversionMethod
                        case 'SVD'
                            EX4 =svd_AK(K,U2D_Kir-U02D_Kir);
                        case 'lsqr'
                            EX4 = lsqr(K,U2D_Kir-U02D_Kir,lsqrTol,200); 
                    end
                    
                    zRec = real(EX4);
                    
                    
                    
                    
            end
            
            

end



function EX4=svd_AK(MTX,U)

% Inversion with Singular Value Decomposition
% Uses Tikhonov regularisation, with generalised cross-validation to
% determine the regularisation parameter beta
                        
                        [XU,XS,XV] = svd(MTX,'econ');
                        xs = diag(XS);
                        xs1  = 1./xs;
            
                        h  = XU*XS;
                        n  = 0;  
                        
                        for beta=logspace(-6,0,400) 
                            n=n+1;
                            xs1b = 1./(xs+beta^2*xs1);
                            hb      = bsxfun(@times,XU',xs1b);
                            Mb      = (eye(size(XU))-h*hb);       
                            trb(n)  = real(trace(Mb));
                            rb(n)   = norm(Mb*U,2); 
                        end
                        
                        % generalised cross-validtion
                        beta=logspace(-6,0,400);
                        [~, indx]=min((rb.^2./trb.^2));
                        beta=beta(indx);
              
                        % Tikhonov regularisation
                        xs1b = 1./(xs+beta^2*xs1);
                        Qb = bsxfun(@times,XV,xs1b.')*XU';
                        
                        MTX1 = Qb;
                        EX4 = MTX1*U;  
end

function EX4Unwr = CentreUnwrap(EX4,UNWRAP)

% unwraps the phase from the middle ('centre') or edge ('sides')
% of the array, and extracts the phase

switch UNWRAP
case 'centre'
                        
                 EX4Unwr1=unwrap(imag(log(EX4(floor(end/2):end)))); 
                 EX4Unwr2=flipud(unwrap(flipud(imag(log(EX4(1:floor(end/2))))))); 
                        
                 EX4Unwr=[EX4Unwr2(1:end-1);EX4Unwr1];
                 
case 'sides'
                         EX4Unwr=unwrap(imag(log(EX4)));
                         
end
end


