
%% References: 

%%          Dolcetti, G., and Krynkin, A. (2020), Matlab Codes for Two-Dimensional
%%      Scattering Surface Reconstruction using Broadband Data, Zenodo

%%          Dolcetti, G., Alkmim, M., Cuenca, J., De Ryck, L., Krynkin, A. (2020), 
%%      Robust Surface Shape Inversion with a Linear Microphones Array, submitted to 
%%      Journal of Sound and Vibrations

%% Author: Giulio Dolcetti g.dolcetti@sheffield.ac.uk
%% Date: 17-07-2020
%% Distribution: Creative Commons Attribution 4.0 International

%%-------------------------------------------------------------------------


%% Random Surface Generator:
%% creates random realisations of a surface with power function spectrum 
%% S(k) = S_0 (k/\kappa_0)^{-\alpha_0} for \kappa_0 < k < \kappa_1, and 
%% S(k) = S_0                          for k < \kappa_0

%% INPUT:
%% sigma_0:         surface standard deviation (m)
%% kappa_0:         saturation wavenumber (rad/m)
%% kappa_1:         wavenumber cut-off (rad/m)
%% alpha_0:         power spectrum slope (-)
%% Dx:              spatial grid size (m)
%% N:               number of grid points (-)

%% OUTPUT
%% x:               surface x-co-ordinate (m)   (N x 1)
%% z:               surface z-co-ordinate (m)   (N x 1)


function [x,z] = RandomSurfaceGenerator(sigma_0,kappa_0,kappa_1,alpha_0,Dx,N)

ks = 2*pi/Dx;
Dk = ks/N;
k = (0:N-1)'*Dk;

% amplitude spectrum
Z = (k/kappa_0).^(-alpha_0/2);
Z(k<kappa_0) = 1;
Z(k>kappa_1) = 0;
Z(k==0) = 0;
% spectrum normalisation
Z = Z/sqrt(trapz(k,abs(Z).^2)/(2*pi));

% random coefficients
RA = randn(size(Z));
RP = rand(size(Z))*2*pi;


x = (0:N-1)'*Dx; x = x-mean(x);
z = sigma_0*real((ifft(Z.*RA.*exp(1i*RP)*N/2,[],1,'symmetric')));







