
%% References: 

%%          Dolcetti, G., and Krynkin, A. (2020), Matlab Codes for Two-Dimensional
%%      Scattering Surface Reconstruction using Broadband Data, Zenodo

%%          Dolcetti, G., Alkmim, M., Cuenca, J., De Ryck, L., Krynkin, A. (2020), 
%%      Robust Surface Shape Inversion with a Linear Microphones Array, submitted to 
%%      Journal of Sound and Vibrations

%% Author: Giulio Dolcetti g.dolcetti@sheffield.ac.uk
%% Date: 17-07-2020
%% Distribution: Creative Commons Attribution 4.0 International

%%-------------------------------------------------------------------------

% Example

clear all

%% surface parameters

sigma0 = 1e-03;             % surface standard deviation (m)
kappa0 = 2*pi/0.050;        % saturation wavenumber (rad/m) 
kappa1 = 2*pi/0.005;        % cut-off wavenumber (rad/m)
alpha0 = 4;                 % surface power spectrum slope (-)

Dx = 1e-03;                 % surface grid size (for forward problem) (m)
N = 2048;                   % number of grid points (forward problem) (-)

%% signal parameters

% the input signal is a Fourier complex spectrum with centre frequency f0 
% and with Nf frequency bands in the bandwidth f0 \pm Wf 

% to use a single frequency, set Wf = 0.

f0 = 17.5e03;                       % centre excitation frequency (Hz)
Wf = f0*0.1;                        % frequency half - bandwidth (Hz).
Nf = 1 + round(Wf/f0/0.0005/2)*2;   % number of frequency bands (-)


%% array parameters

a0 = 20e-03*(20e03/f0);     % equivalent source radius (m) (the source is modelled as
                            % an ideal piston of radius a0, with infinite baffle)
psi0 = pi/3;                % source inclination angle from the horizontal (rad)
z0 = 0.3;                   % source height (m)
zM = z0;                    % receivers height (m)
d0 = 25e-03;                % spacing between microphones (m)   
Nm = 34;                    % number of microphones

%% noise magnitude

AmpNoise = 0.05;    % standard deviation of the relative Gaussian amplitude noise
PhaNoise = 0.05;    % standard deviation of the relative Gaussian phase noise

%% reconstruction parameters

Dxr = 1e-03;        % reconstruction grid size (m)

%%% --------------------------------------------------------------------- %%%

%% surface generation

% generates a random realization of a surface with power-function spectrum,
% with exponent -alpha0, with standard deviation sigma0, cut-off wavenumber 
% kappa1, and saturation wavenumber kappa0.

[x,z] = RandomSurfaceGenerator(sigma0,kappa0,kappa1,alpha0,Dx,N);
CoordSurface = [x,zeros(size(x)),z];
CoordSurface0 = CoordSurface; CoordSurface0(:,end) = 0;

%% array configuration

% builds an array of Nm microphones at height zM and with equidistant
% spacing d0, and a source at height z0.

CoordSource = [z0/tan(pi-psi0) 0 z0 (pi-psi0) a0];

% A horizontal distance of 60e-03 m between the source and the first microphone is
% added to simulate the physical constraint due to the dimensions of the source
xxM = (0:Nm-1)'*d0 + 60e-03 + CoordSource(1); 
CoordMic = [xxM, zeros(size(xxM)), zM*ones(size(xxM))];


figure;
plot(x,z,'k')
hold on;plot(CoordSource(1),CoordSource(3),'sr','MarkerFaceColor','r');
hold on;plot(CoordMic(:,1),CoordMic(:,3),'ok','MarkerFaceColor','k');
title('array geometry');xlabel('x (m)');ylabel('z (m)');

%% Kirchhoff approximation check

lambda0 = 340/f0;

Kir = Kirchhoff(x,z,lambda0);
fprintf('\n minimum Kirchhoff parameter: %.2g \n',min(Kir)*sin(psi0)^3);

%% synthetic signal calculation

% estimates scattered acoustic potential based on Kirchhoff approximation

f = f0 + linspace(-1,1,Nf)*Wf;    % frequency array
If0 = find(f==f0);

lambda = 340./f;

P = zeros(size(CoordMic,1),length(f));  P0 = P;
for if0 = 1:length(f)
    clc
fprintf('\n calculating synthetic acoustic field... %.0f %%\n',if0/length(f)*100);
P(:,if0) = KirchhoffScattering2D(CoordMic,CoordSource,CoordSurface,lambda(if0));
P0(:,if0) = KirchhoffScattering2D(CoordMic,CoordSource,CoordSurface0,lambda(if0));
end

% adds noise
PN = P.*(1+AmpNoise.*randn(size(P))).*exp(1i*2*pi*PhaNoise*randn(size(P)));

figure;
plot(CoordMic(:,1),real(P0(:,If0)),'k',CoordMic(:,1),real(P(:,If0)),'r',...
    CoordMic(:,1),real(PN(:,If0)),'b--','LineWidth',1);
xlabel('x_M (m)');ylabel('Re(P)');legend('P_0','P','P with noise');

%% surface inversion

% stationary phase points
xsp = (CoordSource(1) + CoordMic(:,1))/2;
x0sp = (xsp(1) + xsp(end))/2;
Lambda = xsp(end) - x0sp;           % reconstruction domain
L0 = Lambda*6;
xr = (-L0:Dxr:L0)' + x0sp;          % reconstruction grid

% reconstructs surface shape
fprintf('\n inverting surface shape... \n');
% SA method:
zrSA = SurfaceInversion_MultiFrequency_2D(f,CoordMic,CoordSource,xr,'SA',PN,P0);        
% SA0 method:
zrSA0 = SurfaceInversion_MultiFrequency_2D(f,CoordMic,CoordSource,xr,'SA0',PN,P0);        
% SP method:
zrSP = SurfaceInversion_MultiFrequency_2D(f,CoordMic,CoordSource,xr,'SP',PN,P0);        



figure;plot(x,z,'k-',xr,zrSA,'b--',xr,zrSA0,'r--',xr,zrSP,'g--','LineWidth',1);
xlim(x0sp + [-1 1]*Lambda);legend('target','SA','SA0','SP');
xlabel('x (m)');ylabel('z (m)');


function K = Kirchhoff(x,z,lambda0)

Dx = mean(diff(x));

etax = (z(3:end)-z(1:end-2))/(2*Dx);
etax = [etax(1); etax; etax(end)];
etaxx = (etax(3:end)-etax(1:end-2))/(2*Dx);
etaxx = [etaxx(1); etaxx; etaxx(end)];

Rc = ((1+etax.^2).^(3/2))./abs(etaxx);
K = 4*pi*Rc/lambda0;

end
