---
id: useEditor
title: useEditor()
sidebar_label: useEditor()
---

import {API, Badge} from "@site/src/components";

<Badge type="hook" />


A Hook that provides methods and state information associated with the entire editor.

```tsx
const { connectors, actions, query, ...collected } = useEditor(collector);
```

## Reference
### Parameters
<API items={[
  ["collector", "(state: EditorState, query: Query) => Collected", "A function that collects relevant state information from the editor state. The component will re-render when the values returned by this function changes."]
]} /> 

### Returns

<API items={[
  [null, "Object", [
    ["connectors", "Object", [
      ["select", "(dom: HTMLElement, nodeId: NodeId) => HTMLElement", "Specifies the DOM that when clicked will in turn click the specified Node's user component"],
      ["hover", "(dom: HTMLElement, nodeId: NodeId) => HTMLElement", "Specifies the DOM that when hovered will in turn hover the specified Node's user component"],
      ["drag", "(dom: HTMLElement, nodeId: NodeId) => HTMLElement", "Specifies the DOM that when dragged will move the specified Node's user component. Only applicable if the component is rendered as an immediate child of a &lt;Canvas /&gt; component."],
      ["create", "(dom: HTMLElement, userElement: React.ReactElement) => HTMLElement", "Specifies the DOM that when dragged will create a new instance of the specified User Element at the drop location."]
    ]],
    ["actions", "Object", [
      ["add", "(nodes: Node | Node[], parentId?: NodeId) => void", "Add Node(s) to the given parent node ID. By default the parentId is the id of the Root Node"],
      ["delete", "(nodeID: NodeId) => void", "Delete the specified Node"],
      ["deserialize", "(json: String) => void", "Recreate Nodes from JSON. This will clear all the current Nodes in the editor state with the recreated Nodes"],
      ["move", "(nodeId: NodeId, targetParentId: NodeId, index: number) => void", "Move a Node to the specified parent Node at the given index."],
      ["setProp", "(nodeId: NodeId, update: (props: Object) => void) => void", "Manipulate the props of the given Node"],
      ["setCustom", "(nodeId: NodeId, update: (custom: Object) => void) => void", "Manipulate the custom values of the given Node"],
      ["setHidden", "(nodeId: NodeId, bool: boolean) => void", "When set to true, the User Component of the specified Node will be hidden, but not removed"],
      ["setOptions", "(options: Object) => void", "Update the editor's options. The options object passed is the same as the &lt;Editor /&gt; props."],
      ["selectNode", "(nodeId: NodeId | null) => void", "Select the specified node. You can clear the selection by passing `null`"],
    ]],
    ["query", "Query", [
      ["createNode", "(child: React.ReactElement) => Node", "Create a Node from a React element"],
      ["serialize", "() => String", "Return the current Nodes in JSON"],
      ["getOptions", "() => Object", "Get the options specified in the &lt;Editor /&gt; component"],
      ["getDropPlaceholder", 
        "(sourceNodeId: NodeId, targetNodeId: NodeId, pos: {x: number, y: number}, nodesToDOM?: (node: Node) => HTMLElement = node => node.dom)",
        "Given the target Node and mouse coordinates on the screen, determine the best possible location to drop the source Node. By default, the Node's DOM property is taken into consideration."
      ],
      ["node", "(id: NodeId) => NodeHelpers", "Returns an object containing helper methods to describe the specified Node. Click <a href='/craft.js/r/docs/api/helpers/'>here</a> for more information."]
    ]],
    ["inContext", "boolean", "Returns false if the component is rendered outside of the `<Editor />`. This is useful if you are designing a general component that you also wish to use outside of Craft.js."],
    ["...collected", "Collected", "The collected values returned from the collector"]
  ]
  ]
]} /> 

## Examples

### Collecting state information
```tsx
import {useEditor} from "@craftjs/core";

const Example = () => {
  const { hoveredNodeId } = useEditor((state) => ({
    hoveredNodeId: state.events.hovered
  }));

  return (
    <div>
      The ID of the node currently being hovered is: {hoveredNodeId}
    </div>
  )
}
```

### Updating props
```tsx
import {useEditor} from "@craftjs/core";

const Example = () => {
  const { selectedNodeId, actions: {setProp} } = useEditor((state) => ({
    selectedNodeId: state.events.selected
  }));

  return (
    <a
      onClick={_ => {
        setProp(selectedNodeId, props => {
          props.text = "new value";
        });
      }}
    >
      Update
    </a>
  )
}
```

### Hiding and Deleting a Node
```jsx
const Example = () => {
  const {selectedNodeId, actions} = useEditor((state) => ({
    selectedNodeId: state.events.selected
  }));
  return selectedNodeId && (
    <div>
      <h2>Node selected: {selectedNodeId}</h2>
      <a onClick={() => actions.hide(selectedNodeId)}>Hide</a>
      <a onClick={() => actions.delete(selectedNodeId)}>Delete</a>
    </div>
  )
}
```

### Moving a Node
```jsx
const Example = () => {
  const [sourceId, setSourceId] = useState();
  const [targetId, setTargetId] = useState();
  
  const {selectedNodeId, actions, query} = useEditor((state) => ({
    selectedNodeId: state.events.selected
  }));

  return selectedNodeId && (
    <div>
      <h2>Node selected: {selectedNodeId}</h2>
      <div>
        <input type="text" value={sourceId} placeholder="Source" disabled />
        <button onClick={() => selectedNodeId && setSourceId(selectedNodeId)}>Set selected Node as source</button>
      </div>
      <div>
        <input type="text" value={targetId} placeholder="Target" disabled />
        <button onClick={() => selectedNodeId && setTargetId(selectedNodeId)}>Set selected Node as target</button>
      </div>
      {
        sourceId && targeId ? (
          <button onClick={() => {
            try {
              // .canDropInParent will throw an error message if the conditions failed
              query.canDropInParent(sourceId, targetId); 
              actions.move(sourceId, targetId);
            } catch (e) {
              console.error(e.message);
            } 
          }}>Move Node</button>
        )
      }
    </div>
  )
}
```

### Creating and Adding a new Node
```tsx
import {useEditor} from "@craftjs/core";

const Example = () => {
  const { query, actions } = useEditor((state, query) => ({
    hoveredNodeId: state.events.hovered
  }));

  return (
    <div>
      <a onClick={() => {
        const node = query.createNode(<h2>Hi</h2>);
        actions.add(node);
      }}>Click me to add a new Node</a>
    </div>
  )
}
```


### Getting the currently selected Node's descendants
> Query methods are also accessible from within the collector function.

```tsx
import {useEditor} from "@craftjs/core";

const Example = () => {
  const { selectedDescendants } = useEditor((state, query) => ({
    selectedDescendants: state.events && query.getDeepNodes(state.events.selected).map(node => node.id)
  }));

  return (
    <ul>
      {
        selectedDescendants && selectedDescendants.map(id => <li>{id}</li> )
      }
    </ul>
  )
}
```

### Displaying Drop Indicator for the best possible drop location
```jsx
const Example = () => {
  const [screenClick, setScreenClick] = useState(false);
  const [sourceId, setSourceId] = useState();
  const [targetId, setTargetId] = useState();
  
  const {selectedNodeId, actions, query} = useEditor((state) => ({
    selectedNodeId: state.events.selected
  }));

  const disableScreenClick = useEffect((e) => {
     if(e.key === "Escape") {
       setScreenClick(false);
    }
  }, [screenClick]);

  const clickOnScreen = useEffect((e) => {
    const {clientX: x, clientY: y} = e;
    const dropIndicator = query.getDropIndicator(sourceId, targetId, {x, y});
    actions.setDropIndicator(dropIndicator);
  }, [screenClick]);
  

  useEffect(() => {
    window.addEventListener("click", clickOnScreen);
    window.addEventListener("keyup", disableScreenClick);
    return (() => {
      window.removeEventListener("click", clickOnScreen);
      window.removeEventListener("keyup", disableScreenClick);
    })
  }, [clickOnScreen, disableScreenClick]);

  return selectedNodeId && (
    <div>
      <h2>Node selected: {selectedNodeId}</h2>
      <div>
        <input type="text" value={sourceId} placeholder="Source" disabled />
        <button onClick={() => selectedNodeId && setSourceId(selectedNodeId)}>Set selected Node as source</button>
      </div>
      <div>
        <input type="text" value={targetId} placeholder="Target" disabled />
        <button onClick={() => selectedNodeId && setTargetId(selectedNodeId)}>Set selected Node as target</button>
      </div>
      {
        sourceId && targeId ? (
          <button onClick={() => {
            setScreenClick(true);
          }}>
            {screenClick ? "Click anywhere on the screen to display indicator" : "Start"}
          </button>
        )
      }
    </div>
  )
}
```


## Legacy API
For Class Components, use `connectEditor` instead.

<Badge type="hoc" title={false} />


### Parameters
<API items={[
  ["collector", "(node: Node) => Collected", "A function that collects relevant state information from the corresponding Node. The component will re-render when the values returned by this function changes."]
]} /> 

### Injected Props
<API items={[
  ["...useEditor(collector)", "Object", "Identical return values as the useEditor() hook above"]
]} /> 


### Example
```jsx
import { connectEditor } from "@craftjs/core";

class SidebarInner extends React.Component {
  render() {
    const { actions, query, enabled, currentSelectedNodeId } = this.props;
    return (
      <div>
        <input type="checkbox" value={enabled} onChange={
          e => actions.setOptions(options => options.enabled = !enabled)
        } />
        <button 
          onClick={() => {
            console.log(query.serialize())
          }}
        >
            Serialize JSON to console
        </button>
      </div>
    )
  }
}

export const Sidebar = connectEditor((state) => ({
  currentSelectedNodeId: state.events.selected
}))(SidebarInner);
```
