/*************************************************************

  LSD 7.1 - December 2018
  written by Marco Valente, Universita' dell'Aquila
  and by Marcelo Pereira, University of Campinas
    This file also:  Frederik Schaff, Ruhr-Universitï¿½t Bochum

  Copyright Marco Valente and Marcelo Pereira
    This file also:  Frederik Schaff, Ruhr-Universitï¿½t Bochum
  LSD is distributed under the GNU General Public License

 *************************************************************/

/*************************************************************
    LSD_MAIN.CPP
    Contains:
    - early initialization (namely, of the Log windows)
    - the main cycle: browse a model, run simulation, return to the browser.

    The main functions contained here are:

    - void run( )
    Run the simulation model whose root is r. Running is not only the actual
    simulation run, but also the initialization of result files. Of course, it has
    also to manage the messages from user and from the model at run time.

    - bool alloc_save_mem( );
    Prepare variables to store saved data.
*************************************************************/

#include "decl.h"

// some program defaults
bool grandTotal = false;  // flag to produce or not grand total in batch processing
bool ignore_eq_file = true; // flag to ignore equation file in configuration file
char nonavail[ ] = "NA";  // string for unavailable values (use R default)
char tabs[ ] = "5c 7.5c 10c 12.5c 15c 17.5c 20c"; // Log window tabs
double def_res = 0;     // default equation result
int add_to_tot = false;   // flag to append results to existing totals file (bool)
int max_step = 100;     // default number of simulation runs
int overwConf = true;   // overwrite configuration on run flag (bool)
int saveConf = false;   // save configuration on results saving (bool)
int strWindowOn = true;   // control the presentation of the model structure window (bool)
unsigned seed = 1;      // random number generator initial seed

bool batch_sequential = false;// no-window multi configuration job running
bool brCovered = false;   // browser cover currently covered
bool eq_dum = false;    // current equation is dummy
bool error_hard_thread;   // flag to error_hard() called in worker thread
bool fast;          // safe copy of fast_mode flag
bool log_ok = false;    // control for log window available
bool message_logged = false;// new message posted in log window
bool no_more_memory = false;// memory overflow when setting data save structure
bool no_search;       // disable the standard variable search mechanism
bool no_window = false;   // no-window command line job
bool no_zero_instance = false;// flag to allow deleting last object instance
bool non_var = false;   // flag to indicate INTERACT macro condition
bool on_bar;        // flag to indicate bar is being draw in log window
bool parallel_mode;     // parallel mode (multithreading) status
bool pause_run;       // pause running simulation
bool redrawRoot;      // control for redrawing root window (.)
bool running = false;   // simulation is running
bool save_alt_path = false; // alternate save path flag
bool scroll;        // scroll state in current runtime plot
bool struct_loaded = false; // a valid configuration file is loaded
bool tk_ok = false;     // control for tk ready to operate
bool unsavedData = false; // flag unsaved simulation results
bool unsavedSense = false;  // control for unsaved changes in sensitivity data
bool user_exception = false;// flag indicating exception was generated by user code
bool use_nan;       // flag to allow using Not a Number value
bool worker_ready;      // parallel worker ready flag
bool worker_crashed;    // parallel worker crash flag
char* alt_path = NULL;    // alternative output path
char* eq_file = NULL;   // equation file content
char* exec_file = NULL;   // name of executable file
char* exec_path = NULL;   // path of executable file
char* lsdroot = NULL;   // path of LSD root directory
char* path = NULL;      // path of current configuration
char* sens_file = NULL;   // current sensitivity analysis file
char* simul_name = NULL;  // name of current simulation configuration
char* struct_file = NULL; // name of current configuration file
char equation_name[ MAX_PATH_LENGTH ] = ""; // equation file name
char error_hard_msg1[ TCL_BUFF_STR ];   // buffer for parallel worker title msg
char error_hard_msg2[ TCL_BUFF_STR ];   // buffer for parallel worker log msg
char error_hard_msg3[ TCL_BUFF_STR ];   // buffer for parallel worker box msg
char lastObj[ MAX_ELEM_LENGTH ] = "";   // last shown object for quick reload
char lsd_eq_file[ MAX_FILE_SIZE + 1 ] = ""; // equations saved in configuration file
char msg[ TCL_BUFF_STR ] = "";        // auxiliary Tcl buffer
char name_rep[ MAX_PATH_LENGTH ] = "";    // documentation report file name
char tcl_dir[ MAX_PATH_LENGTH ] = "";   // Tcl/Tk directory
description* descr = NULL;  // model description structure
int actual_steps = 0;   // number of executed time steps
int choice;         // Tcl menu control variable (main window)
int choice_g;               // Tcl menu control variable (structure window)
int cur_plt;        // current graph plot number
int cur_sim;        // current simulation run
int debug_flag = false;   // debug enable control (bool)
int docsv = false;      // produce .csv text results files (bool)
int done_in;        // Tcl menu control variable (log window)
int fast_mode;        // level of LOG messages & runtime plot
int fend;         // last multi configuration job to run
int findex;         // current multi configuration job
int findexSens = 0;     // index to sequential sensitivity configuration filenames
int log_start;        // first period to start logging to file
int log_stop;       // last period to log to file, if any
int macro;          // equations style (macros or C++) (bool)
int max_threads = 1;    // maximum number of parallel threads
int no_res = false;     // do not produce .res results files (bool)
int parallel_disable = false;// flag to control parallel mode
int prof_aggr_time = false; // show aggregate profiling times
int prof_min_msecs = 0;   // profile only variables taking more than X msecs.
int prof_obs_only = false;  // profile only observed variables
int quit = 0;       // simulation interruption mode (0=none)
int series_saved = 0;   // number of series saved
int sim_num = 1;      // simulation number running
int stack;          // LSD stack call level
int stack_info = 0;     // LSD stack control
int t;            // current time step
int when_debug;       // next debug stop time step (0 for none)
int wr_warn_cnt;      // invalid write operations warning counter
long nodesSerial = 1;   // network node's serial number global counter
lsdstack* stacklog = NULL;  // LSD stack
map < string, profile > prof; // set of saved profiling times
object* blueprint = NULL; // LSD blueprint (effective model in use)
object* abmat = NULL;       // LSD abmat (hold reduced model data for analysis)
object* currObj = NULL;   // pointer to current object in browser
object* root = NULL;    // LSD root object
object* wait_delete = NULL; // LSD object waiting for deletion
o_setT obj_list;      // set with all existing LSD objects
sense* rsense = NULL;   // LSD sensitivity analysis structure
variable* cemetery = NULL;  // LSD saved data series (from last simulation run)
FILE* log_file = NULL;    // log file, if any

#ifdef CPP11
  eq_mapT eq_map;       // fast equation look-up map
#endif //#ifdef CPP11

#ifdef LIBZ
  int dozip = true;     // compressed results file flag (bool)
#else
  int dozip = false;
#endif

#ifndef NO_WINDOW
  int i_values[ 4 ];      // user temporary variables copy
  double d_values[ USER_D_VARS ];
  object* o_values[ 10 ];
  netLink* n_values[ 10 ];
  Tcl_Interp* inter = NULL; // global Tcl interpreter in LSD
#endif

#ifdef PARALLEL_MODE
  map< thread::id, worker* > thr_ptr; // worker thread pointers
  thread::id main_thread;   // LSD main thread ID
  worker* workers = NULL;   // multi-thread parallel worker data
#endif


/*********************************
    LSD MAIN
*********************************/
int lsdmain( int argn, char** argv )
{
  char* str;
  int i, j = 0, len, done;
  
  path = new char[ strlen( "" ) + 1 ];
  simul_name = new char[ strlen( DEF_CONF_FILE ) + 1 ];
  strcpy( path, "" );
  strcpy( tcl_dir, "" );
  strcpy( simul_name, DEF_CONF_FILE );
  exec_file = clean_file( argv[ 0 ] );  // global pointer to the name of executable file
  exec_path = clean_path( getcwd( NULL, 0 ) );  // global pointer to path of executable file
  
#ifdef PARALLEL_MODE
  main_thread = this_thread::get_id( );
  max_threads = ( MAX_CORES <= 0 ) ? thread::hardware_concurrency( ) : MAX_CORES;
#else
  max_threads = ( MAX_CORES <= 0 ) ? 4 : MAX_CORES;
#endif
  
  
  root = new object;
  root->init( NULL, "Root" );
  add_description( "Root", "Object", "(no description available)" );
  blueprint = new object;
  blueprint->init( NULL, "Root" );
  abmat = new object;
  abmat->init( NULL, "ABMAT" );
  
  
#ifdef NO_WINDOW
  
  no_window = true;
  findex = 1;
  fend = 0;             // no file number limit
  
  if ( argn < 3 ) {
    fprintf( stderr, "\nThis is the No Window version of LSD.\nCommand line options:\n'-f FILENAME.lsd' to run a single configuration file\n'-f FILE_BASE_NAME -s FIRST_NUM [-e LAST_NUM]' for batch sequential mode\n'-o PATH' to save result file(s) to a different subdirectory\n'-t' to produce comma separated (.csv) text result file(s)\n'-r' for skipping the generation of intermediate result file(s)\n'-g' for the generation of a single grand total file\n'-z' for preventing the generation of compressed result file(s)\n'-c MAX_CORES' for defining the maximum number of CPU cores to use\n" );
    myexit( 1 );
  }
  else {
    for ( i = 1; i < argn; i += 2 ) {
      // read -f parameter : file name or base name
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'f' && 1 + i < argn && strlen( argv[ 1 + i ] ) > 0 ) {
        delete [ ] simul_name;
        simul_name = new char[ strlen( argv[ 1 + i ] ) + 1 ];
        strcpy( simul_name, argv[ 1 + i ] );
        continue;
      }
      
      // read -s parameter : first sequential file to process
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 's' && 1 + i < argn && strlen( argv[ 1 + i ] ) > 0 ) {
        findex = atoi( argv[ i + 1 ] );
        batch_sequential = true;
        continue;
      }
      
      // read -e parameter : last sequential file to process
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'e' && 1 + i < argn && strlen( argv[ 1 + i ] ) > 0 ) {
        fend = atoi( argv[ i + 1 ] );
        continue;
      }
      
      // read -t parameter : produce .csv text results files
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 't' ) {
        i--;          // no parameter for this option
        docsv = true;
        continue;
      }
      
      // read -r parameter : do not produce intermediate .res files
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'r' ) {
        i--;          // no parameter for this option
        no_res = true;
        continue;
      }
      
      // read -g parameter : create grand total file (batch only)
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'g' ) {
        i--;          // no parameter for this option
        grandTotal = true;
        printf( "Grand total file requested ('-g'), please don't run another instance of 'lsd_gnuNW' in this folder!\n" );
        continue;
      }
      
      // read -g parameter : don't create compressed result files
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'z' ) {
        i--;          // no parameter for this option
        dozip = false;
        continue;
      }
      
      // change the path for the output of result files
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'o' && 1 + i < argn && strlen( argv[ 1 + i ] ) > 0 ) {
        results_alt_path( argv[ 1 + i ] );
        continue;
      }
      
      // read -c parameter : max number of cores
      if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'c' && 1 + i < argn && strlen( argv[ 1 + i ] ) > 0 ) {
        j = atoi( argv[ i + 1 ] );
        continue;
      }
      
      fprintf( stderr, "\nOption '%c%c' not recognized.\nThis is the No Window version of LSD.\nCommand line options:\n'-f FILENAME.lsd' to run a single configuration file\n'-f FILE_BASE_NAME -s FIRST_NUM [-e LAST_NUM]' for batch sequential mode\n'-o PATH' to save result file(s) to a different subdirectory\n'-t' to produce comma separated (.csv) text result file(s)\n'-r' for skipping the generation of intermediate result file(s)\n'-g' for the generation of a single grand total file\n'-c MAX_CORES' for defining the maximum number of CPU cores to use\n'-z' for preventing the generation of compressed result file(s)\n", argv[ i ][ 0 ], argv[ i ][ 1 ] );
      myexit( 2 );
    }
  }
  
  if ( ! batch_sequential ) {
    struct_file = new char[ strlen( simul_name ) + 1 ];
    sprintf( struct_file, "%s", simul_name );
    simul_name[ strlen( simul_name ) - 4 ] = '\0';
  }
  else {
    sprintf( msg, "%s_%d.lsd", simul_name, findex );
    struct_file = new char[ strlen( msg ) + 1 ];
    strcpy( struct_file, msg );
  }
  
  FILE* f = fopen( struct_file, "r" );
  
  if ( f == NULL ) {
    fprintf( stderr, "\nFile '%s' not found.\nThis is the no window version of LSD.\nSpecify a -f FILENAME.lsd to run a simulation or -f FILE_BASE_NAME -s 1 for\nbatch sequential simulation mode (requires configuration files:\nFILE_BASE_NAME_1.lsd, FILE_BASE_NAME_2.lsd, etc).\n", struct_file );
    myexit( 3 );
  }
  
  fclose( f );
  
  if ( load_configuration( true ) != 0 ) {
    fprintf( stderr, "\nFile '%s' is invalid.\nThis is the no window version of LSD.\nCheck if the file is a valid LSD configuration or regenerate it using the\nLSD Browser.\n", struct_file );
    myexit( 4 );
  }
  
#ifdef PARALLEL_MODE
  
  if ( j > 0 && j < max_threads )
  { max_threads = j; }
  
#endif
  
#else
  
  for ( i = 1; argv[ i ] != NULL; i++ ) {
    if ( argv[ i ][ 0 ] != '-' || ( argv[ i ][ 1 ] != 'f' && argv[ i ][ 1 ] != 'i' && argv[ i ][ 1 ] != 'c' ) ) {
      log_tcl_error( "Command line parameters", "Invalid option, available options: -i TCL_DIRECTORY / -f MODEL_NAME / -c MAX_CORES" );
      myexit( 1 );
    }
  
    if ( argv[ i ][ 1 ] == 'f' ) {
      delete [ ] simul_name;
      simul_name = new char[ strlen( argv[ i + 1 ] ) + 3 ];
      strcpy( simul_name, argv[ i + 1 ] );
      len = strlen( simul_name );
  
      if ( len > 4 && ! strcmp( ".lsd", simul_name + len - 4 ) )
      { *( simul_name + len - 4 ) = '\0'; }
  
      i++;
    }
  
    if ( argv[ i ][ 1 ] == 'i' ) {
      strcpy( tcl_dir, argv[ i + 1 ] + 2 );
      i++;
    }
  
    // read -c parameter : max number of cores
    if ( argv[ i ][ 0 ] == '-' && argv[ i ][ 1 ] == 'c' ) {
      j = atoi( argv[ i + 1 ] );
      continue;
    }
  }
  
#ifdef PARALLEL_MODE
  
  if ( j > 0 && j < max_threads )
  { max_threads = j; }
  
#endif
  
  // initialize the tcl interpreter
  Tcl_FindExecutable( argv[ 0 ] );
  inter = Tcl_CreateInterp( );
  done = Tcl_Init( inter );
  
  if ( done != TCL_OK ) {
    sprintf( msg, "Tcl initialization directories not found, check the Tcl/Tk installation and configuration or reinstall LSD\nTcl Error = %d : %s", done,  Tcl_GetStringResult( inter ) );
    log_tcl_error( "Create Tcl interpreter", msg );
    myexit( 5 );
  }
  
  // set variables and links in TCL interpreter
  Tcl_LinkVar( inter, "choice", ( char* ) &choice, TCL_LINK_INT );
  Tcl_LinkVar( inter, "debug_flag", ( char* ) &debug_flag, TCL_LINK_BOOLEAN);
  Tcl_LinkVar( inter, "when_debug", ( char* ) &when_debug, TCL_LINK_INT );
  
  // test Tcl interpreter
  cmd( "set choice 1234567890" );
  
  if ( choice != 1234567890 ) {
    log_tcl_error( "Test Tcl", "Tcl failed, check the Tcl/Tk installation and configuration or reinstall LSD" );
    myexit( 6 );
  }
  
  // initialize & test the tk application
  choice = 1;
  done = Tk_Init( inter );
  
  if ( done == TCL_OK )
  { cmd( "if { ! [ catch { package present Tk 8.5 } ] && [ winfo exists . ] } { set choice 0 } { set choice 1 }" ); }
  
  if ( choice ) {
    sprintf( msg, "Tk failed, check the Tcl/Tk installation (version 8.5+) and configuration or reinstall LSD\nTcl Error = %d : %s", done,  Tcl_GetStringResult( inter ) );
    log_tcl_error( "Start Tk", msg );
    myexit( 7 );
  }
  
  tk_ok = true;
  cmd( "tk appname lsd" );
  
  // disable Carbon compatibility in Mac
  cmd( "if [ string equal $tcl_platform(os) Darwin ] { set ::tk::mac::useCompatibilityMetrics 0 }" );
  
  // close console if open (usually only in Mac)
  cmd( "if [ string equal $tcl_platform(os) Darwin ] { foreach i [ winfo interps ] { if { ! [ string equal [ string range $i 0 2 ] lmm ] && ! [ string equal [ string range $i 0 2 ] lsd ] } { send $i \"wm iconify .; wm withdraw .; destroy .\" } } }" );
  
  cmd( "if { [ string first \" \" \"[ pwd ]\" ] >= 0  } { set choice 1 } { set choice 0 }" );
  
  if ( choice ) {
    log_tcl_error( "Path check", "LSD directory path includes spaces, move all the LSD directory in another directory without spaces in the path" );
    cmd( "tk_messageBox -icon error -title Error -type ok -message \"Installation error\" -detail \"The LSD directory is: '[ pwd ]'\n\nIt includes spaces, which makes impossible to compile and run LSD models.\nThe LSD directory must be located where there are no spaces in the full path name.\nMove all the LSD directory in another directory. If exists, delete the 'system_options.txt' file from the \\src directory.\n\nLSD is aborting now.\"" );
    myexit( 8 );
  }
  
  // try to use exec_path to change to the model directory
  if ( strlen( exec_path ) == 0 || ! strcmp( exec_path, "/" ) ) {
    // try to get name from Tcl
    cmd( "if { [ info nameofexecutable ] != \"\" } { set path [ file dirname [ info nameofexecutable ] ] } { set path \"\" }" );
    str = ( char* ) Tcl_GetVar( inter, "path", 0 );
  
    if ( str != NULL && strlen( str ) > 0 ) {
      delete [ ] exec_path;
      exec_path = new char[ strlen( str ) + 1 ];
      strcpy( exec_path, str );
    }
  }
  
  choice = 0;
  cmd( "set path [ file normalize \"%s\" ]", exec_path );
  
  // check if directory is ok and if executable is inside a macOS package
  cmd( "if [ file exists \"$path/modelinfo.txt\" ] { \
			cd \"$path\" \
		} { \
			if [ file exists \"$path/../../../modelinfo.txt\" ] { \
				cd \"$path/../../..\"; \
				set path \"[ pwd ]\" \
			} { \
				set path \"\"; \
				set choice 1 \
			} \
		}" );
  
  if ( choice ) {
    log_tcl_error( "Model files check", "Required model file(s) missing or corrupted, check the model directory and recreate the model if the problem persists" );
    cmd( "tk_messageBox -parent . -title Error -icon error -type ok -message \"File(s) missing or corrupted\" -detail \"Some model files are missing or corrupted.\nPlease recreate your model if the problem persists.\n\nLSD is aborting now.\"" );
    myexit( 200 );
  }
  
  str = ( char* ) Tcl_GetVar( inter, "path", 0 );
  delete [ ] path;
  path = new char[ strlen( str ) + 1 ];
  strcpy( path, str );
  delete [ ] exec_path;
  exec_path = new char[ strlen( str ) + 1 ];
  strcpy( exec_path, str );
  
  // check if LSDROOT already exists and use it if so, if not, search the current directory tree
  cmd( "if [ info exists env(LSDROOT) ] { set RootLsd [ file normalize $env(LSDROOT) ]; if { ! [ file exists \"$RootLsd/src/interf.cpp\" ] } { unset RootLsd } }" );
  
  // do some search for the right path to cope with Mac Acqua package
  choice = 0;
  cmd( "if { ! [ info exists RootLsd ] } { \
			set here [ pwd ]; \
			while { ! [ file exists \"src/interf.cpp\" ] && ! [ string equal [ pwd ] \"/\" ] && [ string length [ pwd ] ] > 3 } { \
				cd .. \
			}; \
			if [ file exists \"src/interf.cpp\" ] { \
				set RootLsd [ pwd ] \
			} { \
				set choice 1 \
			}; \
			cd $here; \
		}" );
  
  if ( choice ) {
    log_tcl_error( "LSDROOT check", "LSDROOT not set, make sure the environment variable LSDROOT points to the directory where LSD is installed" );
    cmd( "tk_messageBox -parent . -title Error -icon error -type ok -message \"LSDROOT not set\" -detail \"Please make sure the environment variable LSDROOT points to the directory where LSD is installed.\n\nLSD is aborting now.\"" );
    myexit( 9 );
  }
  
  cmd( "set env(LSDROOT) $RootLsd" );
  str = ( char* ) Tcl_GetVar( inter, "RootLsd", 0 );
  lsdroot = new char[ strlen( str ) + 1 ];
  strcpy( lsdroot, str );
  len = strlen( lsdroot );
  
  for ( i = 0; i < len; ++i )
    if ( lsdroot[ i ] == '\\' )
    { lsdroot[ i ] = '/'; }
  
  cmd( "set RootLsd \"%s\"", lsdroot );
  
  cmd( "set choice [ file exist \"$RootLsd/lmm_options.txt\" ]" );
  
  if ( choice == 1 ) {
    cmd( "set f [open \"$RootLsd/lmm_options.txt\" r]" );
    cmd( "gets $f sysTerm" );
    cmd( "gets $f HtmlBrowser" );
    cmd( "gets $f fonttype" );
    cmd( "gets $f wish" );
    cmd( "gets $f LsdSrc" );
    cmd( "gets $f dim_character" );
    cmd( "gets $f tabsize" );
    cmd( "close $f" );
  }
  else {
    cmd( "tk_messageBox -parent . -title Warning -icon warning -type ok -message \"Could not locate LMM system options\" -detail \"It may be impossible to open help files and compare the equation files. Any other functionality will work normally. When possible set in LMM the 'Options' in menu 'File'.\"" );
  
    cmd( "set LsdSrc src" );
    cmd( "set tabsize 2" );
  }
  
  i = choice;
  
  // load required Tcl/Tk data, procedures and packages (error coded by file/bit position)
  choice = 0;
  
  // load native Tk windows defaults
  cmd( "if [ file exists \"$RootLsd/$LsdSrc/defaults.tcl\" ] { if { [ catch { source \"$RootLsd/$LsdSrc/defaults.tcl\" } ] != 0 } { set choice [ expr $choice + %d ] } } { set choice [ expr $choice + %d ] }", 0x0100, 0x01 );
  
  // load native Tk procedures for windows management
  cmd( "if [ file exists \"$RootLsd/$LsdSrc/window.tcl\" ] { if { [ catch { source \"$RootLsd/$LsdSrc/window.tcl\" } ] != 0 } { set choice [ expr $choice + %d ] } } { set choice [ expr $choice + %d ] }", 0x0200, 0x02 );
  
  // load native Tcl procedures for external files handling
  cmd( "if [ file exists \"$RootLsd/$LsdSrc/ls2html.tcl\" ] { if { [ catch { source \"$RootLsd/$LsdSrc/ls2html.tcl\" } ] != 0 } { set choice [ expr $choice + %d ] } } { set choice [ expr $choice + %d ] }", 0x0400, 0x04 );
  
  // load additional native Tcl procedures for external files handling
  cmd( "if [ file exists \"$RootLsd/$LsdSrc/lst_mdl.tcl\" ] { if { [ catch { source \"$RootLsd/$LsdSrc/lst_mdl.tcl\" } ] != 0 } { set choice [ expr $choice + %d ] } } { set choice [ expr $choice + %d ] }", 0x0800, 0x08 );
  
  // load module to improve to improve mouse selection
  cmd( "if [ file exists \"$RootLsd/$LsdSrc/dblclick.tcl\" ] { if { [ catch { source \"$RootLsd/$LsdSrc/dblclick.tcl\" } ] != 0 } { set choice [ expr $choice + %d ] } } { set choice [ expr $choice + %d ] }", 0x1000, 0x10 );
  
  if ( choice != 0 ) {
    log_tcl_error( "Source files check failed", "Required Tcl/Tk source file(s) missing or corrupted, check the installation of LSD and reinstall LSD if the problem persists" );
    cmd( "tk_messageBox -parent . -title Error -icon error -type ok -message \"File(s) missing or corrupted\" -detail \"Some critical Tcl files (0x%04x) are missing or corrupted.\nPlease check your installation and reinstall LSD if the problem persists.\n\nLSD is aborting now.\"", choice );
    myexit( 200 + choice );
  }
  
  // create a Tcl command that calls the C discard_change function before killing LSD
  Tcl_CreateCommand( inter, "discard_change", Tcl_discard_change, NULL, NULL );
  
  // create Tcl commands that get and set LSD object/variable properties
  Tcl_CreateCommand( inter, "get_obj_conf", Tcl_get_obj_conf, NULL, NULL );
  Tcl_CreateCommand( inter, "set_obj_conf", Tcl_set_obj_conf, NULL, NULL );
  Tcl_CreateCommand( inter, "get_var_conf", Tcl_get_var_conf, NULL, NULL );
  Tcl_CreateCommand( inter, "set_var_conf", Tcl_set_var_conf, NULL, NULL );
  
  // create a Tcl command to set a c variable when not in a Tcl idle loop
  Tcl_CreateCommand( inter, "set_c_var", Tcl_set_c_var, NULL, NULL );
  
  // create Tcl command to upload series data
  Tcl_CreateObjCommand( inter, "upload_series", Tcl_upload_series, NULL, NULL );
  
  // set platform-specific variables
  if ( i == 0 ) {
    cmd( "if [ string equal $tcl_platform(platform) unix ] { set wish $wishLinux; set sysTerm $sysTermLinux; set HtmlBrowser $browserLinux; set fonttype $fontLinux; set dim_character $fontSizeLinux }" );
#ifdef MAC_PKG
    cmd( "if [ string equal $tcl_platform(os) Darwin ] { set wish $wishMacTk86; set sysTerm $sysTermMac; set HtmlBrowser $browserMac; set fonttype $fontMac; set dim_character $fontSizeMac }" );
#else
    cmd( "if [ string equal $tcl_platform(os) Darwin ] { set wish $wishMacTk85; set sysTerm $sysTermMac; set HtmlBrowser $browserMac; set fonttype $fontMac; set dim_character $fontSizeMac }" );
#endif
    cmd( "if { [ string equal $tcl_platform(platform) windows ] && [ string equal $tcl_platform(machine) intel ] } { set wish $wishWinTk85; set sysTerm $sysTermWindows; set HtmlBrowser $browserWindows; set fonttype $fontWindows; set dim_character $fontSizeWindows }" );
    cmd( "if { [ string equal $tcl_platform(platform) windows ] && [ string equal $tcl_platform(machine) amd64 ] } { set wish $wishWinTk86; set sysTerm $sysTermWindows; set HtmlBrowser $browserWindows; set fonttype $fontWindows; set dim_character $fontSizeWindows }" );
  }
  
  cmd( "if [ string equal $tcl_platform(platform) windows ] { set small_character [ expr $dim_character - $deltaSizeWindows ] } { if [ string equal $tcl_platform(os) Darwin ] { set small_character [ expr $dim_character - $deltaSizeMac ] } { set small_character [ expr $dim_character - $deltaSizeLinux ] } }" );
  cmd( "set font_normal [ list \"$fonttype\" $dim_character ]" );
  cmd( "set font_small [ list \"$fonttype\" $small_character ]" );
  cmd( "set gpterm $gnuplotTerm" );
  
  // set main window
  cmd( "wm withdraw ." );
  cmd( "wm title . \"LSD Browser\"" );
  cmd( "wm protocol . WM_DELETE_WINDOW { if [ string equal [ discard_change ] ok ] { exit } }" );
  cmd( ". configure -menu .m" );    // define here to avoid redimensining the window
  cmd( "icontop . lsd" );
  cmd( "sizetop .lsd" );
  cmd( "setglobkeys ." );       // set global keys for main window
  
  create_logwindow( );
  cmd( "init_canvas_colors" );
  
  read_eq_filename( equation_name );
  
  struct_file = new char[ strlen( simul_name ) + 5 ];
  sprintf( struct_file, "%s.lsd", simul_name );
  
  eq_file = upload_eqfile( );
  strcpy( lsd_eq_file, "" );
  sprintf( name_rep, "report_%s.html", simul_name );
  
  grandTotal = true;        // not in parallel mode: use .tot headers
#endif
  
  // create fast equation look-up map if required
  if ( fast_lookup )
  { init_map( ); }
  
  stacklog = new lsdstack;
  stacklog->prev = NULL;
  stacklog->next = NULL;
  stacklog->ns = 0;
  stacklog->vs = NULL;
  strcpy( stacklog->label, "LSD Simulation Manager" );
  stack = 0;
  
#ifndef NO_WINDOW
  
  while ( 1 ) {
    create( );
    
    try {
      run( );
    }
    catch ( int p ) {         // return point from error_hard() (in object.cpp)
      if ( p != 919293 )    // check throw signature
      { throw; }
      
      quit = 0;
    }
    catch ( ... ) {           // send the rest upward
      throw;
    }
  }
  
#else
  
  run( );
  
#endif
  
  empty_lattice( );
  empty_description( );
  empty_cemetery( );
  blueprint->empty( );
  root->empty( );
  abmat->empty();
  delete abmat;
  delete blueprint;
  delete root;
  delete stacklog;
  delete [ ] struct_file;
  delete [ ] path;
  delete [ ] simul_name;
  delete [ ] lsdroot;
  
  return 0;
}


/*********************************
    RUN
*********************************/
void run( void )
{
  bool batch_sequential_loop = false;
  char bar_done[ 2 * BAR_DONE_SIZE ];
  int i, perc_done, last_done;
  FILE* f;
  clock_t start, end, last_update;
  result* rf;         // pointer for results files (may be zipped or not)
  
#ifdef PARALLEL_MODE
  
  // check if there are parallel computing variables
  if ( parallel_disable || max_threads < 2 )
  { parallel_mode = parallel_ready = false; }
  else {
    parallel_mode = search_parallel( root );
    parallel_ready = true;
  }
  
  // start multi-thread workers
  if ( parallel_mode )
  { workers = new worker[ max_threads ]; }
  
#else
  
  if ( search_parallel( root ) )
  { plog( "\nWarning: parallel mode is not supported under current configuration\n" ); }
  
  parallel_mode = false;
#endif
  
#ifndef NO_WINDOW
  set_buttons_log( true );
  
  prof.clear( );      // reset profiling times
  
  cover_browser( "Running...", "The simulation is being executed", "Use the LSD Log window buttons to interact:\n\n'Stop' :  aborts the simulation\n'Pause' / 'Resume' :  pauses and resumes the simulation\n'Fast' :  accelerates the simulation by hiding information\n'Observe' :  presents more run-time information\n'Debug' :  triggers the debugger at flagged variables" );
  cmd( "wm deiconify .log; raise .log; focus .log" );
#else
  plog( "\nProcessing configuration file %s ...\n", "", struct_file );
#endif
  
  set_fast( 0 );      // should always start on OBSERVE and switch to FAST later
  
  for ( i = 1, quit = 0; i <= sim_num && quit != 2; ++i ) {
    running = true;   // signal simulation is running
    cur_sim = i;    // Update the global variable holding information on the current run in the set of runs
    empty_cemetery( );  // ensure that previous data are not erroneously mixed (sorry Nadia!)
    
#ifndef NO_WINDOW
    prepare_plot( root, i );
#endif
    
    if ( fast_mode < 2 ) {
      if ( parallel_mode )
      { plog( "\nSimulation %d running (up to %d cores)...", "", i, max_threads ); }
      else
      { plog( "\nSimulation %d running...", "", i ); }
    }
    
    // if new batch configuration file, reload all
    if ( batch_sequential_loop ) {
      if ( load_configuration( true ) != 0 ) {
#ifndef NO_WINDOW
        log_tcl_error( "Load configuration", "Configuration file not found or corrupted" );
        cmd( "tk_messageBox -parent . -type ok -icon error -title Error -message \"Configuration file cannot be loaded\" -detail \"Check if LSD still has WRITE access to the model directory.\nLSD will close now.\"" );
#else
        fprintf( stderr, "\nFile '%s' not found or corrupted.\n", struct_file );
#endif
        myexit( 10 );
      }
      
      batch_sequential_loop = false;
    }
    
    // if just another run seed, reload just structure & parameters
    if ( i > 1 )
      if ( load_configuration( true, true ) != 0 ) {
#ifndef NO_WINDOW
        log_tcl_error( "Load configuration", "Configuration file not found or corrupted" );
        cmd( "tk_messageBox -parent . -type ok -icon error -title Error -message \"Configuration file cannot be reloaded\" -detail \"Check if LSD still has WRITE access to the model directory.\nLSD will close now.\"" );
#else
        fprintf( stderr, "\nFile '%s' not found or corrupted.\n", struct_file );
#endif
        myexit( 10 );
      }
      
    // build initial object list for user pointer checking
    if ( ! no_ptr_chk )
    { build_obj_list( true ); }
    
    series_saved = 0;
    
    if ( ! alloc_save_mem( root ) ) {
#ifndef NO_WINDOW
      log_tcl_error( "Memory allocation", "Not enough memory, too many series saved for the memory available" );
      cmd( "tk_messageBox -parent . -type ok -icon error -title Error -message \"Not enough memory\" -detail \"Too many series saved for the available memory. Memory insufficient for %d series over %d time steps. Reduce series to save and/or time steps.\nLSD will close now.\"", series_saved, max_step );
#else
      fprintf( stderr, "\nNot enough memory. Too many series saved for the memory available.\nMemory insufficient for %d series over %d time steps.\nReduce series to save and/or time steps.\n", series_saved, max_step );
#endif
      myexit( 11 );
    }
    
    // reset trace stack
    unwind_stack( );
    
    // new random routine' initialization
    init_random( seed );
    
    // Initialise ABMAT
    abmat_init( );
    
    // reset math error counters
    init_math_error( );
    
    seed++;
    scroll = false;
    pause_run = false;
    debug_flag = false;
    error_hard_thread = false;
    worker_ready = true;
    worker_crashed = false;
    wait_delete = NULL;
    stack_info = 0;
    use_nan = false;
    no_search = false;
    on_bar = false;
    done_in = 0;
    actual_steps = 0;
    perc_done = 0;
    last_done = -1;
    strcpy( bar_done, "" );
    wr_warn_cnt = 0;
    start = last_update = clock( );
    
    for ( t = 1; quit == 0 && t <= max_step; ++t ) {
      // update the percentage done bar, if needed
      if ( perc_done != last_done ) {
        if ( perc_done % 10 == 0 ) {
          char new_perc[ 10 ];
          sprintf( new_perc, "%d%%", perc_done );
          strcat( bar_done, new_perc );
          
          // check if continuing existing bar or starting a new one
          if ( fast_mode == 1 ) {
            if ( on_bar )
            { plog( "%d%%", "bar", perc_done ); }
            else {
              on_bar = true;
              plog( "\n%s", "bar", bar_done );
            }
          }
        }
        else if ( perc_done % ( 100 / ( BAR_DONE_SIZE - 33 ) ) == 0 ) {
          strcat( bar_done, "." );
          
          // check if continuing existing bar or starting a new one
          if ( fast_mode == 1 ) {
            if ( on_bar )
            { plog( ".", "bar" ); }
            else {
              on_bar = true;
              plog( "\n%s", "bar", bar_done );
            }
          }
        }
        
        last_done = perc_done;
      }
      
#ifndef NO_WINDOW
      // restart runtime variables color cycle
      cur_plt = 0;
      
      // adjust "clock" backwards if simulation is paused
      if ( pause_run )
      { t--; }
      
      if ( when_debug == t ) {
        debug_flag = true;
        cmd( "if [ winfo exists .deb ] { wm deiconify .deb; raise .deb; focus -force .deb; update }" );
      }
      
      // only update if simulation not paused
      if ( ! pause_run )
#endif
      {
        if (1 == t) {
          abmat_update_sim_pars( );  //only happens at t==1
        }
        
        root->update( );
        abmat_update( );
      }
      
      
      perc_done = ( 100 * t ) / max_step;
      
#ifndef NO_WINDOW
      
      if ( fast_mode == 0 && ! cur_plt && ! pause_run )
      { plog( "\nSimulation %d step %d done (%d%%)", "", i, t, perc_done ); }
      
      switch ( done_in ) {
        case 1:     // Stop button in Log window / s/S key in Runtime window
          if ( pause_run ) {
            cmd( "wm title .log \"$origLogTit\"" );
            cmd( ".log.but.pause conf -text Pause" );
          }
          
          plog( "\nSimulation stopped at t = %d", "", t );
          quit = 2;
          break;
          
        case 2:     // Fast button in Log window / f/F key in Runtime window
          set_fast( 1 );
          debug_flag = false;
          cmd( "set a [ split [ winfo children . ] ]" );
          cmd( "foreach i $a { if [ string match .plt* $i ] { wm withdraw $i } }" );
          cmd( "if { [ winfo exists .plt%d ] } { .plt%d.fond.go conf -state disabled }", i, i );
          cmd( "if { [ winfo exists .plt%d ] } { .plt%d.fond.shift conf -state disabled }", i, i );
          break;
          
        case 3:     // Debug button in Log window / d/D key in Runtime window
        
          if ( ! pause_run ) {
            when_debug = t + 1;
            debug_flag = true;
            connect_abmat_to_root();
            cmd( "if [ winfo exists .deb ] { wm deiconify .deb; raise .deb; focus -force .deb }" );
            disconnect_abmat_from_root();
          }
          else {    // if paused, just call the data browser
            connect_abmat_to_root();
            double useless = 0;
            deb( root, NULL, "Paused by User", &useless );
            disconnect_abmat_from_root();
          }
          
          break;
          
        case 4:     // Observe button in Log window / o/O key in Runtime window
          set_fast( 0 );
          cmd( "set a [ split [ winfo children . ] ]" );
          cmd( "foreach i $a { if [ string match .plt* $i ] { wm deiconify $i; raise $i } }" );
          cmd( "if { [ winfo exists .plt%d ] } { .plt%d.fond.go conf -state normal }", i, i );
          cmd( "if { [ winfo exists .plt%d ] } { .plt%d.fond.shift conf -state normal }", i, i );
          break;
          
        // plot window DELETE_WINDOW button handler
        case 5:
          if ( pause_run )
          { cmd( "wm title .log \"$origLogTit\"" ); }
          
          cmd( "destroytop .plt%d", i );
          plog( "\nSimulation stopped at t = %d", "", t );
          quit = 2;
          break;
          
        // runtime plot events
        case 7:     // Center button
          cmd( "if { [ winfo exist .plt%d ] && %d > $halfCanvas } { \
							set newpos [ expr %lf - [ expr  $halfCanvas / %lf ] ]; \
							$activeplot.c.c.cn xview moveto $newpos \
						}", i, t, t / ( double ) max_step, ( double ) max_step );
          break;
          
        case 8:     // Scroll checkbox
          scroll = ! scroll;
          break;
          
        case 9:     // Pause simulation
          pause_run = ! pause_run;
          
          if ( pause_run ) {
            cmd( "set origLogTit [ wm title .log ]; wm title .log \"$origLogTit (PAUSED)\"" );
            plog( "\nSimulation paused at t = %d", "", t );
            cmd( ".log.but.pause conf -text Resume" );
          }
          else {
            cmd( "wm title .log \"$origLogTit\"" );
            plog( "\nSimulation resumed" );
            cmd( ".log.but.pause conf -text Pause" );
          }
          
          break;
          
        case 35:    // error exit
          log_tcl_error( "Unexpected termination", "Please try again" );
          cmd( "tk_messageBox -parent . -type ok -icon error -title Error -message \"Unexpected termination\" -detail \"Please try again.\"" );
          myexit( 12 );
          break;
          
        default:
          break;
      }
      
      done_in = 0;
      
      // perform scrolling if enabled
      if ( ! pause_run && scroll )
      { cmd( "if [ winfo exist .plt%d ] { $activeplot.c.c.cn xview scroll 1 units }", i ); }
      
      if ( ( ( float ) clock( ) - last_update ) / CLOCKS_PER_SEC > UPD_PER ) {
        cmd( "update" );
        last_update = clock( );
      }
      
#endif
    } // end of for t
    
    actual_steps = max( t - 1, 1 );
    unsavedData = true;     // flag unsaved simulation results
    running = false;
    deb_log( false );     // close debug log file, if any
    end = clock( );
    
    if ( fast_mode == 1 && on_bar )
    { plog( "100%%", "bar" ); }
    
    if ( fast_mode < 2 )
    { plog( "\nSimulation %d %s (%.2f sec.)\n", "", i, quit == 2 ? "aborted" : "finished", ( float ) ( end - start ) / CLOCKS_PER_SEC ); }
    
    if ( quit == 1 )      // for multiple simulation runs you need to reset quit
    { quit = 0; }
    
#ifndef NO_WINDOW
    cmd( "destroytop .deb" );
    cmd( "update" );
    reset_plot( i );
#endif
    // run user closing function, reporting error appropriately
    user_exception = true;
    close_sim( );
    user_exception = false;
    connect_abmat_to_root(); //after reset_end()??
    reset_end( root );
    root->emptyturbo( );
    
    bool switch_abmat = abmat_series_saved > 0; //in total files only abmat data is considered.
    
    if ( quit != 2 && ( sim_num > 1 || no_window ) ) {
      // save results for multiple simulation runs, if any
      if ( series_saved > 0 || abmat_series_saved > 0 ) {
        // remove existing path, if any, from name in case of alternative output path
        char* alt_name = clean_file( simul_name );
        
        if ( ! no_res ) {
          if ( ! batch_sequential )
          { sprintf( msg, "%s%s%s_%d.%s", save_alt_path ? alt_path : path, strlen( save_alt_path ? alt_path : path ) > 0 ? "/" : "", save_alt_path ? alt_name : simul_name, seed - 1, docsv ? "csv" : "res" ); }
          else
          { sprintf( msg, "%s%s%s_%d_%d.%s", save_alt_path ? alt_path : path, strlen( save_alt_path ? alt_path : path ) > 0 ? "/" : "", save_alt_path ? alt_name : simul_name, findex, seed - 1, docsv ? "csv" : "res" ); }
          
          if ( fast_mode < 2 )
          { plog( "Saving results in file %s%s... ", "", msg, dozip ? ".gz" : "" ); }
          
          
          rf = new result( msg, "wt", dozip, docsv); // create results file object#          
          rf->title( root, 1 );           // write header
          rf->data( root, 0, actual_steps );      // write all data
          delete rf;                  // close file and delete object
          
          if ( fast_mode < 2 )
          { plog( "Done\n" ); }
        }
        
        if ( ! grandTotal || batch_sequential ) { // generate partial total files?
          if ( ! batch_sequential )
          { sprintf( msg, "%s%s%s_%d_%d.%s", save_alt_path ? alt_path : path, strlen( save_alt_path ? alt_path : path ) > 0 ? "/" : "", save_alt_path ? alt_name : simul_name, seed - i, seed - 1 + sim_num - i, docsv ? "csv" : "tot" ); }
          else
          { sprintf( msg, "%s%s%s_%d_%d_%d.%s", save_alt_path ? alt_path : path, strlen( save_alt_path ? alt_path : path ) > 0 ? "/" : "", save_alt_path ? alt_name : simul_name, findex, seed - i, seed - 1 + sim_num - i, docsv ? "csv" : "tot" ); }
        }
        else {                  // generate single grand total file
          sprintf( msg, "%s%s%s.%s", save_alt_path ? alt_path : path, strlen( save_alt_path ? alt_path : path ) > 0 ? "/" : "", save_alt_path ? alt_name : simul_name, docsv ? "csv" : "tot" );
        }
        
        if ( fast_mode < 2 && i == sim_num )    // print only for last
        { plog( "\nSaving totals in file %s%s... ", "", msg, dozip ? ".gz" : "" ); }
        
        if ( i == 1 && grandTotal && ! add_to_tot ) {
          rf = new result( msg, "wt", dozip, docsv, switch_abmat ); // create results file object
          rf->title( root, 0 );         // write header
        }
        else
        { rf = new result( msg, "a", dozip, docsv, switch_abmat ); }  // add results object to existing file
        
        rf->data( root, actual_steps );       // write current data data
        delete rf;                  // close file and delete object
        
        
        
        if ( fast_mode < 2 && i == sim_num )    // print only for last
        { plog( "Done\n" ); }
      }
      else if ( fast_mode < 2 )
      { plog( "Nothing to save: no element selected\n" ); }
      
      
      if ( batch_sequential && i == sim_num ) {   // last run of current batch file?
        findex++;                 // try next file
        sprintf( msg, "%s_%d.lsd", simul_name, findex );
        delete [ ] struct_file;
        struct_file = new char[ strlen( msg ) + 1 ];
        strcpy( struct_file, msg );
        f = fopen( struct_file, "r" );
        
        if ( f == NULL || ( fend != 0 && findex > fend ) ) { // no more file to process
          if ( f != NULL )
          { fclose( f ); }
          
          if ( fast_mode < 2 )
          { plog( "\nFinished processing %s\n", "", simul_name ); }
          
          break;
        }
        
        if ( fast_mode < 2 )
        { plog( "\nProcessing configuration file %s ...\n", "", struct_file ); }
        
        fclose( f );                  // process next file
        
        i = 0;                    // force restarting run count
        batch_sequential_loop = true;       // force reloading configuration
      }
    }
  }
  
  if ( fast_mode == 2 )
  { plog( "\nSimulation %d finished\n", "", i - 1 ); }
  
#ifndef NO_WINDOW
  uncover_browser( );
  set_buttons_log( false );
  show_prof_aggr( );
  cmd( "wm deiconify .log; raise .log; focus .log" );
#endif
  
#ifdef PARALLEL_MODE
  // stop multi-thread workers
  delete [ ] workers;
  workers = NULL;
#endif
  
  quit = 0;
}


/*********************************
    SET_VAR
*********************************/
// function to set a c variable when not in a Tcl idle loop (hardcoded vars only)
#ifndef NO_WINDOW
int Tcl_set_c_var( ClientData cdata, Tcl_Interp* inter, int argc, const char* argv[ ] )
{
  char vname[ MAX_ELEM_LENGTH ];
  int value;
  
  if ( argc != 3 )          // require 2 parameters: variable name and value
  { return TCL_ERROR; }
  
  if ( argv[ 1 ] == NULL || argv[ 2 ] == NULL )
  { return TCL_ERROR; }
  
  if ( ! sscanf( argv[ 1 ], "%99s", vname ) ) // remove unwanted spaces
  { return TCL_ERROR; }
  
  // set the appropriate variable (hardcoded in an else-if chain)
  if ( ! strcmp( vname, "done_in" ) ) {
    if ( ! sscanf( argv[ 2 ], "%d", &value ) )  // transform to integer
    { return TCL_ERROR; }
    
    done_in = value;
  }
  else
  { return TCL_ERROR; }
  
  return TCL_OK;
}
#endif


/*********************************
    SET_FAST
*********************************/
void set_fast( int level )
{
  if ( level > 2 )
  { level = 2; }
  
  if ( level < 0 )
  { level = 0; }
  
  // remove the variables stack when switching to any fast mode
  if ( fast_mode == 0 && level > 0 ) {
    if ( when_debug > 0 || stack_info > 0 || prof_aggr_time ) {
      plog( "\nWarning: %s is active, fast mode command ignored", "",
            when_debug > 0 ? "debugging" : "profiling" );
      return;
    }
    
    unwind_stack( );
    deb_log( false );
  }
  
  fast_mode = level;
  fast = ( level == 0 ) ? false : true;
}


/*********************************
    UNWIND_STACK
*********************************/
void unwind_stack( void )
{
  if ( stacklog != NULL ) {
    // remove stack allocation
    while ( stacklog->prev != NULL ) {
      lsdstack* cur_stack = stacklog;
      stacklog = stacklog->prev;
      delete cur_stack;
    }
    
    // prepare for next run
    stacklog->next = NULL;
    stacklog->ns = 0;
    stacklog->vs = NULL;
    stack = 0;
  }
  else {
#ifndef NO_WINDOW
    log_tcl_error( "Internal error", "LSD trace stack corrupted" );
    cmd( "tk_messageBox -parent . -type ok -icon error -title Error -message \"Internal LSD error\" -detail \"The LSD trace stack is corrupted.\nLSD will close now.\"" );
#else
    fprintf( stderr, "\nLSD trace stack corrupted.\n" );
#endif
    myexit( 28 );
  }
}


/*********************************
    ALLOC_SAVE_MEM
*********************************/
bool alloc_save_mem( object* r )
{
  int toquit = quit;
  bridge* cb;
  object* cur;
  variable* cv;
  
  // for each variable set the data saving support
  for ( cv = r->v; cv != NULL; cv = cv->next ) {
    cv->last_update = 0;
    
    // choose next update step for special updating variables
    if ( cv->delay > 0 || cv->delay_range > 0 ) {
      cv->next_update = 1 + cv->delay;
      
      if ( cv->delay_range > 0 )
      { cv->next_update += rnd_int( 0, cv->delay_range ); }
    }
    
    if ( ( cv->save || cv->savei ) && ! no_more_memory ) {
      if ( cv->num_lag > 0 || cv->param == 1 )
      { cv->start = 0; }
      else
      { cv->start = 1; }
      
      cv->end = max_step;
      
      delete [ ] cv->data;
      
      try {
        cv->data = new double[ max_step + 1 ];
      }
      catch ( bad_alloc& ) {
        set_lab_tit( cv );
        plog( "\nNot enough memory.\nData for %s and subsequent series will not be saved.\n", "", cv->lab_tit );
        cv->save = cv->savei = false;
        no_more_memory = true;
      }
      
      ++series_saved;
      
      if ( cv->num_lag > 0  || cv->param == 1 )
      { cv->data[ 0 ] = cv->val[ 0 ]; }
    }
    else {
      if ( no_more_memory )
      { cv->save = cv->savei = false; }
    }
    
    if ( ( cv->num_lag > 0 || cv->param == 1 ) && cv->data_loaded == '-' ) {
      sprintf( msg, "%s '%s' in object '%s' has not been initialized", cv->param == 1 ? "parameter" : "variable", cv->label, r->label );
      error_hard( msg, "required initialization values missing", "select the object and choose menu 'Data'/'Initial Values'" );
      
      toquit = 2;
    }
  }
  
  for ( cb = r->b; cb != NULL; cb = cb->next )
    for ( cur = cb->head; cur != NULL && quit != 2; cur = go_brother( cur ) )
    { alloc_save_mem( cur ); }
    
  if ( quit != 2 )
  { quit = toquit; }
  
  return ! no_more_memory;
}


/*********************************
    CREATE_LOG_WINDOW
*********************************/
#ifndef NO_WINDOW
void create_logwindow( void )
{
  if ( ! tk_ok )
  { myexit( 7 ); }
  
  cmd( "newtop .log \"LSD Log\" { if { [ discard_change ] == \"ok\" } { exit } { } } \"\"" );
  
  cmd( "set w .log.text" );
  cmd( "frame $w" );
  cmd( "scrollbar $w.scroll -command \"$w.text yview\"" );
  cmd( "scrollbar $w.scrollx -command \"$w.text xview\" -orient hor" );
  cmd( "text $w.text -relief sunken -yscrollcommand \"$w.scroll set\" -xscrollcommand \"$w.scrollx set\" -wrap none -font \"$font_normal\"" );
  cmd( "$w.text configure -tabs {%s}", tabs  );
  
  // Log window tags
  cmd( "$w.text tag configure highlight -foreground red" );
  cmd( "$w.text tag configure tabel" );
  cmd( "$w.text tag configure series -tabs {2c 5c 8c}" );
  cmd( "$w.text tag configure prof1 -tabs {5c 7.5c 9c 11.2c 13.2c 17.5c}" );
  cmd( "$w.text tag configure prof2 -tabs {3c 6c 9c}" );
  
  cmd( "pack $w.scroll -side right -fill y" );
  cmd( "pack $w.text -expand yes -fill both" );
  cmd( "pack $w.scrollx -side bottom -fill x" );
  cmd( "pack $w -expand yes -fill both" );
  
  cmd( "set w .log.but" );
  cmd( "frame $w" );
  cmd( "button $w.stop -width $butWid -text Stop -command {set_c_var done_in 1} -underline 0 -state disabled" );
  cmd( "button $w.pause -width $butWid -text Pause -command {set_c_var done_in 9} -underline 0 -state disabled" );
  cmd( "button $w.speed -width $butWid -text Fast -command {set_c_var done_in 2} -underline 0 -state disabled" );
  cmd( "button $w.obs -width $butWid -text Observe -command {set_c_var done_in 4} -underline 0 -state disabled" );
  cmd( "button $w.deb -width $butWid -text Debug -command {set_c_var done_in 3} -underline 0 -state disabled" );
  cmd( "button $w.help -width $butWid -text Help -command {LsdHelp log.html} -underline 0" );
  cmd( "button $w.copy -width $butWid -text Copy -command {tk_textCopy .log.text.text} -underline 0" );
  
  cmd( "pack $w.stop $w.pause $w.speed $w.obs $w.deb $w.copy $w.help -padx 5 -pady 10 -side left" );
  cmd( "pack $w -padx 10 -side right" );
  
  cmd( "showtop .log none 1 1 0" );
  set_shortcuts_log( ".log", "log.html" );
  
  // replace text widget default insert, delete and replace bindings, preventing the user to change it
  cmd( "rename .log.text.text .log.text.text.internal" );
  cmd( "proc .log.text.text { args } { switch -exact -- [lindex $args 0] { insert { } delete { } replace { } default { return [ eval .log.text.text.internal $args] } } }" );
  
  // a Tcl/Tk version of plog
  cmd( "proc plog cm { .log.text.text.internal insert end $cm; .log.text.text.internal see end }" );
  
  log_ok = true;
}
#endif


/*********************************
    RESET_END
*********************************/
void reset_end( object* r )
{
  bridge* cb;
  object* cur;
  variable* cv;
  
  for ( cv = r->v; cv != NULL; cv = cv->next ) {
    if ( cv->save || cv->abmat )
    { cv->end = t - 1; }
    
    if ( cv->savei == 1 )
    { save_single( cv ); }
  }
  
  for ( cb = r->b; cb != NULL; cb = cb->next ) {
    cur = cb->head;
    
    if ( cur != NULL && cur->to_compute )
      for ( ; cur != NULL; cur = go_brother( cur ) )
      { reset_end( cur ); }
  }
}


/*********************************
    SET_SHORTCUTS_LOG
*********************************/
#ifndef NO_WINDOW
void set_shortcuts_log( const char* window, const char* help )
{
  cmd( "bind %s <F1> { LsdHelp %s }", window, help  );
  cmd( "bind %s <KeyPress-s> {.log.but.stop invoke}; bind %s <KeyPress-S> {.log.but.stop invoke}", window, window );
  cmd( "bind %s <KeyPress-p> {.log.but.pause invoke}; bind %s <KeyPress-P> {.log.but.pause invoke}", window, window );
  cmd( "bind %s <KeyPress-r> {.log.but.pause invoke}; bind %s <KeyPress-R> {.log.but.pause invoke}", window, window );
  cmd( "bind %s <KeyPress-f> {.log.but.speed invoke}; bind %s <KeyPress-F> {.log.but.speed invoke}", window, window );
  cmd( "bind %s <KeyPress-o> {.log.but.obs invoke}; bind %s <KeyPress-O> {.log.but.obs invoke}", window, window );
  cmd( "bind %s <KeyPress-d> {.log.but.deb invoke}; bind %s <KeyPress-D> {.log.but.deb invoke}", window, window );
  cmd( "bind %s <KeyPress-h> {.log.but.help invoke}; bind %s <KeyPress-H> {.log.but.help invoke}", window, window );
  cmd( "bind %s <KeyPress-c> {.log.but.copy invoke}; bind %s <KeyPress-C> {.log.but.copy invoke}", window, window );
  cmd( "bind %s <Control-c> {.log.but.copy invoke}; bind %s <Control-C> {.log.but.copy invoke}", window, window );
  cmd( "bind %s <KeyPress-Escape> {focus -force .}", window );
}


/*********************************
    SET_BUTTONS_LOG
*********************************/
void set_buttons_log( bool on )
{
  const char* state = ( char* ) ( on ? "normal" : "disabled" );
  
  cmd( ".log.but.stop configure -state %s", state );
  cmd( ".log.but.pause configure -state %s", state );
  cmd( ".log.but.speed configure -state %s", state );
  cmd( ".log.but.obs configure -state %s", state );
  cmd( ".log.but.deb configure -state %s", state );
}


/*********************************
    COVER_BROWSER
*********************************/
void cover_browser( const char* text1, const char* text2, const char* text3 )
{
  if ( brCovered )    // ignore if already covered
  { return; }
  
  cmd( "if [ winfo exists .str ] { wm withdraw .str }" );
  
  cmd( "set origMainTit [ wm title . ]; wm title . \"$origMainTit (DISABLED)\"" );
  cmd( "destroy .bbar .m .l" );
  
  cmd( "frame .t" );
  cmd( "label .t.l1 -font {-weight bold} -text \"%s\"", text1  );
  cmd( "label .t.l2 -text \"\n%s\"", text2  );
  cmd( "label .t.l3 -fg red -text \"\nInteraction with the LSD Browser is now disabled\"" );
  cmd( "label .t.l4 -justify left -text \"\n%s\"", text3  );
  cmd( "pack .t.l1 .t.l2 .t.l3 .t.l4 -expand yes -fill y" );
  cmd( "pack .t -fill both -expand yes -padx 10 -pady 10" );
  cmd( "focus .t");
  set_shortcuts_log( ".t", "runtime.html" );
  
  cmd( "update" );
  
  brCovered = true;
  redrawRoot = false;
}


/*********************************
    UNCOVER_BROWSER
*********************************/
void uncover_browser( void )
{
  if ( ! brCovered || running ) // ignore if not covered or running
  { return; }
  
  cmd( "destroytop .deb" );
  cmd( "destroy .t" );
  cmd( "wm title . $origMainTit" );
  cmd( "if { [ string equal [ wm state . ] normal ] && [ winfo exist .str ] && ! [ string equal [ wm state .str ] normal ] } { wm deiconify .str; lower .str }" );
  cmd( "if { [ string equal [ wm state . ] normal ] } { raise .; focus . }" );
  
  cmd( "update" );
  
  brCovered = false;
  redrawRoot = true;
}


/*********************************
    SHOW_PROF_AGGR
*********************************/
struct item {
  const char* var, *obj;
  unsigned int time;
  unsigned int count;
};

bool comp_item( item& item1, item& item2 )
{
  int comp_str = strcmp( item1.obj, item2.obj );
  
  if ( ! comp_str )
  { return item1.time > item2.time; }
  else
  { return comp_str < 0; }
}

void show_prof_aggr( void )
{
  if ( ! prof_aggr_time )
  { return; }
  
  item elem;
  list < item > vars;
  list < item >::iterator it1;
  variable* cv;
  map < string, profile >::iterator it2;
  
  plog( "\nProfiling aggregated results:\n" );
  plog( "\nObject\tElement\tTime (msec.)\tComputation count", "prof2" );
  
  for ( it2 = prof.begin(); it2 != prof.end(); ++it2 ) {
    elem.var = it2->first.c_str( );
    cv = root->search_var( NULL, elem.var );
    elem.obj = ( cv == NULL ) ? NULL : cv->up->label;
    elem.time = 1000 * it2->second.ticks / CLOCKS_PER_SEC;
    elem.count = it2->second.comp;
    vars.push_back( elem );
  }
  
  vars.sort( comp_item );
  
  for ( it1 = vars.begin(); it1 != vars.end(); ++it1 )
  { plog( "\n%-12.12s\t%-12.12s\t%d\t%d", "prof2", it1->obj, it1->var, it1->time, it1->count ); }
  
  plog( "\n" );
}
#endif


/*********************************
    RESULTS_ALT_PATH
*********************************/
//Simple tool to allow changing where results are saved.
void results_alt_path( const char* altPath )
{
  if ( save_alt_path )
  { delete [ ] alt_path; }
  
  if ( strlen( altPath ) == 0 ) {
    save_alt_path = false;
    return;
  }
  
  alt_path = new char[ strlen( altPath ) + 1 ];
  
  if ( sprintf( alt_path, "%s", altPath ) > 0 ) {
    int lstChr = strlen( alt_path ) - 1;
    
    if ( alt_path[ lstChr ] == '\\' || alt_path[ lstChr ] == '/' )
    { alt_path[ lstChr ] = '\0'; }
    
    struct stat sb;
    
    if ( stat( alt_path, &sb ) == 0 && S_ISDIR( sb.st_mode ) ) {
      save_alt_path = true;
      return;
    }
  }
  
  delete [ ] alt_path;
  save_alt_path = false;
  plog( "\nWarning: could not open directory '%s', ignoring '-o' option.\n", "", altPath );
}


/*********************************
    CLEAN_FILE
*********************************/
// remove any path prefixes to filename, if present
char* clean_file( char* filename )
{
  if ( strchr( filename, '/' ) != NULL )
  { return strrchr( filename, '/' ) + 1; }
  
  if ( strchr( filename, '\\' ) != NULL )
  { return strrchr( filename, '\\' ) + 1; }
  
  return filename;
}


/*********************************
    CLEAN_PATH
*********************************/
// remove cygwin path prefix, if present, and replace \ with /
char* clean_path( char* filepath )
{
  int i, len = strlen( "/cygdrive/" );
  
  if ( ! strncmp( filepath, "/cygdrive/", len ) ) {
    char* temp = new char[ strlen( filepath ) + 1 ];
    temp[ 0 ] = toupper( filepath[ len ] );   // copy drive letter
    temp[ 1 ] = ':';              // insert ':'
    strcpy( temp + 2, filepath + len + 1 );   // copy removing prefix
    strcpy( filepath, temp );
    delete [ ] temp;
  }
  
  len = strlen( filepath );
  
  for ( i = 0; i < len; ++i )
    if ( filepath[ i ] == '\\' )          // replace \ with /
    { filepath[ i ] = '/'; }
    
  return filepath;
}


/***************************************
    SEARCH_PARALLEL
***************************************/
bool search_parallel( object* r )
{
  bridge* cb;
  variable* cv;
  
  // search among the variables
  for ( cv = r->v; cv != NULL; cv = cv->next )
    if ( cv->parallel )
    { return true; }
    
  // search among descendants
  for ( cb = r->b; cb != NULL; cb = cb->next )
    if ( cb->head != NULL )
      if ( search_parallel( cb->head ) )
      { return true; }
      
  return false;
}


/*******************************************
    DEB_LOG
    Creates/saves the file "log.txt" and
    enable/disable logging the variables
    computation order and enable/disable the
    debugger
********************************************/
void deb_log( bool on, int time )
{
#ifndef NO_WINDOW

  // check if should turn off
  if ( ! on || parallel_mode || fast_mode != 0 ) {
    // disable debugging
    if ( time > t && when_debug >= time )
    { when_debug = 0; }
    else if ( ( time == 0 && when_debug == t ) || time == t )
    { debug_flag = false; }
    
    // act now?
    if ( time == 0 || t > time ) {
      // close file if open
      if ( log_file != NULL ) {
        fclose( log_file );
        log_file = NULL;
      }
    }
    else
    { log_stop = time; }
  }
  
  // check if should turn on
  if ( on && ! parallel_mode && fast_mode == 0 ) {
    // enable debugging
    if ( time > t )
    { when_debug = time; }
    else if ( time == 0 || time == t ) {
      when_debug = t;
      debug_flag = true;
      cmd( "if [ winfo exists .deb ] { wm deiconify .deb; raise .deb; focus -force .deb; update }" );
    }
    
    // ignore if log already open
    if ( log_file == NULL ) {
      log_file = fopen( "log.txt", "a" );
      log_start = time;
      log_stop = max_step;
    }
  }
  
  if ( on && ( parallel_mode || fast_mode > 0 ) )
    plog( "\nWarning: %s is active, debug command ignored", "",
          parallel_mode ? "parallel processing" : "fast mode" );
          
#endif
}
