(* Content-type: application/vnd.wolfram.mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 12.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       158,          7]
NotebookDataLength[    124202,       3024]
NotebookOptionsPosition[    109607,       2775]
NotebookOutlinePosition[    110256,       2800]
CellTagsIndexPosition[    110213,       2797]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{
Cell["\<\
17-30 January 1995 CHR,  continuing from \"Oseen\", 2 November 1994
Use Zimm (1980) formulation to solve hydrodynamics of a given subunit array. \
Compare various aspects of the Kirkwood/Bloomfield et al. (which I will \
sometimes call \"K/B\") approximation.\
\>", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"dd8c626d-d4db-429b-b842-5b3ee75eaba8"],

Cell[TextData[{
 "CHR started 18 Feb 2020. Test of reproducibility for the ReScience 10-year \
reproducibility challenge (https://rescience.github.io/ten-years/). Chosen \
paper is Robert, C.H. (1995) Estimating Friction Coefficients of Mixed \
Globular/Chain Molecules, such as Protein/DNA Complexes. Biophys J. 69, \
840-48.\n\nAll code in this file came from the original notebook \"s6.0 Zimm \
(1980).nb\" and was run using ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " versions 5.2 and 12.0.\nChanges:\n- added dependence on library s6.1.1 \
(Get[...])\n- added comments\n- copied in MatrixForm tables of original \
results for direct comparison\n- added new final plot that matches Figure 3 \
in the published article for easier comparison\n\nBlue and gray boxes \
indicate results to compare to originals as well as certain comments."
}], "Text",
 CellFrame->True,
 InitializationCell->True,
 FontColor->RGBColor[0, 0, 1],
 Background->GrayLevel[
  0.833326],ExpressionUUID->"4e3a218b-be26-4640-a1f2-d8b9b5531067"],

Cell[BoxData[
 RowBox[{
  RowBox[{"Get", "[", 
   RowBox[{"\"\<ReScience/\>\"", "<>", "\"\<s6.1_repro.m\>\""}], "]"}], 
  ";"}]], "Input",
 InitializationCell->True,
 CellLabel->"In[1]:=",ExpressionUUID->"bc8446c2-357a-4acf-b781-b5ca975580fb"],

Cell[CellGroupData[{

Cell["\<\
General functions to define interaction tensor, velocity perturbations, etc.\
\>", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"eee7abd4-1720-47d7-a2f2-566ce75c99f4"],

Cell["\<\
Off[General::spell];
Off[General::spell1];\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[2]:=",ExpressionUUID->"1ee83d32-0cac-416d-9dc0-227be9be9d10"],

Cell["avg[x_]:=Apply[Plus,x]/Length[x];", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[4]:=",ExpressionUUID->"aa933b91-b2cb-450c-a913-2ccd9b203711"],

Cell["\<\
l[r_]:=Sqrt[r.r];
rr[r_]:=Outer[Times,r,r];
Id[n_]:=IdentityMatrix[n];
zero[n_]:=Table[0,{n},{n}];\
\>", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[5]:=",ExpressionUUID->"f8be7ac8-0a76-4829-9d88-cdd3af80299b"],

Cell["T[r_]:=1/(8 Pi nu l[r])(Id[Length[r]]+rr[r]/l[r]^2);", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[9]:=",ExpressionUUID->"ec8d89f7-da76-4331-bbf5-945812918872"],

Cell[CellGroupData[{

Cell["MatrixForm[T[{x,y,z}]]", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[10]:=",ExpressionUUID->"71d203c7-48ba-49db-a40c-a7a9d5bb1cc2"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      FractionBox[
       RowBox[{"1", "+", 
        FractionBox[
         SuperscriptBox["x", "2"], 
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}]], 
      FractionBox[
       RowBox[{"x", " ", "y"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]], 
      FractionBox[
       RowBox[{"x", " ", "z"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]]},
     {
      FractionBox[
       RowBox[{"x", " ", "y"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]], 
      FractionBox[
       RowBox[{"1", "+", 
        FractionBox[
         SuperscriptBox["y", "2"], 
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}]], 
      FractionBox[
       RowBox[{"y", " ", "z"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]]},
     {
      FractionBox[
       RowBox[{"x", " ", "z"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]], 
      FractionBox[
       RowBox[{"y", " ", "z"}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"], "+", 
           SuperscriptBox["z", "2"]}], ")"}], 
         RowBox[{"3", "/", "2"}]]}]], 
      FractionBox[
       RowBox[{"1", "+", 
        FractionBox[
         SuperscriptBox["z", "2"], 
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}], 
       RowBox[{"8", " ", "nu", " ", "\[Pi]", " ", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["x", "2"], "+", 
          SuperscriptBox["y", "2"], "+", 
          SuperscriptBox["z", "2"]}]]}]]}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{3.7949128050531893`*^9, 3.795266164143448*^9},
 CellLabel->
  "Out[10]//MatrixForm=",ExpressionUUID->"10343a5f-0f91-4084-9a1b-\
bcbd569cc987"]
}, Open  ]],

Cell["Velocity and velocity perturbation functions.", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"ac37265c-0aeb-4626-a25d-05efddfe381e"],

Cell["\<\
V[i_,{p_,radius_}]:=Id[3]/(6 Pi nu radius[[i]])
dV[i_,j_,{p_,radius_}]:=T[p[[j]]-p[[i]]];
u[i_,{p_,radius_}]:={-p[[i,2]],p[[i,1]],0};
z[i_,{p_,radius_}]:={p[[i,2]],-p[[i,1]],0};\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[11]:=",ExpressionUUID->"1f054c95-0db6-45f4-9945-8358f971a3f2"],

Cell["Demonstrate some velocity profiles", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"ac596702-0e90-4447-93e4-c7d3f17f6fb7"],

Cell[TextData[{
 "The perturbation of the velocity of the fluid at a given point r{x,y,z} \
with respect to a frictional sphere is given by the following expression for \
dv for a force f {x,y,z} acting on the center.\n\nThe viscosity nu has \
typically the units of poise, or g/(cm sec). If distances are in cm and force \
is (g cm/sec",
 StyleBox["2",
  FontSize->10,
  FontVariations->{"CompatibilityType"->"Superscript"}],
 "), the velocity given here is cm/sec.\nThe relative velocity dv0 is the \
relative solvent velocity perturbation about a particle travelling with \
whatever velocity results from a unit force acting on it, expressed in terms \
of the distance relative to the particle radius. The factor of the particle \
radius cancels. A positive value says that the sphere is dragging fluid with \
it."
}], "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"fbedbbcd-1ce7-494c-9450-2c98eba0ce76"],

Cell["\<\
ClearAll[dv,dv0];
dv[r_,f_]:=T[r].f;
dv0[r0_,f0_]:=dv[r0,f0] 6 Pi nu; \
\>", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[15]:=",ExpressionUUID->"6da2bbb1-d1cf-4d61-bed3-8614e1ee61e4"],

Cell[CellGroupData[{

Cell["dv[{1,0,0} cm,{1,0,0} g cm/s^2]/.nu->g/(cm s)", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[18]:=",ExpressionUUID->"c1253b67-e300-4faa-a823-c372f457934d"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox[
    SqrtBox[
     SuperscriptBox["cm", "2"]], 
    RowBox[{"4", " ", "\[Pi]", " ", "s"}]], ",", "0", ",", "0"}], 
  "}"}]], "Output",
 CellChangeTimes->{3.794912805237885*^9, 3.795266164458206*^9},
 CellLabel->"Out[18]=",ExpressionUUID->"ff493bac-52f8-4560-90ff-eb39e5b14d80"]
}, Open  ]],

Cell[CellGroupData[{

Cell["dv0[{1,0,0},{1,0,0}]", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[19]:=",ExpressionUUID->"219a4c04-751f-4f0e-b78e-e1be5a51824c"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["3", "2"], ",", "0", ",", "0"}], "}"}]], "Output",
 CellChangeTimes->{3.794912805328967*^9, 3.795266164512966*^9},
 CellLabel->"Out[19]=",ExpressionUUID->"f8fe5fb8-40d5-497c-a06f-480d93685c5c"]
}, Open  ]],

Cell[CellGroupData[{

Cell["dv0[{1.1,0,0},{fx,fy,fz}]", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[20]:=",ExpressionUUID->"6859774f-bd9c-47f1-9612-bd42799852ca"],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"6", " ", 
    RowBox[{"(", 
     RowBox[{"0.`", "\[VeryThinSpace]", "+", 
      FractionBox[
       RowBox[{"0.0723431559508615`", " ", "fx"}], "nu"]}], ")"}], " ", "nu", 
    " ", "\[Pi]"}], ",", 
   RowBox[{"6", " ", 
    RowBox[{"(", 
     RowBox[{"0.`", "\[VeryThinSpace]", "+", 
      FractionBox[
       RowBox[{"0.03617157797543075`", " ", "fy"}], "nu"]}], ")"}], " ", "nu",
     " ", "\[Pi]"}], ",", 
   RowBox[{"6", " ", 
    RowBox[{"(", 
     RowBox[{"0.`", "\[VeryThinSpace]", "+", 
      FractionBox[
       RowBox[{"0.03617157797543075`", " ", "fz"}], "nu"]}], ")"}], " ", "nu",
     " ", "\[Pi]"}]}], "}"}]], "Output",
 CellChangeTimes->{3.794912805413148*^9, 3.795266164614436*^9},
 CellLabel->"Out[20]=",ExpressionUUID->"49293797-898e-4430-a3e4-c7bd1db3ca72"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Plot3D[r={x,y,0};l[N[dv0[r,{1,0,0}]]],
\t{x,-2,2},{y,-2,2},PlotPoints->20,
\tAxesLabel->{\"x\",\"y\",\"z\"}]\
\>", "Input",
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[21]:=",ExpressionUUID->"f37e8b4b-cd36-40d8-8d04-f7a0962fd2e0"],

Cell[BoxData[
 Graphics3DBox[{{}, {}},
  Axes->True,
  AxesLabel->{
    FormBox["\"x\"", TraditionalForm], 
    FormBox["\"y\"", TraditionalForm], 
    FormBox["\"z\"", TraditionalForm]},
  AxesOrigin->{Automatic, Automatic, Automatic},
  BoxRatios->{1, 1, 0.4},
  DisplayFunction->Identity,
  FaceGrids->None,
  FaceGridsStyle->Automatic,
  Method->{"DefaultBoundaryStyle" -> Directive[
      GrayLevel[0.3]], 
    "DefaultGraphicsInteraction" -> {
     "Version" -> 1.2, "TrackMousePosition" -> {True, False}, 
      "Effects" -> {
       "Highlight" -> {"ratio" -> 2}, "HighlightPoint" -> {"ratio" -> 2}, 
        "Droplines" -> {
         "freeformCursorMode" -> True, 
          "placement" -> {"x" -> "All", "y" -> "None"}}}}, "RotationControl" -> 
    "Globe"},
  PlotRange->{Full, Full, All},
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->{Automatic, Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.794912806029675*^9, 3.79526616530217*^9},
 CellLabel->"Out[21]=",ExpressionUUID->"b44ff57b-66b3-49c0-b0fa-ff4d7a1c97bd"]
}, Open  ]]
}, Open  ]],

Cell[TextData[{
 "This 3D plot was not displayed in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " 12.0. (It was calculated here as an exploration, and is not part of the \
original article)."
}], "Text",
 CellFrame->True,
 CellChangeTimes->{{3.795434458282185*^9, 3.795434459785646*^9}},
 FontColor->RGBColor[0, 0, 1],
 Background->GrayLevel[
  0.833326],ExpressionUUID->"7f54a880-0913-48f4-83cf-005bad9924bf"],

Cell[CellGroupData[{

Cell["\<\
Define matrix flatteners to set up the linear hydrodynamics equations:\
\>", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"5e8b84ab-d81c-46c5-84b4-01fb4b61d7b5"],

Cell["\<\
flatmat[M_]:= Module[{Mt,Mf},
\tMt=Transpose[M,{1,3,2,4}];
\tMf=Table[Flatten[Mt[[i,j]]],{i,Length[Mt]},{j,Length[Mt[[1]]]}];
\tN[Flatten[Mf,1]]
];\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[22]:=",ExpressionUUID->"ac8541a0-926b-4b11-9f9e-aeeeb2083c4d"],

Cell["flatvec[x_]:=Flatten[x,1];", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[23]:=",ExpressionUUID->"ee083062-60c5-4dfd-b1cd-2663ccf65525"],

Cell["\<\
complete[M_,geo_]:=Module[{base,n,npoints,Mc},
\t(*\tUgly but necessary, since the additional equations required
\t\tto solve the tableau destroy the \"tensorness\" of the matrix
\t\tof matrices. *)
\tdim=Length[M];
\tn=dim/3-1;
\tMc=Table[0,{dim+1},{dim+1}];
\tDo[\tMc[[i,j]]=M[[i,j]],{i,dim},{j,dim}];
\tDo[\tbase=(i-1)*3;
\t\tMc[[base+1,dim+1]]=Evaluate[u[i,geo]][[1]];
\t\tMc[[base+2,dim+1]]=Evaluate[u[i,geo]][[2]];
\t\tMc[[base+3,dim+1]]=Evaluate[u[i,geo]][[3]],
\t{i,n}];
\tbottom=Flatten[Table[z[i,geo],{i,n}]];
\tDo[ Mc[[dim+1,j]]=bottom[[j]],{j,3*n}];
\tReturn[Mc]
];\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[24]:=",ExpressionUUID->"0ef32b1d-1efb-45bc-a8e8-f3fafbdc8c58"],

Cell["Show tensor->matrix flattening in action:", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"66a8535f-39c9-4240-9196-1cbe3555aba6"],

Cell["\<\
M={\t{{{a1111,a1112},{a1121,a1122}},{{a1211,a1212},{a1221,a1222}}},
\t{{{a2111,a2112},{a2121,a2122}},{{a2211,a2212},{a2221,a2222}}} };\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[25]:=",ExpressionUUID->"7f4e9cf7-6654-476a-a0da-f5283632157e"],

Cell[CellGroupData[{

Cell["MatrixForm[M]", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[26]:=",ExpressionUUID->"058bf719-e40f-4ffb-9e16-a6c05476cf06"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a1111", "a1112"},
         {"a1121", "a1122"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}], 
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a1211", "a1212"},
         {"a1221", "a1222"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}]},
     {
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a2111", "a2112"},
         {"a2121", "a2122"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}], 
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a2211", "a2212"},
         {"a2221", "a2222"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}]}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{3.7949128063367577`*^9, 3.795266166581292*^9},
 CellLabel->
  "Out[26]//MatrixForm=",ExpressionUUID->"e6e8dfec-0b2d-4f42-b051-\
f6661933b070"]
}, Open  ]],

Cell[CellGroupData[{

Cell["MatrixForm[Transpose[M,{1,3,2,4}]]", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[27]:=",ExpressionUUID->"1ec396bf-a54e-408a-b6d2-28c6e7c4ed64"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a1111", "a1112"},
         {"a1211", "a1212"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}], 
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a1121", "a1122"},
         {"a1221", "a1222"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}]},
     {
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a2111", "a2112"},
         {"a2211", "a2212"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}], 
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {"a2121", "a2122"},
         {"a2221", "a2222"}
        },
        GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
        GridBoxSpacings->{"Columns" -> {
            Offset[0.27999999999999997`], {
             Offset[0.7]}, 
            Offset[0.27999999999999997`]}, "Rows" -> {
            Offset[0.2], {
             Offset[0.4]}, 
            Offset[0.2]}}], "\[NoBreak]", ")"}]}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{3.7949128064043207`*^9, 3.795266166624354*^9},
 CellLabel->
  "Out[27]//MatrixForm=",ExpressionUUID->"b469c115-68f0-47c7-bf1e-\
0c49c8334017"]
}, Open  ]],

Cell[CellGroupData[{

Cell["MatrixForm[flatmat[M]]", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[28]:=",ExpressionUUID->"d5113092-c639-418c-8d0a-2fba707f0d3c"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"a1111", "a1112", "a1211", "a1212"},
     {"a1121", "a1122", "a1221", "a1222"},
     {"a2111", "a2112", "a2211", "a2212"},
     {"a2121", "a2122", "a2221", "a2222"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{3.794912806469883*^9, 3.795266166702533*^9},
 CellLabel->
  "Out[28]//MatrixForm=",ExpressionUUID->"d9cd662a-9490-4a1f-b5d4-\
14de100d3cdf"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Random-orientation matrix definition", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"c05606c5-274e-4043-af34-d971f81c998b"],

Cell["\<\
These are from my Nov'94 hydrodynamics program for the random orientation of \
a chain\
\>", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"efb3c98a-06a1-41f0-97f5-3750b357ec53"],

Cell["\<\
Ax[w_]:={\t{1,0,0},
\t\t\t{0,Cos[w],Sin[w]},
\t\t\t{0,-Sin[w],Cos[w]}\t};
Ay[w_]:={\t{Cos[w],0,-Sin[w]},
\t\t\t{0,1,0},
\t\t\t{Sin[w],0,Cos[w]}\t};
Az[w_]:={\t{Cos[w],Sin[w],0},
\t\t\t{-Sin[w],Cos[w],0},
\t\t\t{0,0,1}\t};\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[29]:=",ExpressionUUID->"1ccbb2cc-64b4-4dc2-bda9-0b2b2b350ebc"],

Cell["\<\
AO:=Az[Random[Real,N[{0,2 Pi}]]].Ax[Random[Real,N[{0,2 Pi}]]].
\tAz[Random[Real,N[{0,2 Pi}]]];
\t\t\t\t\t\t(*\tRandom overall orientation matrix *)\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[32]:=",ExpressionUUID->"ef06c4db-2bf8-47c0-9f89-1639614c4987"],

Cell["\<\
rotate[A_,o_]:=Module[{i,j,depth},
\t(*\tRoutine to rotate vectors even when they are assembled
\t\tinto groups and lists of those groups. Rotation matrix
\t\tis A, vector object is o *)
\tdepth=Depth[N[o]];
\tWhich[\tdepth==1,Print[\"Not done: \",depth];Return[o],
\t\t\tdepth==2,Return[N[A].o],
\t\t\tdepth==3,Return[Transpose[N[A].Transpose[o]]],
\t\t\tdepth==4,Return[Map[
\t\t\t\t\t\tTranspose[N[A].Transpose[#]] &,o ]
\t\t\t\t\t ],
\t\t\tdepth>=5,Print[\"Not done: \",depth];Return[o]
\t]
];\t\t\t\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[34]:=",ExpressionUUID->"ab54fce8-9d39-41a5-9a6a-5d3c05207b4f"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Higher level (automating) routines", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"f57a4022-40b1-4c93-b702-9ea1fde1812c"],

Cell["\<\

element[i_,j_,geo_]:=If[And[i<n+1,j<n+1],
\t\t\t\t\t\tIf[i==j,V[i,geo],dV[i,j,geo]],
\t\t\t\t\t\tIf[i<j,-Id[3],If[i==j,zero[3],Id[3]]]
\t\t\t\t\t];\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[35]:=",ExpressionUUID->"f10376da-b9e9-43da-aa89-13ffaeb6d7f6"],

Cell["\<\
ClearAll[hydro];
hydro[geo_]:=Module[{},
\t(*\tSet up and solve tableau AC.xc = yc for xc for a given
\t\trigid set of particles at positions p with radii s and
\t\tfor a given viscosity nu (here nu->1). 'geo' is a
\t\tstructurecontaining a list of coordinates and an
\t\tassociated list of hydrodynamic radii *)
\tn=Length[geo[[1]]];
\tIf[\ttest,
\t\tPrint[\"Hydrodynamics of an assembly of \",n,\" spheres\"];
\t\tPrint[\" {x,y,z}   radius\"];
\t\tPrint[MatrixForm[Transpose[geo]]]
\t];
\tA=Table[element[i,j,geo]/.nu->1,{i,n+1},{j,n+1}];
\tAA=flatmat[A];
\tAC=complete[AA,geo];
\txx=Join[Table[Map[#<>ToString[i] &,{\"fx\",\"fy\",\"fz\"}],{i,n}],
\t\t{{\"ux\",\"uy\",\"uz\",\"omega z\"}}];
\txc=flatvec[xx];\t
\tyy=Join[Table[{0,0,0},{n}],{0,0,1,0}];
\tyc=flatvec[yy];
\tIf[test,Print[\"Matrix is size: \",Dimensions[AC]]];
\tG=Inverse[AC];
\tdata=Append[data,G.yc];
\tIf[test,Return[MatrixForm[Transpose[{xc,G.yc}]]]]
];\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[37]:=",ExpressionUUID->"c6654cea-41d8-49f3-825a-f8efe9b8efff"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Show relationships between equivalent sphere radius and distance in \
\"lollipop\" geometry \
\>", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"5776372b-14c3-461f-a38f-c780f9a48b47"],

Cell["\<\
Copy over the constants used for the equivalent sphere calculations.\
\>", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"0a639cd4-09f4-4bd8-a344-937d5e6dafb8"],

Cell["\<\
el0=3.4;\t\t\t\t(* bp length, A *)
persistence=150 el0;\t(* persistence length, A *)
repeat0=10.4;\t\t\t(* helical repeat of free chain bp/turn *)\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[39]:=",ExpressionUUID->"64025076-4ee2-46ad-8228-54be75977252"],

Cell["\<\
NAvo=6.02 10^23;
ro=1.003;\t\t\t\t(* density, g/cm^3 *)
visc=0.01016;\t\t\t(* viscosity, g/(cm s) or \"Poise\" *)\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[42]:=",ExpressionUUID->"dd215ac0-60e2-4a74-b719-536d4f528f0f"],

Cell["\<\
mbpa=660;\t\t\t\t(* basepair mol. weight, g/mole *)
rbpa=1.8294;\t\t\t(* basepair Stokes radius, A *)
vDNA=0.55;\t\t\t\t(* specific volume of DNA, cm^3/g *)
dDNA=27;\t\t\t\t(* Yamagawa-Fujii (YF) diam, A *)
switchbp=50;\t\t\t(* point at which we switch from YF
\t\t\t\t\t\t   to ellipsoid model, bp *)\
\>", "Input",
 PageBreakWithin->Automatic,
 GroupPageBreakWithin->Automatic,
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[45]:=",ExpressionUUID->"4b8d55ac-9139-4a86-a9a3-07ed35808cb0"],

Cell["\<\
mbpa=660;\t\t\t\t(* basepair mol. weight, g/mole *)
rbpa=1.8294;\t\t\t(* basepair Stokes radius, A *)
vDNA=0.55;\t\t\t\t(* specific volume of DNA, cm^3/g *)
dDNA=27;\t\t\t\t(* Yamagawa-Fujii (YF) diam, A *)
switchbp=50;\t\t\t(* point at which we switch from YF
\t\t\t\t\t\t   to ellipsoid model, bp *)\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[50]:=",ExpressionUUID->"f3b95965-3e81-4ded-ae0b-9ff7dfe87b30"],

Cell[CellGroupData[{

Cell["findoffset", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[55]:=",ExpressionUUID->"a884608e-6fee-4ca7-a509-3808c951be05"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Stokes offset \"\>", "\[InvisibleSpace]", 
   "3.0076424054539217`", "\[InvisibleSpace]", "\<\" A at \"\>", 
   "\[InvisibleSpace]", "50", "\[InvisibleSpace]", "\<\" bp\"\>"}],
  SequenceForm["Stokes offset ", 3.0076424054539217`, " A at ", 50, " bp"],
  Editable->False]], "Print",
 CellChangeTimes->{3.794912806688323*^9, 3.79526616694459*^9},
 CellLabel->
  "During evaluation of \
In[55]:=",ExpressionUUID->"1583f688-0a94-4e04-aca3-8e43ecfa8d6a"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Plot[{rootr2[i/2],rbpb[i]},{i,0.1,10000},
\tPlotRange->{-5000,5000},
\tAxesLabel->{\"DNA\\nlength, bp\",\"\[CapitalARing]\"}]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[56]:=",ExpressionUUID->"91e87679-451e-449a-a1f9-a6a6572a20e4"],

Cell[BoxData[
 GraphicsBox[{{{{}, {}, 
     TagBox[
      {RGBColor[0.368417, 0.506779, 0.709798], AbsoluteThickness[1.6], 
       Opacity[1.], LineBox[CompressedData["
1:eJwdjHk01AkAxzWODtYWZR2hItR6KlftIl/ndijykCS7JZEKoXXHYAhFssPK
pIYpjOM3uWbEuH5m8RwhSqHSiDYiucvW2v2+93mf9/nnu/WMr70HRURE5MQK
/3mhSxC+9KDKNFUiUSW7Q2BKU/JfPOa0FuT7CD1O5CaYiUXKVE7Jw7jGX/+S
6nbcDaapetVowaV78GrXsD46S9doNOTsA6XjiPOpfEuI/L+DmM+9Of+E7QB/
lfH3MSnO8FHpq5pucwePVi1pe90LN9s0TSy7/WGnXN9SPu8Fi+FdTdWKAXhX
3kRTPH0eSVW/c+fcAyA/0vHtrYE3pgVaBqVzAQg2fz0T/uoC+kZES53krmDf
N7FBtq4vxpZdTeg2QXA11nSKmfBHKM1VRyolDIMVfhqvboVAIygr5y8+FYxZ
RVpSUwgqrJ+lLz6nwlVPINw7H4K75SyPH+apGOQoMG+eCEX8nO60kk40Bgob
lcy3hOHSRKqNxZ1ovGBu3HC/OBys/erfqQbH4PkN3hfvlkg4JbLWsLbQMK4b
5fXlazQEI6aGR82uwa71CeXEpgQsUtjNPU03MDd05E0bkYLbNEPmuehUJGm5
u3MnbqE0X+5oWGkajJi6wWkW6fin6qHdTD8dW5biffsTM1HZmr2wqyUDFlGu
5ysyGRCdTG5Mu5KJOOsm8Tvv7yJ1TFvt044siGU72idTmRB/YrHVmM3Ar2uk
0igSuXjuI6sXuT4bh2SktocXsvA53YVtG3MXvWWHeQ36D0Du15K28byHydle
htSrPEy9edknkGUidzvzUxK1AO1/LE/wKplY+FB08NnqQixyXlLpB3Mw4Ej5
pEkUQfPjDYU+YQ7cHDOyLH8uASM3fDTCNxeXZ6+knxsl0HJ53fM4CgspDYep
bgIOZvkencJkFioclrm+CQ/hoRAdmih9H7ncYxmibqWQKdptyqDfx+lR66AI
9TJMSuUrd65/gInrQykbxsrwgfH0sUT8A5iKKhzTelQOLd03EplieThtNSuz
PbIC7eptjeeD8nD4eGAEy6ESVsziCxen80AZpm/Zr8FFlYpRnPBcPhIMjzfx
/+biteTgtas9+eg/yV9fz+chLlh6x06rAizYkKfyY6pQp3jkJ8+SAggrvIVs
q0c4mSV7bU6WjYzJyg275asRUjGk7BfFhuJMDrIGqyG5zmpKaYQNj2Y9+YSH
Ncio3XvvxtFCRHme/egazAc21ppTOYXwijRcddioFv1rrC+ZfF8Eec/iR9XS
deDG0nZmBRYhQ7EpUKa3DiLGD7Nnu4sw1Vk3PudQD/sElZJ2g2KE2PhUTfHr
0WLuqZGUXoxx4mL8oGoDfjGqceBMFcMsjF+wLrEBzm7JSntsSxCp7db+44cG
TJRo9x/JK0Fupq2ky9FGKAmWpquXS1Dok5bdym3EiKhS7FlnAlk7NHW1FUmk
z4nXLRcR0OmwoAqVSNgpOIYElhBo9Put67YyiZhNJtsmCALvuJk+a7eSUDQf
Nh0oJaBvLVU4qknCcZHP5vEIdJ6ZVmMakLjkKYRfEwEKo3qT3DESaKZH9gwQ
oJv2n+2wJ1FvxNpwYIiAlnC2LNZh5b874xb/JQHbnTr2M8dJnBHsPZ0/TIDB
u5fc5UaC7sfYEzFGwLCPtjrpIglLdTGm2gyB1uAcJ3MfEsMuEdv+nCXgurnu
/pIvCfXVq9Kk5gnEuC9aeAWQ2Cq6x2xukUDXtHeUVSgJXZPAHsFXAu70+MfL
YSQCJzC6WYSD+X0slfIIEr27n754utLKUUM126gkJCo57EMUDjhqnyVfRJO4
Z75IExPlwLJZ7mRqLAmHoR7n2pV+5q3HPhBH4lShmVqwGAfe0nZL3+JJ8Lqt
3u4R5+Bf9ci3/A==
        "]]},
      Annotation[#, "Charting`Private`Tag$13298#1"]& ], 
     TagBox[
      {RGBColor[0.880722, 0.611041, 0.142051], AbsoluteThickness[1.6], 
       Opacity[1.], 
       LineBox[{{0.10020407959183673`, 11.670919520742634`}, {
        3.167148533804211, 13.674785292421284`}, {6.234092988016585, 
        15.525671863142279`}, {12.367981896441334`, 18.933537433733463`}, {
        24.63575971329083, 25.054451013040616`}, {27.702704167503207`, 
        26.489465970348643`}, {30.76964862171558, 27.895873425375434`}, {
        36.903537530140326`, 30.634804912098346`}, {46.81125637755102, 
        34.88883506298495}}], LineBox[CompressedData["
1:eJwVVXk01QkDlVAfxlLTRNnKOjIlMTVG3BHGvkYRCjVZs3yKLHlCRY/Hs3vI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        "]], 
       LineBox[CompressedData["
1:eJwVknk41AkcxlE6UdpVE0JZhbBWVqmNb9ouCevIEeUo5diJYjUjV2EwNgbT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        "]], 
       LineBox[{{6332.886997086843, 5000.}, {6332.898431184855, -5000.}}]},
      Annotation[#, 
       "Charting`Private`Tag$13298#2"]& ], {}}, {{}, {}, {}, {}, {}}, {{}, \
{}, {}, {}, {}}}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{
    FormBox["\"DNA\\nlength, bp\"", TraditionalForm], 
    FormBox["\"\[CapitalARing]\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}, {Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{
   "DefaultBoundaryStyle" -> Automatic, 
    "DefaultGraphicsInteraction" -> {
     "Version" -> 1.2, "TrackMousePosition" -> {True, False}, 
      "Effects" -> {
       "Highlight" -> {"ratio" -> 2}, "HighlightPoint" -> {"ratio" -> 2}, 
        "Droplines" -> {
         "freeformCursorMode" -> True, 
          "placement" -> {"x" -> "All", "y" -> "None"}}}}, "DefaultMeshStyle" -> 
    AbsolutePointSize[6], "ScalingFunctions" -> None, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0., 9999.999795920408}, {-5000, 5000}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {0, 0}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.7949128069837923`*^9, 3.795266167187375*^9},
 CellLabel->"Out[56]=",ExpressionUUID->"a9ca3c59-2e36-4983-a2b2-548194a4bc57"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Plot[{rootr2[i/2],rbpb[i]},{i,0.1,2000},
\tAxesLabel->{\"DNA\\nlength, bp\",\"radius, \[CapitalARing]\"}]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[57]:=",ExpressionUUID->"03b7336d-5ae4-43f2-aed0-e9187754460d"],

Cell[BoxData[
 GraphicsBox[{{{{}, {}, 
     TagBox[
      {RGBColor[0.368417, 0.506779, 0.709798], AbsoluteThickness[1.6], 
       Opacity[1.], LineBox[CompressedData["
1:eJwdkHk0FHgAxzXJbFG0Zq1Eg0q7KF6eVZL5MpKGYq2ULsdmc61xrS0j55jc
Yy7HDzMpx6RHu2wHke2iMmLLlHazbmmzvVxhU7bd73uf9/nr88/XKIDtGUhR
UVHZ95H/nOAp83EuaWBw79Xr8+7cZXj0DGjbdI8yRngLx0sDpxhfDO2ujbKd
YTyTBw70UykQj8Vx8r5XRTQvwULpvRqsUFoAnUrDQvmkwey8ISrM/ZWjj9fD
fsDXZvCWJb7JofJZjVYYzxh/b50PVDf/Hhpfw4RHeV+UwzEPqPy/PVA+nN75
NfECW/hLaFCcF5wNHWtMK/2g3l6RIg4+hrH50zGbbILA3BZ2N1z7ON7vN7Ty
9GZj74Wdj5oOB8PcYOTmLr9oPNMv72zfE47lOH5Srz4WPgcGTG91RWJZtUSQ
9WUcTr9D9caIGGwZjvTRCT+NQcNXZcGDsVBL3tGsa5GEsP0TR1MyTmFmVrEi
5loybNQdhZyl8ZjzX/92Ri8VO/SHKAsRCfA+FjjmFsqF1bo3xfKMJKSHTisj
FWkYey2KYt5JRgiTCLPVz8CyLfCiy5pU3Htq/aHtUDpoHb4pq4O4sPfUCb5e
lIFC7UQJ62walm600ZM3ZuKSv2LF60ke3HdXHXR4lYUTb+yblliko/FljuD+
YjbqeH2diWkZ4HqoL4tcn4siljz5pjITDQOXlZts+TB0z2dHa2ZjX2BpVO/a
PNgermvhf5sDaccVOuVcHmaqplz1y3JxwG40wk1fAFfjo7FhPXzceGJXwRcI
sNYzLllb8rHTykx4t1IIv/qeEflWATI4DTtIohBs3X2c6F4BmJoBzmZzQriJ
n/bR04TI660WrQoT4dRP0QFcIxEGhLp09ScisLtMdNa1i0DL+jmi3VmMy3nj
L9jBYkw/czmvUyvG/ROt6edVJXhhspBbuFqCKserhzRqJAjpuOdax5FA2dzI
2szKx3xUeN2SXgk67JShqdP5kNTMDzY75eOzI5SLAlEBRKJaP2p5PsyEzIZk
i0JIWw9cp6oUYIWieK6ypxBxHZ7b9gYUYHulmjc3tghmBXOyiqYCbOWmD+cb
EfzAi6wxMCjE/Ml1jln3CXgtvgXhcYXQWtXWbPNdMdTmt8W6dBViopIXskar
BGTjHzEG5kUw7j9oXV9bgtAYmu+VpCIYWTI6t7BK4RJ0Jkf6qAgvU+1I32Qp
qKrGrb6mBJSTbqkueVIsdfrExIFDYBxH86eKpIg3MdggjydwiH9u3yqRIqws
SVMzgSApKewfZrEUumn2tc+TCBbTsyMZlVKM/jmk/iOPYKFI4ftVkxQjZx1Y
cgHB2+tudhvGpLANyh1dqCLQuUHTG/pLipblXSL/CwTWvz6fLftbil1W17a0
VhPE3Amrp09JwZvkbebXEEwpsk3XfpDiqiLY3bCe4E2vQvdTmgzOTmdNGDcI
tPrFb7t0ZOB+yLl0roXAYvBIN3+NDImsvSZqNwnCR8f5K+kydDq59ipuE4y/
1qAuN5Vh5gg/0ufjrxoT3cNt5jJwmF7BTQ8IzKZKbvEsZPBSU3GnKwhCZs0T
VK1l6G/UGR5+SJA5P334to0MlDmeaHcXQfW7pu0ptjIcko9ZVv9G8OA993OH
nTLQI+ybNB4TvFx0m1lkyDAxmWnN7ib4F0yOTDY=
        "]]},
      Annotation[#, "Charting`Private`Tag$13878#1"]& ], 
     TagBox[
      {RGBColor[0.880722, 0.611041, 0.142051], AbsoluteThickness[1.6], 
       Opacity[1.], 
       LineBox[{{0.10004081428571429`, 11.670807451737794`}, {
        0.7134051693271974, 12.087235061040372`}, {1.3267695243686806`, 
        12.495076022817337`}, {2.553498234451647, 13.288121909271258`}, {
        5.006955654617579, 14.799887175986372`}, {9.913870494949444, 
        17.608007715154578`}, {19.727700175613176`, 22.688477714769306`}, {
        20.341064530654663`, 22.98937610931539}, {20.954428885696146`, 
        23.288694915746387`}, {22.181157595779112`, 23.88277925316846}, {
        24.634615015945045`, 25.053909619074727`}, {29.54152985627691, 
        27.33591590061881}, {39.35535953694064, 31.705421516084424`}, {
        40.02031678378858, 31.99362299447825}, {40.68527403063652, 
        32.28093654344244}, {42.015188524332395`, 32.85296641452072}, {
        44.67501751172415, 33.987074006993254`}, {49.362276785714286`, 
        35.9557192302256}}], LineBox[CompressedData["
1:eJwVk3041AcAx3V5yMtKXkJejp7q2aiRZl6SvrEad9LYZplKTsVx6+5wXu73
i3NedkrJ+8vpfmx5SamkPakQKcYd0WKtZb1QyNEJRbHW/vg8nz8/f32sWVz/
QzQ1NbWdH/nfL8uWvyzz88Y/X2wSsdw8wM25Ehku/A70Up/xmQ1e0JFXiPPY
+7H5b37ri4274enMuXPE4CCyjJv7Q7r9sevs1nuNQWxMFPWetVkdgL/Mz9yV
ex/BmMLGylsnEIE/PLW51cuHi6CsrNlsL46+R806XgwUFboXM2jBeGY1Xs5+
FgteokBbq+8AON9P7RNnJGBOfKzpjgMLTjoeOcRSEkqvgWtOMaHYYj5EW+Al
ok7VZtxXdRCbLVXS6gwR/DctJHVPHMLoZG6U5+1k1DL5Ls7GYbDvOHTOyzQF
TSbyglb/cBh2B4tXhqfiipqoQlvMRpFBUj6jLA3XO+vOCwojcDFEoT35Oh3y
Nidpe3MkwlTujUvsJFDb0f6psJuDy+mP7yalZaCBo+/VMPETihnVya39x3DV
VSv04VIurHYXcKNXZGJ6gJEpXcKDa9Dlm1mhJ0AF1luMZ/AwWzXNNC8/iUFm
sbWPJh/MNftiOQ+y8PtT/SGJiA8zf2GyQf4pSGqfdu2kReFA/YPn1Q7ZUG28
OdqWEAWuiS8RPZiNn4c0xL/MR8En78/H9LQc9O1f3ycVRCPhUjQr1ToX72uj
v5kYiQa3d/0qS3kuvNtZdZmsGPx2SjnCZeeBNnum594fMegMa5f8qp6PImcj
Zt92Aao8rv6oW5uPQbn+Sf0GAfqbrjM2MgogCK0J56+LRbdbf2TKTAFWmrge
XiaNhdFe2rns3EI8tzfComYcbHM8ryXbFeFrnbwKw6g4aCukc5UPikDn3uCz
h+PgUqkRkBpbDGLIRUr4xsMhVTJcYF2CphH3qZ6WeMzHW3oc7yyBelfgcKJ9
AvSWdzQ5HZZijtNjjqIETFWmR5jqlcLi4dR84DIh1jzZ41h/oRQR40q/yzwh
rO233f2ccRpncrOmWweEGEtxK3n8+jRONOiymTsI0OJ9UrxOyZAUUDHrd57A
GqFhiGauDPHHt9yzuEBgO/nIvT1fht3+LbVjFwmIRJx3nlIZFM01Icn1BD5I
MvnbKmVIcy5sunSdwEKxIvjLRhlYa3mMFZ0E3tzwcVs7KgMz0GFQ/pzAqmbD
1UMvZXATPCovHCHg2PLobfnEx16x5eHQMQIxtzn19GkZYC+aeKckMK3ItDH7
V4bRlbvmPpshoBpUmOgbUhgYqKNJaCT0nuS96V1FwfbVnrZv1UnYPdt7P8uU
QvCUMoWuQeLIC2XWJ3QKla9U6le1Pn4zqaupZUNhZ9Kixgs9ErpT94c7NlCQ
Zpd11OmTsJ0uvZVuR2GtVpDkqCGJiLcbEtUdKdwMs9U2MiFxbH4mqM2JwtJX
TPkTUxI17xtdxK4UVGbFmefNSHQtphpv30pB6W7iG29BYuyDz+yHbRSI6p4V
X9FJ/AfRfRKE
        "]]},
      Annotation[#, 
       "Charting`Private`Tag$13878#2"]& ], {}}, {{}, {}, {}, {}, {}}, {{}, \
{}, {}, {}, {}}}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{
    FormBox["\"DNA\\nlength, bp\"", TraditionalForm], 
    FormBox["\"radius, \[CapitalARing]\"", TraditionalForm]},
  AxesOrigin->{0.1, 0},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}, {Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{
   "DefaultBoundaryStyle" -> Automatic, 
    "DefaultGraphicsInteraction" -> {
     "Version" -> 1.2, "TrackMousePosition" -> {True, False}, 
      "Effects" -> {
       "Highlight" -> {"ratio" -> 2}, "HighlightPoint" -> {"ratio" -> 2}, 
        "Droplines" -> {
         "freeformCursorMode" -> True, 
          "placement" -> {"x" -> "All", "y" -> "None"}}}}, "DefaultMeshStyle" -> 
    AbsolutePointSize[6], "ScalingFunctions" -> None, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0.1, 2000}, {0., 1717.1086026869564`}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {
     Scaled[0.05], 
     Scaled[0.05]}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.7949128071486893`*^9, 3.7952661673311787`*^9},
 CellLabel->"Out[57]=",ExpressionUUID->"711e6959-ac4a-4b66-a04f-62c53d0c4a08"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Plot[{rootr2[i/2],rbpb[i]},{i,0.1,200},
\tPlotRange->{0,150},
\tAxesLabel->{\"DNA\\nlength, bp\",\"\[CapitalARing]\"}]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[58]:=",ExpressionUUID->"f2d186d1-3d7d-427c-bd10-7d21f9b4ebc3"],

Cell[BoxData[
 GraphicsBox[{{{{}, {}, 
     TagBox[
      {RGBColor[0.368417, 0.506779, 0.709798], AbsoluteThickness[1.6], 
       Opacity[1.], LineBox[CompressedData["
1:eJwBAQL+/SFib1JlAgAAAB8AAAACAAAA1kJQC96ZuT9wQS9PesLFP6oNa8nm
pcQ/+0sOGZ2M0T/q+S2N3n7MP0Y3GEXfN9g/NOlZCmcY1j8hGAIoBcfiP9ng
70gr5eI/drXE0T4O8D+s3DpojUvxPxUSKVpSYv0/lVrgd75+AECoIFx9cwIM
QIoZs/9WGBBAKsZKVvRMG0BEF74KH44gQOgAQVXTAyxADk+8Ur5/KECxCCbc
8q40QJAXlq28JDBAW48DIzQzO0C41l4VGF40QDbRFKJkHkFAhmy0W0lPOEBY
vhKsTmNEQHTRmoX4lDxA6jfpYDfqR0BDBd3HWmNAQM1HG6iQXEtAHw0zPCRY
QkDdPech+49OQIp8UaIsd0RAT0L2QUsCUUBIV7b3H3JGQP/TN5ZfnVJAGJw3
PRpjSEAFTCRTqC5UQAAAAAAAAElARp1QkcqsVEB4SIF0U35KQCuqhA0n4FVA
K2ARm3d1TEA2ZXQ63nJXQG7fabPalk5ASPAlcXolWUBhZO9dIldQQJlcNWk2
zlpAtQ7N2cxQUUAY+wvsnFhcQNHsjs6WX1JAeIepLYcCXkCWAPQ6VlxTQNtZ
sIhWjl9AI0g9IDVuVEBLDsqIrZxgQLnElH0Xe1VALwRz5FttYUD4do9S73VW
QB92ITg0L2JA68IDpx0yV0AAAAAAAMBiQFDTyc0=
        "]]},
      Annotation[#, "Charting`Private`Tag$14023#1"]& ], 
     TagBox[
      {RGBColor[0.880722, 0.611041, 0.142051], AbsoluteThickness[1.6], 
       Opacity[1.], LineBox[CompressedData["
1:eJwBUQGu/iFib1JlAgAAABQAAAACAAAA1kJQC96ZuT/su+3EcFcnQKoNa8nm
pcQ/PJNKk/ZsJ0Dq+S2N3n7MP1FPKgdwgidANOlZCmcY1j8uUVloPq0nQNng
70gr5eI/FOZGrEwCKECs3DpojUvxPyMCjDtKqihAlVrgd75+AEASjLLHh/Ip
QIoZs/9WGBBAV0QsQMVoLEBEF74KH44gQHLH2vuesjBADk+8Ur5/KED5OukC
294yQJAXlq28JDBAUimCcFTlNEC41l4VGF40QLOaEhGZADdAhmy0W0lPOEBu
X+JsauY4QHTRmoX4lDxAu7fYaDrlOkBDBd3HWmNAQLNqHrVTzTxAHw0zPCRY
QkCt75FjU4o+QIp8UaIsd0RAYsO5P4YxQEBIV7b3H3JGQBi/DQ4YCkFAGJw3
PRpjSEDsPjTi+NpBQGQgvT/X90hA07g/COAYQkBu8onf
        "]], 
       LineBox[CompressedData["
1:eJwVjHtQE3QAxwcFhxleSoGGpwmYeJiEiAQX+Y0JMQVkjx9swoJtPH68Rqig
KCYeSVEEzJOhRww5whWvcYFXA8LJ6IECQ6FcIIx5PCxIHgKDQ1z0x+c+9/nn
s0ucxomzZDAYvuv8728M/hpXGxaKHKgma6c/3GI2l53JCIaWK9kvzWRCakhs
GpSHouj1qixr1hGIYmeEThFhuPtn03TnGwG4tannr+4wNuZiDw/NPAjA/enC
Xnk0ByO6UrmnIhByxoJveBQX5m7fyX0nPoJzkN9bXlIeIsS/RVm6BEHdefWC
w3mCDPlQTOSTICxnFnFmMsKRe3u1wlTOgjlKemPicgQ89mSW+4iOovHsbFx/
Lh+DHql3NtgfQ1JXTpU2WwDnuxPuz345humQRUH/lyfQGhwyx80NhoON7Rf6
vEg0vH99vOVACF5ZsnFfK43Cr8lXhJ/MhMB6m13o7m+FMNqO9YYrQ1Gbrhnz
V3wMZqsrbWQfx9YX7IWL9dGYOvycW/pyGBxitH4CEgOlkzaxVhAG5VCs3+n7
MbBgi54W1oXBMf3syg6+CF3mTtc5KzYklcL0zj9EuOG213OAz0bRVHP1ilAM
r9SKVpcf2CjuGy7ZOyzGoed04lUrDuz7bP2HBBLk242ajISDnL9VNduNEpyr
2kS21HPQXu6s40fHQqOLjzevcTCavKHR5UksDpXteOFNuPictxR8UhoH9Soz
T1TDhaW+ubplKg66t392TFrhQpQx3nLlVDw+DNw/VxfKQ0n95Fdx8/GYt7bx
Gi3nIZQxt7VFmoB9Sy/575ni4Z+UFLFiMQFr4ndY1z4giDytTrhwikK73FFr
X0DQQJbTVRkUPTfLJtK+JrD09s42nqHYdjvvza5Cgu9WmosDsil2tslTz8sI
Fj9tVNtepmjXqDQGOUFRvnKjQk5hFPAclZUEWkWJql1NYTkxcPPdHwnscwbU
s60U/EAvq/yfCJJEdp1O7RTlAW1RRjXBay4yfV4Hxe8VMMnaCITfF1gcv0cx
cPDg4PwdAlNTLtfwiGLEjsNv6iY4WtIh3GygkFknsDb2EigyLSjTSNGXV+cp
0REc8bmYrRynkCvvPdvygEDWfq467SkFsybZ6eRDgrEKtapylmKQZT/cpSd4
79Kyun+eokCyVLxrkGCEmdnrbaJIyRKM9T0i8Nh9S5+4QpGqGs51HSH4zHrh
cdkqhWv/9e05BgL95IF/e9bW/5cKGx6OErh1pZvM5vUOb/Nxf0zwHzQGvFk=

        "]]},
      Annotation[#, 
       "Charting`Private`Tag$14023#2"]& ], {}}, {{}, {}, {}, {}, {}}, {{}, \
{}, {}, {}, {}}}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{
    FormBox["\"DNA\\nlength, bp\"", TraditionalForm], 
    FormBox["\"\[CapitalARing]\"", TraditionalForm]},
  AxesOrigin->{0, 0},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}, {Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{
   "DefaultBoundaryStyle" -> Automatic, 
    "DefaultGraphicsInteraction" -> {
     "Version" -> 1.2, "TrackMousePosition" -> {True, False}, 
      "Effects" -> {
       "Highlight" -> {"ratio" -> 2}, "HighlightPoint" -> {"ratio" -> 2}, 
        "Droplines" -> {
         "freeformCursorMode" -> True, 
          "placement" -> {"x" -> "All", "y" -> "None"}}}}, "DefaultMeshStyle" -> 
    AbsolutePointSize[6], "ScalingFunctions" -> None, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0., 199.99999592040817`}, {0, 150}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {0, 0}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.794912807343568*^9, 3.795266167466639*^9},
 CellLabel->"Out[58]=",ExpressionUUID->"cab04208-cde8-4775-af9c-70fb3a35633d"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Plot[{rootr2[i/2],rbpb[i]},{i,0.1,20},
\tAxesLabel->{\"DNA\\nlength, bp\",\"\[CapitalARing]\"}]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[59]:=",ExpressionUUID->"a8a9d7e7-17f1-4a4a-9581-8cb70d29cd58"],

Cell[BoxData[
 GraphicsBox[{{{}, {}, 
    TagBox[
     {RGBColor[0.368417, 0.506779, 0.709798], AbsoluteThickness[1.6], Opacity[
      1.], LineBox[CompressedData["
1:eJwVjHk41HkAh6dG2ImHSNY5GnJNwnytI8fvo0TtTiubYndRO8LYNrUok5Zk
SEWIomTLFdaDPDXlKlfOVdM4KlfnQ5RoalSotO0f7/M+7z/vSt7en4IW02i0
zV/534KZp6F552qpBM2RqQ0tbZRtEV2Qa1ZPjbBicu1XdFAzW1cdPVt7gxot
Y4nMf++k9l7lF6QPNlIvA1/pKat1U6Q2ak+T0i1K7cn6sDErCWUuWLhUxe2i
NPRP08/E3aOUPiYmb3zeS/V7PWt9qPKEmhdkfjINfkyxq3sVMitfUkwPxowu
pqiOYxd+KxucpcTOSRb5shnK1OuDb1v3IjD5EyNhJh+p6K7zC8YqCghK8fKY
kNLA7vOwWd6qBOu7h/3ylsuBm6iYG750GczvDYUXHFLAIFt14tSUOiwCVym0
9DJQm1ATPNq8AsXDTQ/4UmWEnF/m5BikhZLk1v7yUlU477DbYOuqi+v6PcN8
JzWwnl1Rol7po2v/vXzSqY6scwaT7a0GiFizj1+4XwPuKmP+1ZYs+PqRO9vl
NXH15my+Xbwh+idTOJdrvgVVUpkwJzWCN/Vo+4pAbVTdopUO+xpDVdFXq1mq
g6Ako+btpSbgcfgjqnl6cJt+B8ZyM0h78pyLOEx4i4Q7fw0xx4kX1Z5hEQYY
exiy/+F9NvS5Ok9ObVoJuarEP4WUBba8yKB8lrEgZ5pkI0lbg83V25ivX7AQ
6H/xitsnS0w0rhabigyxTv+QF6fUCollh0dCo40Q4Vc6MfCjNaba9HbHeK3C
lrTcx6l0DmQ6kk6OtjFGsmtozHIOWgvuFNfJjKF8t1xSxyPIXuoxuqnLBLuO
yp5OathA8XK/NDrdFLw/dpR8vGEDN2FFTCzfDO33lY+phX8H/YycQUOOOXJa
CvcwdGwxH08/I/lgjsWN2ra5TbYoKKqWeIrZcO7Yfckq0g7jtLjhfVmrsfYg
90iZjj1mT2YG79prgTz6rV8Me+xRvK2+ac5hDfwnvn8rjHaAz5eA6CyaJYyG
dG3iDNaib3CzaMmAJZiJPZ8P9K9FVFRpQPIWK3gXiaZ//ssRoTmehjE3v7a6
9U3CcYJ/pSg73NQa7+ntQyWjTrj4uFBmn2KNRQxevlKyMzz+qUnjz1pD5Zp7
XSDlApd0H/VQfw5+ONnTWTHuAlu+yH1PEwfjshuNWSkUTg/Y1g2xCeKL3cTX
mYBr8HoNrXQCy4O7xvUigKOafVUDpwh4iyMdMyKB25087tlMguTA26nyBwBf
drxQM5vA87IPeS0AwqRNbzQuEKTGDkQ0xQI50S5itXKCv/cJe3knAFmqQ5JS
J4G+f7x8ST5gjy7W7a6v/6mK9TqFQOwb34bkboKAYFZcWhHA8Ba8Y9wlaNF4
/j6qBDDQqg785j4BJRgY3lgBcAttID9KkNYSKnxZA2R4tw63jRHMs4yuBdQB
D5Z4RyWOEwRHmjzvrQd4/IhKuUmCzGlF9/oGQGBxRZf+lqCBa/c5uRVoeORa
0yIjEHs4mNHaAXp6z9b4dwTT9gnekR1A6lvpiUVzBHYLM6V+/wL9hXHGzfME
D/q0eyXdgPY21Za4TwTH84/Pu90Bdsrn+WOBwDHElVUrBi5VW859+UIQZrxu
k4UE+A+6ISj3
       "]]},
     Annotation[#, "Charting`Private`Tag$14182#1"]& ], 
    TagBox[
     {RGBColor[0.880722, 0.611041, 0.142051], AbsoluteThickness[1.6], Opacity[
      1.], LineBox[CompressedData["
1:eJwVx31QzHkAx/GuTUimCEnPyK5q2/b33b6j69hPw9icUpEejsJ23IaepOlJ
D6IyFdqiqCQ9Ia4MzfVAiR5s0qNS1FRGVKfYHowe0Lk/3vOel6HYd/dhRQUF
Bbuf/f/gqbde2VfLhWGFjn5+LmzQPFZw5saHwrilapGX3diY2mMUe6X8kdBb
bORTsY8N3weSnKTXj4Vdp6LGWAfZIOVB3tWqNUL/mmH/yxI2jIN/5N+zbRDG
W9Gr90PYUJ2LSbD50C5M0E7K6sxgYzY45RvnSL/wzoSXe3U/G/oilSkdjAnX
S3HK35+D5s1x3BuTU8Kzm4O4BV850JcM9/qw54Q7lZ93Z4ZsxOFER9GwXAGj
7e1p75WMwW+J3J+9QgnRTREl4hhjGHe+OZETthDSgb6bq9RMwPU0Wvi0XQVB
t60ymy6ZoKCnuksiX4or+W6vROqmuJlQ23H3ljpGZBOzA5dM8Y9eW4/kt+XY
ckj672k1LhoCO28QmQamB8SfimO5CDDzk+QGrkSJYv3Jd8pmcN1PmpyVNfE8
y6foTLgZOj4mMsVlq3Goc/642nczOAn7nFd5rkFfxNKarSE8qC9y1Xoi14Zc
J/SD1SgPYkbSq56ti7CSe98VBeaQt2VvzmP0MVwX6HPA2xzxI6X2PgEGmJj5
+n31HXPo2WoPSHcYopD3e4hoyBwOI8lCl2VrcXGr5QM1fT7sSvfqfx5ZC4cV
LtOcfXwMPzZt5pSsg9RmRncsmY+Ywsher9D16LUtTLVp4GOsTvdYuKMRnl+0
LB5hMZjUbpUxazZAI4fT6vArg9qcpoKKyQ2IW/7Zd/Ikg7QlosEdDWzsfhZf
zy9isKi4Qx6axIHO0fx3u94x2Hbm7/AIyUb4L975JFOHQC85/fU6xhgDNxv7
Oh0JZqNZl1u/GiNLDeU1MQQ5eaWt9s0mqOPu3NZdSTCkENXjl2qKwW7R4dFx
gunzKUf+9OXC7eXpeD1jAQr2PqyesTTDMTHv4x8eArjMe4SmKvBga5Cwz00q
wMvXdiULunnwMao6d1wmQFDQLY8EB3PUzkrPZswJ4JVuvy680hyKygEOXYwF
3ItK0k5w+DBq23XrlcQC1/tzJzcl8rHMuddGMcMCottlFyXTfCgFWczz2yyw
JclFw8udwXTI3IFUFgWVlGz3rmZQGm6n1SKguNRNK96YEIwbCCtZRymsj2xd
qZVEsOR8rmFYBkWs5st73VKCOebDNZdMihcyse2VFALNg1WrBNcoXE2iz2im
EZSdM/8xmkXhI68eX5lFwLq+Ps8jhyI9dEvz8rsEFRpJtta3KSYvWMapyggq
RSnLlEopNqFh7YsGgm9huoYDPx0x7lqV0EgQMbSd+6iMQsUp+ItKC4EgLdjq
ZAWFgVap5+JXBBoPD/AGKylscwVQHiTonosMrKmlSHaq7al7T/Ap2tD+eh1F
1wKnoJghggsL/zIKq6cQSwKKlD4SHLXsf8bIKIK593VYEwR1iYmDNxopqvqs
y55OEsjsnHPDX1Cwktr2RH8h0DpW7+HWRHFhQh7/ywwBfyS9Ub2FoiM3asOT
WQKdt/OnRn96zV71p1HfCPI8VU1krRQHlbPd8YPA+lF1Z24bRX4pb2Z+nsCN
6oZHtlP8B38TKxA=
       "]]},
     Annotation[#, "Charting`Private`Tag$14182#2"]& ]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{
    FormBox["\"DNA\\nlength, bp\"", TraditionalForm], 
    FormBox["\"\[CapitalARing]\"", TraditionalForm]},
  AxesOrigin->{0.1, 0},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}, {Automatic, 
     Charting`ScaledFrameTicks[{Identity, Identity}]}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{
   "DefaultBoundaryStyle" -> Automatic, 
    "DefaultGraphicsInteraction" -> {
     "Version" -> 1.2, "TrackMousePosition" -> {True, False}, 
      "Effects" -> {
       "Highlight" -> {"ratio" -> 2}, "HighlightPoint" -> {"ratio" -> 2}, 
        "Droplines" -> {
         "freeformCursorMode" -> True, 
          "placement" -> {"x" -> "All", "y" -> "None"}}}}, "DefaultMeshStyle" -> 
    AbsolutePointSize[6], "ScalingFunctions" -> None, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0.1, 20}, {0., 33.62638202409731}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {
     Scaled[0.05], 
     Scaled[0.05]}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.79491280748254*^9, 3.7952661676748877`*^9},
 CellLabel->"Out[59]=",ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],ExpressionUUID->"afbc9d36-8b7a-4b0d-b194-ec34d3202835"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Simulate lollipops to see applicability of Kirkwood/BDvH approximation\
\>", "Subsubsection",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"0c797c40-fed6-49d4-8b87-2fa15993c93c"],

Cell["\<\
I make a two-particle system here that is again like one in my paper. The \
separation is taken as the center of the array of linker spheres (smaller \
spheres) from the previous example. First orient perpendicular to \
sedimentation direction, then nearly parallel (I used a 5 degree angle with \
the z axis. When the linear assembly is exactly parallel, the set of \
equations becomes singular).\
\>", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"fd0e09d5-11a9-4fa2-9d7b-ee09ef5afa35"],

Cell["\<\
test=True;
data={};
nall={};
radiiall={};
uall={};
fzall={};
fBzall={};
AFzall={};
AFBzall={};\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[60]:=",ExpressionUUID->"e8baed60-7586-4ccd-8736-73b12f69f907"],

Cell[CellGroupData[{

Cell["\<\
Do[\tnbp=N[Exp[i/2.2]];
\tpoints={{0,0,0},{57+rootr2[nbp/2],0,0}};
\tradii={57,rbpb[nbp]};

\thydro[{points,radii}];
\tftrue=G.yc;

\tftruez=ftrue[[Range[3,3 n,3]]];
\tutruez=ftrue[[3 n+3]];
\tAAz=AA[[Range[3,3 n,3],Range[3,3 n,3]]];
\tAFz=N[-6 Pi radii AAz.DiagonalMatrix[ftruez]];
\tDo[AFz[[i,i]]=N[6 Pi radii[[i]] utruez],{i,Length[AFz]}];

\tfbloomz=N[Map[#/Apply[Plus,radii] &,radii]];
\tAAz=AA[[Range[3,3 n,3],Range[3,3 n,3]]];
\tAFBz=N[-6 Pi radii AAz.DiagonalMatrix[fbloomz]];
\tDo[AFBz[[i,i]]=N[6 Pi radii[[i]] utruez],{i,Length[AFBz]}];

\tAppendTo[nall,nbp];
\tAppendTo[radiiall,radii];
\tAppendTo[uall,utruez];
\tAppendTo[fzall,N[Apply[Plus,Apply[Plus,Transpose[AFz]]]]];
\tAppendTo[fBzall,N[Apply[Plus,Apply[Plus,Transpose[AFBz]]]]];
\tAppendTo[AFzall,AFz];
\tAppendTo[AFBzall,AFBz],
{i,-2,9}];\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[69]:=",ExpressionUUID->"35adb82c-2295-41dd-b49f-642f7b069a60"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661679873743`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"637e8813-4652-47d5-84eb-8db61c921f2e"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661679975653`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"b40d2aab-23f3-4092-ac2c-3f2c832cbec1"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"57.68476027811842`", ",", "0", ",", "0"}], "}"}], 
      "11.877548930570843`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168002968*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"e1f99adf-9217-46ee-9dfe-d4c383b56601"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168007433*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"0d9b042e-6d69-436f-bf18-35e7252556d9"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680121803`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"bf60ef90-f5c6-4e51-9143-73f9fbc7bc7f"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168015991*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"51d3b554-02de-4c89-bab9-de26d8e9cc75"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"58.078671538817154`", ",", "0", ",", "0"}], "}"}], 
      "12.034323655022332`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168020381*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"fbc13f37-ecf5-4d49-b922-7c36fab3bba6"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.79526616802606*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"894b3220-d5b9-45d1-a3e7-ba47cc0c416e"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168030006*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"d388d2e5-8b86-4b22-bf6c-78ca6affeaac"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680343533`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"9ebf2eb9-8a58-4c44-a992-c40ec28537b5"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"58.6990560800066`", ",", "0", ",", "0"}], "}"}], 
      "12.278815084740968`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680385847`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"00588568-efb9-4e22-88e7-d546d863ed07"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680450153`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"82718771-9ef5-4ac6-87ca-7834ec96deb8"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680510817`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"ef47d120-23a3-416f-91a3-da2c1cf66a73"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168056654*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"8359e9e3-4417-48b7-a1e3-3248d794b38a"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"59.675934953492884`", ",", "0", ",", "0"}], "}"}], 
      "12.658177878173777`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680619717`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"b0af490a-7b74-4904-b4da-783c65dea874"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680682087`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"93abb62f-b6e3-4b09-a9c8-5e228b20c5a2"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680732183`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"9e25ecf0-eb46-42e4-9c8e-af0055118776"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680788307`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"7b6dd9ef-90c9-4b72-a950-5eaa3a7d8b05"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"61.21370026922342`", ",", "0", ",", "0"}], "}"}], 
      "13.242699705429152`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661680844593`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"74b05a97-79da-47ab-a968-644526f98ed8"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168088694*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"4275d928-8a58-4016-9f08-2694cbb4e245"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168093148*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"51b5217f-094e-4f48-992f-1a0bd9a1b440"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.79526616809872*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"b35dd61c-5355-459e-93b8-645cbeb9f2bf"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"63.6332477157901`", ",", "0", ",", "0"}], "}"}], 
      "14.135125827602376`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681034803`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"e7f186e7-e195-4050-b3be-19b14a22d12c"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681083803`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"09d3c716-4b2d-47c8-9bba-6cf875e94713"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168112564*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"117e77ad-ba77-4a16-adbc-ee0a5426e690"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168116391*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"4d8ade43-d107-4fe3-961b-dd280568996c"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"67.43737735001169`", ",", "0", ",", "0"}], "}"}], 
      "15.482728885765113`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681214123`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"e0480394-07e7-4b4d-833a-14e70b2fd225"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681260633`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"a18075a6-20eb-4338-a054-b4643256674c"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168130718*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"d934976c-60d3-4ec0-bab1-7c0839c1f767"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681342*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"c1daae7a-1272-46cd-bbfd-6082e95b11c8"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"73.41142780769728`", ",", "0", ",", "0"}], "}"}], 
      "17.49353584482058`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681386547`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"4907f958-150d-44ad-a562-bf27c97a0241"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168143125*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"3418e482-820c-4ece-9de6-1c912f471847"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168147784*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"d3e154f1-f0a7-4422-8c2b-601aafe50a15"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681515083`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"a52c676a-3ed8-41a3-bb4c-7b1c3b75b302"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"82.77595057310057`", ",", "0", ",", "0"}], "}"}], 
      "20.460026916534517`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168155422*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"013f842d-4e95-4d8a-b9b1-0b96754d159e"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168160471*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"4cbbba01-efac-47b9-89ef-e69f9540a293"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681653137`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"3a2ac742-b9fc-4a28-afce-b6caccb8fc55"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168170686*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"a6863415-48e4-45c4-a50a-bc798ae6230b"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"97.4130217274855`", ",", "0", ",", "0"}], "}"}], 
      "24.79605828146948`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168176283*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"656cab3f-cd3c-4b73-a56a-a6ab962539ff"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681811247`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"6854704d-d29c-4fcb-bf86-c54ff7589467"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168185356*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"8942b2f9-ae87-4c6c-8bdc-416b4944e51f"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681906233`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"14fbb89a-434f-4f9b-bfea-d3647cb65c1d"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"120.18883379815287`", ",", "0", ",", "0"}], "}"}], 
      "31.094880546605896`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661681971607`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"95235eae-058d-4091-bd8b-65e985029151"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168202627*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"921e985a-3f12-4457-87bc-f088a5673356"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Hydrodynamics of an assembly of \"\>", "\[InvisibleSpace]", 
   "2", "\[InvisibleSpace]", "\<\" spheres\"\>"}],
  SequenceForm["Hydrodynamics of an assembly of ", 2, " spheres"],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.7952661682085323`*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"bf1098b8-4ce9-4b2f-9cd9-e526f94c3164"],

Cell[BoxData["\<\" {x,y,z}   radius\"\>"], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168212739*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"3d0ae049-2153-414c-84ba-22a925e82519"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"{", 
       RowBox[{"0", ",", "0", ",", "0"}], "}"}], "57"},
     {
      RowBox[{"{", 
       RowBox[{"155.38271549591818`", ",", "0", ",", "0"}], "}"}], 
      "40.03772654369161`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168217696*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"8e50d289-5088-4ce1-9521-aca70aabe4b0"],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Matrix is size: \"\>", "\[InvisibleSpace]", 
   RowBox[{"{", 
    RowBox[{"10", ",", "10"}], "}"}]}],
  SequenceForm["Matrix is size: ", {10, 10}],
  Editable->False]], "Print",
 CellChangeTimes->{3.7949128075853167`*^9, 3.795266168223288*^9},
 CellLabel->
  "During evaluation of \
In[69]:=",ExpressionUUID->"e82fc9c9-7634-4891-bfc4-63bff7a8b5a8"]
}, Open  ]],

Cell["Results for orientation 1 (aligned in x direction)", "Text",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"3f2e8089-40cf-46f6-9cd4-ce07a2ba234a"],

Cell[CellGroupData[{

Cell["\<\
MatrixForm[N[Transpose[{nall,radiiall,uall,fzall,fBzall}],4]]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[70]:=",ExpressionUUID->"103ed379-464f-4b14-9d5c-00e8bff902a9"],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0.402890321529133`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "11.877548930570843`"}], "}"}], 
      "0.0009162782992554061`", "1.`", "0.9340183206785704`"},
     {"0.6347364189402819`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "12.034323655022332`"}], "}"}], 
      "0.0009155269078632254`", "0.9999999999999999`", 
      "0.9347151620927784`"},
     {"1.`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "12.278815084740968`"}], "}"}], 
      "0.0009143142497740865`", "1.`", "0.9358188084507467`"},
     {"1.5754571033903182`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "12.658177878173777`"}], "}"}], 
      "0.0009123353701727042`", "0.9999999999999999`", "0.937564778929949`"},
     {"2.482065084623012`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "13.242699705429152`"}], "}"}], 
      "0.0009090604123202538`", "0.9999999999999999`", 
      "0.9403102686065898`"},
     {"3.9103870686464153`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "14.135125827602376`"}], "}"}], 
      "0.0009035563500196948`", "1.`", "0.9445559648440504`"},
     {"6.160647084304639`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "15.482728885765113`"}], "}"}], 
      "0.0008941922255138094`", "1.`", "0.9508869995827747`"},
     {"9.705835210448596`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "17.49353584482058`"}], "}"}], 
      "0.0008782491451633617`", "0.9999999999999998`", 
      "0.9597085663966836`"},
     {"15.291127026637104`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "20.460026916534517`"}], "}"}], 
      "0.0008516826460829171`", "0.9999999999999998`", 
      "0.9707018752034525`"},
     {"24.090514692959104`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "24.79605828146948`"}], "}"}], 
      "0.0008096780536246925`", "0.9999999999999999`", 
      "0.9823051890071554`"},
     {"37.95357249735125`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "31.094880546605896`"}], "}"}], 
      "0.0007486240975994874`", "1.0000000000000002`", 
      "0.9920310357192705`"},
     {"59.79422538999145`", 
      RowBox[{"{", 
       RowBox[{"57.`4.", ",", "40.03772654369161`"}], "}"}], 
      "0.0006696849851832519`", "1.0000000000000002`", "0.9978982151196422`"}
    },
    GridBoxAlignment->{"Columns" -> {{Center}}, "Rows" -> {{Baseline}}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 CellChangeTimes->{3.79491280783517*^9, 3.795266168333214*^9},
 CellLabel->
  "Out[70]//MatrixForm=",ExpressionUUID->"fc1fb73e-3152-4742-8d4b-\
1d4ac0abd50e"]
}, Open  ]],

Cell[TextData[{
 "The values in this table match exactly those in the original ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " notebook (but with a different number of significant figures provided), \
copied below:"
}], "Text",
 CellFrame->True,
 FontColor->RGBColor[0, 0, 1],
 Background->GrayLevel[
  0.833326],ExpressionUUID->"19e0ad2a-7a83-4fbf-8f15-4891014b2f88"],

Cell[OutputFormData["\<\
MatrixForm[{{0.4028903215291329984, {57., 11.87754893057077024}, 
 
    0.0009162782992554211008, 1., 0.9340183206785410912}, 
 
   {0.6347364189402818552, {57., 12.03432365502225954}, 
 
    0.0009155269078632304049, 1., 0.9347151620927694008}, 
 
   {1., {57., 12.27881508474089497}, 0.0009143142497740764362, 1., 
 
    0.9358188084507654646}, {1.575457103390318266, 
 
    {57., 12.65817787817370561}, 0.0009123353701727025776, 1., 
 
    0.9375647789299521531}, {2.482065084623011977, 
 
    {57., 13.24269970542908097}, 0.0009090604123202511157, 1., 
 
    0.9403102686065944433}, {3.910387068646415636, 
 
    {57., 14.13512582760230466}, 0.0009035563500196934039, 1., 
 
    0.9445559648440521558}, {6.160647084304639614, 
 
    {57., 15.4827288857650432}, 0.0008941922255138103687, 1., 
 
    0.9508869995827738861}, {9.70583521044859737, 
 
    {57., 17.49353584482051243}, 0.0008782491451633630181, 1., 
 
    0.9597085663966829894}, {15.29112702663710731, 
 
    {57., 20.46002691653445146}, 0.0008516826460829173572, 1., 
 
    0.9707018752034526873}, {24.09051469295910709, 
 
    {57., 24.7960582814694225}, 0.0008096780536246922953, 1., 
 
    0.9823051890071555347}, {37.95357249735125726, 
 
    {57., 31.09488054660584041}, 0.0007486240975994876542, 1., 
 
    0.9920310357192703564}, {59.79422538999145958, 
 
    {57., 40.03772654369157374}, 0.0006696849851832519147, 1., 
 
    0.9978982151196419793}}]\
\>", "\<\
0.4029         {57., 11.88}   0.0009163      1.             0.934

0.6347         {57., 12.03}   0.0009155      1.             0.9347

1.             {57., 12.28}   0.0009143      1.             0.9358

1.575          {57., 12.66}   0.0009123      1.             0.9376

2.482          {57., 13.24}   0.0009091      1.             0.9403

3.91           {57., 14.14}   0.0009036      1.             0.9446

6.161          {57., 15.48}   0.0008942      1.             0.9509

9.706          {57., 17.49}   0.0008782      1.             0.9597

15.29          {57., 20.46}   0.0008517      1.             0.9707

24.09          {57., 24.8}    0.0008097      1.             0.9823

37.95          {57., 31.09}   0.0007486      1.             0.992

59.79          {57., 40.04}   0.0006697      1.             0.9979\
\>"], "Output",
 ImageRegion->{{0, 1}, {0, 
  1}},ExpressionUUID->"b736fcb3-2214-4b0e-84d4-3859f905e7a8"],

Cell[CellGroupData[{

Cell["\<\
ListPlot[Transpose[{nall,fBzall}],
\tPlotJoined->True,
\tAxesLabel->{\"length, bp\",\"F(eq 4)/F(true)\"},
\tPlotRange->{0.9,1}]\
\>", "Input",
 InitializationCell->True,
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"In[71]:=",ExpressionUUID->"61168add-9771-44a2-8670-0c703bc45450"],

Cell[BoxData[
 GraphicsBox[{{}, {{{}, {}, 
     {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
      0.012833333333333334`], AbsoluteThickness[1.6], 
      LineBox[{{0.402890321529133, 0.9340183206785704}, {0.6347364189402819, 
       0.9347151620927784}, {1., 0.9358188084507467}, {1.5754571033903182`, 
       0.937564778929949}, {2.482065084623012, 0.9403102686065898}, {
       3.9103870686464153`, 0.9445559648440504}, {6.160647084304639, 
       0.9508869995827747}, {9.705835210448596, 0.9597085663966836}, {
       15.291127026637104`, 0.9707018752034525}, {24.090514692959104`, 
       0.9823051890071554}, {37.95357249735125, 0.9920310357192705}, {
       59.79422538999145, 0.9978982151196422}}]}}, 
    {RGBColor[0.368417, 0.506779, 0.709798], PointSize[0.012833333333333334`],
      AbsoluteThickness[1.6]}, {
     {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
      0.012833333333333334`], AbsoluteThickness[1.6]}, {}}, {
     {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
      0.012833333333333334`], AbsoluteThickness[1.6]}, {}}}, {{}, {}}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{
    FormBox["\"length, bp\"", TraditionalForm], 
    FormBox["\"F(eq 4)/F(true)\"", TraditionalForm]},
  AxesOrigin->{0, 0.9},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, Automatic}, {Automatic, Automatic}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  Method->{
   "OptimizePlotMarkers" -> True, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0, 59.79422538999145}, {0.9, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {0, 0}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{3.794912808056946*^9, 3.795266168809642*^9},
 CellLabel->"Out[71]=",ExpressionUUID->"06545edb-6730-496d-bf9d-40f900a50a3a"]
}, Open  ]]
}, Open  ]],

Cell["\<\
In the article, the data were plotted in Figure 3 using CricketGraph, but the \
relevant files are no longer readable. Below I mocked up the essential parts \
of this figure using the recalculated data. (Note that only the last 7 data \
points were used in making the published plot.)\
\>", "Text",
 CellFrame->True,
 FontColor->RGBColor[0, 0, 1],
 Background->GrayLevel[
  0.833326],ExpressionUUID->"cbbf5cba-3507-4246-bafa-f4780dff865f"],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"ListPlot", "[", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"Take", "[", 
         RowBox[{"nall", ",", 
          RowBox[{"-", "7"}]}], "]"}], ",", 
        RowBox[{"Take", "[", 
         RowBox[{"fBzall", ",", 
          RowBox[{"-", "7"}]}], "]"}]}], "}"}], "]"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"PlotJoined", "\[Rule]", "True"}], ",", "\[IndentingNewLine]", 
     RowBox[{"FrameLabel", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<length, bp\>\"", ",", "\"\<F(eq 4)/F(true)\>\""}], 
       "}"}]}], ",", "\[IndentingNewLine]", 
     RowBox[{"PlotRange", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "60"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0.92", ",", "1"}], "}"}]}], "}"}]}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"AspectRatio", "\[Rule]", 
      RowBox[{"8.0", "/", "7.5"}]}], ",", "\[IndentingNewLine]", 
     RowBox[{"Frame", "\[Rule]", "True"}]}], "\[IndentingNewLine]", "]"}], 
   ",", "\[IndentingNewLine]", 
   RowBox[{"Graphics", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Dashing", "[", 
       RowBox[{"{", 
        RowBox[{"0.01", ",", "0.015"}], "}"}], "]"}], ",", 
      RowBox[{"HalfLine", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"11", ",", "0.92"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"10.5", ",", "1"}], "}"}]}], "}"}], "]"}]}], "}"}], "]"}]}],
   "\[IndentingNewLine]", "]"}]], "Input",
 CellChangeTimes->{{3.7952664012278223`*^9, 3.795266516663392*^9}, 
   3.795266549097537*^9, 3.7952665947812147`*^9, {3.795266714367305*^9, 
   3.7952667636240797`*^9}, {3.795266797649165*^9, 3.795266871811996*^9}, {
   3.795266913538137*^9, 3.79526693757045*^9}, {3.795267307684555*^9, 
   3.795267341699847*^9}, {3.795267455952443*^9, 3.795267479191345*^9}, {
   3.795267535949451*^9, 3.795267572952033*^9}, {3.795267660387535*^9, 
   3.795267684033229*^9}},
 CellLabel->
  "In[105]:=",ExpressionUUID->"2108b9d5-72c5-4b36-91d8-1a46512255da"],

Cell[BoxData[
 GraphicsBox[{{{}, {{{}, {}, 
      {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
       0.012833333333333334`], AbsoluteThickness[1.6], 
       LineBox[{{3.9103870686464153`, 0.9445559648440504}, {6.160647084304639,
         0.9508869995827747}, {9.705835210448596, 0.9597085663966836}, {
        15.291127026637104`, 0.9707018752034525}, {24.090514692959104`, 
        0.9823051890071554}, {37.95357249735125, 0.9920310357192705}, {
        59.79422538999145, 0.9978982151196422}}]}}, 
     {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
      0.012833333333333334`], AbsoluteThickness[1.6]}, {
      {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
       0.012833333333333334`], AbsoluteThickness[1.6]}, {}}, {
      {RGBColor[0.368417, 0.506779, 0.709798], PointSize[
       0.012833333333333334`], AbsoluteThickness[1.6]}, {}}}, {{}, {}}}, 
   {Dashing[{0.01, 0.015}], 
    TagBox[ConicHullRegionBox[{{11, 0.92}}, {{-0.5, 0.07999999999999996}}],
     "HalfLine"]}},
  AspectRatio->1.0666666666666667`,
  Axes->{True, True},
  AxesLabel->{None, None},
  AxesOrigin->{0, 0.92},
  DisplayFunction->Identity,
  Frame->{{True, True}, {True, True}},
  FrameLabel->{{
     FormBox["\"F(eq 4)/F(true)\"", TraditionalForm], None}, {
     FormBox["\"length, bp\"", TraditionalForm], None}},
  FrameTicks->{{Automatic, Automatic}, {Automatic, Automatic}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  Method->{
   "OptimizePlotMarkers" -> True, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Identity[#]& )[
         Part[#, 1]], 
        (Identity[#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0, 60}, {0.92, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{{0, 0}, {0, 0}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{
  3.795266466558296*^9, 3.7952665185745783`*^9, 3.795266596677375*^9, 
   3.795266765322579*^9, {3.7952668151847773`*^9, 3.795266837124058*^9}, 
   3.7952668737045918`*^9, {3.795266915286901*^9, 3.795266939058607*^9}, 
   3.795267136932729*^9, {3.7952673145739727`*^9, 3.795267343388464*^9}, {
   3.795267511567226*^9, 3.7952675746918707`*^9}, {3.7952676693124447`*^9, 
   3.795267685211239*^9}},
 CellLabel->
  "Out[105]=",ExpressionUUID->"e81a3828-abda-4831-9c9b-df58e9f8b390"]
}, Open  ]]
},
AutoGeneratedPackage->None,
WindowToolbars->{},
CellGrouping->Manual,
WindowSize->{1267, 822},
WindowMargins->{{Automatic, -1}, {Automatic, 92}},
ShowSelection->True,
ShowCellLabel->True,
ShowCellTags->False,
Magnification->1.5,
FrontEndVersion->"12.0 for Linux ARM (32-bit) (June 23, 2019)",
StyleDefinitions->"Default.nb",
PrivateNotebookOptions -> {"ColorPalette" -> {RGBColor, -1}},
RenderingOptions -> {"ObjectDithering" -> True},
RenderingOptions -> {"RasterDithering" -> False}
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[558, 20, 400, 8, 125, "Text",ExpressionUUID->"dd8c626d-d4db-429b-b842-5b3ee75eaba8",
 InitializationCell->True],
Cell[961, 30, 1030, 19, 506, "Text",ExpressionUUID->"4e3a218b-be26-4640-a1f2-d8b9b5531067",
 InitializationCell->True],
Cell[1994, 51, 243, 6, 71, "Input",ExpressionUUID->"bc8446c2-357a-4acf-b781-b5ca975580fb",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[2262, 61, 226, 5, 69, "Subsubsection",ExpressionUUID->"eee7abd4-1720-47d7-a2f2-566ce75c99f4",
 InitializationCell->True],
Cell[2491, 68, 267, 8, 100, "Input",ExpressionUUID->"1ee83d32-0cac-416d-9dc0-227be9be9d10",
 PageBreakWithin->Automatic,
 InitializationCell->True],
Cell[2761, 78, 187, 3, 71, "Input",ExpressionUUID->"aa933b91-b2cb-450c-a913-2ccd9b203711",
 InitializationCell->True],
Cell[2951, 83, 294, 9, 157, "Input",ExpressionUUID->"f8be7ac8-0a76-4829-9d88-cdd3af80299b",
 InitializationCell->True],
Cell[3248, 94, 240, 4, 71, "Input",ExpressionUUID->"ec8d89f7-da76-4331-bbf5-945812918872",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[3513, 102, 211, 4, 71, "Input",ExpressionUUID->"71d203c7-48ba-49db-a40c-a7a9d5bb1cc2",
 InitializationCell->True],
Cell[3727, 108, 4054, 122, 344, "Output",ExpressionUUID->"10343a5f-0f91-4084-9a1b-bcbd569cc987"]
}, Open  ]],
Cell[7796, 233, 178, 3, 54, "Text",ExpressionUUID->"ac37265c-0aeb-4626-a25d-05efddfe381e",
 InitializationCell->True],
Cell[7977, 238, 340, 8, 157, "Input",ExpressionUUID->"1f054c95-0db6-45f4-9945-8358f971a3f2",
 InitializationCell->True],
Cell[8320, 248, 167, 3, 54, "Text",ExpressionUUID->"ac596702-0e90-4447-93e4-c7d3f17f6fb7",
 InitializationCell->True],
Cell[8490, 253, 946, 18, 306, "Text",ExpressionUUID->"fbedbbcd-1ce7-494c-9450-2c98eba0ce76",
 InitializationCell->True],
Cell[9439, 273, 267, 8, 128, "Input",ExpressionUUID->"6da2bbb1-d1cf-4d61-bed3-8614e1ee61e4",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[9731, 285, 234, 4, 71, "Input",ExpressionUUID->"c1253b67-e300-4faa-a823-c372f457934d",
 InitializationCell->True],
Cell[9968, 291, 332, 9, 111, "Output",ExpressionUUID->"ff493bac-52f8-4560-90ff-eb39e5b14d80"]
}, Open  ]],
Cell[CellGroupData[{
Cell[10337, 305, 175, 3, 71, "Input",ExpressionUUID->"219a4c04-751f-4f0e-b78e-e1be5a51824c",
 InitializationCell->True],
Cell[10515, 310, 248, 5, 95, "Output",ExpressionUUID->"f8fe5fb8-40d5-497c-a06f-480d93685c5c"]
}, Open  ]],
Cell[CellGroupData[{
Cell[10800, 320, 180, 3, 71, "Input",ExpressionUUID->"6859774f-bd9c-47f1-9612-bd42799852ca",
 InitializationCell->True],
Cell[10983, 325, 828, 22, 98, "Output",ExpressionUUID->"49293797-898e-4430-a3e4-c7bd1db3ca72"]
}, Open  ]],
Cell[CellGroupData[{
Cell[11848, 352, 305, 8, 128, "Input",ExpressionUUID->"f37e8b4b-cd36-40d8-8d04-f7a0962fd2e0",
 InitializationCell->True],
Cell[12156, 362, 1077, 29, 470, "Output",ExpressionUUID->"b44ff57b-66b3-49c0-b0fa-ff4d7a1c97bd"]
}, Open  ]]
}, Open  ]],
Cell[13260, 395, 419, 11, 116, "Text",ExpressionUUID->"7f54a880-0913-48f4-83cf-005bad9924bf"],
Cell[CellGroupData[{
Cell[13704, 410, 220, 5, 69, "Subsubsection",ExpressionUUID->"5e8b84ab-d81c-46c5-84b4-01fb4b61d7b5",
 InitializationCell->True],
Cell[13927, 417, 310, 9, 158, "Input",ExpressionUUID->"ac8541a0-926b-4b11-9f9e-aeeeb2083c4d",
 InitializationCell->True],
Cell[14240, 428, 181, 3, 38, "Input",ExpressionUUID->"ee083062-60c5-4dfd-b1cd-2663ccf65525",
 InitializationCell->True],
Cell[14424, 433, 740, 21, 398, "Input",ExpressionUUID->"0ef32b1d-1efb-45bc-a8e8-f3fafbdc8c58",
 InitializationCell->True],
Cell[15167, 456, 174, 3, 47, "Text",ExpressionUUID->"66a8535f-39c9-4240-9196-1cbe3555aba6",
 InitializationCell->True],
Cell[15344, 461, 297, 6, 138, "Input",ExpressionUUID->"7f4e9cf7-6654-476a-a0da-f5283632157e",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[15666, 471, 168, 3, 38, "Input",ExpressionUUID->"058bf719-e40f-4ffb-9e16-a6c05476cf06",
 InitializationCell->True],
Cell[15837, 476, 2468, 67, 70, "Output",ExpressionUUID->"e6e8dfec-0b2d-4f42-b051-f6661933b070"]
}, Open  ]],
Cell[CellGroupData[{
Cell[18342, 548, 189, 3, 38, "Input",ExpressionUUID->"1ec396bf-a54e-408a-b6d2-28c6e7c4ed64",
 InitializationCell->True],
Cell[18534, 553, 2468, 67, 70, "Output",ExpressionUUID->"b469c115-68f0-47c7-bf1e-0c49c8334017"]
}, Open  ]],
Cell[CellGroupData[{
Cell[21039, 625, 177, 3, 38, "Input",ExpressionUUID->"d5113092-c639-418c-8d0a-2fba707f0d3c",
 InitializationCell->True],
Cell[21219, 630, 780, 21, 70, "Output",ExpressionUUID->"d9cd662a-9490-4a1f-b5d4-14de100d3cdf"]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[22048, 657, 178, 3, 55, "Subsubsection",ExpressionUUID->"c05606c5-274e-4043-af34-d971f81c998b",
 InitializationCell->True],
Cell[22229, 662, 227, 6, 54, "Text",ExpressionUUID->"efb3c98a-06a1-41f0-97f5-3750b357ec53",
 InitializationCell->True],
Cell[22459, 670, 447, 15, 299, "Input",ExpressionUUID->"1ccbb2cc-64b4-4dc2-bda9-0b2b2b350ebc",
 PageBreakWithin->Automatic,
 InitializationCell->True],
Cell[22909, 687, 310, 7, 128, "Input",ExpressionUUID->"ef06c4db-2bf8-47c0-9f89-1639614c4987",
 InitializationCell->True],
Cell[23222, 696, 729, 20, 442, "Input",ExpressionUUID->"ab54fce8-9d39-41a5-9a6a-5d3c05207b4f",
 PageBreakWithin->Automatic,
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[23988, 721, 176, 3, 69, "Subsubsection",ExpressionUUID->"f57a4022-40b1-4c93-b702-9ea1fde1812c",
 InitializationCell->True],
Cell[24167, 726, 311, 9, 185, "Input",ExpressionUUID->"f10376da-b9e9-43da-aa89-13ffaeb6d7f6",
 InitializationCell->True],
Cell[24481, 737, 1087, 30, 784, "Input",ExpressionUUID->"c6654cea-41d8-49f3-825a-f8efe9b8efff",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[25605, 772, 242, 6, 110, "Subsubsection",ExpressionUUID->"5776372b-14c3-461f-a38f-c780f9a48b47",
 InitializationCell->True],
Cell[25850, 780, 209, 5, 54, "Text",ExpressionUUID->"0a639cd4-09f4-4bd8-a344-937d5e6dafb8",
 InitializationCell->True],
Cell[26062, 787, 372, 9, 128, "Input",ExpressionUUID->"64025076-4ee2-46ad-8228-54be75977252",
 PageBreakWithin->Automatic,
 InitializationCell->True],
Cell[26437, 798, 339, 9, 128, "Input",ExpressionUUID->"dd215ac0-60e2-4a74-b719-536d4f528f0f",
 PageBreakWithin->Automatic,
 InitializationCell->True],
Cell[26779, 809, 527, 12, 214, "Input",ExpressionUUID->"4b8d55ac-9139-4a86-a9a3-07ed35808cb0",
 PageBreakWithin->Automatic,
 InitializationCell->True],
Cell[27309, 823, 464, 10, 214, "Input",ExpressionUUID->"f3b95965-3e81-4ded-ae0b-9ff7dfe87b30",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[27798, 837, 165, 3, 71, "Input",ExpressionUUID->"a884608e-6fee-4ca7-a509-3808c951be05",
 InitializationCell->True],
Cell[27966, 842, 499, 10, 37, "Print",ExpressionUUID->"1583f688-0a94-4e04-aca3-8e43ecfa8d6a"]
}, Open  ]],
Cell[CellGroupData[{
Cell[28502, 857, 288, 7, 128, "Input",ExpressionUUID->"91e87679-451e-449a-a1f9-a6a6572a20e4",
 InitializationCell->True],
Cell[28793, 866, 10922, 201, 323, "Output",ExpressionUUID->"a9ca3c59-2e36-4983-a2b2-548194a4bc57"]
}, Open  ]],
Cell[CellGroupData[{
Cell[39752, 1072, 268, 6, 100, "Input",ExpressionUUID->"03b7336d-5ae4-43f2-aed0-e9187754460d",
 InitializationCell->True],
Cell[40023, 1080, 6349, 125, 339, "Output",ExpressionUUID->"711e6959-ac4a-4b66-a04f-62c53d0c4a08"]
}, Open  ]],
Cell[CellGroupData[{
Cell[46409, 1210, 281, 7, 128, "Input",ExpressionUUID->"f2d186d1-3d7d-427c-bd10-7d21f9b4ebc3",
 InitializationCell->True],
Cell[46693, 1219, 4698, 100, 358, "Output",ExpressionUUID->"cab04208-cde8-4775-af9c-70fb3a35633d"]
}, Open  ]],
Cell[CellGroupData[{
Cell[51428, 1324, 258, 6, 100, "Input",ExpressionUUID->"a8a9d7e7-17f1-4a4a-9581-8cb70d29cd58",
 InitializationCell->True],
Cell[51689, 1332, 19422, 341, 352, 5330, 110, "CachedBoxData", "BoxData", "Output",ExpressionUUID->"afbc9d36-8b7a-4b0d-b194-ec34d3202835"]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[71160, 1679, 220, 5, 69, "Subsubsection",ExpressionUUID->"0c797c40-fed6-49d4-8b87-2fa15993c93c",
 InitializationCell->True],
Cell[71383, 1686, 538, 10, 161, "Text",ExpressionUUID->"fd0e09d5-11a9-4fa2-9d7b-ee09ef5afa35",
 InitializationCell->True],
Cell[71924, 1698, 257, 13, 299, "Input",ExpressionUUID->"e8baed60-7586-4ccd-8736-73b12f69f907",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[72206, 1715, 970, 30, 784, "Input",ExpressionUUID->"35adb82c-2295-41dd-b49f-642f7b069a60",
 InitializationCell->True],
Cell[73179, 1747, 430, 9, 37, "Print",ExpressionUUID->"637e8813-4652-47d5-84eb-8db61c921f2e"],
Cell[73612, 1758, 224, 4, 37, "Print",ExpressionUUID->"b40d2aab-23f3-4092-ac2c-3f2c832cbec1"],
Cell[73839, 1764, 825, 24, 59, "Print",ExpressionUUID->"e1f99adf-9217-46ee-9dfe-d4c383b56601"],
Cell[74667, 1790, 397, 10, 37, "Print",ExpressionUUID->"0d9b042e-6d69-436f-bf18-35e7252556d9"],
Cell[75067, 1802, 430, 9, 37, "Print",ExpressionUUID->"bf60ef90-f5c6-4e51-9143-73f9fbc7bc7f"],
Cell[75500, 1813, 222, 4, 37, "Print",ExpressionUUID->"51d3b554-02de-4c89-bab9-de26d8e9cc75"],
Cell[75725, 1819, 826, 24, 59, "Print",ExpressionUUID->"fbc13f37-ecf5-4d49-b922-7c36fab3bba6"],
Cell[76554, 1845, 396, 10, 37, "Print",ExpressionUUID->"894b3220-d5b9-45d1-a3e7-ba47cc0c416e"],
Cell[76953, 1857, 428, 9, 37, "Print",ExpressionUUID->"d388d2e5-8b86-4b22-bf6c-78ca6affeaac"],
Cell[77384, 1868, 224, 4, 37, "Print",ExpressionUUID->"9ebf2eb9-8a58-4c44-a992-c40ec28537b5"],
Cell[77611, 1874, 826, 24, 59, "Print",ExpressionUUID->"00588568-efb9-4e22-88e7-d546d863ed07"],
Cell[78440, 1900, 399, 10, 37, "Print",ExpressionUUID->"82718771-9ef5-4ac6-87ca-7834ec96deb8"],
Cell[78842, 1912, 430, 9, 37, "Print",ExpressionUUID->"ef47d120-23a3-416f-91a3-da2c1cf66a73"],
Cell[79275, 1923, 222, 4, 37, "Print",ExpressionUUID->"8359e9e3-4417-48b7-a1e3-3248d794b38a"],
Cell[79500, 1929, 828, 24, 59, "Print",ExpressionUUID->"b0af490a-7b74-4904-b4da-783c65dea874"],
Cell[80331, 1955, 399, 10, 37, "Print",ExpressionUUID->"93abb62f-b6e3-4b09-a9c8-5e228b20c5a2"],
Cell[80733, 1967, 430, 9, 37, "Print",ExpressionUUID->"9e25ecf0-eb46-42e4-9c8e-af0055118776"],
Cell[81166, 1978, 224, 4, 37, "Print",ExpressionUUID->"7b6dd9ef-90c9-4b72-a950-5eaa3a7d8b05"],
Cell[81393, 1984, 827, 24, 59, "Print",ExpressionUUID->"74b05a97-79da-47ab-a968-644526f98ed8"],
Cell[82223, 2010, 397, 10, 37, "Print",ExpressionUUID->"4275d928-8a58-4016-9f08-2694cbb4e245"],
Cell[82623, 2022, 428, 9, 37, "Print",ExpressionUUID->"51b5217f-094e-4f48-992f-1a0bd9a1b440"],
Cell[83054, 2033, 221, 4, 37, "Print",ExpressionUUID->"b35dd61c-5355-459e-93b8-645cbeb9f2bf"],
Cell[83278, 2039, 826, 24, 59, "Print",ExpressionUUID->"e7f186e7-e195-4050-b3be-19b14a22d12c"],
Cell[84107, 2065, 399, 10, 37, "Print",ExpressionUUID->"09d3c716-4b2d-47c8-9bba-6cf875e94713"],
Cell[84509, 2077, 428, 9, 37, "Print",ExpressionUUID->"117e77ad-ba77-4a16-adbc-ee0a5426e690"],
Cell[84940, 2088, 222, 4, 37, "Print",ExpressionUUID->"4d8ade43-d107-4fe3-961b-dd280568996c"],
Cell[85165, 2094, 827, 24, 59, "Print",ExpressionUUID->"e0480394-07e7-4b4d-833a-14e70b2fd225"],
Cell[85995, 2120, 399, 10, 37, "Print",ExpressionUUID->"a18075a6-20eb-4338-a054-b4643256674c"],
Cell[86397, 2132, 428, 9, 37, "Print",ExpressionUUID->"d934976c-60d3-4ec0-bab1-7c0839c1f767"],
Cell[86828, 2143, 220, 4, 37, "Print",ExpressionUUID->"c1daae7a-1272-46cd-bbfd-6082e95b11c8"],
Cell[87051, 2149, 826, 24, 59, "Print",ExpressionUUID->"4907f958-150d-44ad-a562-bf27c97a0241"],
Cell[87880, 2175, 397, 10, 37, "Print",ExpressionUUID->"3418e482-820c-4ece-9de6-1c912f471847"],
Cell[88280, 2187, 428, 9, 37, "Print",ExpressionUUID->"d3e154f1-f0a7-4422-8c2b-601aafe50a15"],
Cell[88711, 2198, 224, 4, 37, "Print",ExpressionUUID->"a52c676a-3ed8-41a3-bb4c-7b1c3b75b302"],
Cell[88938, 2204, 825, 24, 59, "Print",ExpressionUUID->"013f842d-4e95-4d8a-b9b1-0b96754d159e"],
Cell[89766, 2230, 397, 10, 37, "Print",ExpressionUUID->"4cbbba01-efac-47b9-89ef-e69f9540a293"],
Cell[90166, 2242, 430, 9, 37, "Print",ExpressionUUID->"3a2ac742-b9fc-4a28-afce-b6caccb8fc55"],
Cell[90599, 2253, 222, 4, 37, "Print",ExpressionUUID->"a6863415-48e4-45c4-a50a-bc798ae6230b"],
Cell[90824, 2259, 823, 24, 59, "Print",ExpressionUUID->"656cab3f-cd3c-4b73-a56a-a6ab962539ff"],
Cell[91650, 2285, 399, 10, 37, "Print",ExpressionUUID->"6854704d-d29c-4fcb-bf86-c54ff7589467"],
Cell[92052, 2297, 428, 9, 37, "Print",ExpressionUUID->"8942b2f9-ae87-4c6c-8bdc-416b4944e51f"],
Cell[92483, 2308, 224, 4, 37, "Print",ExpressionUUID->"14fbb89a-434f-4f9b-bfea-d3647cb65c1d"],
Cell[92710, 2314, 828, 24, 59, "Print",ExpressionUUID->"95235eae-058d-4091-bd8b-65e985029151"],
Cell[93541, 2340, 397, 10, 37, "Print",ExpressionUUID->"921e985a-3f12-4457-87bc-f088a5673356"],
Cell[93941, 2352, 430, 9, 37, "Print",ExpressionUUID->"bf1098b8-4ce9-4b2f-9cd9-e526f94c3164"],
Cell[94374, 2363, 222, 4, 37, "Print",ExpressionUUID->"3d0ae049-2153-414c-84ba-22a925e82519"],
Cell[94599, 2369, 825, 24, 59, "Print",ExpressionUUID->"8e50d289-5088-4ce1-9521-aca70aabe4b0"],
Cell[95427, 2395, 397, 10, 37, "Print",ExpressionUUID->"e82fc9c9-7634-4891-bfc4-63bff7a8b5a8"]
}, Open  ]],
Cell[95839, 2408, 183, 3, 54, "Text",ExpressionUUID->"3f2e8089-40cf-46f6-9cd4-ce07a2ba234a",
 InitializationCell->True],
Cell[CellGroupData[{
Cell[96047, 2415, 224, 5, 71, "Input",ExpressionUUID->"103ed379-464f-4b14-9d5c-00e8bff902a9",
 InitializationCell->True],
Cell[96274, 2422, 2863, 71, 389, "Output",ExpressionUUID->"fc1fb73e-3152-4742-8d4b-1d4ac0abd50e"]
}, Open  ]],
Cell[99152, 2496, 375, 10, 116, "Text",ExpressionUUID->"19e0ad2a-7a83-4fbf-8f15-4891014b2f88"],
Cell[99530, 2508, 2380, 74, 452, "Output",ExpressionUUID->"b736fcb3-2214-4b0e-84d4-3859f905e7a8"],
Cell[CellGroupData[{
Cell[101935, 2586, 290, 8, 157, "Input",ExpressionUUID->"61168add-9771-44a2-8670-0c703bc45450",
 InitializationCell->True],
Cell[102228, 2596, 2230, 49, 346, "Output",ExpressionUUID->"06545edb-6730-496d-bf9d-40f900a50a3a"]
}, Open  ]]
}, Open  ]],
Cell[104485, 2649, 449, 9, 151, "Text",ExpressionUUID->"cbbf5cba-3507-4246-bafa-f4780dff865f"],
Cell[CellGroupData[{
Cell[104959, 2662, 2192, 54, 409, "Input",ExpressionUUID->"2108b9d5-72c5-4b36-91d8-1a46512255da"],
Cell[107154, 2718, 2437, 54, 600, "Output",ExpressionUUID->"e81a3828-abda-4831-9c9b-df58e9f8b390"]
}, Open  ]]
}
]
*)

