library("PMCMRplus")
library("irr")
library("gplots")
#library("broom")

DIGITS <- 2
options(digits=DIGITS)

if (!dir.exists("stat")) dir.create("stat")
if (!dir.exists("fig"))  dir.create("fig")

## Name of the experts
EXPERTS   <- c("01" ,"02" ,"03"   ,"04"       ,"05"    ,"06","07" ,"08"   ,"09"  ,"10"   ,"11"   ,"12"  ,"13"     ,"14"   ,"15")
## Name of the roles used in the project
ROLES     <- c("Prosumer","DER","Aggregator","ESCO","BRP","Supplier","DSO","TSO")
## Reduced set of roles used in the project
ROLES2    <- c("Prosumer","BRP","Supplier","DSO","TSO")
## Name of the Barriers categories
BARRIERS  <- c("Lifestyles","Administration","Technical","Trust","Costs","Standardization")
## Name of the use cases
USECASES  <- c("UC1","UC2","UC3","UC4","UC5","UC7","UC8","UC9","UC11","UC12","UC13","UC15",
               "BUC1","BUC2","BUC3","BUC4")

## Socio-economic description of the experts
KEYWORDS  <- read.csv("data/keywords.csv",row.names="Expert",colClasses=c("character","factor","factor","factor"))
## End critical set of barriers after judgamental correction from the experts
CRITSET   <- read.csv("data/Critical Set.csv",row.names="Roles")
CRITSET[CRITSET == ""]  <- 0
CRITSET[CRITSET == "X"] <- 1
CRITSET <- sapply(CRITSET, as.numeric)  ## make the matrix numeric
row.names(CRITSET) <- ROLES2

## Roles per UC and BUC
## B means big role, and S small role
UCROLES   <- read.csv("data/UC.csv",row.names="ID",colClasses=c("character","character","character","character","character","character","NULL"))
UCROLES[UCROLES == ""]  <- 0
UCROLES[UCROLES == "B"] <- 1
UCROLES[UCROLES == "S"] <- 0.5
UCROLES <- sapply(UCROLES, as.numeric)  ## make the matrix numeric
UCROLES <- t(apply(as.matrix(UCROLES),1, function(x) {x/sum(x)})) ## weight the rows
row.names(UCROLES) <- USECASES

error.bar <- function(x, y, upper, lower=upper, length=0.1,...)
{ arrows(x,y+upper, x, y-lower, angle=90, code=3, length=length, ...) }

## Funtion that read the scores given by the experts
## round: round number
## partner: code of the partner providing the scores
READ <- function(round,partner)
{
  a <- read.csv(paste("data/",round,"/",partner,".csv", sep=""),
                sep = ",", header = T,skip = 2,na.strings = "#DIV/0!")
  a <- a[c(-1,-8,-14,-20,-26,-30),-1]
  row.names(a) <- a$X.1
  a <- a[,-1]

  return(a)
}

## Function that buid the dataframe with the scores of the experts per category
## round: round number
## row: category of barriers to assess
## experts: list of strings with the experts IDs. The scores of the experts are in variables with this name
DATAFRAME_BARRIERS <- function(round,row,experts)
{
  a <- data.frame()
  for (i in experts)
    a <- rbind(a,get(paste(round,"_",i,sep=""))[row,])

  row.names(a) <- experts

  a <- apply(a,2,function(x){x[is.na(x)] = median(x, na.rm=T); return(x) })
  return(as.data.frame(a))
}

## Function that buid the dataframe with the scores of the experts per role
## round: round number
## column: rol to assess
## experts: list of strings with the experts IDs. The scores of the experts are in variables with this name
## barriers: list of strings with the names of the barriers.
DATAFRAME_ROLES <- function(round,column,experts,barriers)
{
  a <- data.frame()
  for (i in experts)
    a <- rbind(a,get(paste(round,"_",i,sep=""))[,column])

  colnames(a)  <- rownames(get(paste(round,"_",i,sep="")))
  row.names(a) <- experts
  a <- as.data.frame(apply(a,2,function(x){x[is.na(x)] = median(x, na.rm=T); return(x) }))

  b <- data.frame()
  b <- cbind((a$Adoption+a$`Ideology & involvement`+a$Disillusion+a$`Reluctance and lack of trust`+a$`Perceived Usefulness`+a$`Ignorance / lack of expertise`)/6,
             (a$`Charging/cost rules`+a$`Decision Making`+a$`Policy/Regulatory Incentives`+a$`Lack of regulation`+a$`Market restrictions`)/5,
             (a$System+a$Algorithms+a$Maturity+a$Deployment+a$Comunication)/5,
             (a$Security+a$Privacy+a$`Stakeholders Cooperation`+a$`Communication strategy`+a$`Emerging Technologies`)/5,
             (a$Investment+a$Pricing+a$Margin)/3,
             (a$`Technology & Market Diversity`+a$`Roles & Objectives`+a$`Business models`+a$Interoperability)/4)

  colnames(b)  <- barriers
  row.names(b) <- experts
  return(as.data.frame(b[,]))
}

## Function that perform the statisticall assessment and provide the results
## Perform the Friedman test with posthoc per barrier and per role
## Perform a Wilcoxon splitting experts' scores per each keyword
## round: integer with the round
## experts: list of strings with the experts IDs. The scores of the experts are in variables with this name
# i experts
# j barriers
# k roles
ASSESSMENT <- function(round, experts)
{
  if (!dir.exists(paste("stat/",round,sep=""))) dir.create(paste("stat/",round,sep=""),recursive=T)
  if (!dir.exists(paste("fig/", round,sep=""))) dir.create(paste("fig/", round,sep=""),recursive=T)

  for (i in experts)
    assign(paste(round,"_",i,sep=""), READ(round,i),pos=1)

  cat("Barrier",ROLES,"\n",sep=",",file=paste("stat/",round,"/",round,"-means-barriers.csv",sep=""))
  cat("Barrier",ROLES,"\n",sep=",",file=paste("stat/",round,"/",round,"-letters-barriers.csv",sep=""))
  cat("Roles","Intraclass correlation coefficient: agreement",
      "Inter-rater agreement","Inter-rater agreement p-value",
      "Homogeneity of Variances","Average interquartile range",
      "Standard Deviation of interquartile ranges",
      "Friedman test p-value","Average power post hoc","\n",sep=",",file=paste("stat/",round,"/",round,"-statistics-barriers.csv",sep=""))
  pdf(width=9,height=9,file=paste("fig/",round,"/",round,"-boxplot-barriers.pdf",sep=""))
  # i experts
  # j barriers
  # k roles
  for (j in row.names(get(paste(round,"_",i,sep=""))))
  {
    assign(paste(round,"_",j,sep=""), DATAFRAME_BARRIERS(round,j,experts),pos=1)
    dat <- as.data.frame(get(paste(round,"_",j,sep="")))

    ca  <- icc(dat,model="oneway",type="agreement")   # https://en.wikipedia.org/wiki/Intraclass_correlation#Interpretation
    kf  <- kappam.fleiss(as.data.frame(t(dat)))       # H0: agreement by chance
    ft  <- fligner.test(as.data.frame(t(dat)))        # H0: equal variances
    f   <- friedman.test(as.matrix(dat))              # H0: no differences
    ff  <- frdAllPairsExactTest(as.matrix(dat))
    ffl <- multcompView::multcompLetters(ff$p.value)
    pp  <- powerMCTests(mu=colMeans(dat),n=length(dat[1,]),test="kwAllPairsConoverTest")
    IQ  <- apply(dat,2,IQR)

    cat(j,colMeans(dat),"\n",sep=",",append=T,file=paste("stat/",round,"/",round,"-means-barriers.csv",sep=""))
    cat(j,ffl$Letters,  "\n",sep=",",append=T,file=paste("stat/",round,"/",round,"-letters-barriers.csv",sep=""))
    cat(j,ca$value,kf$value,kf$p.value,ft$p.value,
        mean(IQ),sd(IQ),f$p.value,pp$avepair,"\n",
        sep=",",append=T,file=paste("stat/",round,"/",round,"-statistics-barriers.csv",sep=""))
    TT <<- paste("Distribution of scores for barrier",j,"\n",
                "Intraclass correlation coefficient: agreement",format(ca$value,digits=DIGITS),"\n",
                "Inter-rater agreement",format(kf$value,digits=DIGITS),"(",format(kf$p.value,digits=DIGITS),")","\n",
                "Homogeneity of Variances",format(ft$p.value,digits=DIGITS),"\n",
                "Average interquartile range",format(mean(IQ),digits=DIGITS),"\n",
                "Standard Deviation of interquartile ranges",format(sd(IQ),digits=DIGITS),"\n",
                "Friedman test p-value",format(f$p.value,digits=DIGITS),"\n",
                "Average power post hoc",format(pp$avepair,digits=DIGITS),"\n",sep=" ")
    par(mar = c(2,4,11,1))
    plot(ff,ylab="Scores",yaxp=c(-5,5,10),main=TT)
  }
  dev.off()

  pdf(width=9,height=9,file=paste("fig/",round,"/",round,"-means-barriers.pdf",sep=""))
    aux <- read.csv(paste("stat/",round,"/",round,"-means-barriers.csv",sep=""))
    row.names(aux) <- aux[,1]
    aux <- aux[,c(-1,-length(names(aux)))]
    heatmap.2(as.matrix(aux),scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
              trace="none",tracecol="black",margins=c(10,10),
              col=heat.colors(11,rev=T),
              main="Mean Scores Barriers")
  dev.off()

  cat("Roles",BARRIERS,"\n",sep=",",file=paste("stat/",round,"/",round,"-means-roles.csv",sep=""))
  cat("Roles",BARRIERS,"\n",sep=",",file=paste("stat/",round,"/",round,"-letters-roles.csv",sep=""))
  cat("Roles","Intraclass correlation coefficient: agreement",
      "Inter-rater agreement","Inter-rater agreement p-value",
      "Homogeneity of Variances","Average interquartile range",
      "Standard Deviation of interquartile ranges",
      "Friedman test p-value","Average power post hoc","\n",sep=",",file=paste("stat/",round,"/",round,"-statistics-roles.csv",sep=""))
  pdf(width=9,height=9,file=paste("fig/",round,"/",round,"-boxplot-roles.pdf",sep=""))
  # i experts
  # j barriers
  # k roles
  for (k in colnames(get(paste(round,"_",i,sep=""))))
  {
    assign(paste(round,"_",k,sep=""), DATAFRAME_ROLES(round,k,experts,BARRIERS),pos=1)
    dat <- as.data.frame(get(paste(round,"_",k,sep="")))

    ca  <- icc(dat,model="oneway",type="agreement")   # https://en.wikipedia.org/wiki/Intraclass_correlation#Interpretation
    kf  <- kappam.fleiss(as.data.frame(t(dat)))       # H0: agreement by chance
    ft  <- fligner.test(as.data.frame(t(dat)))        # H0: equal variances
    f   <- friedman.test(as.matrix(dat))              # H0: no differences
    ff  <- frdAllPairsExactTest(as.matrix(dat))
    ffl <- multcompView::multcompLetters(ff$p.value)
    pp  <- powerMCTests(mu=colMeans(dat),n=length(dat[1,]),test="kwAllPairsConoverTest")
    IQ  <- apply(dat,2,IQR)

    cat(k,colMeans(dat),"\n",sep=",",append=T,file=paste("stat/",round,"/",round,"-means-roles.csv",sep=""))
    cat(k,ffl$Letters,"\n",sep=",",append=T,file=paste("stat/",round,"/",round,"-letters-roles.csv",sep=""))
    cat(k,ca$value,kf$value,kf$p.value,ft$p.value,
        mean(IQ),sd(IQ),f$p.value,pp$avepair,"\n",
        sep=",",append=T,file=paste("stat/",round,"/",round,"-statistics-roles.csv",sep=""))
    TT <<- paste("Distribution of scores for rol",k,"\n",
                 "Intraclass correlation coefficient: agreement",format(ca$value,digits=DIGITS),"\n",
                 "Inter-rater agreement",format(kf$value,digits=DIGITS),"(",format(kf$p.value,digits=DIGITS),")","\n",
                 "Homogeneity of Variances",format(ft$p.value,digits=DIGITS),"\n",
                 "Average interquartile range",format(mean(IQ),digits=DIGITS),"\n",
                 "Standard Deviation of interquartile ranges",format(sd(IQ),digits=DIGITS),"\n",
                 "Friedman test p-value",format(f$p.value,digits=DIGITS),"\n",
                 "Average power post hoc",format(pp$avepair,digits=DIGITS),"\n",sep=" ")
    par(mar = c(2,4,11,1))
    plot(ff,ylab="Scores",yaxp=c(-5,5,10),main=TT)
  }
  dev.off()

  pdf(width=9,height=9,file=paste("fig/",round,"/",round,"-means-roles.pdf",sep=""))
   aux <- read.csv(paste("stat/",round,"/",round,"-means-roles.csv",sep=""))
   row.names(aux) <- aux[,1]
   aux <- aux[,c(-1,-length(names(aux)))]
   heatmap.2(t(as.matrix(aux)),scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
            trace="none",tracecol="black",margins=c(10,10),
            col=heat.colors(11,rev=T),
            main="Mean Scores Roles")
  dev.off()

  # i experts
  # j barriers
  # k roles
  # l keywords
  for (l in colnames(KEYWORDS))
  {
    pdf(width=9,height=9,file=paste("fig/",round,"/",round,"-diff-keyword-",l,".pdf",sep=""))
    cat("barrier,role",levels(KEYWORDS[,l]),"p.value\n",sep=",",file=paste("stat/",round,"/",round,"-diff-keyword-",l,".csv",sep=""))
    cat("barrier",ROLES,"\n",sep=",",file=paste("stat/",round,"/",round,"-pvalue-keyword-",l,".csv",sep=""))
    for (j in row.names(get(paste(round,"_",i,sep=""))))
    {
      aux <- get(paste(round,"_",j,sep=""))
      aux <- cbind(aux,KEYWORDS)

      cat(j,",",sep="",append=T,file=paste("stat/",round,"/",round,"-pvalue-keyword-",l,".csv",sep=""))
      for (k in colnames(get(paste(round,"_",i,sep=""))))
      {
        cat(j,k,tapply(aux[,k],aux[,l],mean),kruskal.test(reformulate(l,k),data=aux)$p.value,"\n",
            append=T,sep=",",file=paste("stat/",round,"/",round,"-diff-keyword-",l,".csv",sep=""))
        cat(kruskal.test(reformulate(l,k),data=aux)$p.value,",",
            sep="",append=T,file=paste("stat/",round,"/",round,"-pvalue-keyword-",l,".csv",sep=""))
        boxplot(reformulate(l,k),data=aux,notch=F,
                ylab="Scores",yaxp=c(-5,5,10),ylim=c(-5,5),xlab="",
                col=c("coral","orchid"),
                main=paste("Score distributions for barrier",j,
                           "and role",k,"\n per keyword",l,sep=" "))
      }
      cat("\n",append=T,file=paste("stat/",round,"/",round,"-pvalue-keyword-",l,".csv",sep=""))
    }
    dev.off()
  }
}

## Function that compares the results of two rounds.
## Produce: boxplot and barplots with the different scores
##          tables with the differences in the scores
## experts: list of strings with the experts IDs. The scores of the experts are in variables with this name
## I: integer with the first round
## E: integer with the second round
# i experts
# j barriers
# k roles
DIFF <- function(experts,I,E)
{
  # Iteration per barrier category
  pdf(width=9,height=8,file=paste("fig/diff",I,"-",E,"-boxplot.pdf",sep=""))
  for (j in row.names(get(paste(I,"_",experts[1],sep=""))))
  {
    f <- as.data.frame(get(paste(I,"_",j,sep="")))
    s <- as.data.frame(get(paste(E,"_",j,sep="")))

    par(xpd=TRUE,mar=c(2, 4, 3, 1))
    boxplot(f$Prosumer  ,s$Prosumer  ,  f$DER       ,s$DER       ,
            f$Aggregator,s$Aggregator,  f$ESCO      ,s$ESCO      ,
            f$BRP       ,s$BRP       ,  f$Supplier  ,s$Supplier  ,
            f$DSO       ,s$DSO       ,  f$TSO       ,s$TSO       ,
            at = c(1,2,    4,5,  7,8,   10,11,
                   13,14, 16,17, 19,20, 22,23),
            ylab="Scores",yaxp=c(-5,5,10),ylim=c(-5,5),boxwex=0.4,xaxt="n",
            col=c("coral","orchid"),
            main=paste("Score distributions for",j,sep=" "))

    axis(1,at = seq(1.5 , length(names(f))*3 , 3),
         labels = names(f),
         tick=FALSE , cex=0.3)

    legend("topright", legend = c(paste("Round",I),paste("Round",E)),
       col=c("coral","orchid"),
       pch = 15, bty = "n", pt.cex = 3, cex = 1.2,  horiz = F, inset = c(0,-0.1))
  }
  dev.off()

  pdf(width=9,height=8,file=paste("fig/diff",I,"-",E,"-barplot.pdf",sep=""))
  for (j in row.names(get(paste(I,"_",experts[1],sep=""))))
  {
    f <- as.data.frame(get(paste(I,"_",j,sep="")))
    s <- as.data.frame(get(paste(E,"_",j,sep="")))

    dat <- data.frame()
    dst <- data.frame()
    for (k in ROLES)
    {
      if(!(k %in% colnames(f))) { f[,k] <- rep(NA,length(f[,1]))}
      if(!(k %in% colnames(s))) { s[,k] <- rep(NA,length(s[,1]))}
      dat <- rbind(dat,t(c( mean(f[,k]), mean(s[,k]))))
      dst <- rbind(dst,t(c( sd(f[,k]),   sd(s[,k]))))
    }

    rownames(dat) <- ROLES
    rownames(dst) <- ROLES
    names(dat) <- c(paste("Round",I),paste("Round",E))
    names(dst) <- c(paste("Round",I,"σ"),paste("Round",E,"σ"))

    par(xpd=TRUE,mar=c(2, 4, 3, 1))
    bp <- barplot(t(dat), beside = T,
            ylab="Scores",yaxp=c(-5,5,10),ylim=c(-5,5),xlab="",
            col=c("coral","orchid"),
            main=paste("Mean scores for",j,sep=" "))
    error.bar(bp,t(dat),t(dst))
    legend("topright", legend = c(paste("Round",I),paste("Round",E)),
           col=c("coral","orchid"),
           pch = 15, bty = "n", pt.cex = 3, cex = 1.2,  horiz = F, inset = c(0,-0.1))

    barplot(t(dat[paste("Round",I)]-dat[paste("Round",E)]),
            ylab="Scores",yaxp=c(-5,5,10),ylim=c(-5,5),xlab="",
            col="coral",main=paste("Differences between mean scores of\n",
                                    paste("round",I),"and",paste("round",E),
                                    "for barrier",j,sep=" "))

    write.table(t(dat),sep=",",file=paste("stat/diff",I,"-",E,"-diff-roles.csv",sep=""))
    write.table(t(dst),sep=",",file=paste("stat/diff",I,"-",E,"-diff-roles.csv",sep=""),append = T,col.names=F)
    write.table(t(dat[paste("Round",I)]-dat[paste("Round",E)]),sep=",",
                file=paste("stat/diff",I,"-",E,"-diff-roles.csv",sep=""),append = T,
                col.names=F,row.names=paste(paste("Round",I),"-",paste("Round",E)))
  }
  dev.off()
}

CATEGORIES <- function(round)
{
  assign(paste(round,"_","Lifestyles",sep=""),(get(paste(round,"_","Adoption",sep=""))+
                                                 get(paste(round,"_","Ideology & involvement",sep=""))+
                                                 get(paste(round,"_","Disillusion",sep=""))+
                                                 get(paste(round,"_","Reluctance and lack of trust",sep=""))+
                                                 get(paste(round,"_","Perceived Usefulness",sep=""))+
                                                 get(paste(round,"_","Ignorance / lack of expertise",sep="")))/6,pos=1)
  assign(paste(round,"_","Administration",sep=""),(get(paste(round,"_","Charging/cost rules",sep=""))+
                                                     get(paste(round,"_","Decision Making",sep=""))+
                                                     get(paste(round,"_","Policy/Regulatory Incentives",sep=""))+
                                                     get(paste(round,"_","Lack of regulation",sep=""))+
                                                     get(paste(round,"_","Market restrictions",sep="")))/5,pos=1)
  assign(paste(round,"_","Technical",sep=""),(get(paste(round,"_","System",sep=""))+
                                                get(paste(round,"_","Algorithms",sep=""))+
                                                get(paste(round,"_","Maturity",sep=""))+
                                                get(paste(round,"_","Deployment",sep=""))+
                                                get(paste(round,"_","Comunication",sep="")))/5,pos=1)
  assign(paste(round,"_","Trust",sep=""),(get(paste(round,"_","Security",sep=""))+
                                            get(paste(round,"_","Privacy",sep=""))+
                                            get(paste(round,"_","Stakeholders Cooperation",sep=""))+
                                            get(paste(round,"_","Communication strategy",sep=""))+
                                            get(paste(round,"_","Emerging Technologies",sep="")))/5,pos=1)
  assign(paste(round,"_","Costs",sep=""),(get(paste(round,"_","Investment",sep=""))+
                                            get(paste(round,"_","Pricing",sep=""))+
                                            get(paste(round,"_","Margin",sep="")))/3,pos=1)
  assign(paste(round,"_","Standardization",sep=""),(get(paste(round,"_","Technology & Market Diversity",sep=""))+
                                                      get(paste(round,"_","Roles & Objectives",sep=""))+
                                                      get(paste(round,"_","Business models",sep=""))+
                                                      get(paste(round,"_","Interoperability",sep="")))/4,pos=1)
}


## Calculate the critical set of the Use Cases
## MEANUC average scores of the barriers of the actors that participate in each use cas
## CONCUC concatenation of the barriers of the actors that participate in each use case
## round round of scores used to calculate the critical set
CRITICAL <- function(round)
{
  if (!dir.exists(paste("stat/",round,sep=""))) dir.create(paste("stat/",round,sep=""),recursive=T)
  if (!dir.exists(paste("fig/", round,sep=""))) dir.create(paste("fig/", round,sep=""),recursive=T)

  MEANUC <<- matrix(nrow = length(USECASES), ncol = length(BARRIERS), dimnames = list(USECASES,BARRIERS))
  CONCUC <<- matrix(nrow = length(USECASES), ncol = length(BARRIERS), dimnames = list(USECASES,BARRIERS))
  for (i in row.names(UCROLES))
  {
    mu  <- 0
    max <- 0
    for (j in colnames(UCROLES))
    {
      mu  <- mu  + UCROLES[i,j]*colMeans(get(paste(round,"_",j,sep="")))
      max <- max + UCROLES[i,j]*CRITSET[j,]
    }
    MEANUC[i,] <<- mu
    CONCUC[i,] <<- max
  }

  pdf(width=9,height=10,file = paste("fig/",round,"/",round,"-UC-heatmaps-average.pdf",sep=""))
    aux <- t(apply(MEANUC,1,rank))
    colnames(aux) <- BARRIERS
    heatmap.2(MEANUC,scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
              trace="none",tracecol="black",margins=c(10,10),
              col=heat.colors(11,rev=T),
              main="Mean Scores")
    heatmap.2(aux,scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
              trace="none",tracecol="black",margins=c(10,10),density.info="none",
              col=rainbow(length(BARRIERS)),main="Ordered Scores")
    write.csv(aux,file = "BORRAME-av.csv")
  dev.off()

  pdf(width=9,height=10,file = paste("fig/",round,"/",round,"-UC-heatmaps-concatenation.pdf",sep=""))
    aux <- t(apply(CONCUC,1,rank))
    colnames(aux) <- BARRIERS
    heatmap.2(CONCUC,scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
              trace="none",tracecol="black",margins=c(10,10),
              col=heat.colors(11,rev=T),
              main="Mean Scores")
    heatmap.2(aux,scale="none",symm=F,Rowv=F,Colv=F,dendrogram="none",
              trace="none",tracecol="black",margins=c(10,10),density.info="none",
              col=rainbow(length(BARRIERS)),main="Ordered Scores")
    write.csv(aux,file = "BORRAME-CO.csv")
  dev.off()

#  MEANUC <- cbind(MEANUC,BARRIERS[apply(MEANUC, 1, which.max)])
#  CONCUC <- cbind(CONCUC,BARRIERS[apply(CONCUC, 1, which.max)])
  write.csv(as.matrix(MEANUC),file=paste("stat/",round,"/",round,"-UC-average.csv",sep=""))
  write.csv(as.matrix(CONCUC),file=paste("stat/",round,"/",round,"-UC-concatenation.csv",sep=""))
}

ASSESSMENT(1,EXPERTS)
ASSESSMENT(2,EXPERTS)
ASSESSMENT(3,EXPERTS)

DIFF(EXPERTS,1,2)
DIFF(EXPERTS,2,3)

CATEGORIES(1)
CATEGORIES(2)
CATEGORIES(3)

CRITICAL(1)
CRITICAL(2)
CRITICAL(3)
