function supplementary_figure_6()
% Generate supplementary figure 6.
% Plot figure panel with overview of clinical, demographic, pathological
% and molecular information for the CRC patients of the NI240 phase III
% clinical trial. Each column represents a patient and each row color-codes
% a feature. Missing data are shown in white. Cell-in-cell (CIC) derived
% features include estimates of absence (CIC=0) or presence (CIC>0) of CIC
% events from tissue microarray sections prepared from 'tumour', 'invasive
% front' and 'normal' tissue.

if exist('figures_and_tables', 'dir')==0
    mkdir('figures_and_tables');
end

% Load data
data = get_data();

% Select columns to use in heatmaps
tumour_proteins = {'T_TRAIL', 'T_DR4', 'T_DR5', 'T_FLIP', 'T_CASP8'};
proteins = {'TRAIL', 'DR4', 'DR5', 'FLIP', 'CASP8'};

% Initialize HeatmapCovariatePlot
p = HCP.HeatmapCovariatePlot(data,...
    'font_name', 'arial',...
    'font_size', 4,......
    'subblock_title_margin', 5,...
    'left_labels_space', 20,...
    'covariate_row_margin', 1,...
    'covariate_legend_entry_width', 12,...
    'covariate_legend_text_margin', 1,...
    'heatmap_labels_font_size', 4,...
    'enable_sample_lines', false);

b0 = p.blocks.AddSubBlock('title', 'CIC events');
b0.AddCovariateRow('cic_events_class_tumour',...
    'cmap', [[31,120,180]./255; [227,26,28]./255],...
    'title', 'Tumour tissue');
b0.AddCovariateRow('cic_events_class_invasive_front',...
    'cmap', [[31,120,180]./255; [227,26,28]./255],...
    'title', 'Invasive front tissue');
b0.AddCovariateRow('cic_events_class_normal',...
    'cmap', [[31,120,180]./255; [227,26,28]./255],...
    'title', 'Normal tissue');

% Add sub-block with annotations for clinical-demographic patient features
b1 = p.blocks.AddSubBlock('title', 'Clinical');
b1.AddCovariateRow('stage',...
    'cmap', brewermap(numel(categories(data.stage)), 'Reds'),...
    'title', 'Stage');
b1.AddCovariateRow('T_stage',...
    'cmap', brewermap(numel(categories(data.T_stage)), 'Reds'),...
    'title', 'T stage');
b1.AddCovariateRow('N_stage',...
    'cmap', brewermap(numel(categories(data.N_stage)), 'Reds'),...
    'title', 'N stage');
b1.AddCovariateRow('age_bin',...
    'cmap', [[107,174,214]./255; [33,113,181]./255; [8,48,107]./255],...
    'title', 'Age',...
    'unit', '[years]');
b1.AddCovariateRow('sex',...
    'cmap', [107, 174, 214; 250, 159, 181]./255,...
    'title', 'Sex');
b1.AddCovariateRow('lymphovascular_invasion',...
    'cmap', brewermap(numel(categories(data.lymphovascular_invasion)), 'Set1'),...
    'title', 'LVI');
b1.AddCovariateRow('type_of_surgery',...
    'cmap', brewermap(numel(categories(data.type_of_surgery)), 'Set1'),...
    'title', 'Surgery');
b1.AddCovariateRow('tumour_site',...
    'cmap', [[166,206,227]./255; [31,120,180]./255; [255,127,0]./255;...
    [77, 175, 74]./255],...
    'title', 'Location');
b1.AddCovariateRow('cancer_type',...
    'cmap', [[31,120,180]./255; [255,127,0]./255],...
    'title', 'Cancer type');
b1.AddCovariateRow('radiotherapy',...
    'cmap', [[31,120,180]./255; [0.5,0.5,0.5]; ],...
    'title', 'Radiotherapy');

% Add sub-block with annotations for treatment
b2 = p.blocks.AddSubBlock('title', 'Trial treatment');
b2.AddCovariateRow('treatment',...
    'cmap', [[0.5,0.5,0.5]; [31,120,180]./255],...
    'title', 'Treatment');
b2.AddCovariateRow('chemotherapy_cycles_bin',...
    'cmap', [[0.5, 0.5, 0.5]; [166,206,227]./255; [31,120,180]./255],...
    'title', 'Chemotherapy cycles',...
    'unit', '[#]');
b2.AddCovariateRow('chemotherapy_within_6weeks_of_surgery',...
    'cmap', [[0.5, 0.5, 0.5]; [31,120,180]./255; [227,26,28]./255],...
    'title', '<=6 weeks of surgery');
b2.AddCovariateRow('chemotherapy_at_2weekly_intervals',...
    'cmap', [[0.5, 0.5, 0.5]; [31,120,180]./255; [227,26,28]./255],...
    'title', 'Two-weekly intervals');
b2.AddCovariateRow('chemotherapy_number_cycles_delayed',...
    'cmap', [[0.5, 0.5, 0.5]; [31,120,180]./255; [227,26,28]./255],...
    'title', 'Delayed cycles',...
    'unit', '[#]');

% Add sub-block with annotations for survival
b3 = p.blocks.AddSubBlock('title', 'Survival');
b3.AddCovariateRow('DSS_months',...
    'cmap', brewermap(100, 'Blues'),...
    'clim', [0, 120],...
    'title', 'DSS',...
    'unit', '[months]');
b3.AddCovariateRow('DSS_event',...
    'cmap', brewermap(numel(categories(data.DSS_event)), 'Set1'),...
    'title', 'Death by CRC');
b3.AddCovariateRow('DFS_months',...
    'cmap', brewermap(100, 'Blues'),...
    'clim', [0, 120],...
    'title', 'DFS',...
    'unit', '[months]');
b3.AddCovariateRow('DFS_event',...
    'cmap', brewermap(numel(categories(data.DFS_event)), 'Set1'),...
    'title', 'Relapse');

% Add sub-block for proteins in tumour tissue
b3 = p.blocks.AddSubBlock('title', 'T proteins');

b3.AddHeatmap(tumour_proteins,...
    'clim', [0, 12],...
    'title', 'Expression [a.u.]',...
    'standardize', false,...
    'cmap', brewermap(12, 'YlOrRd'),...
    'show_sample_dendrogram', false,...
    'show_feature_dendrogram', false,...
    'height_per_feature', 2,...
    'show_feature_labels', true,...
    'show_sample_labels', false,...
    'bottom_labels_space', 10,...
    'feature_cluster_cmap', brewermap(5, 'Set1'));

% Add sub-block for proteins in tumour tissue
b4 = p.blocks.AddSubBlock('title', '(T - N) proteins');
b4.AddHeatmap(proteins,...
    'clim', [-12, 12],...
    'title', 'Expression [a.u.]',...
    'standardize', false,...
    'cmap', brewermap(12, 'RdYlBu'),...
    'show_sample_dendrogram', false,...
    'show_feature_dendrogram', false,...
    'height_per_feature', 2,...
    'show_feature_labels', true,...
    'show_sample_labels', false,...
    'bottom_labels_space', 10,...
    'xlabel', 'Patients');

% Plot
p.Plot();

% Export figure as pdf
p.Export(fullfile('figures_and_tables', 'supplementary_figure_6.pdf'),...
    '-w185',...
    '-h210',...
    '-rx');
end

function data = get_data()
% Get data, as generated by the python function
% 'data_util.export_data_for_supplementary_figure_6()', and prepare data
% for plotting with the 'HCP' toolbox.

data = readtable(...
    fullfile('processed_data', 'clinical_aggregated_cic_proteins.csv'));

data.cic_events_class_tumour = categorical(data.cic_events_class_tumour,...
    {'CIC=0', 'CIC>0'});
data.cic_events_class_invasive_front = categorical(data.cic_events_class_invasive_front,...
    {'CIC=0', 'CIC>0'});
data.cic_events_class_normal = categorical(data.cic_events_class_normal,...
    {'CIC=0', 'CIC>0'});

data.TRAIL = data.protein_median_TRAIL_tumour - data.protein_median_TRAIL_normal;
data.DR4 = data.protein_median_DR4_tumour - data.protein_median_DR4_normal;
data.DR5 = data.protein_median_DR5_tumour - data.protein_median_DR5_normal;
data.FLIP = data.protein_median_FLIP_tumour - data.protein_median_FLIP_normal;
data.CASP8 = data.protein_median_CASP8_tumour - data.protein_median_CASP8_normal;

data.Properties.VariableNames{'protein_median_TRAIL_tumour'} = 'T_TRAIL';
data.Properties.VariableNames{'protein_median_DR4_tumour'} = 'T_DR4';
data.Properties.VariableNames{'protein_median_DR5_tumour'} = 'T_DR5';
data.Properties.VariableNames{'protein_median_FLIP_tumour'} = 'T_FLIP';
data.Properties.VariableNames{'protein_median_CASP8_tumour'} = 'T_CASP8';

data = sortrows(data,...
    {'cic_events_class_tumour', 'cic_events_class_invasive_front',...
    'cic_events_class_normal', 'FLIP', 'DR4','TRAIL', 'stage',...
    'tumour_site', 'age_bin'},...
    'ascend');

data.stage = categorical(data.stage, [2, 3]);
data.detailed_stage = categorical(data.detailed_stage,...
    {'1', '2A', '2B', '3A', '3B', '3C'});
data.T_stage = categorical(data.T_stage, {'1-2', '3', '4'});
data.N_stage = categorical(data.N_stage, {'0', '1', '2-3'});
data.grade = categorical(data.grade, [1, 2, 3]);
data.type_of_surgery = categorical(data.type_of_surgery);

data.radiotherapy = categorical(data.radiotherapy,...
    {'radiotherapy', 'no-radiotherapy'});

data.received_randomised_treatment = categorical(...
    data.received_randomised_treatment, {'randomized', 'not_randomized'});
data.treatment = categorical(data.treatment, {'observation', 'chemotherapy'});
data.chemotherapy_full_dose_given = categorical(data.chemotherapy_full_dose_given);

data.chemotherapy_within_6weeks_of_surgery(data.treatment=='observation') = {'none'};
data.chemotherapy_within_6weeks_of_surgery = categorical(...
    data.chemotherapy_within_6weeks_of_surgery,...
    {'none', 'within_6weeks', 'not_within_6weeks'});
data.chemotherapy_within_6weeks_of_surgery = renamecats(...
    data.chemotherapy_within_6weeks_of_surgery,...
    {'within_6weeks', 'not_within_6weeks'},...
    {'yes', 'no'});

data.chemotherapy_at_2weekly_intervals(data.treatment=='observation') = {'none'};
data.chemotherapy_at_2weekly_intervals = categorical(...
    data.chemotherapy_at_2weekly_intervals,...
    {'none', '2weekly_intervals', 'not_2weekly_intervals'});
data.chemotherapy_at_2weekly_intervals = renamecats(...
    data.chemotherapy_at_2weekly_intervals,...
    {'2weekly_intervals', 'not_2weekly_intervals'},...
    {'yes', 'no'});

data.chemotherapy_cycles_bin = categorical(data.chemotherapy_cycles_bin,...
    {'none', 'incomplete', 'full'});

data.chemotherapy_number_cycles_delayed = categorical(...
    data.chemotherapy_number_cycles_delayed);
data.chemotherapy_number_cycles_delayed(data.treatment=='observation') = 'none';
data.chemotherapy_number_cycles_delayed = mergecats(...
    data.chemotherapy_number_cycles_delayed, {'1', '2'}, '1-2');
data.chemotherapy_number_cycles_delayed = reordercats(...
    data.chemotherapy_number_cycles_delayed, {'none', '0', '1-2'});

data.age_bin = categorical(data.age_bin, {'under 60', '60 to 70', 'over 70'});
data.sex = categorical(data.sex, {'male', 'female'});
data.cancer_type = categorical(data.cancer_type);
data.tumour_site = categorical(data.tumour_site);
data.lymphovascular_invasion = categorical(data.lymphovascular_invasion);

data.DSS_event = categorical(data.DSS_event);
data.DSS_event = reordercats(...
    renamecats(data.DSS_event, {'0', '1'}, {'no', 'yes'}), {'yes', 'no'});
data.DFS_event = categorical(data.DFS_event);
data.DFS_event = reordercats(...
    renamecats(data.DFS_event, {'0', '1'}, {'no', 'yes'}), {'yes', 'no'});
end
