function TCGACutaneousMelanomaCancer()

if exist(fullfile('+HCP', '+examples', 'figures'), 'dir')==0
    mkdir(fullfile('+HCP', '+examples', 'figures'));
end

% Load data
data = HCP.examples.get.TCGACutaneousMelanomaCancer();

% Select columns to use in heatmaps
immune_cells = data.Properties.VariableNames(24:49);
proteins = data.Properties.VariableNames(50:end);

% Initialize HeatmapCovariatePlot
p = HCP.HeatmapCovariatePlot(data,...
    'font_name', 'arial',...
    'subblock_title_margin', 5,...
    'left_labels_space', 25,...
    'covariate_row_margin', 1,...
    'heatmap_labels_font_size', 4);

% Add sub-block with annotations for clinical-demographic patient features
b1 = p.blocks.AddSubBlock('title', 'Clinical');
b1.AddCovariateRow('stage',...
    'title', 'Stage');
b1.AddCovariateRow('age',...
    'clim', [18, 100],...
    'title', 'Age',...
    'unit', '[years]');
b1.AddCovariateRow('sex',...
    'cmap', [107, 174, 214; 250, 159, 181]./255,...
    'title', 'Sex');
b1.AddCovariateRow('overall_survival',...
    'cmap', brewermap(100, 'Blues'),...
    'clim', [0, 60],...
    'title', 'Overall survival',...
    'unit', '[months]');
b1.AddCovariateRow('death',...
    'cmap', [0, 0, 0; 0.5, 0.5, 0.5],...
    'title', 'Death');

% Add sub-block with annotations for pathological features specific to
% cutaneous melanoma
b2 = p.blocks.AddSubBlock('title', 'Pathological');
b2.AddCovariateRow('uv_signature',...
    'cmap', [224, 130, 20; 27, 120, 55]./255,...
    'title', 'UV signature');
b2.AddCovariateRow('uv_rate',...
    'clim', [0.0, 1.0],...
    'title', 'UV rate');
b2.AddCovariateRow('pigment_score',...
    'cmap', brewermap(numel(categories(data.pigment_score)), 'Reds'),...
    'title', 'Pigment score');
b2.AddCovariateRow('breslow',...
    'title', 'Breslow',...
    'clim', [0, 75]);
b2.AddCovariateRow('ulceration',...
    'title', 'Ulceration');

% Add sub-block with annotations for mutational status features
b3 = p.blocks.AddSubBlock('title', 'Mutations');
b3.AddCovariateRow('total_mutations',...
    'clim', [0, 2500],...
    'title', 'Total mutations');
b3.AddCovariateRow('mutation_subtypes',...
    'cmap', brewermap(numel(categories(data.mutation_subtypes)), 'Spectral'),...
    'title', 'Mutation subtype');
cmap_mutations = [77, 175, 74; 152, 78, 163]./255;
b3.AddCovariateRow('BRAF_mut',...
    'cmap', cmap_mutations,...
    'title', 'BRAF');
b3.AddCovariateRow('KRAS_mut',...
    'cmap', cmap_mutations,...
    'title', 'KRAS');
b3.AddCovariateRow('NRAS_mut',...
    'cmap', cmap_mutations,...
    'title', 'NRAS');
b3.AddCovariateRow('HRAS_mut',...
    'cmap', cmap_mutations,...
    'title', 'HRAS');
b3.AddCovariateRow('NF1_mut',...
    'cmap', cmap_mutations,...
    'title', 'NF1');

% Add sub-block with annotations for copy number alteration features
cmap_cnas = [244, 109, 67; 254, 224, 144; 116, 173, 209]./255;
b4 = p.blocks.AddSubBlock('title', 'CNA');
b4.AddCovariateRow('BRAF_cna',...
    'cmap', cmap_cnas,...
    'title', 'BRAF');
b4.AddCovariateRow('KRAS_cna',...
    'cmap', cmap_cnas,...
    'title', 'KRAS');
b4.AddCovariateRow('NRAS_cna',...
    'cmap', cmap_cnas,...
    'title', 'NRAS');
b4.AddCovariateRow('HRAS_cna',...
    'cmap', cmap_cnas,...
    'title', 'HRAS');
b4.AddCovariateRow('NF1_cna',...
    'cmap', cmap_cnas,...
    'title', 'NF1');

% Add sub-block for heatmap for immune scores
b5 = p.blocks.AddSubBlock('title', 'Immune profiles');
b5.AddHeatmap(immune_cells,...
    'clim', [0, 1],...
    'show_sample_dendrogram', false,...
    'show_feature_dendrogram', false,...
    'show_sample_labels', false);

% Add sub-block for heatmap with clustering for protein expression
b6 = p.blocks.AddSubBlock('title', 'Protein profiles');
b6.AddHeatmap(proteins,...
    'standardize', true,...
    'show_sample_dendrogram', true,...
    'show_feature_dendrogram', true,...
    'linkage', 'ward',...
    'dendrogram_linewidth', 0.5,...
    'n_sample_clusters', 3,...
    'n_feature_clusters', 5,...
    'show_feature_labels', false,...
    'height_per_feature', 0.25);

% Plot
p.Plot();

% Export figure as pdf
p.Export(fullfile('+HCP', '+examples', 'figures',...
    'tcga_cutaneous_melanoma_cancer.pdf'),...
    '-w180',...
    '-rx');
end
