// Copyright © 2017 Will Rowe <w.p.m.rowe@gmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

package cmd

import (
	"fmt"
	"log"
	"os"
	"strings"

	"github.com/spf13/cobra"
	"github.com/will-rowe/groot/src/misc"
	"github.com/will-rowe/groot/src/reporting"
	"github.com/will-rowe/groot/src/version"
)

// the command line arguments
var (
	bamFile   *string  // a BAM file to generate report from
	covCutoff *float64 // breadth of coverage theshold
	lowCov    *bool    // reports ARGs which don't have 5' or 3' coverage
)

// the report command (used by cobra)
var reportCmd = &cobra.Command{
	Use:   "report",
	Short: "Generate a report from the output of groot align",
	Long: `Generate a report from the output of groot align.

	This will report gene, read count, gene length, coverage cigar to STDOUT as tab separated values.

	Coverage cigar is present to help debug and indicates if the reference gene is covered (M) or not (D).`,
	Run: func(cmd *cobra.Command, args []string) {
		runReport()
	},
}

/*
  A function to initialise the command line arguments
*/
func init() {
	RootCmd.AddCommand(reportCmd)
	bamFile = reportCmd.Flags().String("bamFile", "", "BAM file generated by groot alignment (will use STDIN if not provided)")
	covCutoff = reportCmd.Flags().Float64P("covCutoff", "c", 0.97, "coverage cutoff for reporting ARGs")
	lowCov = reportCmd.Flags().Bool("lowCov", false, "reports ARGs which don't have 5' or 3' coverage (overrides -c option)")
}

/*
  A function to check user supplied parameters
*/
func reportParamCheck() error {
	// if no BAM files provided, check STDIN
	if *bamFile == "" {
		stat, err := os.Stdin.Stat()
		if err != nil {
			return fmt.Errorf("error with STDIN")
		}
		if (stat.Mode() & os.ModeNamedPipe) == 0 {
			return fmt.Errorf("no STDIN found")
		}
		log.Printf("\tBAM file: using STDIN")
		// check the provided BAM files
	} else {
		if _, err := os.Stat(*bamFile); err != nil {
			if os.IsNotExist(err) {
				return fmt.Errorf("BAM file does not exist: %v", *bamFile)
			} else {
				return fmt.Errorf("can't access BAM file (check permissions): %v", *bamFile)
			}
		}
		splitFilename := strings.Split(*bamFile, ".")
		if splitFilename[len(splitFilename)-1] != "bam" {
			return fmt.Errorf("the BAM file does not have a `.bam` extension: %v", *bamFile)
		}
		log.Printf("\tBAM file: %v", *bamFile)
	}
	if *covCutoff > 1.0 {
		return fmt.Errorf("supplied coverage cutoff exceeds 1.0 (100%%): %v", *covCutoff)
	}
	return nil
}

/*
  The main function for the report sub-command
*/
func runReport() {
	logFH := misc.StartLogging(*logFile)
	defer logFH.Close()
	log.SetOutput(logFH)
	log.Printf("i am groot (version %s)", version.GetVersion())
	log.Printf("starting the report subcommand")
	// check the supplied files and then log some stuff
	log.Printf("checking parameters...")
	misc.ErrorCheck(reportParamCheck())
	log.Printf("\tcoverage cutoff: %.2f", *covCutoff)
	log.Printf("\tprocessors: %d", *proc)
	bamReader := reporting.NewBAMreader()
	if *bamFile != "" {
		bamReader.InputFile = *bamFile
	}
	// lowCov overrides user supplied covCutoff
	if *lowCov == true {
		bamReader.LowCov = true
		bamReader.CoverageCutoff = 0.97
	} else {
		bamReader.CoverageCutoff = *covCutoff
	}
	bamReader.Run()
	log.Println("finished")

} // end of report main function
