package edu.uiuc.ncsa.myproxy.oa4mp.oauth2.loader;

import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.OA2SE;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.OA2ServiceTransaction;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.claims.BasicClaimsSourceImpl;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.claims.LDAPClaimsSource;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.cm.CMConfig;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.cm.CMConfigs;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.cm.ClientManagementConstants;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.json.JSONStoreProviders;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.json.MultiJSONStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.storage.*;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.storage.clients.OA2ClientConverter;
import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.storage.clients.OA2ClientProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.ClientApprovalProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.OA4MPConfigTags;
import edu.uiuc.ncsa.myproxy.oa4mp.server.ServiceConstantKeys;
import edu.uiuc.ncsa.myproxy.oa4mp.server.ServiceEnvironmentImpl;
import edu.uiuc.ncsa.myproxy.oa4mp.server.admin.adminClient.AdminClientStore;
import edu.uiuc.ncsa.myproxy.oa4mp.server.admin.adminClient.AdminClientStoreProviders;
import edu.uiuc.ncsa.myproxy.oa4mp.server.admin.adminClient.MultiDSAdminClientStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.admin.transactions.DSTransactionProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.admin.transactions.OA4MPIdentifierProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.servlet.AbstractConfigurationLoader;
import edu.uiuc.ncsa.myproxy.oa4mp.server.storage.MultiDSClientApprovalStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.storage.MultiDSClientStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.storage.filestore.DSFSClientApprovalStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.storage.filestore.DSFSClientStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.storage.sql.provider.DSSQLClientApprovalStoreProvider;
import edu.uiuc.ncsa.myproxy.oa4mp.server.util.ClientApprovalMemoryStore;
import edu.uiuc.ncsa.myproxy.oa4mp.server.util.ClientApproverConverter;
import edu.uiuc.ncsa.qdl.config.QDLConfigurationConstants;
import edu.uiuc.ncsa.qdl.config.QDLConfigurationLoader;
import edu.uiuc.ncsa.qdl.config.QDLEnvironment;
import edu.uiuc.ncsa.security.core.IdentifiableProvider;
import edu.uiuc.ncsa.security.core.Identifier;
import edu.uiuc.ncsa.security.core.configuration.Configurations;
import edu.uiuc.ncsa.security.core.configuration.provider.CfgEvent;
import edu.uiuc.ncsa.security.core.configuration.provider.TypedProvider;
import edu.uiuc.ncsa.security.core.exceptions.GeneralException;
import edu.uiuc.ncsa.security.core.util.DebugUtil;
import edu.uiuc.ncsa.security.core.util.IdentifierProvider;
import edu.uiuc.ncsa.security.core.util.MyLoggingFacade;
import edu.uiuc.ncsa.security.delegation.server.issuers.AGIssuer;
import edu.uiuc.ncsa.security.delegation.server.issuers.ATIssuer;
import edu.uiuc.ncsa.security.delegation.server.issuers.PAIssuer;
import edu.uiuc.ncsa.security.delegation.server.storage.ClientApprovalStore;
import edu.uiuc.ncsa.security.delegation.server.storage.ClientStore;
import edu.uiuc.ncsa.security.delegation.storage.Client;
import edu.uiuc.ncsa.security.delegation.storage.ClientApprovalKeys;
import edu.uiuc.ncsa.security.delegation.storage.TransactionStore;
import edu.uiuc.ncsa.security.delegation.token.TokenForge;
import edu.uiuc.ncsa.security.oauth_2_0.OA2ConfigurationLoaderUtils;
import edu.uiuc.ncsa.security.oauth_2_0.OA2Constants;
import edu.uiuc.ncsa.security.oauth_2_0.OA2TokenForge;
import edu.uiuc.ncsa.security.oauth_2_0.server.AGI2;
import edu.uiuc.ncsa.security.oauth_2_0.server.ATI2;
import edu.uiuc.ncsa.security.oauth_2_0.server.PAI2;
import edu.uiuc.ncsa.security.oauth_2_0.server.claims.ClaimSource;
import edu.uiuc.ncsa.security.oauth_2_0.server.claims.ClaimSourceConfiguration;
import edu.uiuc.ncsa.security.oauth_2_0.server.config.LDAPConfiguration;
import edu.uiuc.ncsa.security.oauth_2_0.server.config.LDAPConfigurationUtil;
import edu.uiuc.ncsa.security.servlet.ServletDebugUtil;
import edu.uiuc.ncsa.security.storage.data.MapConverter;
import edu.uiuc.ncsa.security.storage.sql.ConnectionPool;
import edu.uiuc.ncsa.security.storage.sql.ConnectionPoolProvider;
import edu.uiuc.ncsa.security.util.jwk.JSONWebKeyUtil;
import edu.uiuc.ncsa.security.util.jwk.JSONWebKeys;
import org.apache.commons.configuration.tree.ConfigurationNode;

import javax.inject.Provider;
import java.io.File;
import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import static edu.uiuc.ncsa.myproxy.oa4mp.server.admin.transactions.OA4MPIdentifierProvider.TRANSACTION_ID;
import static edu.uiuc.ncsa.security.core.configuration.Configurations.*;
import static edu.uiuc.ncsa.security.oauth_2_0.OA2ConfigTags.*;
import static edu.uiuc.ncsa.security.oauth_2_0.OA2Constants.*;

/**
 * <p>Created by Jeff Gaynor<br>
 * on 9/23/13 at  1:50 PM
 */
public class OA2ConfigurationLoader<T extends ServiceEnvironmentImpl> extends AbstractConfigurationLoader<T> {
    /**
     * Default is 15 days. Internally the refresh lifetime (as all date-ish things) are in milliseconds
     * though the configuration file is assumed to be in seconds.
     */
    public long REFRESH_TOKEN_LIFETIME_DEFAULT = 15 * 24 * 3600 * 1000L;
    public int CLIENT_SECRET_LENGTH_DEFAULT = 258; //This is divisible by 3 and greater than 256, so when it is base64 encoded there will be no extra characters.

    public OA2ConfigurationLoader(ConfigurationNode node) {
        super(node);
    }

    public OA2ConfigurationLoader(ConfigurationNode node, MyLoggingFacade logger) {
        super(node, logger);
    }

    @Override
    public T createInstance() {
        try {
            initialize();
            T se = (T) new OA2SE(loggerProvider.get(),
                    getTransactionStoreProvider(),
                    getClientStoreProvider(),
                    getMaxAllowedNewClientRequests(),
                    getRTLifetime(),
                    getClientApprovalStoreProvider(),
                    getMyProxyFacadeProvider(),
                    getMailUtilProvider(),
                    getMP(),
                    getAGIProvider(),
                    getATIProvider(),
                    getPAIProvider(),
                    getTokenForgeProvider(),
                    getConstants(),
                    getAuthorizationServletConfig(),
                    getUsernameTransformer(),
                    getPingable(),
                    getMpp(),
                    getMacp(),
                    getClientSecretLength(),
                    getScopes(),
                    getClaimSource(),
                    getLdapConfiguration(),
                    isRefreshTokenEnabled(),
                    isTwoFactorSupportEnabled(),
                    getMaxClientRefreshTokenLifetime(),
                    getJSONWebKeys(),
                    getIssuer(),
                    // getMLDAP(),
                    isUtilServerEnabled(),
                    isOIDCEnabled(),
                    getMultiJSONStoreProvider(),
                    getCmConfigs(),
                    getQDLEnvironment(),
                    isScitokenEnabled(),
                    isRFC8693Enabled(),
                    isWLCGEnabled());

            if (getClaimSource() instanceof BasicClaimsSourceImpl) {
                ((BasicClaimsSourceImpl) getClaimSource()).setOa2SE((OA2SE) se);
            }
            return se;
        } catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            throw new GeneralException("Error: Could not create the runtime environment", e);
        }
    }

    protected QDLEnvironment getQDLEnvironment() {
        ConfigurationNode node = getFirstNode(cn, QDLConfigurationConstants.CONFIG_TAG_NAME);
        if (node == null) {
            return new QDLEnvironment();// no op. This is disabled.
        }
        // Note that the first argument is the name fo the file. In server mode this won't be available anyway
        // and is optional.
        QDLConfigurationLoader loader = new QDLConfigurationLoader("(none)", node,loggerProvider.get());
        return loader.load();
    }

    HashMap<String, String> constants;

    @Override
    public HashMap<String, String> getConstants() {
        if (constants == null) {
            constants = new HashMap<String, String>();
            // OAuth 1.0a callback constant. This is used to as a key for http request parameters
            constants.put(ServiceConstantKeys.CALLBACK_URI_KEY, REDIRECT_URI);
            constants.put(ServiceConstantKeys.TOKEN_KEY, AUTHORIZATION_CODE);
            constants.put(ServiceConstantKeys.FORM_ENCODING_KEY, FORM_ENCODING);
            constants.put(ServiceConstantKeys.CERT_REQUEST_KEY, CERT_REQ);
            constants.put(ServiceConstantKeys.CERT_LIFETIME_KEY, CERT_LIFETIME);
            constants.put(ServiceConstantKeys.CONSUMER_KEY, OA2Constants.CLIENT_ID);
        }
        return constants;
    }

    Boolean utilServerEnabled = null;

    protected Boolean isUtilServerEnabled() {
        if (utilServerEnabled == null) {
            try {
                utilServerEnabled = Boolean.parseBoolean(getFirstAttribute(cn, OA4MPConfigTags.ENABLE_UTIL_SERVLET));
            } catch (Throwable t) {
                // use default which is to enable. We let this be null to trigger pulling the value, if any, out of the
                // the configuration
                utilServerEnabled = Boolean.TRUE;
            }
        }
        return utilServerEnabled;
    }
    Boolean rfc8693Enabled = null;
    protected Boolean isRFC8693Enabled() {
           if (rfc8693Enabled == null) {
               try {
                   rfc8693Enabled = Boolean.parseBoolean(getFirstAttribute(cn, OA4MPConfigTags.ENABLE_RFC8693_SUPPORT));
               } catch (Throwable t) {
                   // use default which is to disabled. We let this be null to trigger pulling the value, if any, out of the
                   // the configuration
                   rfc8693Enabled = Boolean.TRUE;
               }
           }
           return rfc8693Enabled;
       }

    Boolean scitokenEnabled = null;

    protected Boolean isScitokenEnabled() {
        if (scitokenEnabled == null) {
            try {
                scitokenEnabled = Boolean.parseBoolean(getFirstAttribute(cn, OA4MPConfigTags.ENABLE_SCITOKEN_SUPPORT));
            } catch (Throwable t) {
                // use default which is to disabled. We let this be null to trigger pulling the value, if any, out of the
                // the configuration
                scitokenEnabled = Boolean.FALSE;
            }
        }
        return scitokenEnabled;
    }
    Boolean wlcgEnabled = null;

    protected Boolean isWLCGEnabled() {
        if (wlcgEnabled == null) {
            try {
                wlcgEnabled = Boolean.parseBoolean(getFirstAttribute(cn, OA4MPConfigTags.ENABLE_WLCG_SUPPORT));
            } catch (Throwable t) {
                // use default which is to disabled. We let this be null to trigger pulling the value, if any, out of the
                // the configuration
                wlcgEnabled = Boolean.FALSE;
            }
        }
        return wlcgEnabled;
    }
    protected CMConfigs createDefaultCMConfig() {
        CMConfigs cmConfigs = new CMConfigs();
        String serverAddress = getServiceAddress().toString();
        if (!serverAddress.endsWith("/")) {
            serverAddress = serverAddress + "/";
        }
        CMConfig tempCfg = new CMConfig(ClientManagementConstants.OA4MP_VALUE,
                URI.create(serverAddress + ClientManagementConstants.DEFAULT_OA4MP_ENDPOINT),
                true);
        cmConfigs.put(tempCfg);
        tempCfg = new CMConfig(ClientManagementConstants.RFC_7591_VALUE,
                URI.create(serverAddress + ClientManagementConstants.DEFAULT_RFC7591_ENDPOINT),
                true);
        cmConfigs.put(tempCfg);
        // NOTE there is no difference in endpoints for RFC 7591 and 7592! The question is if
        // the client management protocol endpoint also supports RFC 7592.
        tempCfg = new CMConfig(ClientManagementConstants.RFC_7592_VALUE,
                URI.create(serverAddress + ClientManagementConstants.DEFAULT_RFC7591_ENDPOINT),
                true);

        cmConfigs.put(tempCfg);
        return cmConfigs;
    }

    /*
    A typical entry we are parsing looks like this

     <clientManagement>
        <api protocol="rfc7951" enable="true" endpoint="oidc-cm"/>
        <api protocol="rfc7952" enable="true" endpoint="oidc-cm"/>
        <api protocol="oa4mp" enable="true"  url="https://foo.bar/oauth2/clients"/>
     </clientManagement>

     Note that EITHER the endpoint is given (and the full url is then constructed here)
     or the url is given. Giving the url has right of way. The configuration object
     only has URLs and they are resolved here from the server address.

     In this example, the endpoints for the RFCs are constructed but the native OA4MP endpoint
     is explicitly given. These ar eneeded since responses must include an actual address for
     clients to come to for future updates, etc. 
     */
    public CMConfigs getCmConfigs() {
        if (cmConfigs == null) {
            List kids = cn.getChildren(ClientManagementConstants.CLIENT_MANAGEMENT_TAG);
            if (kids == null || kids.isEmpty()) {
                cmConfigs = createDefaultCMConfig();
                return cmConfigs; // missing element (which is fine)  so jump out...
            }
            String serverAddress = getServiceAddress().toString();
            // need to loop through all kids.
            for (Object object : kids) {
                ConfigurationNode sn = (ConfigurationNode) object;
                if (sn.getName().equals(ClientManagementConstants.API_TAG)) {
                    try {
                        CMConfig cfg = CMConfigs.createConfigEntry(
                                getFirstAttribute(sn, ClientManagementConstants.PROTOCOL_ATTRIBUTE),
                                serverAddress,
                                getFirstAttribute(sn, ClientManagementConstants.ENDPOINT_ATTRIBUTE),
                                getFirstAttribute(sn, ClientManagementConstants.FULL_URL_ATTRIBUTE),
                                getFirstAttribute(sn, ClientManagementConstants.ENDPOINT_ATTRIBUTE)
                        );
                        cmConfigs.put(cfg);
                    } catch (Throwable t) {
                        ServletDebugUtil.warn(this, "error loading client management api entry \"" + t.getMessage() + "\"");
                    }
                }
            }
            if (cmConfigs.isEmpty()) {
                cmConfigs = createDefaultCMConfig();
                ServletDebugUtil.warn(this, "Warning: none of the entries in the client managment element parsed. Using defaults...");

            }

        }
        return cmConfigs;
    }

    CMConfigs cmConfigs;

    public MultiJSONStoreProvider getMultiJSONStoreProvider() {
        if (multiJSONStoreProvider == null) {

            multiJSONStoreProvider = new MultiJSONStoreProvider(cn,
                    isDefaultStoreDisabled(),
                    loggerProvider.get(),
                    null,
                    null);
            multiJSONStoreProvider.addListener(JSONStoreProviders.getJSMSP(cn));
            multiJSONStoreProvider.addListener(JSONStoreProviders.getJSFSP(cn));
            multiJSONStoreProvider.addListener(JSONStoreProviders.getMariaJS(cn, getMariaDBConnectionPoolProvider()));
            multiJSONStoreProvider.addListener(JSONStoreProviders.getMySQLJS(cn, getMySQLConnectionPoolProvider()));
            multiJSONStoreProvider.addListener(JSONStoreProviders.getPostgresJS(cn, getPgConnectionPoolProvider()));

        }
        return multiJSONStoreProvider;
    }

    protected MultiJSONStoreProvider multiJSONStoreProvider;


    protected MultiDSAdminClientStoreProvider macp;

    protected MultiDSAdminClientStoreProvider getMacp() {
        if (macp == null) {
            macp = new MultiDSAdminClientStoreProvider(cn,
                    isDefaultStoreDisabled(),
                    loggerProvider.get(),
                    null,
                    null,
                    AdminClientStoreProviders.getAdminClientProvider());
            macp.addListener(AdminClientStoreProviders.getACMP(cn));
            macp.addListener(AdminClientStoreProviders.getACFSP(cn));
            macp.addListener(AdminClientStoreProviders.getMariaACS(cn, getMariaDBConnectionPoolProvider()));
            macp.addListener(AdminClientStoreProviders.getMysqlACS(cn, getMySQLConnectionPoolProvider()));
            macp.addListener(AdminClientStoreProviders.getPostgresACS(cn, getPgConnectionPoolProvider()));
            AdminClientStore acs = (AdminClientStore) macp.get();
        }
        return macp;
    }

    protected JSONWebKeys getJSONWebKeys() {
        ConfigurationNode node = getFirstNode(cn, "JSONWebKey");
        if (node == null) {
            warn("Error: No signing keys in the configuration file. Signing is not available");
            //throw new IllegalStateException();
            return new JSONWebKeys(null);
        }
        String json = getNodeValue(node, "json", null); // if the whole thing is included
        JSONWebKeys keys = null;
        try {
            if (json != null) {
                keys = JSONWebKeyUtil.fromJSON(json);
            }
            String path = getNodeValue(node, "path", null); // points to a file that contains it all
            if (path != null) {
                keys = JSONWebKeyUtil.fromJSON(new File(path));
            }
        } catch (Throwable t) {
            throw new GeneralException("Error reading signing keys", t);
        }

        if (keys == null) {
            throw new IllegalStateException("Error: Could not load signing keys");
        }
        keys.setDefaultKeyID(getFirstAttribute(node, "defaultKeyID"));
        return keys;
    }

    @Override
    public Provider<AGIssuer> getAGIProvider() {
        if (agip == null) {
            return new Provider<AGIssuer>() {
                @Override
                public AGIssuer get() {
                    return new AGI2(getTokenForgeProvider().get(), getServiceAddress(), isOIDCEnabled());
                }
            };
        }
        return agip;
    }

    Provider<AGIssuer> agip = null;

    @Override
    public Provider<ClientApprovalStore> getClientApprovalStoreProvider() {
        return getCASP();
    }

    @Override
    public Provider<ClientStore> getClientStoreProvider() {
        return getCSP();
    }


    @Override
    protected MultiDSClientApprovalStoreProvider getCASP() {
        if (casp == null) {
            casp = new MultiDSClientApprovalStoreProvider(cn, isDefaultStoreDisabled(), loggerProvider.get());
            final ClientApprovalProvider caProvider = new ClientApprovalProvider();
            ClientApprovalKeys caKeys = new ClientApprovalKeys();
            caKeys.identifier("client_id");
            final ClientApproverConverter cp = new ClientApproverConverter(caKeys, caProvider);
            casp.addListener(new DSFSClientApprovalStoreProvider(cn, cp));
            casp.addListener(new DSSQLClientApprovalStoreProvider(cn, getMySQLConnectionPoolProvider(), OA4MPConfigTags.MYSQL_STORE, cp));
            casp.addListener(new DSSQLClientApprovalStoreProvider(cn, getMariaDBConnectionPoolProvider(), OA4MPConfigTags.MARIADB_STORE, cp));
            casp.addListener(new DSSQLClientApprovalStoreProvider(cn, getPgConnectionPoolProvider(), OA4MPConfigTags.POSTGRESQL_STORE, cp));

            casp.addListener(new TypedProvider<ClientApprovalStore>(cn, OA4MPConfigTags.MEMORY_STORE, OA4MPConfigTags.CLIENT_APPROVAL_STORE) {

                @Override
                public Object componentFound(CfgEvent configurationEvent) {
                    if (checkEvent(configurationEvent)) {
                        return get();
                    }
                    return null;
                }

                @Override
                public ClientApprovalStore get() {
                    return new ClientApprovalMemoryStore(caProvider, cp);
                }
            });
        }
        return casp;
    }

    public class OA4MP2TProvider extends DSTransactionProvider<OA2ServiceTransaction> {
        public OA4MP2TProvider(IdentifierProvider<Identifier> idProvider) {
            super(idProvider);
        }

        @Override
        public OA2ServiceTransaction get(boolean createNewIdentifier) {
            return new OA2ServiceTransaction(createNewId(createNewIdentifier));
        }
    }

    long rtLifetime = -1L;

    protected long getRTLifetime() {
        if (rtLifetime < 0) {
            String x = getFirstAttribute(cn, REFRESH_TOKEN_LIFETIME);
            // Fixes OAUTH-214
            if (x == null || x.length() == 0) {
                rtLifetime = REFRESH_TOKEN_LIFETIME_DEFAULT;
            } else {
                try {
                    rtLifetime = Long.parseLong(x) * 1000; // The configuration file has this in seconds. Internally this is ms.
                } catch (Throwable t) {
                    rtLifetime = REFRESH_TOKEN_LIFETIME_DEFAULT;
                }
            }
        }
        return rtLifetime;

    }

    String issuer = null;

    protected String getIssuer() {
        if (issuer == null) {
            String x = getFirstAttribute(cn, ISSUER);
            // Fixes OAUTH-214
            if (x == null || x.length() == 0) {
                return null;
            } else {
                issuer = x;
            }
        }
        return issuer;

    }

    long maxClientRefreshTokenLifetime = -1L;

    protected long getMaxClientRefreshTokenLifetime() {
        if (maxClientRefreshTokenLifetime < 0) {
            String x = getFirstAttribute(cn, MAX_CLIENT_REFRESH_TOKEN_LIFETIME);
            // Fixes OAUTH-214
            if (x == null || x.length() == 0) {
                maxClientRefreshTokenLifetime = 13 * 30 * 24 * 3600 * 1000L; // default of 13 months.
            } else {
                try {
                    maxClientRefreshTokenLifetime = Long.parseLong(x) * 1000; // The configuration file has this in seconds. Internally this is ms.
                } catch (Throwable t) {
                    maxClientRefreshTokenLifetime = 13 * 30 * 24 * 3600 * 1000L; // default of 13 months.
                }
            }
        }
        return maxClientRefreshTokenLifetime;
    }

    Boolean oidcEnabled = null;

    public boolean isOIDCEnabled() {
        if (oidcEnabled == null) {
            String x = getFirstAttribute(cn, OIDC_SUPPORT_ENABLED);
            if (x == null) {
                oidcEnabled = Boolean.TRUE; // default.
            } else {
                try {
                    oidcEnabled = Boolean.valueOf(x);
                } catch (Throwable t) {
                    info("COuld not parse OIDC enabled flag, setting default to true");
                    oidcEnabled = Boolean.TRUE;
                }
            }
        }
        return oidcEnabled;
    }


    public boolean isRefreshTokenEnabled() {
        if (refreshTokenEnabled == null) {
            String x = getFirstAttribute(cn, REFRESH_TOKEN_ENABLED);
            if (x == null) {
                refreshTokenEnabled = Boolean.FALSE;
            } else {
                try {
                    refreshTokenEnabled = Boolean.valueOf(x);
                } catch (Throwable t) {
                    info("Could not parse refresh token enabled attribute. Setting default to false.");
                    refreshTokenEnabled = Boolean.FALSE;
                }
            }
        }
        return refreshTokenEnabled;
    }

    Boolean twoFactorSupportEnabled = null;

    public boolean isTwoFactorSupportEnabled() {
        if (twoFactorSupportEnabled == null) {
            String x = getFirstAttribute(cn, ENABLE_TWO_FACTOR_SUPPORT);
            if (x == null) {
                twoFactorSupportEnabled = Boolean.FALSE;
            } else {
                try {
                    twoFactorSupportEnabled = Boolean.valueOf(x);
                } catch (Throwable t) {
                    info("Could not parse two factor enabled attribute. Setting default to false.");
                    twoFactorSupportEnabled = Boolean.FALSE;
                }
            }

        }
        return twoFactorSupportEnabled;
    }

    public void setRefreshTokenEnabled(boolean refreshTokenEnabled) {
        this.refreshTokenEnabled = refreshTokenEnabled;
    }

    Boolean refreshTokenEnabled = null;
    Collection<String> scopes = null;
    protected ClaimSource claimSource;

    public ClaimSource getClaimSource() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        DebugUtil.trace(this, "Getting scope handler " + claimSource);
        if (claimSource == null) {
            // This gets the scopes if any and injects them into the scope handler.
            if (0 < cn.getChildrenCount(SCOPES)) {
                String scopeHandlerName = getFirstAttribute(Configurations.getFirstNode(cn, SCOPES), SCOPE_HANDLER);
                if (scopeHandlerName != null) {
                    Class<?> k = Class.forName(scopeHandlerName);
                    Object x = k.newInstance();
                    if (!(x instanceof ClaimSource)) {
                        throw new GeneralException("The scope handler specified by the class name \"" +
                                scopeHandlerName + "\" does not extend the ScopeHandler " +
                                "interface and therefore cannot be used to handle scopes.");
                    }
                    claimSource = (ClaimSource) x;
                } else {
                    info("Scope handler attribute found in configuration, but no value was found for it. Skipping custom loaded scope handling.");
                }
            }

            // no scopes element, so just use the basic handler.
            if (claimSource == null) {

                DebugUtil.trace(this, "No server-wide configured Scope handler");
                if (getLdapConfiguration().isEnabled()) {
                    DebugUtil.trace(this, "   LDAP scope handler enabled, creating default");
                    claimSource = new LDAPClaimsSource(getLdapConfiguration(), myLogger);
                } else {
                    DebugUtil.trace(this, "   LDAP scope handler disabled, creating basic");
                    ClaimSourceConfiguration claimSourceConfiguration = new ClaimSourceConfiguration();
                    claimSourceConfiguration.setEnabled(false);
                    claimSource = new BasicClaimsSourceImpl();
                    claimSource.setConfiguration(claimSourceConfiguration);
                }
            }
            claimSource.setScopes(getScopes());
            DebugUtil.trace(this, "   Actual scope handler = " + claimSource.getClass().getSimpleName());

        }
        return claimSource;
    }

    LDAPConfiguration ldapConfiguration;

    protected LDAPConfiguration getLdapConfiguration() {
        if (ldapConfiguration == null) {
            LDAPConfigurationUtil ldapConfigurationUtil = new LDAPConfigurationUtil();
            ldapConfiguration = ldapConfigurationUtil.getLdapConfiguration(myLogger, cn);
        }
        return ldapConfiguration;

    }

    public Collection<String> getScopes() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        if (scopes == null) {
            scopes = OA2ConfigurationLoaderUtils.getScopes(cn);
        }
        return scopes;
    }

    public int getClientSecretLength() {
        if (clientSecretLength < 0) {
            String x = getFirstAttribute(cn, CLIENT_SECRET_LENGTH);
            if (x != null) {
                try {
                    clientSecretLength = Integer.parseInt(x);
                } catch (Throwable t) {
                    clientSecretLength = CLIENT_SECRET_LENGTH_DEFAULT;
                }
            } else {
                clientSecretLength = CLIENT_SECRET_LENGTH_DEFAULT;
            }
        }
        return clientSecretLength;
    }

    int clientSecretLength = -1; // Negative (illegal value) to trigger parsing from config file on load. Default is 258.


    public static class ST2Provider extends DSTransactionProvider<OA2ServiceTransaction> {

        public ST2Provider(IdentifierProvider<Identifier> idProvider) {
            super(idProvider);
        }

        @Override
        public OA2ServiceTransaction get(boolean createNewIdentifier) {
            return new OA2ServiceTransaction(createNewId(createNewIdentifier));
        }
    }

    public static class OA2MultiDSClientStoreProvider extends MultiDSClientStoreProvider {
        public OA2MultiDSClientStoreProvider(ConfigurationNode config, boolean disableDefaultStore, MyLoggingFacade logger) {
            super(config, disableDefaultStore, logger);
        }

        public OA2MultiDSClientStoreProvider(ConfigurationNode config, boolean disableDefaultStore, MyLoggingFacade logger, String type, String target, IdentifiableProvider clientProvider) {
            super(config, disableDefaultStore, logger, type, target, clientProvider);
        }

        @Override
        public ClientStore getDefaultStore() {
            logger.info("Using default in memory client store");
            return new OA2ClientMemoryStore(clientProvider);
        }
    }

    @Override
    protected MultiDSClientStoreProvider getCSP() {
        if (csp == null) {
            OA2ClientConverter converter = new OA2ClientConverter(getClientProvider());
            csp = new OA2MultiDSClientStoreProvider(cn, isDefaultStoreDisabled(), loggerProvider.get(), null, null, getClientProvider());

            csp.addListener(new DSFSClientStoreProvider(cn, converter, getClientProvider()));
            csp.addListener(new OA2ClientSQLStoreProvider(getMySQLConnectionPoolProvider(),
                    OA4MPConfigTags.MYSQL_STORE,
                    converter, getClientProvider()));
            csp.addListener(new OA2ClientSQLStoreProvider(getMariaDBConnectionPoolProvider(),
                    OA4MPConfigTags.MARIADB_STORE,
                    converter, getClientProvider()));
            csp.addListener(new OA2ClientSQLStoreProvider(getPgConnectionPoolProvider(),
                    OA4MPConfigTags.POSTGRESQL_STORE,
                    converter, getClientProvider()));
            csp.addListener(new TypedProvider<ClientStore>(cn, OA4MPConfigTags.MEMORY_STORE, OA4MPConfigTags.CLIENTS_STORE) {

                @Override
                public Object componentFound(CfgEvent configurationEvent) {
                    if (checkEvent(configurationEvent)) {
                        return get();
                    }
                    return null;
                }

                @Override
                public ClientStore get() {
                    return new OA2ClientMemoryStore(getClientProvider());
                }
            });
        }
        return csp;
    }

    protected OA2SQLTransactionStoreProvider createSQLTSP(ConfigurationNode config,
                                                          ConnectionPoolProvider<? extends ConnectionPool> cpp,
                                                          String type,
                                                          MultiDSClientStoreProvider clientStoreProvider,
                                                          Provider<? extends OA2ServiceTransaction> tp,
                                                          Provider<TokenForge> tfp,
                                                          MapConverter converter) {
        return new OA2SQLTransactionStoreProvider(config, cpp, type, clientStoreProvider, tp, tfp, converter);
    }

    protected Provider<TransactionStore> getTSP(IdentifiableProvider tp,
                                                OA2TConverter<? extends OA2ServiceTransaction> tc) {
        if (tsp == null) {
            final IdentifiableProvider tp1 = tp; // since this is referenced in an inner class below.
            OA2MultiTypeProvider storeProvider = new OA2MultiTypeProvider(cn, isDefaultStoreDisabled(), loggerProvider.get(), tp);
            storeProvider.addListener(createSQLTSP(cn,
                    getMySQLConnectionPoolProvider(),
                    OA4MPConfigTags.MYSQL_STORE,
                    getCSP(),
                    tp,
                    getTokenForgeProvider(),
                    tc));
            storeProvider.addListener(createSQLTSP(cn,
                    getMariaDBConnectionPoolProvider(),
                    OA4MPConfigTags.MARIADB_STORE,
                    getCSP(),
                    tp,
                    getTokenForgeProvider(),
                    tc));
            storeProvider.addListener(createSQLTSP(cn,
                    getPgConnectionPoolProvider(),
                    OA4MPConfigTags.POSTGRESQL_STORE,
                    getCSP(),
                    tp,
                    getTokenForgeProvider(),
                    tc));

            storeProvider.addListener(new OA2FSTStoreProvider(cn, tp, getTokenForgeProvider(), tc));
            storeProvider.addListener(new TypedProvider<TransactionStore>(cn, OA4MPConfigTags.MEMORY_STORE, OA4MPConfigTags.TRANSACTIONS_STORE) {
                @Override
                public Object componentFound(CfgEvent configurationEvent) {
                    if (checkEvent(configurationEvent)) {
                        return get();
                    }
                    return null;
                }

                @Override
                public TransactionStore get() {
                    return new OA2MTStore(tp1);
                }

            });
            tsp = storeProvider;
        }
        return tsp;
    }

    @Override
    protected Provider<TransactionStore> getTSP() {
        IdentifiableProvider tp = new ST2Provider(new OA4MPIdentifierProvider(TRANSACTION_ID, false));
        OA2TransactionKeys keys = new OA2TransactionKeys();
        OA2TConverter<OA2ServiceTransaction> tc = new OA2TConverter<OA2ServiceTransaction>(keys, tp, getTokenForgeProvider().get(), getClientStoreProvider().get());
        return getTSP(tp, tc);
    }


    @Override
    public Provider<TransactionStore> getTransactionStoreProvider() {
        return getTSP();
    }

    @Override
    public Provider<TokenForge> getTokenForgeProvider() {
        return new Provider<TokenForge>() {
            @Override
            public TokenForge get() {
                return new OA2TokenForge(getServiceAddress().toString());
            }
        };
    }

    @Override
    public Provider<ATIssuer> getATIProvider() {
        return new Provider<ATIssuer>() {
            @Override
            public ATIssuer get() {
                return new ATI2(getTokenForgeProvider().get(), getServiceAddress(), isOIDCEnabled());
            }
        };
    }

    @Override
    public Provider<PAIssuer> getPAIProvider() {
        return new Provider<PAIssuer>() {
            @Override
            public PAIssuer get() {
                return new PAI2(getTokenForgeProvider().get(), getServiceAddress(), isOIDCEnabled());
            }
        };
    }


    @Override
    public IdentifiableProvider<? extends Client> getClientProvider() {
        return new OA2ClientProvider(new OA4MPIdentifierProvider(OA2Constants.CLIENT_ID, false));
    }

    @Override
    public String getVersionString() {
        return "OAuth 2 for MyProxy, version " + VERSION_NUMBER;
    }

}
