% PURPOSE : 
%      Compute the first term of RMES (method based on CoVaR approximation)
% USAGE:  
%      CoVaR_term      = RMES_CoVaR_term(var1,var2,cov,rho,VaR2,teta,alpha)
% INPUTS:
%      var1            = Y1 estimated variance
%      var2            = Y2 estimated variance
%      rho             = Y1 Y2 estimated correlation
%      VaR2            = Y2 estimated value at risk
%      teta            = [var1;rho;var2] MLE parameters vector
%      alpha           = alpha order of the Value at Risk
% OUTPUTS:
%      CoVaR_term      = First term of RMES quantity

function CoVaR_term = RMES_CoVaR_term(var1,var2,rho,VaR2,teta,alpha)

CoVaR_term=integral(@first,0,1);

function [first_term]=first(beta)
    
% ------------------------------------ % 
% 1 - Computation of CoVaRt(beta,teta) %
% ------------------------------------ % 

x=-norminv(beta);   

H0=-norminv(alpha);  
                    
% Cumulants computation in i (Kappa_10,Kappa_20,...,Kappa_80)
n=8;                                   
ki0=NaN(n,1);                          % Vector of Kappa_10,Kappa_20,...,Kappa_80
ki0(1)=normpdf(H0)./normcdf(-H0);
ki0(2)=H0*normpdf(H0)./normcdf(-H0)-(normpdf(H0)./normcdf(-H0)).^2+1;
u=normpdf(H0)./normcdf(-H0);           % u(x) in the papier
v=1./u;                                % v(x) in the papier
dv=NaN(n-1,1);
dv(1)=1-H0*v;                          % dv1 => dv(x)/dx
dv(2)=v-H0*dv(1);                      % dv2 => d^2v(x)/dx^2
du=NaN(n-1,1);
du(1)=-u.^2*dv(1);                     % du1 => du(x)/dx
du(2)=-u*(2*du(1)*dv(1)+u*dv(2));      % du2 => du(x)/dx

for i=3:n-1                            % Computation of dv3...dv8 & du3...du8 
    dv(i)=(i-1)*dv(i-2)-H0*dv(i-1);
    cumsumj=0;
    for j=1:i
        if (i-j==0)
            sumj=u*dv(j);
        else
            sumj=(factorial(i)./(factorial(j)*factorial(i-j)))*du(i-j)*dv(j);
        end
        cumsumj=cumsumj+sumj;
    end
    du(i)=-u*cumsumj;
end
    
ki0(3:n)=du(2:n-1);

% Cumulants computation in j (Kappa_01,Kappa_02,...,Kappa_08)
k0j=NaN(n,1);                         % Vector of Kappa_01,Kappa_02,...,Kappa_08
k0j(1)=rho*ki0(1);
k0j(2)=rho.^2*(ki0(2)-1)+1;
for j=3:n
    k0j(j)=rho.^j*ki0(j);
end

% Ajustment's coefficients
c=k0j(3)./k0j(2).^(3./2);
d=k0j(4)./k0j(2).^2;
e=k0j(5)./k0j(2).^(5./2);
f=k0j(6)./k0j(2).^3;
g=k0j(7)./k0j(2).^(7./2);
h=k0j(8)./k0j(2).^4;

% Ajustments I, II, III, IV, V, et VI 
% Source : "The Percentile Points of Distributions Having Known Cumulants" (1960) - Fisher & Cornish
I   = c*(x.^2-1)./6;                                                                               
II  = d*(x.^3-3*x)./24 - c.^2*(2*x.^3-5*x)./36;                                                    
III = e*(x.^4-6*x.^2+3)./120 - c*d*(x.^4-5*x.^2+2)./24 +...                                        
      c.^3*(12*x.^4-53*x.^2+17)./324;
IV  = f*(x.^5-10*x.^3+15*x)./720 - c*e*(2*x.^5-17*x.^3+21*x)./180 -...                             
      d.^2*(3*x.^5-24*x.^3+29*x)./384 + c.^2*d*(14*x.^5-103*x.^3+107*x)./288 -...
      c.^4*(252*x.^5-1688*x.^3+1511*x)./7776;
V   = g*(x.^6-15*x.^4+45*x.^2-15)./5040 -...                                                       
      c*f*(x.^6-13*x.^4+33*x.^2-9)./432 - d*e*(x.^6-12*x.^4+29*x.^2-8)./240 +...
      c.^2*e*(16*x.^6-181*x.^4+393*x.^2-90)./1080 + c*d.^2*(12*x.^6-129*x.^4+271*x.^2-64)./576 -...
      c.^3*d*(80*x.^6-803*x.^4+1513*x.^2-304)./1296 + c.^5*(960*x.^6-8937*x.^4+15062*x.^2-2651)./29160;
VI  = h*(x.^7-21*x.^5+105*x.^3-105*x)./40320 -c*g*(2*x.^7-37*x.^5+160*x.^3-135*x)./5040 -...       
      d*f*(x.^7-17*x.^5+69*x.^3-57*x)./1152 - e.^2*(2*x.^7-33*x.^5+132*x.^3-108*x)./3600 +...
      c.^2*f*(18*x.^7-293*x.^5+1100*x.^3-795*x)./5184 + c*d*e*(18*x.^7-273*x.^5+974*x.^3-695*x)./1440 +...
      d.^3*(9*x.^7-131*x.^5+451*x.^3-321*x)./3072 - c.^3*e*(396*x.^7-5708*x.^5+18755*x.^3-11811*x)./19440 -...
      c.^2*d.^2*(594*x.^7-8193*x.^5+26006*x.^3-16367*x)./13824 + ...
      c.^4*d*(5148*x.^7-67004*x.^5+195259*x.^3-109553*x)./62208 -...
      c.^6*(154440*x.^7-1887684*x.^5+5033714*x.^3-2542637*x)./4199040;
    
L0_opp=-(sqrt(k0j(2))*(I+II+III+IV+V+VI+x)+k0j(1)); % Be carreful, to obtain -L, we need to take the opposite of L

CoVaR_chou=L0_opp*sqrt(var1);                          

% -------------------------------------- % 
% 2 - Computation of CoVaRt(beta,teta+z) %
% -------------------------------------- % 

z=10.^-8;

if teta==1
    var1_z=var1+z;
    rho_z=rho;
end

if teta==2
    var1_z=var1;
    rho_z=rho+z;
end

if teta==3
    var1_z=var1;
    rho_z=rho;
end

k0j_z=NaN(n,1);                         
k0j_z(1)=rho_z*ki0(1);
k0j_z(2)=rho_z.^2*(ki0(2)-1)+1;
for j=3:n
    k0j_z(j)=rho_z.^j*ki0(j);
end

c_z=k0j_z(3)./k0j_z(2).^(3./2);
d_z=k0j_z(4)./k0j_z(2).^2;
e_z=k0j_z(5)./k0j_z(2).^(5./2);
f_z=k0j_z(6)./k0j_z(2).^3;
g_z=k0j_z(7)./k0j_z(2).^(7./2);
h_z=k0j_z(8)./k0j_z(2).^4;

I_z   = c_z*(x.^2-1)./6;                                                                        
II_z  = d_z*(x.^3-3*x)./24 - c_z.^2*(2*x.^3-5*x)./36;                                                 
III_z = e_z*(x.^4-6*x.^2+3)./120 - c_z*d_z*(x.^4-5*x.^2+2)./24 +...                                      
        c_z.^3*(12*x.^4-53*x.^2+17)./324;
IV_z  = f_z*(x.^5-10*x.^3+15*x)./720 - c_z*e_z*(2*x.^5-17*x.^3+21*x)./180 -...                           
        d_z.^2*(3*x.^5-24*x.^3+29*x)./384 + c_z.^2*d_z*(14*x.^5-103*x.^3+107*x)./288 -...
        c_z.^4*(252*x.^5-1688*x.^3+1511*x)./7776;
V_z   = g_z*(x.^6-15*x.^4+45*x.^2-15)./5040 -...                                                   
        c_z*f_z*(x.^6-13*x.^4+33*x.^2-9)./432 - d_z*e_z*(x.^6-12*x.^4+29*x.^2-8)./240 +...
        c_z.^2*e_z*(16*x.^6-181*x.^4+393*x.^2-90)./1080 + c_z*d_z.^2*(12*x.^6-129*x.^4+271*x.^2-64)./576 -...
        c_z.^3*d_z*(80*x.^6-803*x.^4+1513*x.^2-304)./1296 + c_z.^5*(960*x.^6-8937*x.^4+15062*x.^2-2651)./29160;
VI_z  = h_z*(x.^7-21*x.^5+105*x.^3-105*x)./40320 -c_z*g_z*(2*x.^7-37*x.^5+160*x.^3-135*x)./5040 -...       
        d_z*f_z*(x.^7-17*x.^5+69*x.^3-57*x)./1152 - e_z.^2*(2*x.^7-33*x.^5+132*x.^3-108*x)./3600 +...
        c_z.^2*f_z*(18*x.^7-293*x.^5+1100*x.^3-795*x)./5184 + c_z*d_z*e_z*(18*x.^7-273*x.^5+974*x.^3-695*x)./1440 +...
        d_z.^3*(9*x.^7-131*x.^5+451*x.^3-321*x)./3072 - c_z.^3*e_z*(396*x.^7-5708*x.^5+18755*x.^3-11811*x)./19440 -...
        c_z.^2*d_z.^2*(594*x.^7-8193*x.^5+26006*x.^3-16367*x)./13824 + ...
        c_z.^4*d_z*(5148*x.^7-67004*x.^5+195259*x.^3-109553*x)./62208 -...
        c_z.^6*(154440*x.^7-1887684*x.^5+5033714*x.^3-2542637*x)./4199040;
    
L0_opp_z=-(sqrt(k0j_z(2))*(I_z+II_z+III_z+IV_z+V_z+VI_z+x)+k0j_z(1)); 

CoVaR_chou_z=L0_opp_z*sqrt(var1_z);   

% -------------------------------------- % 
% 3 - Numerical CoVaR derivative in teta %
% -------------------------------------- % 

d_CoVaR=(CoVaR_chou_z-CoVaR_chou)./z;

% -------------------------------------- % 
% 4 - Computation of RMES first term     %
% -------------------------------------- % 

f12=normpdf(CoVaR_chou,0,sqrt(var1)).*normcdf(VaR2,rho*sqrt(var2/var1)*CoVaR_chou,sqrt(var2-rho^2*var2));

first_term=f12.*d_CoVaR;

end

end
