%-------------------------------------------------------------------------------------------------------
% PURPOSE: "Backtesting Marginal Expected Shortfall and Related Systemic Risk Measures"
%           Banulescu, Hurlin, Leymarie, and Scaillet (2019)
%-------------------------------------------------------------------------------------------------------
% MES, SRISK, & SES backtesting exercises at daily horizon h=1 (short-term) at the end of each month 
% for the 95 firms (using a GJR-DCC model with normal innovations)
%-------------------------------------------------------------------------------------------------------
% J. Leymarie, September 25, 2017.
% LEO, University of Orleans
%-------------------------------------------------------------------------------------------------------
% Notes : 
% - The code is indiced according to time (Ex : year=2005 and month=1 means that test statistics are computed 
%   using data available until jan. 2005).
% - The excel file "ref_ticker.xlsx" reports the ticker reference associated with each firm=1,...,95 on the 
%   excel database "bdd_for_matlab.xlsx".
% - The parameters of the GJR-DCC model are estimated with the MFE Toolbox.
% - The main outputs of this code are daily MES, daily LRMES, daily SRISK estimations, Ht process, 
%   test statistics, p-values, and adjusted systemic risk measures.
% - The backtesting is computed considering a window size of one month (end of month) between each backtest. 
% - 2 estimation scheme can be considered in this code :
%   * recursive estimation scheme as in Brownlees and Engle (2007) (rw should be set to 0) 
%   * rolling window estimation scheme (rw should be set to 1) 
%-------------------------------------------------------------------------------------------------------

clear ; clc ; close all ;

%================%
%== Parameters ==%
%================%

filename='bdd_for_matlab.xlsx';                       % Excel file 

database=xlsread(filename);                           % Data importation 

nb_firm=95;                                           % Number of firm

k=0.08;                                               % Prudential capital fraction k (for SRISK computation)

rw=1;                                                 % 1 if rolling window estimation scheme, 0 if recursive estimation scheme

rw_size=500;                                          % Size of the in-sample rolling windows (in case rw=1)

N=500;                                                % Out-of-sample size for the empirical application

first_year=2005;                                      % First year of backtesting computation

last_year=2016;                                       % Last year of backtesting computation

alpha=0.05;                                           % alpha level for quantile of Y2

mmax=20;                                              % Maximum lag autocovariance for IND test

options=optimset('Display','off');                    % Options

%======================%
%=== Initialization ===%
%======================%

uc_risk=NaN((last_year-first_year+1)*12,nb_firm);     % UC statistics with estimation risk

ind_risk=NaN((last_year-first_year+1)*12,nb_firm,mmax);% IND statistics with estimation risk

uc_c=uc_risk; uc_c_kernel=uc_risk;                    % UC robust statistics

ind_c=ind_risk; ind_c_kernel=ind_risk;                % IND robust statistics

p_uc_risk=uc_risk; p_ind_risk=ind_risk;               % UC and IND p_values with estimation risk

p_uc_c=uc_risk;p_uc_c_kernel=uc_risk;                 % UC robust p_values

p_ind_c=ind_risk;p_ind_c_kernel=ind_risk;             % IND robust p_values 

H_global=NaN(N,(last_year-first_year+1)*12,nb_firm);  % Cumulative joint violation process Ht

theta_global=NaN(11,(last_year-first_year+1)*12,nb_firm);% Estimated parameters

V_asy_global=NaN(11,11,(last_year-first_year+1)*12,nb_firm);% Covariance matrix of estimated parameters

MES=NaN((last_year-first_year+1)*12,nb_firm);         % Estimated daily MES

LRMES=NaN((last_year-first_year+1)*12,nb_firm);       % Estimated daily LRMES

SRISK=NaN((last_year-first_year+1)*12,nb_firm);       % Estimated daily SRISK

MES_adj=NaN((last_year-first_year+1)*12,nb_firm);     % Daily adjusted MES

LRMES_adj=NaN((last_year-first_year+1)*12,nb_firm);   % Daily adjusted LRMES

SRISK_adj=NaN((last_year-first_year+1)*12,nb_firm);   % Daily adjusted SRISK 

alpha_adj=NaN((last_year-first_year+1)*12,nb_firm);   % Adjusted coverage level

for firm=1:nb_firm                                    % Loop for backtesting firm

for s=1:(last_year-first_year+1)*12                   % Loop for backtesting year and month

ref=ceil(s/12)-1;                                     % Year reference

year=first_year+ref;                                  % Last year for the full sample

month=s-ref*12;                                       % Last month for the full sample

data=database(database(:,1)==firm,:);                 % Data for the current firm  

year_base=data(data(:,2)<year,:);                     % Data available before year                

year_inprog=data(data(:,2)==year,:);                  % Data available for the current year

month_inprog=year_inprog(year_inprog(:,3)<=month,:);  % Selection of available month of the current year

full=[year_base;month_inprog];                        % In and out sample data until start of the sample

if (rw==0) && (length(full)<500+N)                    % If recursive scheme, a minimum of 500 + N OOS obs. required
                                                                                                         
    continue
    
end

if (rw==1) && (length(full)<rw_size+N)                % If rolling scheme, a minimum of rw_size + N obs. required

    continue 
    
end

if (full(end,2)~=year) || (full(end,3)~=month)        % If no enough data available, consider the next firm
    
    break
    
end

out=full(end-N+1:end,:);                              % Out-of-sample data

in_full=full(1:end-N,:);                              % In-sample data

if rw==1
    
    in=in_full(end-rw_size+1:end,:);
    
else
   
    in=in_full;
    
end

T=length(in);                                         % In sample size

TT=T+N;                                               % Full sample size

in_out=[in;out];                                      % Full sample data

Y=in_out(:,[4,5]);                                    % Full sample returns
    
%============================
%=== Estimation In-Sample ===
%============================
    
[theta_in,loglik,Hcond,V_asy,scores,diagnostic] = dcc(Y(1:T,:),[],1,0,1,1,1,1,[],[],[],[],options); 
                         
% If an error occurs in the estimation routine, use the line below to help the algorithm
% theta_init=[0.149 0.065 0.075 0.855 ...                      
%             0.057 0.001 0.268 0.786 ...                    
%             0.663 0.041 0.754]; 
% [theta_in,loglik,Hcond,V_asy,scores,diagnostic] = dcc(Y(1:T,:),[],1,0,1,1,1,1,[],[],[],theta_init,options); 

theta_global(:,s,firm)=theta_in';             % I record theta_in for future use

V_asy_global(:,:,s,firm)=V_asy;               % I record V_asy for future use

np=length(theta_in);                          % Number of model parameter estimates

garch_hat=[theta_in(1:4)' theta_in(5:8)'];    % GJR-GARCH parameter estimates

dcc_hat=theta_in(10:11);                      % DCC parameter estimates (without rho_bar)

Q_bar_hat=[1 theta_in(9);theta_in(9) 1];      % uncondional correlation estimate

%==============================================================================%
%==== Evaluation of the correlations-variances in-sample and out-of-sample ====%
%==============================================================================%

[Sigma_hat,rho_hat,]= ...                                              % Evaluation over the full sample (initial condition based on in-sample period)
    gjrdcc_evaluate([Y;NaN(1,2)],garch_hat,dcc_hat,Q_bar_hat,T);       % We add a NaN(2,1) vector to get the Y's T+N+1 cov. matrix

Sig_TNh=Sigma_hat(:,:,end);                                            % We extract T+N+1 cov. matrix given T+N 

VaR2_TNh=sqrt(Sig_TNh(2,2))*norminv(alpha);                            % We extract T+N+1 Y2's VaR given T+N 

Sigma_hat=Sigma_hat(:,:,1:TT);                                         % We keep only cov. matrix T=1,...TT in the following

rho_hat=rho_hat(1:TT);                                                 % We keep only corr. vector T=1,...TT in the following

S2_hat=NaN(TT,2);                                                      % Estimated cond. variances (in-sample and out-of-sample)

S2_hat(:,1)=Sigma_hat(1,1,:);S2_hat(:,2)=Sigma_hat(2,2,:);             % Estimated cond. variances (in-sample and out-of-sample)

var1_in=S2_hat(1:T,1);var2_in=S2_hat(1:T,2);                           % Conditional variances (in-sample)

var1_out=S2_hat(T+1:end,1);var2_out=S2_hat(T+1:end,2);                 % Conditional variances (out-of-sample)

rho_in=rho_hat(1:T);rho_out=rho_hat(T+1:end);                          % Cond. correlations (in-sample and out-of-sample)

VaR2_in=sqrt(var2_in)*norminv(alpha);                                  % Value-at-Risk of level alpha for Y2 (in-sample)

VaR2_out=sqrt(var2_out)*norminv(alpha);                                % Value-at-Risk of level alpha for Y2 (in-sample)

Y1_out=Y(T+1:end,1);                                                   % Y1 OOS

Y2_out=Y(T+1:end,2);                                                   % Y2 OOS

Y_out=[Y1_out Y2_out];                                                 % [Y1 Y2] OOS         

Sigma_out=Sigma_hat(:,:,T+1:end);                                      % OOS cond. variance matrices

%========================================================================%
%===         Estimated daily MES given T+N (exact method)             ===%
%========================================================================%

detsig=det(Sig_TNh);isig=inv(Sig_TNh);e=isig(1,1);...                  % Some inputs for the bivariate pdf
    f=isig(1,2);g=isig(2,1); h=isig(2,2);              

xfxy = @(x,y) x.*((1/(2.*pi.*sqrt(detsig))).*...                       % x * bivariate pdf  
    exp(-0.5.*(x.^2.*e+y.*g.*(x)+x.*f.*y+y.^2.*h)));                                       

MES(s,firm) = -(integral2...                                           % MES estimation at time T+N+1 given T+N (exact)
    (xfxy,-inf,inf,-inf,VaR2_TNh,'AbsTol',0,'RelTol',1e-10)/alpha)/100;   

%========================================================================%
%===        Estimated daily SRISK given T+N (simulated method)        ===%
%========================================================================%

Sig_12=[sqrt(Sig_TNh(1,1))*sqrt(1-(Sig_TNh(1,2)/(sqrt(Sig_TNh(1,1)*Sig_TNh(2,2))))^2) , ...
    (Sig_TNh(1,2)/(sqrt(Sig_TNh(1,1)*Sig_TNh(2,2))))*sqrt(Sig_TNh(1,1)) ; 0 , sqrt(Sig_TNh(2,2))];

eps=normrnd(0,1,2,5000000);                                            % Pseudo sample generation

pseudo_Y=(Sig_12*eps)'/100;

LRMES(s,firm)=mean(exp(pseudo_Y(pseudo_Y(:,2)<(VaR2_TNh/100),1))-1);   % LRMES estimation at time T+N+1 given T+N
        
SRISK(s,firm)=(k*out(end,6)-(1-k)*out(end,7)*(1+LRMES(s,firm)));       % SRISK estimation at time T+N+1 given T+N

if SRISK(s,firm)<0
    
    SRISK(s,firm)=0;                                                   % Negative SRISK is set to 0

end

%=======================================================================================%
%=== Derivatives of the cond. variance matrices with respect to estimated parameters ===%
%=======================================================================================%

z=10.^-8;                                                                              % Incremental value for derivatives

dSigma=NaN(np,3,TT);                                                                   % Initialization matrix of derivatives

for i=1:np                                                                             % Loop on the derivatives

    theta_in_z=theta_in;                                                               % Initialization 

    theta_in_z(i)=theta_in_z(i)+z;                                                     % Augmented value

    garch_hat_z=[theta_in_z(1:4)' theta_in_z(5:8)'];                                   % GARCH parameters

    dcc_hat_z=theta_in_z(10:11);                                                       % DCC parameters (without rho_bar)

    Q_bar_hat_z=[1 theta_in_z(9);theta_in_z(9) 1];                                     % Uncondional correlation

    [Sigma_hat_z,rho_hat_z,]= gjrdcc_evaluate(Y,garch_hat_z,dcc_hat_z,Q_bar_hat_z,T);  % Evaluation over the full sample (initial condition based on in-sample period)

    dSigma(i,1,:)=(reshape(Sigma_hat_z(1,1,:),TT,1)-reshape(Sigma_hat(1,1,:),TT,1))/z; % Derivatives of cond. variance var1

    dSigma(i,2,:)=(rho_hat_z-rho_hat)/z;                                               % Derivatives of cond. correlation

    dSigma(i,3,:)=(reshape(Sigma_hat_z(2,2,:),TT,1)-reshape(Sigma_hat(2,2,:),TT,1))/z; % Derivatives of cond. variance var2

end                                                                                    % End loop on the derivatives
        
dSigma_out=dSigma(:,:,T+1:end);                                                        % Out-of-sample derivatives
        
%=====================================================%
%=== Computation of H(alpha) series and RMES terms ===%
%=====================================================%

H_risk=zeros(N,1);                                                                     % Ht(alpha) series (with estimation risk)

RMESt_kernel=NaN(np,N);                                                                % RMES terms (kernel correction): derivatives with respect to estimated parameters

RMESt=NaN(np,N);                                                                       % RMES terms: derivatives with respect to estimated parameters     

parfor i=1:N                                                                           % Loop on out-of-sample periods
            
    if Y2_out(i)<=VaR2_out(i)                                                          % With estimation risk
                
        H_risk(i)=1-mvncdf([Y1_out(i) VaR2_out(i)],zeros(1,2),Sigma_out(:,:,i))/alpha; % H(t)=1-u12 (with estimation risk)
                
    end                                                                                % End of if condition
            
  RMESt_kernel(:,i)=dSigma_out(:,:,i)*H_normal_derivatives_kernel(Sigma_out(:,:,i),Y_out(i,:),alpha,1/N);  % Correction term RMES (kernel method)               

  RMESt(:,i)=dSigma_out(:,:,i)*[RMES_CoVaR_term(var1_out(i),var2_out(i),rho_out(i),VaR2_out(i),1,alpha);...% Correction term RMES (covar method)
       RMES_CoVaR_term(var1_out(i),var2_out(i),rho_out(i),VaR2_out(i),2,alpha);...
       RMES_VaR2_term(var1_out(i),var2_out(i),rho_out(i),VaR2_out(i),3,alpha)];

end                                                                                    % End of loop on i
  
H_global(:,s,firm)=H_risk;

%========================================%
%=== UC_MES - Unconditionnal backtest ===%  
%========================================%

% With Estimation Risk

uc_risk(s,firm)=sqrt(N)*(mean(H_risk)-alpha/2)/sqrt(alpha*(1/3-alpha/4)); % Statistic UC with estimation risk

p_uc_risk(s,firm)=2*normcdf(-abs(uc_risk(s,firm)));                       % UC p_value

% Robust Test Statistics

RMES_kernel=mean(RMESt_kernel,2);                                         % RMES mean (Kernel methodology)

RMES=mean(RMESt,2);                                                       % RMES mean (CoVaR methodology)

uc_c_kernel(s,firm)=sqrt(N)*(mean(H_risk)-alpha/2)/sqrt(alpha*(1/3-alpha/4)+N/T*RMES_kernel'*T*V_asy*RMES_kernel);   % Statistic UC robust (Kernel methodology)

p_uc_c_kernel(s,firm)=2*normcdf(-abs(uc_c_kernel(s,firm)));               % UC p_value

uc_c(s,firm)=sqrt(N)*(mean(H_risk)-alpha/2)/sqrt(alpha*(1/3-alpha/4)+N/T*RMES'*T*V_asy*RMES); % Statistic UC robust (CoVaR methodology)

p_uc_c(s,firm)=2*normcdf(-abs(uc_c(s,firm)));                             % UC p_value

%========================================%
%===    Adjusted MES, LRMES, SRISK    ===%  
%========================================%

[alpha_tilde, fval] = fminsearch('adjusted_alpha',alpha,options,Y1_out,Y2_out,var2_out,Sigma_out,alpha);

alpha_adj(s,firm)=alpha_tilde;                                            % I record the adjusted coverage level

%=============================================%
%=== Adjusted MES given T+N (exact method) ===%
%=============================================%

VaR2_TNh_tilde=sqrt(Sig_TNh(2,2))*norminv(alpha_tilde);                   % We extract T+N+1 Y2's VaR given T+N 

MES_adj(s,firm) = -(integral2...                                          % MES estimation at time T+N+1 given T+N (exact)
    (xfxy,-inf,inf,-inf,VaR2_TNh_tilde,'AbsTol',0,'RelTol',1e-10)/alpha_tilde)/100;   

%===================================================%
%=== Adjusted SRISK given T+N (simulated method) ===%
%===================================================%

LRMES_adj(s,firm)=mean(exp(pseudo_Y(pseudo_Y(:,2)<(VaR2_TNh_tilde/100),1))-1);   % LRMES estimation at time T+N+1 given T+N
        
SRISK_adj(s,firm)=(k*out(end,6)-(1-k)*out(end,7)*(1+LRMES_adj(s,firm)));  % SRISK estimation at time T+N+1 given T+N

if SRISK_adj(s,firm)<0
    
    SRISK_adj(s,firm)=0;                                                  % Negative SRISK is set to 0

end

%========================================%
%=== IND_MES - Conditionnal backtest  ===%
%========================================%
       
gammaHt_hat_risk=NaN(mmax,1);                                             % Autocovariances of Ht(alpha)

rhoHt_hat_risk=NaN(mmax,1);                                               % Autocorrelations of Ht(alpha)

R_kernel=NaN(mmax,np);                                                    % Correction term R (Ht kernel numerical derivative)  

R=NaN(mmax,np);                                                           % Correction term R (CoVaR methodology)                            
          
for m=1:mmax                                                              % Loop on the lag

    % Autocorrelations with estimation risk

    Hlag_risk=[NaN(m,1) ; H_risk(1:N-m)];                                          % Lagged values with Estimation Risk

    gammaHt_hat_risk(m)=(1/(N-m))*nansum((H_risk-alpha/2).*(Hlag_risk-alpha/2));   % Autocovariances of H(t) with Estimation Risk

    rhoHt_hat_risk(m)=gammaHt_hat_risk(m)/var(H_risk);                             % Autocorrelations of H(t) with Estimation Risk

    % Matrix Ri and Rj 

    R_kernel(m,:)=(1/(alpha*(1/3-alpha/4)))*(nansum(repmat((Hlag_risk-alpha/2),1,np).*RMESt_kernel')/(N-m));   % Be careful: RMES must be a vector in the conditional case, do not use repmat

    R(m,:)=(1/(alpha*(1/3-alpha/4)))*(nansum(repmat((Hlag_risk-alpha/2),1,np).*RMESt')/(N-m));                 % Be careful: RMES must be a vector in the conditional case, do not use repmat

    % With estimation risk

    ind_risk(s,firm,m)=N*sum(rhoHt_hat_risk(1:m).^2);                         % Statistic IND with estimation risk 

    p_ind_risk(s,firm,m)=1-chi2cdf(ind_risk(s,firm,m),m);                     % IND p-value

    % Robust test statistics

    delta=eye(m)+N*R(1:m,:)*V_asy*R(1:m,:)';                                  % Variance-covariance matrix of the corrected statistic

    delta_kernel=eye(m)+N*R_kernel(1:m,:)*V_asy*R_kernel(1:m,:)';             % Variance-covariance matrix of the corrected statistic (kernel)

    ind_c_kernel(s,firm,m)=N*rhoHt_hat_risk(1:m)'*pinv(delta_kernel)*rhoHt_hat_risk(1:m);  % Statistic IND robust (Ht kernel numerical derivative)

    p_ind_c_kernel(s,firm,m)=1-chi2cdf(ind_c_kernel(s,firm,m),m);            % IND p-value

    ind_c(s,firm,m)=N*rhoHt_hat_risk(1:m)'*pinv(delta)*rhoHt_hat_risk(1:m);   % Statistic IND robust (CoVaR methodology)

    p_ind_c(s,firm,m)=1-chi2cdf(ind_c(s,firm,m),m);                           % IND p-value

end                                                    % End loop on lags

fprintf('  Firm reference              : %1.0f  \n',firm)
fprintf('  End of in-sample period     : year = %1.0f    month = %1.0f \n',in(end,2),in(end,3)),
fprintf('  End of out_of-sample period : year = %1.0f    month = %1.0f \n',year,month)

end 

end

%=============================================%
%== Date vector construction (for graphics) ==%
%=============================================%

date=NaN(1,12*(last_year-first_year+1));
j=1;
for year=first_year:last_year
date(j*12-11:j*12)=datenum(year,1:12,31);
j=j+1;
end 
