%---------------------------------------------------------------------------------------------------------------------------------
% PURPOSE: Compute the partial derivative of the H(teta) function in the
% Bivariate Normal case with respect to variance(Y1), correlation(Y1,Y2) and variance(Y2)
% with a smoothing derivative kernel instead of the indicator function
%---------------------------------------------------------------------------------------------------------------------------------
% function dH=H_normal_derivatives_kernel(Sigma,Y,alpha,h)
%
% Inputs:
%   Sigma: Variance-Covariance matric of the bivariate normal distribution
%   Y: 2xT vector (y1,y2). T can be equal to 1
%   alpha: coverage rate for the VaR
%   h: Bandwith parameter of the kernel smoothing
%
% Outputs:  dH is a (3,T) matrix
%   dH(1,:): vector of derivatives with respect to variance(Y1)
%   dH(2,:): vector of derivatives with respect to correlation(Y1,Y2)
%   dH(3,:): vector of derivatives with respect to variance(Y2)
%
% ---------------------------------------------------------------------- 
% C. Hurlin and J. Leymarie. May 10, 2016
% LEO, University of Orleans
% ----------------------------------------------------------------------

function dH = H_normal_derivatives_kernel(Sigma,Y,alpha,h)

N=size(Y,1);                                                             % Sample size 

dH=NaN(3,N);                                                             % Partial derivatives (var1, corr12, var2)

z=10^-8;                                                                 % Increment for numerical derivatives

%==========================%
%=== H(theta) function  ===%
%==========================%

Y1=Y(:,1);Y2=Y(:,2);                                                     % Returns Y1 and Y2

corr_mat=corrcov(Sigma);                                                 % Correlation matrix

corr12=corr_mat(1,2);                                                    % Estimated correlation parameter Y1 Y2

VaR2=sqrt(Sigma(2,2))*norminv(alpha);                                    % VaR for asset 2

H=zeros(N,1);                                                            % Ht(alpha) series

if N>1                                                                   % Vector case
    
    Y1_s=Y1(Y2<=VaR2);                                                   % Value of Y1 for which Y<VaR2
       
    u12=mvncdf([Y1_s VaR2*ones(size(Y1_s))],0,Sigma)/alpha;              % Computation for u12 for those observations
  
    H(Y2<=VaR2)=1-u12;                                                   % Ht(alpha) series (evaluated for Y1 for which Y2<VaR2

elseif (N==1)&&(Y2<=VaR2)                                                % Scalar case                                                            
    
    H=1-mvncdf([Y1 VaR2],0,Sigma)/alpha;                                 % Computation for u12 for those observations
  
end                                                                      % End condition if

%==========================%
%=== Partial derivative ===%
%==========================%

for j=1:3                                                                % Loop on the partial derivatives

    Sigma_z=Sigma;                                                       % Initialization
    
    if j==1                                                              % Derivative with respect to var1
        
        Sigma_z=[Sigma(1,1)+z sqrt((Sigma(1,1)+z)*Sigma(2,2))*corr12 ; ...
        sqrt((Sigma(1,1)+z)*Sigma(2,2))*corr12 Sigma(2,2)];              % Augmented var-cov matrix
        
    end                                                                  % End condition j==1
    
    if j==2                                                              % Derivative with respect to corr12
        
        Sigma_z=[Sigma(1,1) sqrt(Sigma(1,1)*Sigma(2,2))*(corr12+z) ; ...
        sqrt(Sigma(1,1)*Sigma(2,2))*(corr12+z) Sigma(2,2)];              % Augmented var-cov matrix
       
    end                                                                  % End condition j==2
        
    if j==3                                                              % Derivative with respect to var2
        
        Sigma_z=[Sigma(1,1) sqrt(Sigma(1,1)*(Sigma(2,2)+z))*corr12 ; ...     
        sqrt(Sigma(1,1)*(Sigma(2,2)+z))*corr12 Sigma(2,2)+z];            % Augmented var-cov matrix
        
    end                                                                  % End condition j==3
    
    VaR2_z=sqrt(Sigma_z(2,2))*norminv(alpha);                            % Augmented  VaR
    
    H_z=zeros(N,1);                                                      % Ht(alpha) series
    
    if N>1                                                               % Vector case
    
        Y1_s=Y1(Y2<=VaR2_z);                                             % Value of Y1 for which Y<VaR2
    
        u12_z=mvncdf([Y1_s VaR2_z*ones(size(Y1_s))],0,Sigma_z)/alpha;    % Computation for u12 for those observations
    
        H_z(Y2<=VaR2_z)=1-u12_z;                                         % Ht(alpha) series (evaluated for Y1 for which Y2<VaR2
    
    elseif (N==1)&&(Y2<=VaR2_z)                                          % Scalar case                
        
        H_z=1-mvncdf([Y1 VaR2_z],0,Sigma_z)/alpha;                       % Computation for u12 for those observations
        
    end                                                                  % End if condition on the size of Y
    
    if j==3                                                              % Case derivative with respect to var2
                   
        u2=normcdf(Y2,0,sqrt(Sigma(2,2)));                               % Computation of u2
        
        u12=mvncdf([Y1 VaR2*ones(size(Y1))],0,Sigma)/alpha;              % Computation of u12

        dFy2=-normpdf(Y2/sqrt(Sigma(2,2))).*Y2./(2*Sigma(2,2)^(3/2));    % Derivative of FY2
          
        dkernel=(1/h)*(normpdf(u2/h)-normpdf((u2-alpha)/h)).*dFy2;       % Derivative of kernel smoothing
          
        dH(j,:)=(H_z-H)'/z + ((1-u12).*dkernel)';                        % Partial derivative for j=3
        
    else                                                                 % Partial derivative
        
        dH(j,:)=(H_z-H)'/z;                                              % Partial derivatives for corr12 and var1
        
    end                                                                  % End condition j==3

end                                                                      % End of loop on j 

end                                                                      %  End of function
