%---------------------------------------------------------------------------------------------------------------------------------
% PURPOSE: Compute the partial derivative of the ht(teta) function in the
% Bivariate Normal case with respect to variance(Y1), correlation(Y1,Y2) and variance(Y2)
% with a smoothing derivative kernel instead of the indicator function
%---------------------------------------------------------------------------------------------------------------------------------
% => Delta CoVaR backtesting case <=
%---------------------------------------------------------------------------------------------------------------------------------
% function [dh1 , dh2] = h_normal_derivatives_kernel_deltacovar(Sigma,Y,z,alpha,btilde,h)
%
% Inputs:
%   Sigma: Variance-Covariance matrix of the bivariate normal distribution
%   Y: N*2 vector (y1,y2). N can be equal to 1
%   z: Increment for numerical derivatives
%   alpha: coverage rate for the VaR
%   btilde: Second coverage rate for the VaR
%   h: Bandwith parameter of the kernel smoothing
%
% Outputs:  dh1 and dh3 are (3,N) matrix
%  -  dh1 => h1t partial derivatives
%            * dh1(1,:): vector of derivatives with respect to variance(Y1)
%            * dh1(2,:): vector of derivatives with respect to correlation(Y1,Y2)
%            * dh1(3,:): vector of derivatives with respect to variance(Y2)
%  -  dh2 => h2t partial derivatives
%            * Same construction than dh1
% ---------------------------------------------------------------------- 
% J. Leymarie. October 24, 2016
% LEO, University of Orleans
% ----------------------------------------------------------------------

function [dh1 , dh2] = h_normal_derivatives_kernel_deltacovar(Sigma,Y,z,alpha,btilde,h)

N=size(Y,1);                                                                 % Sample size 

dh1=NaN(3,N);                                                                % h1t partial derivatives (var1, corr12, var2)

dh2=NaN(3,N);                                                                % h2t partial derivatives (var1, corr12, var2)

%===========================
%=== ht(theta) function  ===
%===========================

Y1=Y(:,1);Y2=Y(:,2);                                                         % Returns Y1 and Y2

corr_mat=corrcov(Sigma);                                                     % Correlation matrix

corr12=corr_mat(1,2);                                                        % Estimated correlation parameter Y1 Y2

binf=btilde(1);                                                              % Inferior beta side level for Y2

bsup=btilde(2);                                                              % Superior beta side level for Y2

VaR2alpha=sqrt(Sigma(2,2))*norminv(alpha);                                   % alpha VaR for Y2

VaR2binf=sqrt(Sigma(2,2))*norminv(binf);                                     % binf VaR for Y2

VaR2bsup=sqrt(Sigma(2,2))*norminv(bsup);                                     % bsup VaR for Y2

u1=mvncdf([Y1 VaR2alpha*ones(N,1)],0,Sigma)/alpha;                           % Y1 cdf given Y2<VaRalpha

u1b=(mvncdf([Y1 VaR2bsup*ones(N,1)],0,Sigma)-...                             % Y1 cdf given Y2 in [VaRbinf,VaRbsup]
    mvncdf([Y1 VaR2binf*ones(N,1)],0,Sigma))/(bsup-binf);  

u2=normcdf(Y2,0,sqrt(Sigma(2,2)));                                           % Y2 marginal cdf                          

%==========================
%=== Partial derivative ===
%==========================

for j=1:3                                                                    % Loop on the partial derivatives

    Sigma_z=Sigma;                                                           % Initialization
    
    if j==1                                                                  % Derivative with respect to var1
        
        Sigma_z=[Sigma(1,1)+z sqrt((Sigma(1,1)+z)*Sigma(2,2))*corr12 ; ...
        sqrt((Sigma(1,1)+z)*Sigma(2,2))*corr12 Sigma(2,2)];                  % Augmented var-cov matrix
        
    end                                                                      % End condition j==1
    
    if j==2                                                                  % Derivative with respect to corr12
        
        Sigma_z=[Sigma(1,1) sqrt(Sigma(1,1)*Sigma(2,2))*(corr12+z) ; ...
        sqrt(Sigma(1,1)*Sigma(2,2))*(corr12+z) Sigma(2,2)];                  % Augmented var-cov matrix
       
    end                                                                      % End condition j==2
        
    if j==3                                                                  % Derivative with respect to var2
        
        Sigma_z=[Sigma(1,1) sqrt(Sigma(1,1)*(Sigma(2,2)+z))*corr12 ; ...     
        sqrt(Sigma(1,1)*(Sigma(2,2)+z))*corr12 Sigma(2,2)+z];                % Augmented var-cov matrix
        
    end                                                                      % End condition j==3
    
    VaR2alpha_z=sqrt(Sigma_z(2,2))*norminv(alpha);                           % Augmented alpha VaR

    VaR2binf_z=sqrt(Sigma_z(2,2))*norminv(binf);                             % Augmented binf VaR

    VaR2bsup_z=sqrt(Sigma_z(2,2))*norminv(bsup);                             % Augmented bsup VaR

    u1_z=mvncdf([Y1 VaR2alpha_z*ones(N,1)],0,Sigma_z)/alpha;                 % Augmented Y1 cdf given Y2<VaRalpha

    u1b_z=(mvncdf([Y1 VaR2bsup_z*ones(N,1)],0,Sigma_z)-...                   % Augmented Y1 cdf given Y2 in [VaRbinf,VaRbsup]
    mvncdf([Y1 VaR2binf_z*ones(N,1)],0,Sigma_z))/(bsup-binf);  

    u2_z=normcdf(Y2,0,sqrt(Sigma_z(2,2)));                                   % Augmented Y2 marginal cdf                          
                               
    du1=(u1_z-u1)/z;                                                         % Derivative of the Y1 cdf given Y2<VaRalpha
     
    du1b=(u1b_z-u1b)/z;                                                      % Derivative of the Y1 cdf given Y2 in [VaRbinf,VaRbsup]
    
    du2=(u2_z-u2)/z;                                                         % Derivative of the Y2 marginal cdf  
    
    r1_dkernel1=(1/h)*(normpdf(u1/h)-normpdf((u1-alpha)/h)).*du1;            % 1st derivative of kernel smoothing for h1t
    
    r1_dkernel2=(1/h)*(normpdf(u2/h)-normpdf((u2-alpha)/h)).*du2;            % 2nd derivative of kernel smoothing for h1t
        
    r2_dkernel1=(1/h)*(normpdf(u1b/h)-normpdf((u1b-alpha)/h)).*du1b;         % 1st derivative of kernel smoothing for h2t
    
    r2_dkernel2=(1/h)*(normpdf((u2-binf)/h)-normpdf((u2-bsup)/h)).*du2;      % 2nd derivative of kernel smoothing for h2t
    
    dh1(j,:)=r1_dkernel1.*(u2<=alpha)+r1_dkernel2.*(u1<=alpha);              % h1t partial derivatives

    dh2(j,:)=r2_dkernel1.*((u2<=bsup)-(u2<=binf))+r2_dkernel2.*(u1b<=alpha); % h2t partial derivatives
       
end                                                           % End of loop on j 

end                                                           % End of function

