%-------------------------------------------------------------------------------------------------------------
% Purpose: Simulate conditional correlations and covariances from
% GJRDCC-GARCH parameters estimates, initial conditions and a sample of returns
%-------------------------------------------------------------------------------------------------------------
% Function [Sigmat,rhot]= gjrdcc_evalute(Y,garch_hat,dcc_hat,Q_bar_hat,T)
% Inputs 
%             - Y: (TTx2) vector of observations
%             - garch_hat: (4x2) matrix of GJRGARCH parameters for the two models (first column: asset 1)
%             - dcc_hat: (1x2) or (2x1) vector of DCC parameter
%             - Q_bar_hat: matrix of unconditional correlation
%             - T : size of the in-sample period (to determinate the initial conditions)
% Outputs
%              - Sigmat: (2x2xTT) series of 2x2 matrix of conditional variances
%              - rhot: vector of cond. correlations
%              - Q_in: (2x2xTT) matrices used in the computation of conditional correlation
% ---------------------------------------------------------------------
% C. Hurlin & J. Leymarie - June 20, 2016.
% LEO, University of Orleans
% ---------------------------------------------------------------------

function [Sigmat,rhot,Q_in]= gjrdcc_evaluate(Y,garch_hat,dcc_hat,Q_bar_hat,T)

TT=length(Y);                                  % Sample size

St=NaN(TT,2);                                  % Initialization of the cond. variances

rhot=NaN(TT,1);                                % Initialization of the cond. correlations

Q_in=NaN(2,2,TT);                              % Initialization of Q matrices

back_cast=NaN(1,size(Y,2)); 

for num=1:size(Y,2)

    epsilon=Y(1:T,num);

    back_cast_length = max(floor(length(epsilon)^(1/2)),1);

    back_cast_weights = .05*(.9.^(0:back_cast_length ));

    back_cast_weights = back_cast_weights/sum(back_cast_weights);

    back_cast(num) = back_cast_weights*((epsilon(1:back_cast_length+1)).^2);

    if back_cast==0

        back_cast=mean(epsilon.^2);

    end

end

St(1,:)=garch_hat(1,:)+garch_hat(2,:).*mean(Y(1:T,:).^2)+...
        garch_hat(3,:).*0.5.*mean(Y(1:T,:).^2)+...
        garch_hat(4,:).*back_cast;
    
for t=2:TT

    St(t,:)=garch_hat(1,:)+garch_hat(2,:).*Y(t-1,:).^2+...
        garch_hat(3,:).*(Y(t-1,:)<0).*Y(t-1,:).^2+...
        garch_hat(4,:).*St(t-1,:);

end

Y_std=Y./sqrt(St);    

back_cast = zeros(2);     

w = .06*.94.^(0:sqrt(T));    

w = w'/sum(w);

for t=1:length(w)

    back_cast = back_cast + w(t)*Y_std(t,:)'*Y_std(t,:);

end

Q1_in=(1-dcc_hat(1)-dcc_hat(2))*Q_bar_hat + ...             
    dcc_hat(1)*back_cast+dcc_hat(2)*back_cast; 

R1_in=diag(diag(1./sqrt(Q1_in)))*Q1_in*...                  
    diag(diag(1./sqrt(Q1_in)));

rhot(1)=R1_in(2,1);                            

for t=2:TT     

    Q_in(:,:,t)=(1-dcc_hat(1)-dcc_hat(2))*Q_bar_hat + ...                  
        dcc_hat(1)*Y_std(t-1,:)'*Y_std(t-1,:)+dcc_hat(2)*Q1_in;  

    Q1_in=Q_in(:,:,t); 

    R_in= diag(diag(1./sqrt(Q_in(:,:,t))))*Q_in(:,:,t)*diag(diag(1./sqrt(Q_in(:,:,t))));

    rhot(t)=R_in(1,2);  

end 

Sigmat=NaN(2,2,TT);                                                                                                               % Conditional variances matrices

Sigmat(1,1,:)=St(:,1);                                                                                                               % Conditional variances

Sigmat(2,2,:)=St(:,2);                                                                                                               % Conditional variances

Sigmat(1,2,:)=rhot.*sqrt(St(:,1).*St(:,2));                                                                                  % Conditional covariances

Sigmat(2,1,:)=Sigmat(1,2,:);      

end
