%-------------------------------------------------------------------------------------------------------
% PURPOSE: "Backtesting Marginal Expected Shortfall and Related Systemic Risk Measures"
%           Banulescu, Hurlin, Leymarie, and Scaillet (2019)
%-------------------------------------------------------------------------------------------------------
% Delta CoVaR backtesting exercise at daily horizon h=1 (short-term) at the end of each month
% for the 95 firms (using a GJR-DCC model with normal innovations)
%-------------------------------------------------------------------------------------------------------
% J. Leymarie, October 04, 2017.
% LEO, University of Orleans
%-----------------------------------------------------------------------------------------------------------
% Notes:
% - The code is indiced according to time (Ex : year=2005 and month=1 means that test statistics are computed 
%   using data available until jan. 2005).
% - The excel file "ref_ticker.xlsx" reports the ticker reference associated with each firm=1,...,95 on the 
%   excel database "bdd_for_matlab.xlsx".
% - The parameters of the GJR-DCC model are estimated with the MFE Toolbox.
% - The main outputs of this code are daily Delta CoVaR forecasts, ht process, 
%   test statistics, p-values.
% - The backtesting is computed considering a window size of one month (end of month) between each backtest. 
% - 2 estimation scheme can be considered in this code :
%   * recursive estimation scheme (rw should be set to 0) 
%   * rolling window estimation scheme (rw should be set to 1) 
% - Only the kernel correction to estimation risk is provided (the one reported in the article).
%-------------------------------------------------------------------------------------------------------

clear ; clc ; close all ;  tic

%==================%
%=== Parameters ===%
%==================%

filename='bdd_for_matlab.xlsx';                       % Excel file 

database=xlsread(filename);                           % Data importation 

nb_firm=95;                                           % Number of firm

rw=0;                                                 % 1 if rolling window estimation scheme, 0 if recursive estimation scheme

rw_size=250;                                          % Size of the in-sample rolling windows (if rw=1)

N=250;                                                % Out-of-sample size for the empirical application

first_year=2005;                                      % First year of backtesting computation

last_year=2016;                                       % Last year of backtesting computation

alpha=0.05;                                           % alpha level for quantile of Y2

btilde=[0.25;0.75];binf=btilde(1);bsup=btilde(2);     % Two-sided Y2 truncation

options=optimset('Display','off');                    % Options for parameter's estimation

covar_options = optimset('TolX',10^-12);              % Covar termination tolerance for numerical accuracy

%======================%
%=== Initialization ===%
%======================%

uc_dcovar_risk=NaN((last_year-first_year+1)*12,nb_firm);              % uc dcovar with estimation risk

uc_dcovar_ck=uc_dcovar_risk;                                          % uc dcovar robust statistics

puc_dcovar_risk=uc_dcovar_risk;                                       % uc p-values with estimation risk

puc_dcovar_ck=uc_dcovar_risk;                                         % uc dcovar robust p-values

uc_h1_risk=uc_dcovar_risk; uc_h2_risk=uc_dcovar_risk;                 % uc covar1 and uc covar2 with estimation risk

puc_h1_risk=uc_dcovar_risk; puc_h2_risk=uc_dcovar_risk;               % uc p-values with estimation risk

uc_h1_c=uc_dcovar_risk; uc_h2_c=uc_dcovar_risk;                       % uc covar1 and covar2 robust statistics

puc_h1_c=uc_dcovar_risk; puc_h2_c=uc_dcovar_risk;                     % uc covar1 and covar2 robust p-values

h_global=NaN(N,2,(last_year-first_year+1)*12,nb_firm);                % CoVaR joint violation process

covar1_out=NaN((last_year-first_year+1)*12,nb_firm);                  % covar1 estimates

covar2_out=NaN((last_year-first_year+1)*12,nb_firm);                  % covar2 estimates

dcovar_out=NaN((last_year-first_year+1)*12,nb_firm);                  % Delta CoVaR estimates

theta_global=NaN(11,(last_year-first_year+1)*12,nb_firm);             % Estimated parameters

V_asy_global=NaN(11,11,(last_year-first_year+1)*12,nb_firm);          % Covariance matrix of estimated parameters

for firm=1:nb_firm                                                    % Loop for backtesting firm

for s=1:(last_year-first_year+1)*12                                   % Loop for backtesting year and month

ref=ceil(s/12)-1;                                                     % Year reference

year=first_year+ref;                                                  % Last year for the in-sample period

month=s-ref*12;                                                       % Last month for the in-sample period

data=database(database(:,1)==firm,:);                                 % Data for the current firm  

year_base=data(data(:,2)<year,:);                                     % Data available before year                

year_inprog=data(data(:,2)==year,:);                                  % Data available for the current year

month_inprog=year_inprog(year_inprog(:,3)<=month,:);                  % Selection of available month of the current year

full=[year_base;month_inprog];                                        % In and out sample data until beginning of the sample

if (rw==0) && (length(full)<500+N)                                    % If recursive scheme, a minimum of 500 + N obs. required
                                                                                                                         
    continue
    
end

if (rw==1) && (length(full)<rw_size+N)                                % If rolling scheme, a minimum of rw_size + N obs. required

    continue 
    
end

if (full(end,2)~=year) || (full(end,3)~=month)                        % If no enough data available, go to next firm
    
    break
    
end

out=full(end-N+1:end,:);                                              % Out-of-sample data

in_full=full(1:end-N,:);                                              % In-sample data

if rw==1
    
    in=in_full(end-rw_size+1:end,:);
    
else
   
    in=in_full;
    
end

T=length(in);                                                         % In sample size

TT=T+N;                                                               % Full sample size

in_out=[in;out];                                                      % Full sample data

Y=in_out(:,[4,5]);                                                    % Full sample returns
    
%============================
%=== Estimation In-Sample ===
%============================
        
[theta_in,loglik,Hcond,V_asy,scores,diagnostic] = dcc(Y(1:T,:),[],1,0,1,1,1,1,[],[],[],[],options); 

% If an error occurs in the estimation routine, use the line below to help the algorithm
% theta_init=[0.149 0.065 0.075 0.855 ...                      
%             0.057 0.001 0.268 0.786 ...                    
%             0.663 0.041 0.754]; 
% [theta_in,loglik,Hcond,V_asy,scores,diagnostic] = dcc(Y(1:T,:),[],1,0,1,1,1,1,[],[],[],theta_init,options); 

theta_global(:,s,firm)=theta_in';                                     % I record theta_in for future use

V_asy_global(:,:,s,firm)=V_asy;                                       % I record V_asy for future use

np=length(theta_in);                                                  % Number of model parameter estimates

garch_hat=[theta_in(1:4)' theta_in(5:8)'];                            % GJR-GARCH parameter estimates

dcc_hat=theta_in(10:11);                                              % DCC parameter estimates (without rho_bar)

Q_bar_hat=[1 theta_in(9);theta_in(9) 1];                              % Uncondional correlation parameter estimate
   
%================================================================%
%==== Evaluation of the correlations-variances out-of-sample ====%
%================================================================%

[Sigma_hat,rho_hat,]= ...                                              % Evaluation over the full sample (initial condition based on in-sample period)
    gjrdcc_evaluate([Y;NaN(1,2)],garch_hat,dcc_hat,Q_bar_hat,T);       % We add a NaN(2,1) vector to get the Y's T+N+1 cov. matrix       
                                                                       
Sig_TNh=Sigma_hat(:,:,end);                                            % We extract T+N+1 cov. matrix given T+N 

VaR2alpha_TNh=sqrt(Sig_TNh(2,2))*norminv(alpha);                       % We extract T+N+1 Y2's alpha level VaR given T+N 

VaR2binf_TNh=sqrt(Sig_TNh(2,2))*norminv(binf);                         % We extract T+N+1 Y2's binf level VaR given T+N 

VaR2bsup_TNh=sqrt(Sig_TNh(2,2))*norminv(bsup);                         % We extract T+N+1 Y2's bsup level VaR given T+N 

Sigma_hat=Sigma_hat(:,:,1:TT);                                         % We keep only cov. matrix T=1,...TT in the following

rho_hat=rho_hat(1:TT);                                                 % We keep only corr. vector T=1,...TT in the following
                                                                     
S2_hat=NaN(TT,2);                                                      % Estimated cond. variances (in-sample and out-of-sample)

S2_hat(:,1)=Sigma_hat(1,1,:);S2_hat(:,2)=Sigma_hat(2,2,:);             % Estimated cond. variances (in-sample and out-of-sample)

Sigma_out=Sigma_hat(:,:,T+1:end);                                      % Estimated OOS cond. variance matrices

var1_out=S2_hat(T+1:end,1);var2_out=S2_hat(T+1:end,2);                 % Estimated OOS cond. variances

rho_out=rho_hat(T+1:end);                                              % Estimated OOS cond. correlations

VaR2alpha_out=sqrt(var2_out)*norminv(alpha);                           % Estimated OOS Value-at-Risk of level alpha for Y2
    
VaR2binf_out=sqrt(var2_out)*norminv(binf);                             % Estimated OOS Value-at-Risk of level binf for Y2            
    
VaR2bsup_out=sqrt(var2_out)*norminv(bsup);                             % Estimated OOS Value-at-Risk of level bsup for Y2 
   
%=========================================================%
%==== Estimated Delta CoVaR at time T+N+1 (given T+N) ====%
%=========================================================%

obj = @(x) (mvncdf([x;VaR2alpha_TNh],zeros(2,1),Sig_TNh)/normcdf(VaR2alpha_TNh,0,sqrt(Sig_TNh(2,2)))-alpha)^2;

x0=0; covar1_out(s,firm)=fminsearch(obj,x0,covar_options)/100;                 % CoVaR(alpha,alpha) 

obj = @(x) ((mvncdf([x;VaR2bsup_TNh],zeros(2,1),Sig_TNh)-mvncdf([x;VaR2binf_TNh],zeros(2,1),Sig_TNh))/...
    (normcdf(VaR2bsup_TNh,0,sqrt(Sig_TNh(2,2)))-normcdf(VaR2binf_TNh,0,sqrt(Sig_TNh(2,2))))-alpha)^2;

x0=0; covar2_out(s,firm)=fminsearch(obj,x0,covar_options)/100;                 % "Median" CoVaR(alpha,btilde)

dcovar_out(s,firm)=-(covar1_out(s,firm)-covar2_out(s,firm));                   % (positive) Delta CoVaR estimate at time T+N+1 given T+N
 
%===================================================%
%==== Extraction of asset returns out-of-sample ====%
%===================================================%

Y1_out=Y(T+1:end,1);                                                   % Y1 OOS

Y2_out=Y(T+1:end,2);                                                   % Y2 OOS

Y_out=[Y1_out Y2_out];                                                 % [Y1 Y2] OOS        

%=======================================================================================%
%=== Derivatives of the cond. variance matrices with respect to estimated parameters ===%
%=======================================================================================%

z=10.^-8;                                                                              % Incremental value for derivatives

dSigma=NaN(np,3,TT);                                                                   % Initialization matrix of derivatives

for i=1:np                                                                             % Loop on the derivatives

    theta_in_z=theta_in;                                                               % Initialization 

    theta_in_z(i)=theta_in_z(i)+z;                                                     % Augmented value

    garch_hat_z=[theta_in_z(1:4)' theta_in_z(5:8)'];                                   % GARCH parameters

    dcc_hat_z=theta_in_z(10:11);                                                       % DCC parameters (without rho_bar)

    Q_bar_hat_z=[1 theta_in_z(9);theta_in_z(9) 1];                                     % Uncondional correlation

    [Sigma_hat_z,rho_hat_z,]= gjrdcc_evaluate(Y,garch_hat_z,dcc_hat_z,Q_bar_hat_z,T);  % Evaluation over the full sample (initial condition based on in-sample period)

    dSigma(i,1,:)=(reshape(Sigma_hat_z(1,1,:),TT,1)-reshape(Sigma_hat(1,1,:),TT,1))/z; % Derivatives of cond. variance var1

    dSigma(i,2,:)=(rho_hat_z-rho_hat)/z;                                               % Derivatives of cond. correlation

    dSigma(i,3,:)=(reshape(Sigma_hat_z(2,2,:),TT,1)-reshape(Sigma_hat(2,2,:),TT,1))/z; % Derivatives of cond. variance var2

end                                                                                                                                                  % End loop on the derivatives
        
dSigma_out=dSigma(:,:,T+1:end);                                                        % Out-of-sample derivatives

%============================================================%
%=== Computation of [h1,h2] series and ER corrected terms ===%
%============================================================%
 
%-----------------------------%
%--   With Estimation Risk  --%
%-----------------------------%

ht_risk=NaN(N,2);                                                         % [h1t,h2t] violations (with ER)

for i=1:N
    
    h1t_risk=((mvncdf([Y_out(i,1) VaR2alpha_out(i)],0,Sigma_out(:,:,i)))/alpha<alpha).*(Y_out(i,2)<VaR2alpha_out(i));
    
    h2t_risk=((mvncdf([Y_out(i,1) VaR2bsup_out(i)],0,Sigma_out(:,:,i))-mvncdf([Y_out(i,1) VaR2binf_out(i)],0,Sigma_out(:,:,i)))/(bsup-binf)<alpha).*...
        (Y_out(i,2)>VaR2binf_out(i))&(Y_out(i,2)<VaR2bsup_out(i));
    
    ht_risk(i,:)=[h1t_risk h2t_risk];
    
end

htbar_risk=mean(ht_risk)';                                                % h bar violation (with ER)

muh=[alpha^2;alpha*(bsup-binf)];                                          % h bar expectation under H0

sigmah=[alpha^2*(1-alpha^2) -alpha^3*(bsup-binf);...                      % h bar variance under H0
    -alpha^3*(bsup-binf) alpha*(bsup-binf)*(1-alpha*(bsup-binf))];

h_global(:,:,s,firm)=ht_risk;                                             % I stack ht joint violation in h_global

uc_dcovar_risk(s,firm)=N*(htbar_risk-muh)'*pinv(sigmah)*(htbar_risk-muh); % uc_dcovar_risk statistic (with ER)

puc_dcovar_risk(s,firm)=1-chi2cdf(uc_dcovar_risk(s,firm),2);              % p-value of uc_dcovar_risk (with ER)
 
uc_h1_risk(s,firm)=sqrt(N)*(htbar_risk(1)-muh(1))/sqrt(sigmah(1,1));      % uc for h1 violation (with ER)

puc_h1_risk(s,firm)=2*normcdf(-abs(uc_h1_risk(s,firm)));                  % p-value of uc_h1 (with ER)

uc_h2_risk(s,firm)=sqrt(N)*(htbar_risk(2)-muh(2))/sqrt(sigmah(2,2));      % uc for h2 violation (with ER)

puc_h2_risk(s,firm)=2*normcdf(-abs(uc_h2_risk(s,firm)));                  % p-value of uc_h2 (with ER)

%---------------------------------------------------------------------%
%-- ER corrected by deriving the smoothed [h1,h2] (kernel smoother) --%
%---------------------------------------------------------------------%

r1covart_k=NaN(np,N);                                                     % Initialization of r1covart_k

r2covart_k=NaN(np,N);                                                     % Initialization of r2covart_k

for i=1:N

    [dh1 , dh2] = ...                                                     % kernel h1t, h2t derivatives with respect to [var1;rho;var2]
        h_normal_derivatives_kernel_deltacovar...
        (Sigma_out(:,:,i),Y_out(i,:),z,alpha,btilde,1/N);
    
    r1covart_k(:,i)=dSigma_out(:,:,i)*dh1;                                % h1t kernel derivatives with respect to theta 

    r2covart_k(:,i)=dSigma_out(:,:,i)*dh2;                                % h2t kernel derivatives with respect to theta

end

r1covar_k=mean(r1covart_k,2);                                             % kernel r1covar term

r2covar_k=mean(r2covart_k,2);                                             % kernel r2covar term

sigma_ck=sigmah+...                                                       % h bar corrected variance from ER
    [N*r1covar_k'*V_asy*r1covar_k N*r1covar_k'*V_asy*r2covar_k ; ...
     N*r1covar_k'*V_asy*r2covar_k N*r2covar_k'*V_asy*r2covar_k];

uc_dcovar_ck(s,firm)=N*(htbar_risk-muh)'*pinv(sigma_ck)*(htbar_risk-muh); % uc_dcovar_ck statistic (corrected from ER)

puc_dcovar_ck(s,firm)=1-chi2cdf(uc_dcovar_ck(s,firm),2);                  % p-value of uc_dcovar_ck

uc_h1_c(s,firm)=sqrt(N)*(htbar_risk(1)-muh(1))...                         % uc for h1 violation (corrected from ER)
    /sqrt(sigmah(1,1)+N*r1covar_k'*V_asy*r1covar_k);      

puc_h1_c(s,firm)=2*normcdf(-abs(uc_h1_c(s,firm)));                        % p-value of uc_h1 (corrected from ER)

uc_h2_c(s,firm)=sqrt(N)*(htbar_risk(2)-muh(2))/...                        % uc for h2 violation (corrected from ER)
    sqrt(sigmah(2,2)+N*r2covar_k'*V_asy*r2covar_k);      

puc_h2_c(s,firm)=2*normcdf(-abs(uc_h2_c(s,firm)));                        % p-value of uc_h2 (corrected from ER)

fprintf('  Firm reference              : %1.0f  \n',firm)
fprintf('  End of in-sample period     : year = %1.0f    month = %1.0f \n',in(end,2),in(end,3)),
fprintf('  End of out_of-sample period : year = %1.0f    month = %1.0f \n',year,month)

end 

end

%=============================================%
%== Date vector construction (for graphics) ==%
%=============================================%

date=NaN(1,12*(last_year-first_year+1));
j=1;
for year=first_year:last_year
date(j*12-11:j*12)=datenum(year,1:12,31);
j=j+1;
end
  