%---------------------------------------------------------------------------------------------------------------
% Purpose: Simulate pseudo-sample returns (loop on simulation number is vectorized for the speed time execution)
%---------------------------------------------------------------------------------------------------------------
% Function pseudo_Y = pseudo_sample(Y,garch_hat,dcc_hat,S2_hat,Q_hat,Q_bar_hat,h,nb_pseudo,pseudo_epst,T)
% Inputs 
%             - Y: (TTx2) vector of observations
%             - garch_hat: (4x2) matrix of GJRGARCH parameters for the two models (first column: asset 1)
%             - dcc_hat: (1x2) or (2x1) vector of DCC parameter
%             - S2_hat: (TTx2) Conditionnal variances
%             - Q_hat : (2x2xTT) Q matrices
%             - Q_bar_hat: matrix of unconditional correlation
%             - h: horizon of the systemic risk measure
%             - nb_pseudo: number of pseudo-sample used
%             - pseudo_epst: (hx2xnb_pseudo) random sample 
%             - T : size of the in-sample period (to determinate the initial conditions)
% Outputs
%             - Pseudo_Y: (hx2xnb_pseudo) pseudo Y series
% ---------------------------------------------------------------------
% J. Leymarie - August 4, 2016.
% LEO, University of Orleans
% ---------------------------------------------------------------------

function pseudo_Y=pseudo_sample(Y,garch_hat,dcc_hat,S2_hat,Q_hat,Q_bar_hat,h,nb_pseudo,pseudo_epst,T)

pseudo_Y=NaN(h,2,nb_pseudo);           % Initialization of the nb_pseudo sample of Y returns
  
pseudo_Qlag=Q_hat(:,:,T);              % Initial condition for the Q matrix
    
for t=1:h

    if t==1

    pseudo_St=repmat(garch_hat(1,:),1,1,nb_pseudo)+...
        repmat(garch_hat(2,:).*Y(T,:).^2,1,1,nb_pseudo)+...  
        repmat(garch_hat(3,:).*(Y(T,:)<0).*Y(T,:).^2,1,1,nb_pseudo)+...
        repmat(garch_hat(4,:).*S2_hat(T,:),1,1,nb_pseudo);

    pseudo_rtstd=repmat(Y(T,:),1,1,nb_pseudo)./repmat(sqrt(S2_hat(T,:)),1,1,nb_pseudo);                                                     

    pseudo_Stlag=pseudo_St;

    pseudo_Q=repmat((1-dcc_hat(1)-dcc_hat(2)),2,2,nb_pseudo).*repmat(Q_bar_hat,1,1,nb_pseudo) + ...                  
        repmat(dcc_hat(1),2,2,nb_pseudo).*bsxfun(@times,permute(pseudo_rtstd,[2 1 3]),pseudo_rtstd)+...
        repmat(dcc_hat(2),2,2,nb_pseudo).*repmat(Q_hat(:,:,T),1,1,nb_pseudo);

    pseudo_Qlag=pseudo_Q; 

    R_in=[ones(1,1,nb_pseudo) , pseudo_Q(1,2,:)./sqrt(pseudo_Q(1,1,:).*pseudo_Q(2,2,:)) ;...
         pseudo_Q(2,1,:)./sqrt(pseudo_Q(1,1,:).*pseudo_Q(2,2,:)) , ones(1,1,nb_pseudo)];

    pseudo_rhot=R_in(1,2,:);

    sigma12=[sqrt(pseudo_St(:,1,:)).*sqrt(1-pseudo_rhot.^2) , pseudo_rhot.*sqrt(pseudo_St(:,1,:));...
        zeros(1,1,nb_pseudo) , sqrt(pseudo_St(:,2,:))];

    pseudo_Y(t,:,:)=sum(sigma12.*repmat(pseudo_epst(t,:,:),2,1,1),2);

    else 

    pseudo_St=repmat(garch_hat(1,:),1,1,nb_pseudo)+...
        repmat(garch_hat(2,:),1,1,nb_pseudo).*pseudo_Y(t-1,:,:).^2+...
        repmat(garch_hat(3,:),1,1,nb_pseudo).*(pseudo_Y(t-1,:,:)<0).*pseudo_Y(t-1,:,:).^2+...
        repmat(garch_hat(4,:),1,1,nb_pseudo).*pseudo_Stlag;

    pseudo_rtstd=pseudo_Y(t-1,:,:)./sqrt(pseudo_Stlag);

    pseudo_Stlag=pseudo_St;

    pseudo_Q=repmat((1-dcc_hat(1)-dcc_hat(2)),2,2,nb_pseudo).*repmat(Q_bar_hat,1,1,nb_pseudo) + ...                  
        repmat(dcc_hat(1),2,2,nb_pseudo).*bsxfun(@times,permute(pseudo_rtstd,[2 1 3]),pseudo_rtstd)+...
        repmat(dcc_hat(2),2,2,nb_pseudo).*pseudo_Qlag;

    pseudo_Qlag=pseudo_Q;  

    R_in=[ones(1,1,nb_pseudo) , pseudo_Q(1,2,:)./sqrt(pseudo_Q(1,1,:).*pseudo_Q(2,2,:)) ;...
         pseudo_Q(2,1,:)./sqrt(pseudo_Q(1,1,:).*pseudo_Q(2,2,:)) , ones(1,1,nb_pseudo)];

    pseudo_rhot=R_in(1,2,:);

    sigma12=[sqrt(pseudo_St(:,1,:)).*sqrt(1-pseudo_rhot.^2) , pseudo_rhot.*sqrt(pseudo_St(:,1,:));...
        zeros(1,1,nb_pseudo) , sqrt(pseudo_St(:,2,:))];

    pseudo_Y(t,:,:)=sum(sigma12.*repmat(pseudo_epst(t,:,:),2,1,1),2);

    end 

end

end