
# DESCRIPTION: this file contains the code needed to reproduce Table 4 of the
# working paper "Climate certainty impacts citizens' willingness to take mitigating action".
# See the Supplementary Materials for an explanation of all variables used.

# TO USE: expect the code to take at least 24 hours to run
# download the data file "ClimateCertaintyRaw.csv" from doi: 10.5281/zenodo.3711760
# Packages bayesm, mvtnorm, actuar, coda, EnvStats must be installed
# Set working directory to the folder on your computer where the data file is stored, e.g.:
setwd("C:/Users/Reichl/Desktop/R")


library(bayesm)
library(mvtnorm)
library(actuar)
library(coda)
library(EnvStats)



#load data
rawData <- read.csv2("ClimateCertaintyRaw.csv", sep=";", dec=".")

#check the data was read correctly
View(rawData)


# data must be sorted with respect to variable "Country"
cleanData = rawData[order(rawData$Country),]

# check number of cases, should be 15951
length(cleanData[,1])


nObs = length(cleanData[,1])
nSubs = length(unique(cleanData$Country))

nCountryVars= 4
gammaMat =  array(0,dim=c(nSubs,nCountryVars+1))


i<-1
h<-1
obs=1
obsList<-array()
while(i < (nObs+1)) {
  compare<-cleanData[i,"Country"]
  p=0
  if(i==nObs) {
    obsList[h]=obs
    gammaMat[h,] = c(cleanData[i,"Country"],
                     cleanData$Damage_CC[i],
                     cleanData$SCC[i],
                     cleanData$DonationIndex[i],
                     cleanData$Progress_Paris[i]
    )
    break()
  } else {
    if(cleanData[i+1,"Country"]==compare){
      obs = obs+1
    } else {
      gammaMat[h,] = c(cleanData[i,"Country"],
                       cleanData$Damage_CC[i],
                       cleanData$SCC[i],
                       cleanData$DonationIndex[i],
                       cleanData$Progress_Paris[i]
      )
      p=1
    }
  }
  if(p==1) {
    obsList[h]=obs
    h=h+1
    obs=1

  }
  i=i+1
}

# setting the intercept for all
big1s  = array(1,dim=c(nObs,1))

# setting the matrix with individual-level explanatory variables
bigX   =     cbind(cleanData$ConsumptionMWh,
                   cleanData$MobilityMWh,
                   cleanData$HeatingMWh,
                   cleanData$CollectiveEfficacy,
                   cleanData$Male,
                   cleanData$Urban,
                   cleanData$NumberChildren,
                   cleanData$HouseholdSize,
                   cleanData$Age35to44,
                   cleanData$Age45to54,
                   cleanData$AgeOver55,
                   cleanData$AncillaryBenefit_jobs,
                   cleanData$AncillaryBenefit_environment,
                   cleanData$Status_belowAverage,
                   cleanData$Status_aboveAverage,
                   cleanData$Econ_left,
                   cleanData$Econ_right,
                   cleanData$Social_left,
                   cleanData$Social_right,
                   cleanData$NaturalCauses,
                   cleanData$HumanCauses,
                   cleanData$CC_no,
                   cleanData$CC_Probably,
                   cleanData$CC_Yes,
                   cleanData$Income
)


# setting the vector with revealed donation decisions of all
Choice = cleanData$Donation

# setting the matrix with country-specific explanatory variables
bigZ4Gamma = cbind(array(1,dim=c(nSubs,1)),gammaMat[,-1])

# setting the vector with country-specific intercepts
bigZ4Delta = array(1,dim=c(nSubs,1))



############## START # sampler to get a draw of latent WTD #######################
sampleYAll0to5<-function(yAll, Choice, newXBeta, new1Alpha, newSigExt, nSubs) {


  for(i in 1:nSubs){

    thisIndex = (sum(obsList[1:i])-obsList[i]+1):sum(obsList[1:i])

    yUnit        <- as.vector(Choice[thisIndex])
    xbUnit       <- newXBeta[thisIndex]
    alphaUnit    <- new1Alpha[thisIndex]
    sigUnit      <- newSigExt[thisIndex]

    alim = as.vector(yUnit)

    blim = as.vector(alim+1)
    blim = ifelse(yUnit == 5, 100, blim)   #lower limit set to 100 for avoiding spuriously high WTPs for some

    for (j in 1:obsList[i]) {
        yAll[thisIndex[j]] <- rlnormTrunc(1, meanlog = alphaUnit[j] + xbUnit[j], sdlog = sigUnit[j], min = alim[j], max = blim[j])
    }
  }

  yAll = ifelse(log(yAll)== Inf,    100  ,yAll)
  yAll = ifelse(log(yAll)==-Inf, -0.0001 ,yAll)


  return(yAll)
}
############ END # sampler to get a draw of latent WTD #########################




########### START # sampler to obtain a draw of coeff_i = random intercept/coefficent  ##############################
runiregGCoeffi = function(bigY, bigX, newsig, vCoeffPrior, mCoeffPrior) {
  nb = length(bigX[1,])
  XpXScaled = crossprod(bigX/newsig)
  XpyScaled = crossprod(bigX/newsig,bigY/newsig)
  IR = backsolve(chol(XpXScaled + vCoeffPrior), diag(nb))
  btilde = crossprod(t(IR)) %*% (XpyScaled + vCoeffPrior %*% mCoeffPrior)
  betac = btilde + IR %*% rnorm(nb)
  list(betadraw = betac)
}
########### END   # sampler to obtain a draw of coeff_i = random intercept/coefficent   ##############################

########### START # sampler to obtain a draw of coeff = fixed coefficent  ##############################
runiregGCoeff = function(bigY, bigX, newsig, vCoeffPrior, mCoeffPrior) {
  nb = length(bigX[1,])
  Xscaled=as.matrix(bigX/newsig)
  XpXScaled = crossprod(Xscaled)
  XpyScaled = crossprod(Xscaled,bigY/newsig)
  IR = backsolve(chol(XpXScaled + vCoeffPrior), diag(nb))
  btilde = crossprod(t(IR)) %*% (XpyScaled + vCoeffPrior %*% mCoeffPrior)
  betac = btilde + IR %*% rnorm(nb)
  list(betadraw = betac)
}
########### END   # sampler to obtain a draw of coeff = fixed coefficent   ##############################


########### START # sampler to obtain a draw of sig2  #######################################
runiregGSigmasq = function(res, c0, C0, ni) {
  s = t(res) %*% res
  sigmasq = rinvgamma(1,c0+ni/2,scale = C0+s/2)
  list(sigmasqdraw = sigmasq)
}
########### END   # sampler to obtain a draw of sig2  #######################################


########### START # sampler to obtain a draw of C0  #######################################
runiregGC0 = function(nSubs, newSig, c0, g0, G0) {
  C0=rgamma(1,g0+c0*nSubs, G0+sum((1/newSig)^2))
  list(C0draw = C0)

}
########### END   # sampler to obtain a draw of sig2  #######################################




####### prepares variables and vectors reuqired for Gibbs sampling
set.seed(1)
sim<-20000
keep<-1

# extends the design matrix by 1 column to accomodate PN score
bigX = cbind(bigX,rnorm(nObs)) # extends the design matrix by 1 column to accomodate PN score


# derives the dimension of all explanatory vectors/matrices
nx = length(bigX[1,])   # +1 is for the PN variable
ng = length(bigZ4Gamma[1,])
nd = length(bigZ4Delta[1,])


# sets prior parameters for country-specific models parameters
gammaNu  = 4 #(ng + 1)
gammaV   = as.matrix(gammaNu * 1)/2
gammaA   = diag(0.01, nrow=ng)
gammaBar = as.matrix(rep(0,ng))
c0 = 1
newC0 = 10
g0 = 1
G0 = 0.01

# sets prior parameters for fixed coefficients
betaA   = diag(0.01, nrow=nx)
betaBar = array(0,dim=c(nx))


# prepares vectros/matrices for storing the Gibbs sampler draws
yAllDraws       <- array(0,dim=c(sim/keep,nObs))
betaDraws       <- array(0,dim=c(sim/keep,nx))
alphaDraws      <- array(0,dim=c(sim/keep,nSubs))
sigmaDraws      <- array(0,dim=c(sim/keep,nSubs))
sigmaGammaDraws <- array(0,dim=c(sim/keep))
gammaDraws      <- array(0,dim=c(sim/keep,ng))
sigmaDeltaDraws <- array(0,dim=c(sim/keep,nx,nx))
deltaDraws      <- array(0,dim=c(sim/keep,nd,nx))
C0Draws         <- array(0,dim=c(sim/keep))

# sets the starting values of all parameters for Gibbs sampling
newSig<-rep(5,nSubs)
newSigExt<-rep(1,sum(obsList))
newVGammaInv = 1
newGamma = rep(0,ng)
newVDeltaInv = diag(1/100,nrow=nx)
newDelta= rep(0,nx)
newAlpha = rep(0,nSubs)
newBeta  = array(0,dim=c(nx))
newMalpha = array(0,dim=c(nSubs,1))
new1Alpha = rep(0,nObs)
newXBeta = rep(1,nObs)
newYallL = Choice

####### END # preparation of vectors needed during sampling #############################


###### START # the Gibbs sampler of the posterior distribution as described in section 2 of the Supplementary Materials#####
j<-2
i<-1
while(i < (sim+1)) {

  newYallL = sampleYAll0to5(newYallL, Choice, newXBeta, new1Alpha, newSigExt, nSubs)
  newYallN = log(newYallL)

  for(h in 1:nSubs){

    thisIndex = (sum(obsList[1:h])-obsList[h]+1):sum(obsList[1:h])

    yi           <- as.matrix(newYallN[thisIndex])
    newXBetai    <- as.matrix(newXBeta[thisIndex])
    bigXi        <- as.matrix(bigX[thisIndex,])
    big1si       <- as.matrix(big1s[thisIndex,])


    newMalphai = newMalpha[h,]
    alphai    <- as.vector(runiregGCoeffi(yi-newXBetai, big1si, newSig[h], newVGammaInv, newMalphai)$betadraw)
    newAlpha[h] = alphai
    new1Alpha[thisIndex] = newAlpha[h]

    newSig[h] = sqrt(as.vector(runiregGSigmasq((yi - new1Alpha[thisIndex] - newXBeta[thisIndex]), c0, newC0, obsList[h])$sigmasqdraw))
    newSigExt[thisIndex] = newSig[h]

  }

  newBeta   = as.matrix(runiregGCoeff(as.vector(newYallN-new1Alpha),  bigX , newSigExt, betaA, betaBar)$betadraw)
  bigX[,nx] = rnorm(n = nObs, mean = cleanData$PNscore_mean, sd = cleanData$PNscore_sd)
  newXBeta  = as.matrix(bigX) %*% newBeta


  if(i<50) {
    newSig<-rep(1,nSubs)
    newSigExt<-rep(1,nObs)
  }

  newRegDraw   =  rmultireg(as.matrix(newAlpha), bigZ4Gamma, gammaBar, gammaA, gammaNu, gammaV)
  newGamma     <- newRegDraw$B
  newVGamma    <- newRegDraw$Sigma
  newVGammaInv <- 1/newVGamma
  newMalpha    <- bigZ4Gamma%*%newGamma


  newC0 = runiregGC0(nSubs, newSig, c0, g0, G0)$C0draw


  if(i%%keep==0) {
    yAllDraws[j,]<- newYallL
    betaDraws[j,]<- newBeta
    alphaDraws[j,]<-newAlpha
    sigmaDraws[j,]<-newSig
    sigmaGammaDraws[j]<-newVGamma
    gammaDraws[j,]<-newGamma
    C0Draws[j]<-newC0

    j<-j+1
  }


  if(i%%100==0) {

    print(c(newBeta))
    print(c(j*keep,nObs,nx))

    par(mfrow=c(2,4))
      for(gI in 1:3) {
        plot(gammaDraws[c(1:(j-1)),gI],type="l")
      }
    for(gI in (nx-(5)+1):(nx)) {
      plot(betaDraws[c(1:(j-1)),gI],type="l")
    }

  }

  i<-i+1
}

###### END # the Gibbs sampler of the posterior distribution as described in section 2 of the Supplementary Materials #####


###### START # calculates the marginal efffects - code will produce results in Table 4 ####################################
burn  =  5000
stopI = j#25000
library(MCMCpack)

binaryB = rep(0, ng)
for(i in 1:nx) {
  if (length(unique(bigX[,i])) == 2) binaryB[i] = 1 else binaryB[i] = 0
}
#for intercept var set binary manually to 1
#binaryB[1] = 1

deltaBDraws = array(0,dim=c(length(burn:stopI),nx))
for(h in burn:(stopI-1)){
    if((h-burn+1) %% 1000 == 0) {print(h)}
      for(nI in 1:nx){
        deltaBDraws[h-burn+1, nI] = mean(exp(log(yAllDraws[h,])
                                                      + (-bigX[,nI] * betaDraws[h,nI]) + (if(binaryB[nI] == 1) 1 else(mean(bigX[,nI])+1)) * betaDraws[h,nI]
                                                      ) -
                                                  exp(log(yAllDraws[h,])
                                                      + (-bigX[,nI] * betaDraws[h,nI]) + (if(binaryB[nI] == 1) 0 else(mean(bigX[,nI]))) * betaDraws[h,nI]
                                                    )
                                              )
    }
}


margBetaOutHP<-array(0,dim=c(nx,7))
for(w in 1:nx){
  margBetaOutHP[w,3]<-mean(deltaBDraws[,w])
  margBetaOutHP[w,6]<-sd(deltaBDraws[,w])
  margBetaOutHP[w,7]<-sum(ifelse(deltaBDraws[,w]>0,1,0))/length(deltaBDraws[,w])

  margBetaOutHP[w,1]<-HPDinterval(as.mcmc(c(deltaBDraws[,w])), prob=0.95)[,1]
  margBetaOutHP[w,5]<-HPDinterval(as.mcmc(c(deltaBDraws[,w])), prob=0.95)[,2]
  margBetaOutHP[w,2]<-HPDinterval(as.mcmc(c(deltaBDraws[,w])), prob=0.90)[,1]
  margBetaOutHP[w,4]<-HPDinterval(as.mcmc(c(deltaBDraws[,w])), prob=0.90)[,2]
}
round(margBetaOutHP[,],8)


binaryG = rep(0, ng)
for(i in 1:ng) {
  if (length(unique(bigZ4Gamma[,i])) == 2) binaryG[i] = 1 else binaryG[i] = 0
}
binaryG[1] = 1

bigZ4GammaExtT = array(0,dim=c(ng,nObs))
for(cI in 1:nSubs) {
  thisIndex = (sum(obsList[1:cI])-obsList[cI]+1):sum(obsList[1:cI])
  bigZ4GammaExtT[,thisIndex] = bigZ4Gamma[cI,]
}
bigZ4GammaExt = t(bigZ4GammaExtT)



deltaGDraws = array(0,dim=c(length(burn:stopI),nx))
for(h in burn:(stopI-1)){
    if((h-burn+1) %% 1000 == 0) {print(h)}
    for(nI in 1:ng){

      deltaGDraws[h-burn+1, nI] = mean(exp(log(yAllDraws[h,])
                                                    + (-bigZ4GammaExt[,nI] * gammaDraws[h,nI]) + (if(binaryG[nI] == 1) 1 else(mean(bigZ4GammaExt[,nI])+1)) * gammaDraws[h,nI]
                                                ) -
                                                exp(log(yAllDraws[h,])
                                                    + (-bigZ4GammaExt[,nI] * gammaDraws[h,nI]) + (if(binaryG[nI] == 1) 0 else(mean(bigZ4GammaExt[,nI]))) * gammaDraws[h,nI]
                                                )
                                           )
    }
}


margGammaOutHP<-array(0,dim=c(ng,7))
for(w in 1:ng){
  margGammaOutHP[w,3]<-mean(deltaGDraws[,w])
  margGammaOutHP[w,6]<-sd(deltaGDraws[,w])
  margGammaOutHP[w,7]<-sum(ifelse(deltaGDraws[,w]>0,1,0))/length(deltaGDraws[,w])

  margGammaOutHP[w,1]<-HPDinterval(as.mcmc(c(deltaGDraws[,w])), prob=0.95)[,1]
  margGammaOutHP[w,5]<-HPDinterval(as.mcmc(c(deltaGDraws[,w])), prob=0.95)[,2]
  margGammaOutHP[w,2]<-HPDinterval(as.mcmc(c(deltaGDraws[,w])), prob=0.90)[,1]
  margGammaOutHP[w,4]<-HPDinterval(as.mcmc(c(deltaGDraws[,w])), prob=0.90)[,2]
}
round(margGammaOutHP[-1,],8)

###### START # calculates the marginal efffects - code will produce results in Table 4 ########################################## START # calculates the marginal efffects - code will produce results in Table 4 ####################################
