#'========================================================================================
#' Project: Van Dijk et al. (2021), Nature Food
#' Subject: Code to replicate figures in the main paper and si
#' Author:  Michiel van Dijk
#' Contact: michiel.vandijk@wur.nl
#' Website: www.michielvandijk.org
#'========================================================================================

# ========================================================================================
# SET UP ---------------------------------------------------------------------------------
# ========================================================================================

# Load/install pacman for p_load, which installs all necessary packages on the fly.
if(!require(pacman)) install.packages("pacman")
library(pacman)

# Load key packages
p_load("tidyverse", "scales", "glue", "cowplot", "readxl", "RColorBrewer", "ggpubr", "patchwork")


# ========================================================================================
# LOAD DATA ------------------------------------------------------------------------------
# ========================================================================================

# Make sure to copy all scripts and data from the repository to the same folder.
# Replace the parameter "folder" with the name of the folder on your machine where
# you stored the data.
folder <- "C:/temp"

# gfsp_db
gfsp_db_file <- file.path(folder, "gfsp_db_v1.0.0.xlsx")
db_raw <- read_excel(gfsp_db_file, sheet = "gfsp_db")
db_hist <-read_excel(gfsp_db_file, sheet = "hist_db")

# slr_db_file
slr_db_file <- file.path(folder, "slr_db_v1.0.0.xlsx")
study_level <-  read_excel(slr_db_file, sheet = "study_level") %>%
  rename(study_year = year)
study_short_clean <- read_excel(slr_db_file, sheet = "studies") %>%
  dplyr::select(study_short_clean, study_short, study_year = year)
model_raw <- read_excel(slr_db_file, sheet = "model")
driver_raw <- read_excel(slr_db_file, sheet = "driver")
indicator_raw <- read_excel(slr_db_file, sheet = "indicator")

# Conduct meta-regression - see script for regression results
source(file.path(folder, "nf_meta_regression.r"))


# ========================================================================================
# PREPARATION ----------------------------------------------------------------------------
# ========================================================================================

# Only select baseline projections
db <- db_raw %>%
  filter(type == "Baseline")

# Create factor variables
db <- db %>%
  mutate(ssp = factor(ssp, levels = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5", "No class")),
         rcp = factor(rcp, levels = c("NOCC", "RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5", "No class")),
         pure_ssp = factor(pure_ssp, levels = c("Pure SSP", "Derived SSP", "No class")),
         pure_rcp = factor(pure_rcp, levels = c("Pure RCP", "Derived RCP", "No class")))

# Add study_short_clean
db <- db %>%
  left_join(study_short_clean)


# ========================================================================================
# FIGURE 1 -------------------------------------------------------------------------------
# ========================================================================================

col8   <- c("#999999", "#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7")
col12 <- c("#88CCEE", "#CC6677", "#DDCC77", "#117733", "#332288", "#AA4499", "#44AA99",
           "#999933", "#882255", "#661100", "#6699CC", "#888888")

# Single-multi over time
p_year <- dplyr::select(study_level, study_short, study_year) %>%
  mutate(study_year = as.integer(study_year)) %>%
  unique() %>%
  group_by(study_year) %>%
  count

range <- data.frame(study_year = c(min(p_year$study_year): max(p_year$study_year)))
single_multi <- study_level %>%
  dplyr::select(study_year, single_multi_model) %>%
  group_by(study_year, single_multi_model) %>%
  count() %>%
  spread(single_multi_model, n) %>%
  left_join(range,.) %>%
  ungroup() %>%
  mutate(multi = ifelse(is.na(multi), 0, multi),
         single = ifelse(is.na(single), 0, single),
         n = multi + single,
         total = cumsum(n))

n_type <- single_multi %>%
  dplyr::select(-n, -total) %>%
  gather(type, value, -study_year)

fig_n_studies <- ggplot() +
  geom_col(data = n_type, aes(x = study_year, y = value, fill = type), width = 0.5, color = "black") +
  geom_line(data = single_multi, aes(x = study_year, y = total), size = 2) +
  labs(x = "", y = "number of studies", fill = "") +
  theme_classic(base_size = 14) +
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank()) +
  scale_x_continuous(breaks = c(min(p_year$study_year):max(p_year$study_year)), expand = c(0.01,0.01)) +
  scale_y_continuous(breaks = pretty_breaks(n = 10), expand = expansion(mult = c(0, .1))) +
  scale_fill_manual(values = col8[c(2,3)], breaks = c("single", "multi"), labels = c("Single-model study", "Multi-model study")) +
  theme(legend.position=c(0.5,0.9), legend.direction = "vertical") +
  geom_vline(xintercept = 2007.5, linetype = "dashed") +
  annotate("rect", xmin = 2005, xmax = 2010, ymin = 32, ymax = 48, fill = "white") +
  annotate("text", x = 2007.5, y = 40, label = "2007/2008 \n Global food price crisis", size = 3.5)


# ========================================================================================
# FIGURE 2 -------------------------------------------------------------------------------
# ========================================================================================

# Figure 2a
n_method <- study_level %>%
  group_by(methodology) %>%
  count

study <- study_level %>%
  mutate(methodology = recode(methodology, "Statistical extrapolation" = "Statistical \n extrapolation",
                              "Simulation models" = "Simulation \n models",
                              "Combination of methods" = "Combination \n of methods"))

fig_methods <- ggplot(data = study, aes(x = fct_infreq(methodology), fill = methodology)) +
  geom_bar(color = "black") +
  labs(x = "", y = "Number of studies") +
  theme_classic(base_size = 12) +
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank()) +
  scale_y_continuous(breaks = pretty_breaks(n = 10), expand = expansion(mult = c(0, .1))) +
  theme(legend.position="none") +
  scale_fill_manual(values =  col8[c(4,6,7)])

# Figure 2b
model <- model_raw %>%
  mutate(type2 = ifelse(!type %in% c("CGE", "PE"), "Other", type),
         type2 = factor(type2, levels = c("CGE", "PE", "Other")))

n_model <- model %>%
  count(model)

n_model_type <- model %>%
  count(type)

fig_models <- ggplot(model, aes(x = fct_rev(fct_infreq(model)), fill = type2)) +
  geom_bar(color = "black", size = 0.01, width = 1) +
  labs(x = "", y = "Number of studies", fill = "Type of model") +
  theme_classic(base_size = 12) +
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank()) +
  scale_y_continuous(expand = expansion(mult = c(0, .1)), breaks = pretty_breaks(10)) +
  scale_fill_manual(values = col8[c(1,2,3)]) +
  theme(legend.position = c(0.7, 0.6)) +
  coord_flip()

# Figure 2
fig_methods_models <- plot_grid(fig_methods,
                                fig_models,
                                align = 'h', rel_widths = c(1, 1), labels = "auto")



# ========================================================================================
# FIGURE 3--------------------------------------------------------------------------------
# ========================================================================================

# Figure 3a
driver <- driver_raw %>%
  left_join(study_short_clean) %>%
  mutate(driver = ifelse(driver == "Urbanisation", "Urbanization", driver))

col_drivers <- rep(col8[c(1,2)], 6)
names(col_drivers) <- names(sort(table(driver$driver), decreasing = T))

n_driver <- driver %>%
  count(driver)

fig_drivers <- ggplot(data = driver, aes(x = fct_rev(fct_infreq(driver)), fill = driver)) +
  geom_bar(color = "black", size = 0.01, width = 1) +
  labs(x = "", y = "Number of studies") +
  theme_classic(base_size = 10) +
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank()) +
  scale_y_continuous(expand = expansion(mult = c(0, .1)), breaks = pretty_breaks(10)) +
  scale_fill_manual(values = col_drivers) +
  theme(legend.position="none") +
  coord_flip()

# Figure 3b
indicator <- indicator_raw %>%
  pivot_longer(names_to = "variable",
               values_to = "code",
               !c(study_short, year)) %>%
  mutate(code = ifelse(variable %in% c("CU", "FP") & code %in% c("AN", "AH"), "A", code),
         code = case_when(
           variable %in% "other" & is.na(code) ~"N",
           variable %in% "other" & !is.na(code) ~ "A",
           TRUE ~ code),
         variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU", "other"),
                           labels = c("Per capita\nfood\ndemand", "Total food\n demand",
                                      "Population\nat risk\nof hunger",
                                      "Food price", "Childhood\nundernutrition", "Other"))) %>%
  filter(variable != "Total food\n demand", code != c("N"))

n_indicator <- indicator %>%
  count(variable)

fig_ind_cov <- ggplot(data = indicator, aes(x = variable, fill = variable)) +
  geom_bar(color = "black", size = 0.01, width = 1) +
  scale_y_continuous(expand = expansion(mult = c(0, 0.1))) +
  scale_fill_manual(values = col8[c(3,4,6:8)]) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic(base_size = 10) +
  guides(fill = F)

# Figure 3
fig_driv_cov <- plot_grid( fig_drivers, fig_ind_cov,
                          align = 'h', rel_widths = c(1, 1),
                          labels = "auto", axis = "tb")


# ========================================================================================
# FIGURE 4a,b ----------------------------------------------------------------------------
# ========================================================================================

# set colors
ssp_col <- c(brewer.pal(n = 5, name = "Set1"), "grey50")
names(ssp_col) <- c("SSP3", "SSP2", "SSP1", "SSP5", "SSP4", "No class")
bw_col <- c("NOCC" = "grey30", "RCP8.5" = "grey60")


# PREPARE DATA ---------------------------------------------------------------------------

# Select historical data
calo_hist <- db_hist %>%
  filter(variable == "CALO", year >= 2000)

# Select CALO data
calo_db <- db %>%
  filter(variable == "CALO",
         ssp != "No class", rcp != "No class")

# Only select NOCC
calo_nocc <- calo_db %>%
  filter(rcp == "NOCC")

# Calculate mean for NOCC, SSP (pure and derived)
calo_nocc_mean <- calo_nocc %>%
  filter(rcp %in% c("NOCC")) %>%
  group_by(year, ssp) %>%
  summarize(value = mean(proj, na.rm = T), .groups = "drop")

# Calculate levels for point estimates
calo_pe_l <- calo_pe %>%
  mutate(fit_l = ((fit/100)+1)*calo_hist$value[calo_hist$year == 2010],
         upper_l = ((upper/100)+1)*calo_hist$value[calo_hist$year == 2010],
         lower_l = ((lower/100)+1)*calo_hist$value[calo_hist$year == 2010],
         variable = "CALO")

# Select lower and upper range of 95% error bars
q05_calo <- min(calo_pe_l$lower_l[calo_pe_l$rcp == "NOCC"])
q95_calo <- max(calo_pe_l$upper_l[calo_pe_l$rcp == "NOCC"])
q05_calo_cc <- min(calo_pe_l$lower_l[calo_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])
q95_calo_cc <- max(calo_pe_l$upper_l[calo_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])

# Select 2050 data
calo_cc_2050 <- calo_db %>%
  filter(
    year == 2050) %>%
  group_by(ssp, rcp) %>%
  mutate(max_y = max(proj, na.rm = T),
         min_y = min(proj, na.rm = T),
         mean = mean(proj, na.rm = T)) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}"))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
calo_cc_2050_lab <- calo_cc_2050 %>%
  ungroup() %>%
  dplyr::select(rcp, ssp, n_stud_scen) %>%
  unique

# Set limits
y_lo_calo <- 2200
y_up_calo <- 4000

# labels for ssp as we remove facet strip
ssp_label_calo = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                            label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                            x = c(1.5, 1.5, 1.5, 1.5, 1.5),
                            y = rep(c(y_up_calo),5))

# labels for ssp all rcp as we remove facet strip
ssp_label_calo_all = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                                label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                                x = c(3, 3, 3, 2, 1.5),
                                y = rep(c(y_up_calo),5))

# Shade for bandwidth
bw_calo <- data.frame(xmin = c(-Inf, -Inf),
                      xmax = c(Inf, Inf),
                      ymin = c(q05_calo, q05_calo_cc),
                      ymax = c(q95_calo, q95_calo_cc),
                      calo = c("NOCC", "RCP8.5"))

# DRAW PLOT ------------------------------------------------------------------------------------------

# Line plot
# Add special legend that covers shape and linetype
fig_calo_nocc <- ggplot() +
  geom_hline(yintercept = calo_hist$value[calo_hist$year == 2010], size = 0.5, linetype = "solid") +
  geom_line(data = calo_nocc,
            aes(x = year, y = proj, colour = ssp, linetype = pure_ssp,
                group = interaction(projection, study_short, model)),
            size = 1, alpha = 0.3) +
  geom_line(data = calo_nocc_mean, aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_point(data = filter(calo_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050)),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_point(data = calo_nocc,
             aes(x = year, y = proj, shape = pure_ssp),
             size = 5, colour = "transparent") +
  geom_line(data = calo_hist,
            aes(x = year, y = value),
            colour = "black", size = 2) +
  scale_colour_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  theme_classic() +
  theme(panel.border = element_rect(colour = "black", fill=NA)) +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_calo, y_up_calo),
                     sec.axis = sec_axis(~((./calo_hist$value[calo_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10)),
                     breaks = scales::pretty_breaks(n = 10)) +
  labs(x = NULL, y = "Per capita food consumption (kcal/cap/day)", colour = NULL, linetype = NULL, shape = NULL) +
  guides(colour = guide_legend(override.aes=list(shape = NA))) +
  guides(lty = guide_legend(override.aes=list(color = "black", alpha = 1)))


# 2050 plot, only NOCC and RCP8.5
fig_calo_cc <- ggplot(data = filter(calo_cc_2050, rcp %in% c("NOCC", "RCP8.5"))) +
  geom_rect(data = bw_calo,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = calo),
            alpha = 0.2) +
  geom_hline(yintercept = calo_hist$value[calo_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = filter(calo_pe_l, rcp %in% c("NOCC", "RCP8.5")),
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = filter(calo_pe_l, rcp %in% c("NOCC", "RCP8.5")),
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x", ) +
  scale_y_continuous(labels = scales::comma,
                     limits = c(y_lo_calo, y_up_calo),
                     sec.axis = sec_axis(~((./calo_hist$value[calo_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Per capita food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "", colour = NULL, linetype = NULL, shape = NULL, fill = NULL) +
  rremove("grid") +
  guides(color = F, shape = F, fill = F) +
  geom_text(data = filter(calo_cc_2050_lab, rcp %in% c("NOCC", "RCP8.5")),
            aes(x = rcp, y = 3900, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_calo, aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())

# 2050 plot, all rcp
fig_calo_cc_all <- ggplot(data = calo_cc_2050) +
  geom_rect(data = bw_calo,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = calo),
            alpha = 0.2) +
  geom_hline(yintercept = calo_hist$value[calo_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = calo_pe_l,
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = calo_pe_l,
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x", ) +
  scale_y_continuous(labels = scales::comma,
                     limits = c(y_lo_calo, y_up_calo),
                     sec.axis = sec_axis(~((./calo_hist$value[calo_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Per capita food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "Per capita food consumption (kcal/cap/day)",
       colour = NULL, linetype = NULL, shape = NULL, fill = NULL) +
  rremove("grid") +
  guides(fill = F) +
  geom_text(data = calo_cc_2050_lab,
            aes(x = rcp, y = 3900, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_calo_all, aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())


# ========================================================================================
# FIGURE 4c,d ----------------------------------------------------------------------------
# ========================================================================================

# PREPARE DATA ---------------------------------------------------------------------------
# Select historical data
cons_hist <- db_hist %>%
  filter(variable %in% c("CONS"), year >= 2000)

# Select CONS data
cons_db <- db %>%
  filter(variable == "CONS",
         ssp != "No class", rcp != "No class")

# Only select NOCC
cons_nocc <- cons_db %>%
  filter(rcp == "NOCC")

# Calculate mean for NOCC, SSP (pure and derived)
cons_nocc_mean <- cons_nocc %>%
  filter(rcp %in% c("NOCC")) %>%
  group_by(year, ssp) %>%
  summarize(value = mean(proj, na.rm = T), .groups = "drop")

# Calculate levels for point estimates
cons_pe_l <- cons_pe %>%
  mutate(fit_l = ((fit/100)+1)*cons_hist$value[cons_hist$year == 2010],
         upper_l = ((upper/100)+1)*cons_hist$value[cons_hist$year == 2010],
         lower_l = ((lower/100)+1)*cons_hist$value[cons_hist$year == 2010],
         variable = "CONS")

# Select lower and upper range of 95% error bars
q05_cons <- min(cons_pe_l$lower_l[cons_pe_l$rcp == "NOCC"])
q95_cons <- max(cons_pe_l$upper_l[cons_pe_l$rcp == "NOCC"])
q05_cons_cc <- min(cons_pe_l$lower_l[cons_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])
q95_cons_cc <- max(cons_pe_l$upper_l[cons_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])

# Select 2050 data
cons_cc_2050 <- cons_db %>%
  filter(
    year == 2050) %>%
  group_by(ssp, rcp) %>%
  mutate(max_y = max(proj, na.rm = T),
         min_y = min(proj, na.rm = T),
         mean = mean(proj, na.rm = T)) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}"))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
cons_cc_2050_lab <- cons_cc_2050 %>%
  ungroup() %>%
  dplyr::select(rcp, ssp, n_stud_scen) %>%
  unique

# Set limits
y_lo_cons <- 5.5
y_up_cons <- 13.5

# labels for ssp as we remove facet strip
ssp_label_cons = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                            label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                            x = c(1.5, 1.5, 1.5, 1.5, 1.5),
                            y = rep(c(y_up_cons),5))

# labels for ssp as we remove facet strip
ssp_label_cons_all = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                                label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                                x = c(3, 3, 3, 2, 1.5),
                                y = rep(c(y_up_cons),5))

# Shade for bandwidth
bw_cons <- data.frame(xmin = c(-Inf, -Inf),
                      xmax = c(Inf, Inf),
                      ymin = c(q05_cons, q05_cons_cc),
                      ymax = c(q95_cons, q95_cons_cc),
                      cons = c("NOCC", "RCP8.5"))


# DRAW PLOT ------------------------------------------------------------------------------

# line plot
# Added a hack to make sure SSP2 is plotted over SSP3 as the former is more important
fig_cons_nocc <- ggplot() +
  geom_hline(yintercept = cons_hist$value[cons_hist$year == 2010], size = 0.5, linetype = "solid") +
  geom_line(data = cons_nocc,
            aes(x = year, y = proj, colour = ssp, linetype = pure_ssp,
                group = interaction(projection, study_short, model)),
            size = 1, alpha = 0.3) +
  geom_line(data = filter(cons_nocc_mean, ssp != "SSP2"),
            aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_line(data = filter(cons_nocc_mean, ssp == "SSP2"),
            aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_point(data = filter(cons_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050), ssp != "SSP2"),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_point(data = filter(cons_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050), ssp == "SSP2"),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_line(data = cons_hist, aes(x = year, y = value),
            colour = "black", size = 2) +
  scale_colour_manual(values = ssp_col) +
  theme_classic() +
  theme(legend.position = c(0.1, 0.7),
        panel.border = element_rect(colour = "black", fill=NA)) +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_cons, y_up_cons),
                     sec.axis = sec_axis(~((./cons_hist$value[cons_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10)),
                     breaks = scales::pretty_breaks(n = 10)) +
  labs(x = NULL, y = "Total food consumption (1e15 kcal)", colour = NULL, linetype = NULL) +
  guides(linetype = F, colour = F)


# 2050 plot, only NOCC and RCP8.5
fig_cons_cc <- ggplot(data = filter(cons_cc_2050, rcp %in% c("NOCC", "RCP8.5"))) +
  geom_rect(data = bw_cons,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = cons),
            alpha = 0.2) +
  geom_hline(yintercept = cons_hist$value[cons_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = filter(cons_pe_l, rcp %in% c("NOCC", "RCP8.5")),
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = filter(cons_pe_l, rcp %in% c("NOCC", "RCP8.5")),
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x") +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_cons, y_up_cons),
                     sec.axis = sec_axis(~((./cons_hist$value[cons_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Global food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "", colour = NULL, linetype = NULL, fill = NULL) +
  rremove("grid") +
  guides(color = F, shape = F, fill = F) +
  geom_text(data = filter(cons_cc_2050_lab, rcp %in% c("NOCC", "RCP8.5")),
            aes(x = rcp, y = 13, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_cons, aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())

# 2050 plot, all rcp
fig_cons_cc_all <- ggplot(data = cons_cc_2050) +
  geom_rect(data = bw_cons,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = cons),
            alpha = 0.2) +
  geom_hline(yintercept = cons_hist$value[cons_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = cons_pe_l,
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = cons_pe_l,
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x") +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_cons, y_up_cons),
                     sec.axis = sec_axis(~((./cons_hist$value[cons_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Global food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL,  y = "Global food consumption (1e15 kcal)",
       colour = NULL, linetype = NULL, fill = NULL) +
  rremove("grid") +
  guides(color = F, shape = F, fill = F) +
  geom_text(data = cons_cc_2050_lab,
            aes(x = rcp, y = 13, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_cons_all, aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())


# ========================================================================================
# FIGURE 4 -------------------------------------------------------------------------------
# ========================================================================================

# Get legends
l1 <- get_legend(fig_calo_nocc +
                   theme(legend.position = "bottom",
                         legend.direction = "horizontal") +
                   theme(legend.text=element_text(size=11)) +
                   theme(legend.key.width = unit(3, "line")) +
                   guides(colour = guide_legend(nrow = 1, override.aes=list(shape = NA))) +
                   guides(lty = guide_legend(override.aes=list(color = "black", alpha = 1))))

# Prepare plot components
p1 <- cowplot::plot_grid(
  fig_calo_nocc +  theme(legend.position="none"),
  fig_cons_nocc, ncol = 1, align = "v", axis = "lr", labels = c("a", "c"))

p2 <- cowplot::plot_grid(
  fig_calo_cc  + theme(legend.position="none"),
  fig_cons_cc, ncol = 1, align = "v", axis = "lr", labels = c("b", "d"))

p3 <- cowplot::plot_grid(p1, p2, nrow = 1, rel_widths = c(1, 1), align = "h")

fig_calo_cons <- cowplot::plot_grid(p3, l1, ncol = 1, rel_heights = c(1, 0.1))
rm(p1, p2, p3)


# ========================================================================================
# FIGURE 5 -------------------------------------------------------------------------------
# ========================================================================================

# PREPARE DATA ---------------------------------------------------------------------------

# Select historical data
prh_hist <- db_hist %>%
  filter(variable == "PRH", year >= 2000)

# Select CALO Baseline
prh_db <- db %>%
  filter(variable == "PRH",
         ssp != "No class", rcp != "No class")

# Only select NOCC
prh_nocc <- prh_db %>%
  filter(rcp == "NOCC")

# Calculate mean for NOCC, SSP (pure and derived)
prh_nocc_mean <- prh_nocc %>%
  filter(rcp %in% c("NOCC")) %>%
  group_by(year, ssp) %>%
  summarize(value = mean(proj, na.rm = T), .groups = "drop")

# Calculate levels for point estimates
prh_pe_l <- prh_pe %>%
  mutate(fit_l = ((fit/100)+1)*prh_hist$value[prh_hist$year == 2010],
         upper_l = ((upper/100)+1)*prh_hist$value[prh_hist$year == 2010],
         lower_l = ((lower/100)+1)*prh_hist$value[prh_hist$year == 2010],
         variable = "PRH")

# Select lower and upper range of 95% error bars
q05_prh <- min(prh_pe_l$lower_l[prh_pe_l$rcp == "NOCC"])
q95_prh <- max(prh_pe_l$upper_l[prh_pe_l$rcp == "NOCC"])
q05_prh_cc <- min(prh_pe_l$lower_l[prh_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])
q95_prh_cc <- max(prh_pe_l$upper_l[prh_pe_l$rcp %in% c("RCP2.6", "RCP4.5", "RCP6.0", "RCP8.5")])

# Select 2050 data
prh_cc_2050 <- prh_db %>%
  filter(
    year == 2050) %>%
  group_by(ssp, rcp) %>%
  mutate(max_y = max(proj, na.rm = T),
         min_y = min(proj, na.rm = T),
         mean = mean(proj, na.rm = T)) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}"))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
prh_cc_2050_lab <- prh_cc_2050 %>%
  ungroup() %>%
  dplyr::select(rcp, ssp, n_stud_scen) %>%
  unique

# Set limits
y_lo_prh <- -500
y_up_prh <- 1400

# labels for ssp as we remove facet strip
ssp_label_prh = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                           label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                           x = c(1.5, 1.5, 1.5, 1.5, 1),
                           y = rep(c(1200),5))

# labels for ssp all as we remove facet strip
ssp_label_prh_all = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                               label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                               x = c(3, 3, 3, 1.5, 1.5),
                               y = rep(c(1200),5))

# Shade for bandwidth
bw_prh <- data.frame(xmin = c(-Inf, -Inf),
                     xmax = c(Inf, Inf),
                     ymin = c(q05_prh, q05_prh_cc),
                     ymax = c(q95_prh, q95_prh_cc),
                     rcp = c("NOCC", "RCP8.5"))

# DRAW PLOT ------------------------------------------------------------------------------

# line plot
fig_prh_nocc <- ggplot() +
  geom_hline(yintercept = prh_hist$value[prh_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_line(data = prh_nocc,
            aes(x = year, y = proj, colour = ssp, linetype = pure_ssp,
                group = interaction(projection, study_short, model)),
            size = 1, alpha = 0.3) +
  geom_line(data = prh_nocc_mean,
            aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_point(data = filter(prh_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050)),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_line(data = prh_hist,
            aes(x = year, y = value),
            colour = "black", size = 2) +
  scale_colour_manual(values = ssp_col) +
  theme_classic() +
  theme(legend.position = c(0.1, 0.7),
        panel.border = element_rect(colour = "black", fill=NA)) +
  scale_y_continuous(labels = scales::comma,
                     limits = c(y_lo_prh, y_up_prh),
                     sec.axis = sec_axis(~((./prh_hist$value[prh_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10)),
                     breaks = scales::pretty_breaks(n = 10)) +
  labs(x = NULL, y = "People at risk of hunger (million persons)", colour = NULL, linetype = NULL) +
  guides(shape = F, color = F, lty = F) +
  coord_cartesian(ylim=c(0, 1200))

# 2050 plot, only NOCC and RCP8.5
fig_prh_cc <- ggplot(data = filter(prh_cc_2050, rcp %in% c("NOCC", "RCP8.5"))) +
  geom_rect(data = bw_prh,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = rcp),
            alpha = 0.2) +
  geom_hline(yintercept = prh_hist$value[prh_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = filter(prh_pe_l, rcp %in% c("NOCC", "RCP8.5")),
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = filter(prh_pe_l, rcp %in% c("NOCC", "RCP8.5")),
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x") +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./prh_hist$value[prh_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Food prhumumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "", colour = NULL, linetype = NULL, fill = NULL) +
  rremove("grid") +
  guides(color = F, shape = F, fill = F) +
  geom_text(data = filter(prh_cc_2050_lab, rcp %in% c("NOCC", "RCP8.5")),
            aes(x = rcp, y = 1150, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_prh,
            aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank()) +
  coord_cartesian(ylim=c(0, 1200))

# 2050 plot, all rcp
fig_prh_cc_all <- ggplot(data = prh_cc_2050) +
  geom_rect(data = bw_prh,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = rcp),
            alpha = 0.2) +
  geom_hline(yintercept = prh_hist$value[prh_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_errorbar(data = prh_pe_l,
                aes(ymin = lower_l, ymax = upper_l, x = rcp, color = ssp),
                size = 1, width = 0.2) +
  geom_point(data = prh_pe_l,
             aes(y = fit_l, x = rcp, color = ssp),
             size = 3, shape = 19) +
  scale_color_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  scale_fill_manual(values = bw_col) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x") +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./prh_hist$value[prh_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Food prhumumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "People at risk of hunger (million persons)"
       , colour = NULL, linetype = NULL, fill = NULL) +
  rremove("grid") +
  guides(color = F, shape = F, fill = F) +
  geom_text(data = prh_cc_2050_lab,
            aes(x = rcp, y = 1150, label = n_stud_scen),
            size = 3) +
  geom_text(data = ssp_label_prh_all,
            aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank()) +
  coord_cartesian(ylim=c(0, 1200))

p1 <- cowplot::plot_grid(fig_prh_nocc, fig_prh_cc,
                         nrow = 1, rel_widths = c(1, 1),
                         labels = "auto", axis = "tb")

fig_prh <- cowplot::plot_grid(p1, l1, ncol = 1, rel_heights = c(1, 0.1))


# ========================================================================================
# SUMMARY TABLE BANDWIDHT META-REGRESSION ------------------------------------------------
# ========================================================================================

# bandwidth db
bw_raw <- bind_rows(
  bind_rows(calo_pe_l, cons_pe_l, prh_pe_l) %>%
    filter(rcp == "NOCC") %>%
    group_by(variable, rcp) %>%
    mutate(bw_up = max(upper),
           bw_lo = min(lower),
           bw_up_l = max(upper_l),
           bw_lo_l = min(lower_l)),
  bind_rows(calo_pe_l, cons_pe_l, prh_pe_l) %>%
    filter(rcp != "NOCC") %>%
    group_by(variable) %>%
    mutate(bw_up = max(upper),
           bw_lo = min(lower),
           bw_up_l = max(upper_l),
           bw_lo_l = min(lower_l))
  )  %>%
  arrange(variable, rcp, ssp) %>%
  dplyr::select(variable, ssp, rcp, everything())

# bandwidth reference values for paper
bw <- bw_raw %>%
  mutate(across(where(is.numeric), round)) %>%
  pivot_longer(!c(ssp, variable, rcp),
               names_to = "indicator",
               values_to = "value") %>%
  mutate(value = ifelse(value >= 0, paste0("+", value), as.character(value)),
         value = ifelse(indicator %in% c("fit", "upper", "lower", "bw_up", "bw_lo"),
                        paste0(value, "%"), value)) %>%
  pivot_wider(names_from = indicator,
              values_from = value)


# ========================================================================================
# EXTENDED DATA FIGURE 1 -----------------------------------------------------------------
# ========================================================================================

# Available on request

# ========================================================================================
# EXTENDED DATA FIGURE 2 -----------------------------------------------------------------
# ========================================================================================

fig_drivers_study <- ggplot(data = driver,
                            aes(x = fct_rev(fct_infreq(driver)),
                                y = reorder(study_short_clean, desc(study_year)), fill = driver)) +
  geom_tile() +
  labs(x = NULL, y = NULL) +
  theme_classic(base_size = 14) +
  scale_fill_manual(values = col12) +
  theme(legend.position="none",
        axis.text.x = element_text(angle = 45, hjust = 1))


# ========================================================================================
# EXTENDED DATA FIGURE 3 -----------------------------------------------------------------
# ========================================================================================

l2 <- get_legend(fig_calo_cc_all +
                   theme(legend.position = "bottom",
                         legend.direction = "horizontal") +
                   theme(legend.text=element_text(size=11)) +
                   theme(legend.key.width = unit(3, "line")) +
                   guides(colour = guide_legend(nrow = 1)) +
                   guides(shape = guide_legend(override.aes = list(size = 3))))

fig_calo_cons_cc_all <- cowplot::plot_grid(fig_calo_cc_all + theme(legend.position="none"),
                                           fig_cons_cc_all, l2,
                                           ncol = 1,  align = "v", axis = "lr", rel_heights = c(1, 1, 0.1))



# ========================================================================================
# EXTENDED DATA FIGURE 4 -----------------------------------------------------------------
# ========================================================================================

fig_prh_cc_all <- cowplot::plot_grid(fig_prh_cc_all + theme(legend.position="none"),
                                     l2,
                                     ncol = 1,  align = "v", axis = "lr", rel_heights = c(1, 0.1))


# ========================================================================================
# EXTENDED DATA FIGURE 5 -----------------------------------------------------------------
# ========================================================================================

# PREPARE DATA ---------------------------------------------------------------------------

# Select historical data
pop_hist <- db_hist %>%
  filter(variable %in% c("POP"), year >= 2000)

# Calculate kcal consumption for db
pop_db <- db %>%
  filter(variable == "POP",
         ssp != "No class", rcp != "No class")

# Only select NOCC
pop_nocc <- pop_db %>%
  filter(rcp == "NOCC")

# Calculate mean for NOCC, SSP
pop_nocc_mean <- pop_nocc %>%
  filter(rcp %in% c("NOCC")) %>%
  group_by(year, ssp) %>%
  summarize(value = mean(proj, na.rm = T))

# select 2050 data
pop_cc_2050 <- pop_db %>%
  filter(
    rcp %in% c("NOCC"),
    ssp %in% c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
    year == 2050) %>%
  group_by(ssp, rcp) %>%
  mutate(max_y = max(proj, na.rm = T),
         min_y = min(proj, na.rm = T),
         mean = mean(proj, na.rm = T)) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}"))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
pop_cc_2050_lab <- pop_cc_2050 %>%
  ungroup() %>%
  dplyr::select(rcp, ssp, n_stud_scen) %>%
  unique

# Set limits
y_lo_pop <- 6000
y_up_pop <- 11000

# labels for ssp as we remove facet strip
ssp_label_pop = data.frame(ssp = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                           label = c("SSP1", "SSP2", "SSP3", "SSP4", "SSP5"),
                           x = c(1, 1, 1, 1, 1),
                           y = rep(c(y_up_pop),5))


# DRAW PLOT ------------------------------------------------------------------------------

# line plot
# Added a hack to make sure SSP2 is plotted over SSP3 as the former is more important
fig_pop_nocc <- ggplot() +
  geom_hline(yintercept = pop_hist$value[pop_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_line(data = pop_nocc,
            aes(x = year, y = proj, colour = ssp, linetype = pure_ssp,
                group = interaction(projection, study_short, model)),
            size = 1, alpha = 0.3) +
  geom_line(data = filter(pop_nocc_mean, ssp != "SSP2"),
            aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_line(data = filter(pop_nocc_mean, ssp == "SSP2"),
            aes(x = year, y = value, colour = ssp),
            size = 2) +
  geom_point(data = filter(pop_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050), ssp != "SSP2"),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_point(data = filter(pop_nocc_mean, year %in% c(2010, 2020, 2030, 2040, 2050), ssp == "SSP2"),
             aes(x = year, y = value, colour = ssp),
             shape = 21, size = 3, fill = "white", stroke = 1.5) +
  geom_line(data = pop_hist, aes(x = year, y = value),
            colour = "black", size = 2) +
  scale_colour_manual(values = ssp_col) +
  theme_classic() +
  theme(legend.position = c(0.1, 0.7),
        panel.border = element_rect(colour = "black", fill=NA)) +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_pop, y_up_pop),
                     sec.axis = sec_axis(~((./pop_hist$value[pop_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10)),
                     breaks = scales::pretty_breaks(n = 10)) +
  labs(x = NULL, y = "Global population (million persons)", colour = NULL, linetype = NULL) +
  guides(linetype = F, colour = F)


# box plot
fig_pop_cc <- ggplot(data = pop_cc_2050) +
  geom_hline(yintercept = pop_hist$value[pop_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_errorbar(aes(ymin = min_y, ymax = max_y, x = rcp),
                width = 0.2) +
  geom_boxplot(aes(x = rcp, y = proj, fill = ssp),
               outlier.shape = NA, coef = 0) +
  geom_jitter(aes(x = rcp, y = proj, shape = pure_ssp),
              width = 0.3, color = "grey40") +
  geom_point(aes(x = rcp, y = mean),
             shape = 18, size = 3) +
  scale_fill_manual(values = ssp_col) +
  scale_shape_manual(values = c(0,17)) +
  theme_bw() +
  facet_grid(~ssp, scales = "free_x", space = "free_x") +
  scale_y_continuous(labels = scales::comma, limits = c(y_lo_pop, y_up_pop),
                     sec.axis = sec_axis(~((./pop_hist$value[pop_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Global population relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 1)) +
  labs(x = NULL, y = "", colour = NULL, linetype = NULL) +
  rremove("grid") +
  guides(fill = F, shape = F) +
  geom_text(data = pop_cc_2050_lab, aes(x = rcp, y = 10500, label = n_stud_scen), size = 3) +
  geom_text(data = ssp_label_pop, aes(x = x, y = y, label = label)) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())

p1 <- cowplot::plot_grid(fig_pop_nocc, fig_pop_cc,
                         nrow = 1, rel_widths = c(1, 1),
                         labels = "auto", axis = "tb")

fig_pop <- cowplot::plot_grid(p1, l1, ncol = 1, rel_heights = c(1, 0.1))
rm(p1)


# ========================================================================================
# FIGURE S1 ------------------------------------------------------------------------------
# ========================================================================================

indicator_si <- indicator_raw %>%
  pivot_longer(names_to = "variable",
               values_to = "code",
               !c(study_short, year)) %>%
  mutate(code = ifelse(variable %in% c("CU", "FP") & code %in% c("AN", "AH"), "A", code),
         code = case_when(
           variable %in% "other" & is.na(code) ~"N",
           variable %in% "other" & !is.na(code) ~ "A",
           TRUE ~ code),
         code = factor(code,
                       levels = c("N", "AN", "AX", "AH", "A"),
                       labels = c("Not covered", "Not possible to extract", "Extracted but not comparable",
                                  "Extracted", "Covered")),
         variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU", "other"),
                           labels = c("Per capita\nfood demand", "Total food\n demand", "Population at\nrisk of hunger",
                                      "Food price", "Childhood\nundernutrition", "Other"))) %>%
  filter(variable != "Total food\n demand") %>%
  group_by(variable, code) %>%
  summarise(n = n()) %>%
  mutate(sh = n / sum(n))

col_ind = c("white", col8[c(1:4)])
names(col_ind) <- c("Not covered", "Extracted but not comparable", "Not possible to extract",
                    "Extracted", "Covered")

fig_ind_cov_si <- ggplot(data = indicator_si,
                         aes(x = variable, y = n,
                             fill = code,
                             label = scales::percent(sh, accuracy = 1))) +
  geom_bar(stat = "identity", color =  "black") +
  geom_text(position=position_stack(vjust=0.5)) +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col_ind) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic(base_size = 14) +
  theme(legend.position = "bottom")


# ========================================================================================
# FIGURE S2 ------------------------------------------------------------------------------
# ========================================================================================

# Available on request


# ========================================================================================
# FIGURE S3 ------------------------------------------------------------------------------
# ========================================================================================

# ERROR!
study_ind_db <- db_raw %>%
  dplyr::select(study_short, variable, year) %>%
  filter(!variable %in% c("POP", "XPRP")) %>%
  unique %>%
  left_join(study_short_clean) %>%
  mutate(variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU"),
                           labels = c("Per capita\nfood demand", "Total food\n demand",
                                      "Population at\n risk of hunger",
                                      "Food price", "Childhood\nundernutrition"))) %>%
  filter(variable != "Total food\n demand")

fig_ind_stud_db <- ggplot(data = study_ind_db,
                          aes(x = variable, y = reorder(study_short_clean, desc(study_year)),
                              fill = variable)) +
  geom_tile() +
  labs(x = NULL, y = NULL) +
  theme_classic(base_size = 14) +
  scale_fill_manual(values = col8) +
  scale_y_discrete(expand = c(0,0)) +
  scale_x_discrete(expand = c(0,0)) +
  theme(legend.position="none",
        panel.border = element_rect(colour = "black", fill = NA))


# ========================================================================================
# FIGURE S4 ------------------------------------------------------------------------------
# ========================================================================================

ssp_share <- db %>%
  filter(variable != "POP", year == 2010) %>%
  mutate(variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU"),
                           labels = c("Per capita\nfood demand", "Total food\n demand", "Population at\n risk of hunger",
                                      "Food price", "Childhood\nundernutrition")),
         pure_ssp = factor(pure_ssp, levels = c("Derived SSP", "No class", "Pure SSP"))) %>%
  group_by(variable, pure_ssp) %>%
  summarise(n = n()) %>%
  filter(variable != "Total food\n demand") %>%
  mutate(sh = n / sum(n))

rcp_share <- db %>%
  filter(variable != "POP", year == 2010) %>%
  mutate(variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU"),
                           labels = c("Per capita\nfood demand", "Total food\n demand", "Population at\n risk of hunger",
                                      "Food price", "Childhood\nundernutrition")),
         pure_rcp = factor(pure_rcp, levels = c("Derived RCP", "No class", "Pure RCP"))) %>%
  group_by(variable, pure_rcp) %>%
  summarise(n = n()) %>%
  filter(variable != "Total food\n demand") %>%
  mutate(sh = n / sum(n))

fig_ssp_pure_sh <- ggplot(data = ssp_share,
                          aes(x = variable, y = n,
                              fill = pure_ssp,
                              label = scales::percent(sh, accuracy = 1))) +
  geom_bar(stat = "identity", color =  "black") +
  geom_text(data = ssp_share, position=position_stack(vjust=0.5)) +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of projections") +
  theme_classic(base_size = 14) +
  theme(legend.position = "bottom")

fig_rcp_pure_sh <- ggplot(data = rcp_share,
                          aes(x = variable, y = n,
                              fill = pure_rcp,
                              label = scales::percent(sh, accuracy = 1))) +
  geom_bar(stat = "identity", color =  "black") +
  geom_text(data = filter(rcp_share, pure_rcp != "No class"), position=position_stack(vjust=0.5)) +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8[c(5:8)]) +
  labs(fill = "", x = "", y = "Number of projections") +
  theme_classic(base_size = 14) +
  theme(legend.position = "bottom")

fig_pure_sh <- cowplot::plot_grid(
  fig_ssp_pure_sh,
  fig_rcp_pure_sh, nrow = 1, align = "v", axis = "lr", labels = c("a", "b"))


# ========================================================================================
# FIGURE S5 ------------------------------------------------------------------------------
# ========================================================================================

type_n <- db_raw %>%
  dplyr::select(type, policy_shock, variable, study_short, projection, model) %>%
  filter(variable != "POP") %>%
  mutate(variable = factor(variable,
                           levels = c("CALO", "CONS", "PRH", "FP", "CU"),
                           labels = c("Per capita\nfood demand", "Total food\n demand", "Population at\n risk of hunger",
                                      "Food price", "Childhood\nundernutrition"))) %>%
  filter(variable != "Total food\n demand") %>%
  distinct() %>%
  group_by(type, policy_shock, variable) %>%
  count()

fig_bl_ps <- ggplot(data = type_n,
                    aes(x = type, y = n,
                        fill = policy_shock)) +
  geom_bar(stat = "identity", color =  "black") +
  facet_wrap(~variable) +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col12) +
  labs(fill = "", x = "", y = "Number of projections") +
  theme_bw(base_size = 14) +
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank()) +
  theme(legend.position = "bottom")


# ========================================================================================
# FIGURE S6 ------------------------------------------------------------------------------
# ========================================================================================

# See nf_meta_regression.r


# ========================================================================================
# FIGURE S7 ------------------------------------------------------------------------------
# ========================================================================================

# See nf_meta_regression.r


# ========================================================================================
# FIGURE S8 ------------------------------------------------------------------------------
# ========================================================================================

# select policy shock
ps <- db_raw %>%
  filter(type == "Policy shock")

# CALO ----------------------------------------------------------------------------------
# Select CALO data
calo_ps_db <- ps %>%
  filter(variable == "CALO",
         year == 2050) %>%
  mutate(id = glue("{projection} ({model})"))

# Set color and shape
ps_col <- brewer.pal(9, "Set1")[c(2,1,3)]
names(ps_col) <- sort(unique(calo_ps_db$pure_ssp))
ps_shape <- c(0:8, 15:17)
names(ps_shape) <- sort(unique(calo_ps_db$policy_shock))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
calo_ps_2050_lab <- calo_ps_db %>%
  ungroup() %>%
  filter(year == 2050) %>%
  group_by(rcp) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}")) %>%
  dplyr::select(rcp, n_stud_scen) %>%
  unique

# Plot
fig_calo_ps <- ggplot() +
  geom_rect(data = bw_calo,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = calo),
            alpha = 0.2) +
  geom_hline(yintercept = calo_hist$value[calo_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(data = calo_ps_db,
              aes(y = proj, x = rcp, shape = policy_shock , color = pure_ssp),
              width = 0.3, alpha = 0.7) +
  scale_colour_manual(values = ps_col) +
  scale_shape_manual(values = ps_shape) +
  scale_fill_manual(values = bw_col) +
  theme_bw(base_size = 14) +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./calo_hist$value[calo_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Per capita food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL,  y = "Per capita food consumption (kcal/cap/day)",
       colour = "Class", shape = "Policy shock", fill = NULL) +
  rremove("grid") +
  geom_text(data = calo_ps_2050_lab,
            aes(x = rcp, y = 4250, label = n_stud_scen),
            size = 3) +
  guides(
    shape = guide_legend(title.position="top", title.hjust = 0.5),
    color = guide_legend(title.position="top", title.hjust = 0.5),
    fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank(),
        axis.title.y = element_text(size = 10))


# CONS -----------------------------------------------------------------------------------
# Select CONS data
cons_ps_db <- ps %>%
  filter(variable == "CONS", type == "Policy shock",
         year == 2050) %>%
  mutate(id = glue("{projection} ({model})"))

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
cons_ps_2050_lab <- cons_ps_db %>%
  ungroup() %>%
  filter(year == 2050) %>%
  group_by(rcp) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}")) %>%
  dplyr::select(rcp, n_stud_scen) %>%
  unique

# Plot
fig_cons_ps <- ggplot() +
  geom_rect(data = bw_cons,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = cons),
            alpha = 0.2) +
  geom_hline(yintercept = cons_hist$value[cons_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(data = cons_ps_db,
              aes(y = proj, x = rcp, shape = policy_shock , color = pure_ssp),
              width = 0.3, alpha = 0.7) +
  scale_colour_manual(values = ps_col) +
  scale_shape_manual(values = ps_shape) +
  scale_fill_manual(values = bw_col) +
  theme_bw(base_size = 14) +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./cons_hist$value[cons_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Total food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "Total food consumption (1e15 kcal)",
       colour = "Class", shape = "Policy shock", fill = NULL) +
  rremove("grid") +
  geom_text(data = cons_ps_2050_lab,
            aes(x = rcp, y = 13, label = n_stud_scen),
            size = 3) +
  guides(fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank(),
        axis.title.y = element_text(size = 10))


# PRH ------------------------------------------------------------------------------------
# Select PRH data
prh_ps_db <- ps %>%
  filter(variable == "PRH", type == "Policy shock",
         year == 2050)

# labels. Need to be unique as geom_text results in overplotting and hence poor quality
prh_ps_2050_lab <- prh_ps_db %>%
  ungroup() %>%
  filter(year == 2050) %>%
  group_by(rcp) %>%
  mutate(n_stud = length(unique(study_short)),
         n_scen = length(unique(interaction(projection, study_short, model))),
         n_stud_scen = glue("{n_stud}\\{n_scen}")) %>%
  dplyr::select(rcp, n_stud_scen) %>%
  unique

# Plot
fig_prh_ps <- ggplot() +
  geom_rect(data = bw_prh,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = rcp),
            alpha = 0.2) +
  geom_hline(yintercept = prh_hist$value[prh_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_jitter(data = prh_ps_db,
              aes(y = proj, x = rcp, shape = policy_shock , color = pure_ssp),
              width = 0.3, alpha = 0.7) +
  scale_colour_manual(values = ps_col) +
  scale_shape_manual(values = ps_shape) +
  scale_fill_manual(values = bw_col) +
  theme_bw(base_size = 14) +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./prh_hist$value[prh_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "People at risk of hunger relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "People at risk of hunger (million persons)",
       colour = "Class", shape = "Policy shock", fill = NULL) +
  rremove("grid") +
  geom_text(data = prh_ps_2050_lab,
            aes(x = rcp, y = 1150, label = n_stud_scen),
            size = 3) +
  guides(fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank(),
        axis.title.y = element_text(size = 10)) +
  coord_cartesian(ylim=c(0, 1200))


## COMBINE PLOTS
l <- get_legend(fig_calo_ps)
p1 <- cowplot::plot_grid(fig_calo_ps +  theme(legend.position="none"),
                         fig_cons_ps +  theme(legend.position="none"),
                         fig_prh_ps +  theme(legend.position="none"),
                         ncol = 1, rel_heights = c(1, 1, 1),
                         labels = "auto", axis = "lr", align = "hv")

fig_ps <- cowplot::plot_grid(p1, l, nrow = 1, rel_widths = c(1, 0.15))
rm(l, p1)


# ========================================================================================
# FIGURE S9 ------------------------------------------------------------------------------
# ========================================================================================

study <- study_level %>%
  mutate(in_gfsp_db_v1.1 = ifelse(in_gfsp_db_v1.1 == "N", "Excluded", "Included"),
         study_year = ifelse(study_year < 2012, "< 2012", ">= 2012"),
         jb_published = case_when(
           type_of_publication %in% c("Journal article", "Book (chapter)") ~ "Published",
           type_of_publication %in% c("Report/working paper") ~ "Unpublished"),
         j_published = case_when(
           type_of_publication %in% c("Journal article") ~ "Published",
           type_of_publication %in% c("Report/working paper", "Book (chapter)") ~ "Unpublished"),
         methodology = gsub(" ", "\n", methodology),
         scenario = factor(scenario, levels = c("SSP", "SRES", "Other"))
  ) %>%
  pivot_longer(
    !c(study_short, in_gfsp_db_v1.1),
    names_to = "variable",
    values_to = "value",
  ) %>%
  group_by(in_gfsp_db_v1.1, variable, value) %>%
  summarize(n = n()) %>%
  mutate(sh = n / sum(n))

p1 <- ggplot(data = filter(study, variable == "methodology"),
             aes(x = in_gfsp_db_v1.1, y = n, fill = value)) +
  geom_bar(stat = "identity", color = "black") +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic() +
  theme(legend.position = "bottom") +
  labs(title = "Methodology") +
  theme(plot.title = element_text(hjust = 0.5)) +
  geom_text(aes(label = scales::percent(sh, accuracy = 1)), position=position_stack(vjust=0.5))

p2 <- ggplot(data = filter(study, variable == "single_multi_model"),
             aes(x = in_gfsp_db_v1.1, y = n, fill = value)) +
  geom_bar(stat = "identity", color = "black") +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic() +
  theme(legend.position = "bottom") +
  labs(title = "Single/multi model study") +
  theme(plot.title = element_text(hjust = 0.5)) +
  geom_text(aes(label = scales::percent(sh, accuracy = 1)), position=position_stack(vjust=0.5))

p3 <- ggplot(data = filter(study, variable == "j_published"),
             aes(x = in_gfsp_db_v1.1, y = n, fill = value)) +
  geom_bar(stat = "identity", color = "black") +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic() +
  theme(legend.position = "bottom") +
  labs(title = "(Un)published study") +
  theme(plot.title = element_text(hjust = 0.5)) +
  geom_text(aes(label = scales::percent(sh, accuracy = 1)), position=position_stack(vjust=0.5))

p4 <- ggplot(data = filter(study, variable == "study_year"),
             aes(x = in_gfsp_db_v1.1, y = n, fill = value)) +
  geom_bar(stat = "identity", color = "black") +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic() +
  theme(legend.position = "bottom") +
  labs(title = "Study year") +
  theme(plot.title = element_text(hjust = 0.5)) +
  geom_text(aes(label = scales::percent(sh, accuracy = 1)), position=position_stack(vjust=0.5))

p5 <- ggplot(data = filter(study, variable == "scenario"),
             aes(x = in_gfsp_db_v1.1, y = n, fill = value)) +
  geom_bar(stat = "identity", color = "black") +
  scale_y_continuous(expand = expansion(mult = c(0, 0.05))) +
  scale_fill_manual(values = col8) +
  labs(fill = "", x = "", y = "Number of studies") +
  theme_classic() +
  theme(legend.position = "bottom") +
  labs(title = "Scenario type") +
  theme(plot.title = element_text(hjust = 0.5)) +
  geom_text(aes(label = scales::percent(sh, accuracy = 1)), position=position_stack(vjust=0.5))

fig_study_comp <- p1 + p2 + p3 + p4 + p5  +
  plot_annotation(tag_levels = 'a')
rm(p1, p2, p3, p4, p5)


# ========================================================================================
# FIGURE S10 -----------------------------------------------------------------------------
# ========================================================================================

# CALO -----------------------------------------------------------------------------------

# Select CALO data
calo_nc_db <- db %>%
  filter(variable == "CALO",
         !(ssp != "No class" & rcp != "No class"),
         year == 2050) %>%
  mutate(id = glue("{projection} ({model})"))

# Set color and shape
nc_col <- brewer.pal(9, "Set1")[c(2,1,5,4)]
names(nc_col) <- sort(unique(calo_nc_db$study_short_clean))
nc_shape <- c(1:6)
names(nc_shape) <- sort(unique(calo_nc_db$id))

# Plot
fig_calo_nc <- ggplot() +
  geom_rect(data = bw_calo,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = calo),
            alpha = 0.2) +
  geom_hline(yintercept = calo_hist$value[calo_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_point(data = calo_nc_db,
             aes(y = proj, x = rcp, shape = id, color = study_short_clean),
             size = 3) +
  scale_shape_manual(values = nc_shape) +
  scale_fill_manual(values = bw_col) +
  scale_color_manual(values = nc_col) +
  theme_bw() +
  scale_y_continuous(labels = scales::comma,
                     limits = c(y_lo_calo, y_up_calo),
                     sec.axis = sec_axis(~((./calo_hist$value[calo_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Per capita food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL,  y = "Per capita food consumption (kcal/cap/day)",
       colour = "Study", shape = "Projection (model)", fill = NULL) +
  rremove("grid") +
  guides(
    shape = guide_legend(title.position="top", title.hjust = 0.5),
    color = guide_legend(title.position="top", title.hjust = 0.5),
    fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())


# CONS -----------------------------------------------------------------------------------

# Select CONS data
cons_nc_db <- db %>%
  filter(variable == "CONS",
         !(ssp != "No class" & rcp != "No class"),
         year == 2050) %>%
  mutate(id = glue("{projection} ({model})"))

# Plot
fig_cons_nc <- ggplot() +
  geom_rect(data = bw_cons,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = cons),
            alpha = 0.2) +
  geom_hline(yintercept = cons_hist$value[cons_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_point(data = cons_nc_db,
             aes(y = proj, x = rcp, shape = id, color = study_short_clean),
             size = 3) +
  scale_shape_manual(values = nc_shape) +
  scale_fill_manual(values = bw_col) +
  scale_color_manual(values = nc_col) +
  theme_bw() +
  scale_y_continuous(labels = scales::comma,
                     limits = c(y_lo_cons, y_up_cons),
                     sec.axis = sec_axis(~((./cons_hist$value[cons_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "Total food consumption relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "Total food consumption (1e15 kcal)",
       colour = NULL, shape = NULL, fill = NULL) +
  rremove("grid") +
  guides(fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank())


# PRH ------------------------------------------------------------------------------------

# Select PRH data
prh_nc_db <- db %>%
  filter(variable == "PRH",
         !(ssp != "No class" & rcp != "No class"),
         year == 2050) %>%
  mutate(id = glue("{projection} ({model})"))

# Plot
fig_prh_nc <- ggplot() +
  geom_rect(data = bw_prh,
            aes(xmin = xmin, xmax = xmax, ymin = ymin, ymax = ymax, fill = rcp),
            alpha = 0.2) +
  geom_hline(yintercept = prh_hist$value[prh_hist$year == 2010],
             size = 0.5, linetype = "solid") +
  geom_point(data = prh_nc_db,
             aes(y = proj, x = rcp, shape = id, color = study_short_clean),
             size = 3) +
  scale_shape_manual(values = nc_shape) +
  scale_fill_manual(values = bw_col) +
  scale_color_manual(values = nc_col) +
  theme_bw() +
  scale_y_continuous(labels = scales::comma,
                     sec.axis = sec_axis(~((./prh_hist$value[prh_hist$year == 2010])-1)*100,
                                         breaks = pretty_breaks(n = 10),
                                         name = "People at risk of hunger relative to 2010 (%)"),
                     breaks = scales::pretty_breaks(n = 10)) +
  scale_x_discrete(expand = expansion(add = 0.5)) +
  labs(x = NULL, y = "People at risk of hunger (million persons)",
       colour = NULL, shape = NULL, fill = NULL) +
  rremove("grid") +
  guides(fill = F) +
  theme(strip.background = element_blank(),
        strip.text.x = element_blank()) +
  coord_cartesian(ylim=c(0, 1200))

l <- get_legend(fig_calo_nc +
                  theme(legend.position = "bottom",
                        legend.direction = "vertical",
                        legend.box = "horizontal") +
                  theme(legend.text=element_text(size=11)) +
                  theme(legend.key.width = unit(3, "line")))

p1 <- cowplot::plot_grid(fig_calo_nc +  theme(legend.position="none"),
                         fig_cons_nc +  theme(legend.position="none"),
                         fig_prh_nc +  theme(legend.position="none"),
                         nrow = 1, rel_widths = c(1, 1, 1),
                         labels = "auto", axis = "tb")

fig_nc <- cowplot::plot_grid(p1, l, ncol = 1, rel_heights = c(1, 0.6))
rm(l, p1)


