import numpy as np
import matplotlib.pyplot as plt

from pymor.basic import (
    InstationaryModel,
    LincombOperator,
    NumpyMatrixOperator,
    ProjectionParameterFunctional,
)
from pymor.algorithms.timestepping import ImplicitEulerTimeStepper

T_final = 1.0
num_time_steps = 100
parameter_values = [1e2, 1e-2, 1e-3, 1e-4]

Ai = [NumpyMatrixOperator.from_file(f"A{i}.mtx") for i in range(5)]
A = LincombOperator(
    Ai, [1.0] + [ProjectionParameterFunctional("cookies", 4, (i,)) for i in range(4)]
)
B = NumpyMatrixOperator.from_file("B.txt")
C = NumpyMatrixOperator.from_file("C.txt")
E = NumpyMatrixOperator.from_file("E.mtx")

m = InstationaryModel(
    T_final,
    A.source.zeros(),
    -A,
    B.H,
    E,
    output_functional=C,
    time_stepper=ImplicitEulerTimeStepper(nt=num_time_steps),
)
U, output = m.solve(parameter_values, return_output=True)

np.savetxt("U.txt", U.to_numpy())

fig, ax = plt.subplots()
for i in range(4):
    ax.plot(
        np.linspace(0, T_final, num_time_steps + 1),
        output.to_numpy()[:, i],
        label=f"Output {i + 1}",
    )
ax.grid()
ax.set_xlabel("Time")
ax.set_ylabel("Temperature")
ax.legend()
fig.savefig("output.png", dpi=150)
