import dolfin as df
import numpy as np
from scipy.sparse import csr_matrix
from scipy.io import mmwrite, savemat


def convert_matrix(M):
    return csr_matrix(
        df.as_backend_type(M).mat().getValuesCSR()[::-1], shape=(V.dim(), V.dim())
    )


mesh = df.Mesh("cookie.xml")
print(f"hmin = {mesh.hmin():.3e}")
print(f"hmax = {mesh.hmax():.3e}")

subdomains = df.MeshFunction("size_t", mesh, "cookie_physical_region.xml")
boundaries = df.MeshFunction("size_t", mesh, "cookie_facet_region.xml")

V = df.FunctionSpace(mesh, "CG", 1)
u = df.TrialFunction(V)
v = df.TestFunction(V)

BC = df.DirichletBC(V, df.Constant(0.0), boundaries, 2)

dxi = df.Measure("dx", mesh, subdomain_data=subdomains)
dsi = df.Measure("ds", mesh, subdomain_data=boundaries)

Ai = [df.dot(df.grad(u), df.grad(v)) * dxi(i) for i in range(1, 6)]

for i in range(len(Ai)):
    if int(df.__version__.split(".")[0]) < 2019:
        A = df.assemble(Ai[i], keep_diagonal=True)
        # 2018.1 fails later if 0 diagonal entries are not preserved
    else:
        A = df.assemble(Ai[i])
    if i == 0:
        BC.apply(A)
    else:
        BC.zero(A)
    A = -convert_matrix(A)
    mmwrite("A" + str(i), A)
    Ai[i] = A

B = df.assemble(v * dsi(4))
BC.apply(B)
B = np.array(B)
B = B[:, np.newaxis]
np.savetxt("B.txt", B)

area = df.assemble(1 * dxi(2))
C1 = np.array(df.assemble(v * dxi(2))) / area
C2 = np.array(df.assemble(v * dxi(3))) / area
C3 = np.array(df.assemble(v * dxi(4))) / area
C4 = np.array(df.assemble(v * dxi(5))) / area
C = np.vstack([C1, C2, C3, C4])
np.savetxt("C.txt", C)

E = df.assemble(u * v * dxi)
BC.apply(E)
E = convert_matrix(E)
mmwrite("E", E)

savemat(
    "ABCE.mat",
    dict({"A" + str(i): A for i, A in enumerate(Ai)}, B=B, C=C, E=E),
    format="5",
)
