package ColomEtAlOffLatticeExample;

import HAL.GridsAndAgents.AgentGrid2D;
import HAL.GridsAndAgents.SphericalAgent2D;
import HAL.Gui.OpenGL2DWindow;
import HAL.Rand;
import HAL.Tools.FileIO;
import HAL.Tools.Internal.Gaussian;
import HAL.Util;
import org.apache.commons.cli.*;

import java.util.ArrayList;
import java.util.HashMap;

import static HAL.Util.RGB256;

class CellOL extends SphericalAgent2D<CellOL, ColomExampleOffLattice>{
    int type;
    int color;
    int clone_label;
    int subclone_label;
    boolean edge_subclone;
    double forceSum;//used with contact inhibition calculation
    public void Init(int type, int clone_label, int subclone_label, boolean edge_subclone){
        this.type =type;
        this.clone_label=clone_label;
        this.subclone_label=subclone_label;
        this.edge_subclone=edge_subclone;
        this.radius= G.RADIUS;
        GetColor();
    }
    public void GetColor(){
        if (subclone_label==-1){
            // Not a subclone. Colour based on WT or Mutant
            if (type== ColomExampleOffLattice.WT){
                color= ColomExampleOffLattice.WT;
            } else {
                color= ColomExampleOffLattice.MUTANT;
            }
        } else {
            // Subclone - colour by edge or core
            if (edge_subclone){
                color= ColomExampleOffLattice.EDGE_SUBCLONE;
            } else {
                color= ColomExampleOffLattice.CORE_SUBCLONE;
            }
        }
    }
    public void AssignSubCloneLabel(int subclone_label, double neighbourhoodSize){
        //Finish this. Define what an edge clone is.
        this.subclone_label=subclone_label;
        edge_subclone = false;
        for (CellOL cell : G.IterAgentsRad(Xpt(), Ypt(), neighbourhoodSize)){
            if(cell.type == ColomExampleOffLattice.WT){
                edge_subclone=true;
                break;
            }
        }
        GetColor();
    }
    double ForceCalc(double overlap, CellOL other){
        if(overlap<0) {
            return 0;//if cells aren't actually overlapping, then there is no force response
        }
        return G.FORCE_SCALER*overlap;//this constant scaling of the overlap is called Hooke's law!
    }
    public void CalcMove(){
        //sets x and y velocity components of cell
        //G.neighborList.clear();
        //G.GetAgentsRad(G.neighborList,G.neighborInfo,Xpt(),Ypt(),G.RADIUS*2);
        forceSum=SumForces(G.RADIUS*2,this::ForceCalc);
        //forceSum=SumForces(G.neighborList,G.neighborInfo,this::ForceCalc);
    }
    public boolean CanDivide(double div_bias,double inhib_weight){
        if (forceSum==0){
            return true;
        }
        return G.rn.Double()<Math.tanh(div_bias-forceSum*inhib_weight);
    }
    public void MoveDiv(){
        //move cell and reduce x and y velocity components by friction constant
        ForceMove();
        ApplyFriction(G.FRICTION);
        // Need to add some death
        if((type == ColomExampleOffLattice.WT &&(G.rn.Double() < G.WT_DIE_PROB)||(type == ColomExampleOffLattice.MUTANT &&(G.rn.Double() < G.MUTANT_DIE_PROB)))){
            // cell will die
            Dispose();
            return;
        }
        //compute whether division can occur, using the constants
        if((type == ColomExampleOffLattice.WT &&CanDivide(G.WT_DIV_BIAS, G.WT_INHIB_WEIGHT))||(type == ColomExampleOffLattice.MUTANT &&CanDivide(G.MUTANT_DIV_BIAS, G.MUTANT_INHIB_WEIGHT))){
                Divide(radius*2.0/3.0, G.divCoordStorage, G.rn).Init(type, clone_label, subclone_label, edge_subclone);
        }
    }
}

public class ColomExampleOffLattice extends AgentGrid2D<CellOL> {
    static final int WHITE=RGB256(248,255,252), WT = Util.BLACK, MUTANT =RGB256(0,150,0);  // Cell type colours
    static final int CORE_SUBCLONE=Util.BLUE, EDGE_SUBCLONE=Util.RED;  // Subclone colours
    double RADIUS=1;

    double FORCE_SCALER=0.25;
    double FRICTION=0.5;

    //default mutation properties. These can be overwritten by the command line options.
    double WT_DIV_BIAS=0.1;
    double MUTANT_DIV_BIAS=0.1;
    double WT_INHIB_WEIGHT=0.2;
    double MUTANT_INHIB_WEIGHT=0.2;
    double WT_DIE_PROB=0.05;
    double MUTANT_DIE_PROB=0.05;
    ArrayList<CellOL> neighborList=new ArrayList<>();
    ArrayList<double[]> neighborInfo=new ArrayList<>();
    double[]divCoordStorage=new double[2];
    Rand rn=new Rand(0);
    Gaussian gn =new Gaussian();
    FileIO out;

    public ColomExampleOffLattice(int x, int y) {
        super(x, y, CellOL.class,true,true);
    }
    public ColomExampleOffLattice(int x, int y, String outFileName) {
        super(x, y, CellOL.class,true,true);
        out=new FileIO(outFileName,"w");
    }
    public static void main(String[] args) {
        Options options = new Options();
        options.addOption("h", "help", false,"Show this help message");
        // Options for the simulation
        options.addOption("xdim", true,"x-dimension of grid. Default=250");
        options.addOption("ydim", true,"y-dimension of grid. Default=250");
        options.addOption("pix", true,"pixels per unit length. Default=3");
        options.addOption("initPop", true,"initial number of cells. Default=xdim*ydim/3");
        options.addOption("maxSteps", true,"Number of simulation steps. Default=850");
        options.addOption("writeSteps", true,"Number of simulation steps between recording results. Default=10");
        options.addOption("drawSteps", true,"Number of simulation steps between movie frames. Default=5");
        options.addOption("outFile", true,"Name of output csv file. Default='OffLatticeSimulationResults.csv'");
        options.addOption("videoOutDir", true,
                "Output directory for video frames. Directory must exist before running. " +
                        "The jpg frames can be combined into a movie. " +
                        "For example, using ffmpeg> ffmpeg -y -r 12 -i \"off_lattice_%06d.jpg\" -b:v 10000k -c:v libx264 -pix_fmt yuv420p output_video.mp4 \nDefault=null");
        options.addOption("mutantInduction", true,"Proportion of initial cells that are mutant. Default=0.015");
        options.addOption("subcloneInductionStep", true,"Simulation step to induce subclones. Default=250");
        options.addOption("subcloneInduction", true,"Proportion of cells that are labelled as a subclone founder. Default=0.075");
        options.addOption("edgeProximity", true,
                "Distance threshold for defining edge and core subclones. Default=3.5");

        // Options for the mutant and wt cell properties
        options.addOption("run_example", false,
                "Runs with example non-neutral parameters. MUTANT_DIV_BIAS = 0.13, all others left as default. " +
                        "Ignores any other arguments that change MUTANT or WT cell properties");
        options.addOption("WT_DIV_BIAS", true,
                "Division bias of WT cells. Higher values mean fitter cells. Default=0.1");
        options.addOption("MUTANT_DIV_BIAS", true,
                "Division bias of mutant cells. Higher values mean fitter cells. Default=0.1");
        options.addOption("WT_INHIB_WEIGHT", true,
                "Inhibition of WT cells based on neighbouring cells. Lower values mean fitter cells. Default=0.2");
        options.addOption("MUTANT_INHIB_WEIGHT", true,
                "Inhibition of mutant cells based on neighbouring cells. Lower values mean fitter cells. Default=0.2");
        options.addOption("WT_DIE_PROB", true,
                "Probability for a WT cell to differentiate (die) at each time step. Lower values mean fitter cells. Default=0.05");
        options.addOption("MUTANT_DIE_PROB", true,
                "Probability for a mutant cell to differentiate (die) at each time step. Lower values mean fitter cells. Default=0.05");

        CommandLineParser parser = new DefaultParser();
        try {
            CommandLine cmd = parser.parse(options, args);
            if (cmd.hasOption("help")){
                HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp( "ExampleOffLattice1", options );

            } else {
                // default values for the simulation
                int x = 250;
                int y = 250;
                double pix = 3;
                int maxSteps = 850;
                int writeSteps = 10;
                int drawSteps = 5;
                double mutantInduction = 0.015;
                int subcloneInductionStep = 250;
                double subcloneInduction = 0.075;
                double edgeProximity = 3.5;
                String outFile = "OffLatticeSimulationResults.csv";
                String videoOutDir = null;
                if (cmd.hasOption("xdim")) {
                    x = Integer.parseInt(cmd.getOptionValue("xdim"));
                }
                if (cmd.hasOption("ydim")) {
                    y = Integer.parseInt(cmd.getOptionValue("ydim"));
                }
                if (cmd.hasOption("pix")) {
                    pix = Double.parseDouble(cmd.getOptionValue("pix"));
                }
                int initPop = x * y / 3;
                if (cmd.hasOption("initPop")) {
                    initPop = Integer.parseInt(cmd.getOptionValue("initPop"));
                }
                if (cmd.hasOption("maxSteps")) {
                    maxSteps = Integer.parseInt(cmd.getOptionValue("maxSteps"));
                }
                if (cmd.hasOption("writeSteps")) {
                    writeSteps = Integer.parseInt(cmd.getOptionValue("writeSteps"));
                }
                if (cmd.hasOption("drawSteps")) {
                    drawSteps = Integer.parseInt(cmd.getOptionValue("drawSteps"));
                }
                if (cmd.hasOption("outFile")) {
                    outFile = cmd.getOptionValue("outFile");
                }
                if (cmd.hasOption("videoOutDir")) {
                    videoOutDir = cmd.getOptionValue("videoOutDir");
                }
                if (cmd.hasOption("mutantInduction")) {
                    mutantInduction = Double.parseDouble(cmd.getOptionValue("mutantInduction"));
                }
                if (cmd.hasOption("subcloneInductionStep")) {
                    subcloneInductionStep = Integer.parseInt(cmd.getOptionValue("subcloneInductionStep"));
                }
                if (cmd.hasOption("subcloneInduction")) {
                    subcloneInduction = Double.parseDouble(cmd.getOptionValue("subcloneInduction"));
                }
                if (cmd.hasOption("edgeProximity")) {
                    edgeProximity = Double.parseDouble(cmd.getOptionValue("edgeProximity"));
                }

                //to record output, call the constructor with an output filename
                ColomExampleOffLattice ex = new ColomExampleOffLattice(x, y, outFile);
                if (ex.out != null) {
                    ex.out.Write("CloneType,TimeStep,CloneSizes\n");
                }

                // default values for the mutant properties are defined above in the class (not in the main function)
                if (cmd.hasOption("run_example")) {
                    ex.MUTANT_DIV_BIAS = 0.13;
                } else {
                    if (cmd.hasOption("WT_DIV_BIAS")) {
                        ex.WT_DIV_BIAS = Double.parseDouble(cmd.getOptionValue("WT_DIV_BIAS"));
                    }
                    if (cmd.hasOption("MUTANT_DIV_BIAS")) {
                        ex.MUTANT_DIV_BIAS = Double.parseDouble(cmd.getOptionValue("MUTANT_DIV_BIAS"));
                    }
                    if (cmd.hasOption("WT_INHIB_WEIGHT")) {
                        ex.WT_INHIB_WEIGHT = Double.parseDouble(cmd.getOptionValue("WT_INHIB_WEIGHT"));
                    }
                    if (cmd.hasOption("MUTANT_INHIB_WEIGHT")) {
                        ex.MUTANT_INHIB_WEIGHT = Double.parseDouble(cmd.getOptionValue("MUTANT_INHIB_WEIGHT"));
                    }
                    if (cmd.hasOption("WT_DIE_PROB")) {
                        ex.WT_DIE_PROB = Double.parseDouble(cmd.getOptionValue("WT_DIE_PROB"));
                    }
                    if (cmd.hasOption("MUTANT_DIE_PROB")) {
                        ex.MUTANT_DIE_PROB = Double.parseDouble(cmd.getOptionValue("MUTANT_DIE_PROB"));
                    }
                }

                OpenGL2DWindow vis = new OpenGL2DWindow("Off Lattice Example", (int) (x * pix), (int) (y * pix), x, y);
                ex.Setup(initPop, mutantInduction);
                // Running the simulation
                int i = 0;
                while (i < maxSteps && !vis.IsClosed()) {//check for click on close button on window
                    vis.TickPause(0);
                    if (i == subcloneInductionStep) {
                        ex.InduceSubclones(subcloneInduction, edgeProximity);
                    }
                    ex.StepCells(i);
                    if (ex.out != null && (i % writeSteps == 0)) {
                        ex.RecordOut(ex.out, i);
                    }
                    if (i % drawSteps == 0) {
                        ex.DrawCells(vis);
                        if (videoOutDir != null) {
                            vis.ToJPG(videoOutDir + "/off_lattice_" + String.format("%06d", i / drawSteps) + ".jpg");
                        }
                    }

                    i++;
                }
                if (ex.out != null) {
                    ex.out.Close();//be sure to call Close when finished writing output to make sure everything is recorded.
                }
                vis.Close();
            }
        } catch (ParseException e){
            e.printStackTrace();
        }
    }
    public void Setup(double initPop,double propMutant){
        for (int i = 0; i < initPop; i++) {
            RandomPointInRect(divCoordStorage);
            double x = rn.Double()*xDim;
            double y = rn.Double()*yDim;
            //create a new agent, and set the type depending on a comparison with the random number generator
            NewAgentPT(divCoordStorage[0],divCoordStorage[1]).Init(rn.Double()<propMutant? MUTANT : WT, i,
                    -1, false);
        }
    }

    public void RandomPointInRect(double[] ret) {
        double x = rn.Double() * xDim;
        double y = rn.Double() * yDim;
        ret[0] = x;
        ret[1] = y;
    }

    public void DrawCells(OpenGL2DWindow vis){
        vis.Clear(WHITE);
//        for (CellOL cell : this) {
//            //draw "cytoplasm" of cell
//            vis.Circle(cell.Xpt(),cell.Ypt(),cell.radius,CYTOPLASM);
//        }
        for (CellOL cell : this) {
            //draw colored "nucleus" on top of cytoplasm
            vis.Circle(cell.Xpt(), cell.Ypt(), cell.radius , cell.color);
        }
        vis.Update();
    }
    public void StepCells(int step){
        for (CellOL cell : this) {
            cell.CalcMove();//calculation of forces before any agents move, for simultaneous movement
        }
        for (CellOL cell : this) {
            cell.MoveDiv();//movement and division
        }
    }
    public void InduceSubclones(double induction, double neighbourhoodSize){
        int s=0;
        for (CellOL cell : this) {
            if(rn.Double()<induction&&cell.type==MUTANT){
                cell.AssignSubCloneLabel(s, neighbourhoodSize);
                s++;
            }
        }
    }
    public void RecordOut(FileIO writeHere, int step){
        // Edit here to change output record.
        // Output clone sizes
        // Time step, then list of clone sizes.
        // Each time step has one line for WT, one line for Mutant, one line for core subclone and one line for edge subclone sizes. 
        String outlineWT="WT,"+step;
        String outlineMutant="MUTANT,"+step;
        String outlineCore="CORE,"+step;
        String outlineEdge="EDGE,"+step;
        HashMap<Integer, Integer> cloneSizesWT = new HashMap<Integer, Integer>();
        HashMap<Integer, Integer> cloneSizesMutant = new HashMap<Integer, Integer>();
        HashMap<Integer, Integer> subcloneSizesCore = new HashMap<Integer, Integer>();
        HashMap<Integer, Integer> subcloneSizesEdge = new HashMap<Integer, Integer>();
        for (CellOL cell : this) {
            if(cell.type == WT){
                if(!cloneSizesWT.containsKey(cell.clone_label)){
                    cloneSizesWT.put(cell.clone_label, 1);
                } else {
                    cloneSizesWT.put(cell.clone_label, cloneSizesWT.get(cell.clone_label)+1);
                }
            } else{
                if(!cloneSizesMutant.containsKey(cell.clone_label)){
                    cloneSizesMutant.put(cell.clone_label, 1);
                } else {
                    cloneSizesMutant.put(cell.clone_label, cloneSizesMutant.get(cell.clone_label)+1);
                }
            }

            if(cell.subclone_label != -1){
                if(cell.edge_subclone){
                    if(!subcloneSizesEdge.containsKey(cell.subclone_label)){
                        subcloneSizesEdge.put(cell.subclone_label, 1);
                    } else {
                        subcloneSizesEdge.put(cell.subclone_label, subcloneSizesEdge.get(cell.subclone_label)+1);
                    }
                } else {
                    if(!subcloneSizesCore.containsKey(cell.subclone_label)){
                        subcloneSizesCore.put(cell.subclone_label, 1);
                    } else {
                        subcloneSizesCore.put(cell.subclone_label, subcloneSizesCore.get(cell.subclone_label)+1);
                    }
                }
            }
        }

        //Output one line per clone type, and one line per subclone type
        writeLine(writeHere, outlineWT, cloneSizesWT);
        writeLine(writeHere, outlineMutant, cloneSizesMutant);
        writeLine(writeHere, outlineCore, subcloneSizesCore);
        writeLine(writeHere, outlineEdge, subcloneSizesEdge);
    }

    public void writeLine(FileIO writeHere, String lineStart, HashMap<Integer, Integer> counter){
        writeHere.Write(lineStart);
        counter.forEach( (k, v) ->
                writeHere.Write("," + v)
        );
        writeHere.Write("\n");
    }
}
