function [PA,stats]=param_distribution(mgca,nr_id,fig_num,options)
% This function a) computes and plots the empirical distribution of the groups of individual models (IMs) passed by argument,
% and b) calculates their statistics.  In principle, each group of models is supposed to be the result of one test (robot 
% trajectory), which results in a group of IMs. This function then helps to study the variability of the IMs in different tests.
%  
% SYNTAX: param_distribution(mgca,nr_id,fig_num,[options]);
%
% INPUTS
% - mgca: model groups cell array. A cell array containing the groups of models to plot the distributions and calculate the 
%       statistics of. mgca{t_i}{n_i} contains the model of the n_i'th node corresponding to the t_i'th group (test)
% - nr_id: id of the node on top of the robot. Its corresponding model is extracted using all the data from all the nodes (reference model)
% - fig_num: figure where to plot the comparison. If empty, no plot will be produced.
% - options: optional configuration parameters (see the def_opts structure in the code).
%
% RETURNS
% - PA: Parameters Array, containing all the parameters of all the IMs, indexed as PA(test index,node index,parameter index).
% - stats: cell with structures containing the statistics of the model parameters. stats{t_i} contains the statistics associated
%       to group/test t_i (that is, to the group of models in mgca{t_i}).

% example:
%	options=struct( ...
%		'PlotGlobalEmpiricalDensity', false, ...
%		'PlotRefModel',false, ...
%		'Labels',{{'Bcourt','Lobby','Office'}}); %Note the double brackets!
%	param_distribution({m1,m2,m3},21,1,options);
%

%Default options
def_opts=struct(...
    'PlotEmpiricalDensities',true, ...      % The densities for the parameters for each test
    'PlotRefModel',true, ...                % The model built with all the data from all the nodes in each test. Plot only a symbol (see RefModelSymbol) in the value of the parameter.
    'PlotGlobalEmpiricalDensity',false, ...  % The density for parameters of all nodes together (same test)
    'PlotDots',true, ...                    % One dot for each value of the parameter for each node.
    
    'NumMeshPoints',2^8, ...                % Number of points in the mesh used by the kernel density estimator
    'RefModelSymbol','.', ...               % Symbol to use to plot the value of the parameters for the reference models
    'PlotLabels',true, ...                  % Do we want to plot the labels or not?
    'Labels',{{}}, ...                      % Labels for the data (legends). Empty by default in the def_opts, so it can be easily identified later
    'XLabels',{{}}, ...
    'YLabels',{{}}, ...
    'Titles',{{}}, ...                      % Titles of the subplots (one for each parameter)
    'Grid',false, ...                       % Grid on or off
    'Linewidth',2, ...                      % Width of the lines used for the densities
    'DotSize',4, ...                        % Marker size for the dots
    'KRange',0.5, ...                       % Coefficient for addition of range when estimation of the density 
    'KSep',0.2, ...                         % Coefficient of space in Y axis dedicated to the plot of the points
    'Colors',{{}}, ...	                    % of the plots for each test
    'Min',[], ...
    'Max',[], ...
    'UseKde',false);                        % Use kde function (true) for the density estimation or gkdeb (false)
    
if nargin < 3
  error("Expected 3 arguments");
elseif nargin == 3
  options=def_opts;
elseif nargin == 4
  %User provided options
  if !isstruct(options)
    error("Param_distribution:The argument options must be a structure");
  endif
  fs = {'PlotEmpiricalDensities','PlotGlobalEmpiricalDensity','PlotDots','PlotRefModel','NumMeshPoints','RefModelSymbol', ...
	'PlotLabels','Labels','XLabels','YLabels','Titles','Grid','Linewidth','DotSize','KRange','KSep','Colors','Min','Max','UseKde'};
  for nm=1:length(fs)
      if ~isfield(options,fs{nm}), options.(fs{nm}) = def_opts.(fs{nm}); end
  endfor
endif

%Check that the number of labels is the same than the number of tests
if !isempty(options.Labels) && numel(options.Labels)!=numel(mgca)
  error("Number of labels must be equal to the number of tests to compare");
endif
%If the labels are not specified, create legends
if isempty(options.Labels)
  for t_i=1:numel(mgca)
    options.Labels{t_i}=['test ' sprintf("%i",t_i)];
  endfor
endif
%Check that the number of colors is the same than the number of tests
if !isempty(options.Colors) && numel(options.Colors)!=numel(mgca)
  error("Number of Colors must be equal to the number of tests to compare");
endif
%If the colors are not specified, pick the colors randomly
if isempty(options.Colors)
  for t_i=1:numel(mgca)+1 %One extra for the global model, if required
    options.Colors{t_i}=rand(1,3);
  endfor
endif

%Number of parameters. Needed to add NaN as parameter values when nodes are missing
num_p=0;
for t_i=1:numel(mgca)
  for n_i=1:numel(mgca{t_i}) %node index
    if (isempty(mgca{t_i}{n_i}))
      continue
    endif
    num_p=length(mgca{t_i}{n_i}.theta);
    break;
  endfor
endfor
if num_p==0 error("No parameters in models???") endif

%Form an array with all the parameters from all the groups/tests and nodes
for t_i=1:numel(mgca)
  for n_i=1:numel(mgca{t_i}) %node index
    if (n_i==nr_id) || isempty(mgca{t_i}{n_i})
      PA(t_i,n_i,:)=repmat(NaN,1,num_p);
      continue
    endif
    PA(t_i,n_i,:)=mgca{t_i}{n_i}.theta; %Parameters Array
  endfor
  
  %The statistics
  for p_i=1:num_p
    data_=squeeze(PA(t_i,:,p_i));     %all the p_i'th parameter values corresponding to the test t_i
    data_=data_(1:numel(mgca{t_i}));  %Remove possible extra zeros coming from comparing tests with different number of nodes
    data_(find(isnan(data_)))=[];     %Remove possible NaN from empty nodes and the node from the robot
    stats{t_i}.params(p_i,:)=data_;
    stats{t_i}.mean(1,p_i)=mean(data_);
    stats{t_i}.max(1,p_i)=max(data_);
    stats{t_i}.min(1,p_i)=min(data_);
    stats{t_i}.std_dev(1,p_i)=std(data_);
    stats{t_i}.median(1,p_i)=median(data_);
    par_values{t_i}{p_i}=data_;
  endfor
  clear data_;
endfor

if isempty(fig_num)
    return
endif

num_subplots=num_p;

fh=figure(fig_num);clf
%For each parameter one subplot
for p_i=1:num_p
  ph=subplot(num_subplots,1,p_i); %plot handler
  if options.PlotEmpiricalDensities
    d_max=0; %to find the highest point in the plot
    for t_i=1:numel(mgca) %for each test, one curve and line
      if !isempty(options.Min)
        MIN=options.Min;
      else
        MIN=min(par_values{t_i}{p_i})-range(par_values{t_i}{p_i})*options.KRange;
      endif
      if !isempty(options.Max)
        MAX=options.Max;
      else
        MAX=max(par_values{t_i}{p_i})+range(par_values{t_i}{p_i})*options.KRange;
      endif
      if options.UseKde
        [~,density{t_i},xmesh{t_i}]=kde(par_values{t_i}{p_i},options.NumMeshPoints,MIN,MAX);
      else
        clear p; p.lB=MIN;p.uB=MAX;
        p=gkdeb(par_values{t_i}{p_i},p); density{t_i}=p.pdf'; xmesh{t_i}=p.x;
      endif
      d_max=max([d_max;density{t_i}]);
    endfor
    for t_i=1:numel(mgca)
      color=options.Colors{t_i};
      if options.PlotLabels 	
        legnd=options.Labels{t_i};
        plot(xmesh{t_i},density{t_i},[';' legnd ';' '-'],'Color',color,'LineWidth',options.Linewidth);hold on;
      else
        plot(xmesh{t_i},density{t_i},'-','Color',color);hold on;
      endif
    endfor
  endif
  
  %A pdf with all the parameters as if they had come from the same test (global model). 
  %Note that this is different than the reference model.
  if (numel(mgca)>1 && options.PlotGlobalEmpiricalDensity)
    data_gm=squeeze(PA(:,:,p_i));
    data_gm=data_gm(:,1:numel(mgca{t_i})); %Remove possible extra zeros coming from comparing tests with different number of nodes
    data_gm(find(isnan(data_gm)))=[];
    if !isempty(options.Min)
      MIN=options.Min;
    else
      MIN=min(data_gm)-range(data_gm)*options.KRange;
    endif
    if !isempty(options.Max)
      MAX=options.Max;
    else
      MAX=max(data_gm)+range(data_gm)*options.KRange;
    endif
    if options.UseKde
      [~,density_gm,xmesh_gm]=kde(data_gm,options.NumMeshPoints,MIN,MAX);
    else
      clear p; p.lB=MIN; p.uB=MAX;
      p=gkdeb(data_gm,p); density_gm=p.pdf'; xmesh_gm=p.x;
    endif 
    legnd=['all']; 
    plot(xmesh_gm,density_gm,[';' legnd ';' '--']);
  endif 
  if !isempty(options.Titles)
    title(options.Titles{p_i});
  endif
  if isempty(options.XLabels)
    xlabel(sprintf("p%i", p_i));
  else
    xlabel(options.XLabels{p_i});
  endif
  if !isempty(options.YLabels)
    ylabel(options.YLabels{p_i});
  endif
  
  %Plot the dots corresponding to parameter values
  dy=linspace(0,options.KSep*d_max,numel(mgca)+2); %add two to leave some margin up and down
  for t_i=1:numel(mgca)
    color=options.Colors{t_i};
    if options.PlotDots && !options.PlotRefModel
      sep=-dy(t_i+1); %leave one for the upper margin
      plot(par_values{t_i}{p_i},sep*ones(1,length(par_values{t_i}{p_i})),'.','Color',color,'markersize',options.DotSize);
      limits=axis();
      limits(3)=-options.KSep*d_max; %adjust y_min
      axis(limits);
    elseif options.PlotDots && options.PlotRefModel
      sep=-dy(t_i+1); %leave one for the upper margin
      plot(par_values{t_i}{p_i},sep*ones(1,length(par_values{t_i}{p_i})),'.','Color',color,'markersize',options.DotSize);
      plot(mgca{t_i}{nr_id}.theta(p_i),sep,options.RefModelSymbol,'Color',color,'markersize',2*options.DotSize);
      limits=axis();
      limits(3)=-options.KSep*d_max; %adjust y_min
      axis(limits);
    elseif options.PlotRefModel && !options.PlotDots 
      sep=-options.KSep*d_max;
      plot(mgca{t_i}{nr_id}.theta(p_i),sep,options.RefModelSymbol,'Color',color,'markersize',2*options.DotSize);
      %readjust the axis so that the text fits
      limits=axis();
      limits(3)=2*sep; %y_min down a bit to fit the symbols
      axis(limits);
    endif
  endfor  
  set(ph,'YTick',[]); %remove the Y ticks
  %Do we want the grid on?
  if options.Grid
    grid on; 
  endif
  %Plot the line at zeros
  limits=axis();
  plot(limits(1:2),[0 0],'-k');
endfor

figure(fh);

