function model=extract_rssi_models(node_s,robot_s, filter_options, model_type, fig_number)
% Extract RSSI-distance model for each individual node (besides the one on the robot) 
% and for all the nodes
%
% The data is filtered first with the indexed_waverage function. The type of averaging can be
% controlled with the filter_options argument, that is the list of parameters to be pased to
% the filter_rssi function
%
% SYNTAX: model=extract_rssi_model(node_s,robot_s, filter_options, model_type)
%	  model=extract_rssi_model(node_s,robot_s, filter_options, model_type, fig_number)
%	  
%
% INPUTS
%	node_s: cell array with info from the nodes
%	robot_s: structure with info about the position of the robot
%	filter_options: parameters to be passed to the filter_rssi function.
%			If no filtering is required, put 'NO_FILTER'.
%	model_type: cell array with information about the type of model to extract and the required
%	parameters. The possible types are
%		- PN: polynomial with normal error. Requires the degree of the polynomial
%		- PSN: polynomial with skewed normal error. Requires the degree of the polynomial
%		- PBN: polynomial with Bi-normal error. Requires the degree of the polynomial
%
%	fig_number: figure used for plotting results. If not specified, or if equal to zero, no plot
%		will be produced
%
% RETURNS
%	model: cell array indexed with the rx_node id, each cell containing a structure with the model
%		parameters for that node. The specific fields depend on the model.
%
% 	The model that uses all the data is in cell number 13 (model{13}). The node on top of the robot
%	uses this id,
%
%
% EXAMPLES:
%	model=extract_rssi_models(node_s,robot_s,'NO_FILTER',{'PN',1})
%	extracts polynomial models of degree 1 with normal error
%
%	model=extract_rssi_models(node_s,robot_s,'NO_FILTER',{'PN',3})
%	extracts polynomial models of degree 3 with normal error
%

% 	m5=extract_rssi_models(oh_av{t_i}.nodes_s,oh_av{t_i}.robot_s,'NO_FILTER',{'PLN',1,5});

if (nargin < 4)
	error ("Wrong number of arguments");
endif

if ((nargin == 5) && (fig_number!=0))
  plot_me=true;
else
  plot_me=false;
  fig_number=0;
endif

if ( (!iscell(filter_options) && !ischar(filter_options)) || ...
     (ischar(filter_options) && !strcmp(filter_options,'NO_FILTER')) )
  error("filter_options must be either a cell array or \'NO_FILTER\'"); 
endif

d_all=[];
rssi_all=[];
lr=0;

nr_id=robot_s.nr_id; %node robot id

for n_i=1:numel(node_s) %don't study the node from the robot with itself neither the sink!
  %Check that the node is not empty.
  if (isempty(node_s{n_i}) || (node_s{n_i}.rx_id == nr_id) || isnan(node_s{n_i}.pos)) continue; endif

  %[d,rssi,traj,nts,ierr]=get_data_from_node(node_s,robot_s,filter_options);
  [d,rssi,traj,nts,ierr]=get_data_from_node(node_s{n_i},robot_s,filter_options);

  %Check ierr
  if(ierr==2)
    %skip this node
    continue;
  endif
  
  % --- Model extraction --- %
  model{n_i}=extract_rssi_model(d,rssi,model_type);
  if isnan(model{n_i}.lr)
    keyboard
  endif
  lr+=model{n_i}.lr;
  
  % Add the true position of the node
  model{n_i}.t_pos=node_s{n_i}.pos;
    
  % --- Collect the data for global model ---
  rssi_all=[rssi_all ; rssi];
  d_all=[d_all ; d];

  if (plot_me==1)
    plot_rssi_model(model{n_i},fig_number)
    subplot(2,1,1);
    title(sprintf("RSSI vs. distance for node %i",n_i));
    subplot(2,1,2);
    title(sprintf("Residuals for node %i",n_i));
    printf("Node%i \t",n_i);
    printf_model_params(model{n_i});
    kbhit();%wait for a key hit
  endif
  fflush(stdout);
endfor

printf("Log-likelihood of residuals for IMs:%i\n",lr);

%Global model with all the data
model{nr_id}=extract_rssi_model(d_all,rssi_all,model_type);

if (plot_me)
  plot_rssi_model(model{nr_id},fig_number)
  subplot(2,1,1);
  title("RSSI vs. distance (All nodes)");
  subplot(2,1,2);
  title("Residuals (All nodes)");
  printf("CM(%i)\t",nr_id);
  printf_model_params(model{nr_id});
  printf('CM(lr):%f\n',model{nr_id}.lr);
  %printf("All: \tp=[%f,%f].res_sdev=%f\n",model{nr_id}.p(1),model{nr_id}.p(2),model{nr_id}.res_sdev);
  fflush(stdout);
endif
