function  model=extract_pln_model(d,rssi,p_deg,d0=1)
% Fit the model RSSI = p(log10(d)) + N(0,Qcov) to the rssi - distance data, 
% where p is a n degree polynomial function of the log of the distance d, and N(0,Qcov)
% a normally distributed error with zero mean and Qcov covariance.
%
% SYNTAX: model=extract_pln_model(d,rssi,p_deg)
%
% Arguments
%	d: 
%	rssi:
%	p_deg: degree of the polynomial to fit the model to.
%
% Returns
%	model: cell array with the following fields
%		model.type: string identifying the type of model. "PN" (Polynomial with Normal error)
%		model.typeID: numeric ID identifying the type. For PN type this is 1 
%		model.typeCell: cell identifying the model and polynomial degree, that is {"PN",1}
%		model.rssi: rssi used for the fitting, sorted with distance
%		model.d: distance used for the fitting
%		model.p: vector with fitting parameters
%		model.std: standard deviation of the Gaussian
%		model.var: variance of the Gaussian
%		model.res: the residuals
%		model.yf: y points of the fitted curve for the given d 
%

if (nargin < 3)
  error ("Wrong number of arguments");
endif


%Get rid of NaN rssi measurements
idx=find(isnan(rssi));
rssi(idx)=[];
d(idx)=[];

%Sort the data so that we can plot it nicely in order
if !issorted(d)
  [d, index]=sort(d);
  rssi=rssi(index);
endif
DN=d/d0;%Normalization once that the data is ordered!
[p,s]=polyfit(log10(DN),rssi,p_deg);
%The path loss exponent is the coefficient for the linear term, that is, p(end-1)
%Divide by -10 to obtain the path loss exponent. Be careful to take this into account when 
%calculating the likelihood
p(end-1)/=-10;

model.type='PLN'; 
model.typeID=9;
model.typeCell={"PLN",p_deg};
model.p_deg=p_deg;
model.rssi=rssi;
model.d=d; %Not normalized, as is the original data
model.d0=d0;
model.p=p;
%The residuals
model.res=rssi-s.yf; 
%The resuduals are already in logaritmic scale, and thus are normally distributed. ML estimates for the parameters
%http://en.wikipedia.org/wiki/Log-normal_distribution
model.std=std(model.res);
model.var=var(model.res);
model.yf=s.yf;
model.C=s.C; %The unscaled covariance matrix, formally equal (X'*X)^-1
%All the parameters of the model in a vector
model.theta=[p model.std];

%These are the mean and std.dev for the whole distribution. Needed for comparison with mixture of gaussians
model.MU=mean(model.res); %Should be zero
model.S=model.std;
%Handler to a function that calculates the likelihood
%Usage: model.likelihood(model,z,d);
model.likelihood=@(m,z,d) calculate_likelihood_P(m,m.theta,z,d);
model.loglikelihood=@(m,z,d) log(m.likelihood(m,z,d));
%Log likelihood of the training data. The better the model, the higher this number.
model.lr=sum(model.loglikelihood(model,rssi,d));

%Labels for the parameters
for i=1:numel(p)
  model.plabel{i}=['p' sprintf("%i",i)];
endfor
model.plabel{i+1}=['\sigma'];

endfunction
