function zp=calculate_prediction(m,theta,d)
%Calculates the predicted RSS measurement for the distance d according to the model m
%
% SYNTAX
%	zp=calculate_prediction(m,theta,d)
%
% INPUTS
%	- m: the model.
%	- theta: KxT matrix with the K vectors of model parameters of dimension T
%	- d: NxK matrix containing the distances.
%
% RETURNS
%	- z: NxK matrix containing the predicted RSS measurements
%

if nargin != 3
  error("calculate_prediction: expected 3 arguments");
  print_usage ();
endif


%----------------------------------------------------
%         Check that the data provided is consistent
%----------------------------------------------------
%Check correct dimensions of the arguments
[Kt,Tt]=size(theta);
[Nd,Kd]=size(d);
if Tt!=numel(m.theta)
  error("calculate_prediction: wrong number of parameters in theta.");
endif
if Kt!=1 && Kt!=Kd
  error("calculate_prediction: check K");
endif

%Log Normal error (LN) 
if (m.typeID==9)
  %Separate p and std
  p=theta(:,1:numel(m.p));
  p(:,end-1)*=-10; %Because of the path loss exponent
  %The model predictions (averages for the normal distributions)
  dp=permute(d,[1 3 2]);
  Dp=bsxfun(@power,log10(dp/m.d0),[columns(p)-1:-1:0]);
  zp=sum(Dp.*permute(p,[3 2 1]),2);
  zp=permute(zp,[1 3 2]);
  %zp(find(d==0))=sign(p(:,1))*(-Inf); %The dominant factor is the one with the largest exponent

else
  error("calculate_prediction: Model type not supported");
endif


%%%%%%%%%%%%%%%%%%%%%%%%% TESTING %%%%%%%%%%%%%%%%%%%%%%%%

%!test
%!shared N, K
% In all these cases, the predictions should be exactly the distances
%! N=100;K=10;
%! m1=create_init_model({"PN",1},{[1 0],1});
%! m2=create_init_model({"PMIXKN",1,2},{[1 0],[0.5 0.5],[0 0],[10 10],[1 1]});
%! d=rand(N,K);
%! zp=calculate_prediction(m1,m1.theta,d);
%! assert(zp,d);
%! zp=calculate_prediction(m2,m2.theta,d);
%! assert(zp,d);

%!test
% The predictions of these ones should be equal
%! d=rand(N,K);
%! m1=create_init_model({"PLN",1},{[2 -50],5});
%! P0=-50-20*log10(8);
%! m2=create_init_model({"PLN",1},{[2 P0],5,8}); %d0=8
%! m3=create_init_model({"PLNMIXKN",3,2},{[0 0 -20 -50],[0.5 0.5],[0 0],[5 5],[1 1]});
%! zp1=calculate_prediction(m1,m1.theta,d);
%! zp2=calculate_prediction(m2,m2.theta,d);
%! zp3=calculate_prediction(m3,m3.theta,d);
%! assert(zp1,zp2,1e-10);
%! assert(zp1,zp3,1e-10);
%! assert(zp2,zp3,1e-10);



