function l=calculate_likelihood_P(m,theta,z,d)
% Calculates the likelihood of measurements z given the distances d according to the model m with parameters theta.
%
% The same type of model is used for all the measurements, but with different model parameters (Individual Models)
%
% INPUTS
%	- m: the model.
%	- theta: KxMD matrix with the K vectors of model parameters of dimension MD
%	- z: NxK matrix containing the RSS measurements 
%	- d: NxK matrix containing the distances.
%
% OUTPUT
%	- l: NxK matrix of the likelihoods.
%
% The likelihoods of the N RSS measurements from the j-th column of z are calculated with the model parameters
% of the j-th row of theta. That is: l(i,j) = lik(z(i,j)|d(i,j),theta(j,:))
%
% If z and d are arrays, they must be of the same size. If z is a scalar, it is expanded to construct
% a matrix the same size as d. If d is a scalar, it is expanded to construct a matrix the same size as z.
%
% If theta is a row vector, the same model parameters are used for all RSS. This is equivalent to
% calculate_likelihood(m,z,d)
%
% If the rows of theta are greater than the columns of z and d, z and d are expanded to have the same number of
% columns that rows of theta.
%
% The dimension of l is the common size of z and d after any necessary expansions.
%

if nargin != 4
  error("calculate_likelihood_P: expected 4 arguments");
  print_usage ();
endif

%Check dimensions of inputs.
[errorcode, a, b] = common_size (z, d');
if (!isscalar (z) || !isscalar (d))
  [retval,z,d] = common_size(z,d);
  if (retval > 0)
    error ("calculate_likelihood_P: z and d must be of common size or scalars");
  endif
endif
  
cols_cs=max([columns(z) columns(d)]);
num_models=rows(theta);
if num_models!=cols_cs
  if rows(theta)==1
    theta=repmat(theta,[cols_cs 1]);
  elseif columns(z)==1
    z=repmat(z,[1,num_models]);
    d=repmat(d,[1,num_models]);
  endif
endif

% Models normally distributed
% Polynomial with Normal error (PN) and Log Normal (PLN)
if (m.typeID==1) || (m.typeID==9)
  %The model predictions (averages for the normal distributions)
  Z=calculate_prediction(m,theta,d); %NxK
  %The standard deviations
  s=theta(:,numel(m.p)+1)'; %1xK
  %Finally the likelihoods
  %l=normpdf(z,Z,repmat(s',[rows(z) 1]));
  l= stdnormal_pdf( (z-Z) ./ s ) ./ s ; %This is faster
  %                -----    ---    ---
  %                  NxK    1xK    1xK
else
  error("calculate_likelihood_P: model type not supported");
endif

%!test
% In these cases z should be normally distributed
%! sigma=2;
%! m1=create_init_model({"PN",1},{[1 0],sigma});
%! m2=create_init_model({"PMIXKN",1,1},{[1 0],[1],[0],[sigma],[1]});
%! N=100;K=1;
%! z=sigma*randn(N,K);
%! d=zeros(size(z));
%! l1=calculate_likelihood_P(m1,m1.theta,z,d);
%! assert(l1,normpdf(z,0,sigma));
%! l2=calculate_likelihood_P(m2,m2.theta,z,d);
%! assert(l1,normpdf(z,0,sigma));
%
% Testing higher order polynomials and individual models.
% The PN and PMIXKN with one mixture component should be the same
%! m1=create_init_model({"PN",3},{[0.1 0.1 1 0],1});
% The following MoG has 2 equal components, so it is equivalent to a single gaussian
%! m2=create_init_model({"PMIXKN",3,2},{[0.1 0.1 1 0],[0.5 0.5],[0 0],[1 1],[1 1]});
%! N=100;K=3;
%! z=rand(N,K);
%! d=z; %for fun only
%! theta1=repmat(m1.theta,[K,1]);
%! theta1+=rand(size(theta1));
%! l1=calculate_likelihood_P(m1,theta1,z,d);
%! theta2=[theta1(:,1:numel(m1.p)) repmat([0.5 0.5 0 0],rows(theta1),1) theta1(:,end) theta1(:,end)];
%! l2=calculate_likelihood_P(m2,theta2,z,d);
%! assert(l1,l2,1e-10);

%!test
% The PLN and PLNMIXKN with one mixture component should be the same
%! m1=create_init_model({"PLN",1},{[2 -50],5});
%  Note that the PLNMIXKN does not use the path loss exponent as a parameter, but -10n
%! m2=create_init_model({"PLNMIXKN",1,1},{[-20 -50],[1],[0],[5],[1]});
%! N=100;K=3;
%! z=-70-20*randn(N,K);
%! d=10+rand(size(z)); %for fun, ensuring positive values
%! theta1=repmat(m1.theta,[K,1]);
%! theta1+=rand(size(theta1));
%! l1=calculate_likelihood_P(m1,theta1,z,d);
%! theta2=[theta1(:,1:numel(m1.p)) repmat([1 0],rows(theta1),1) theta1(:,end)];
%! theta2(:,1)*=-10;
%! l2=calculate_likelihood_P(m2,theta2,z,d);
%! assert(l1,l2,1e-10);
%
% Testing higher order polynomials and individual models.
% The PLN and PLNMIXKN with one mixture component should be the same
%! m1=create_init_model({"PLN",3},{[0.1 0.1 2 -50],1});
% The following MoG has 2 equal components, so it is equivalent to a single gaussian
%! m2=create_init_model({"PLNMIXKN",3,2},{[0.1 0.1 -20 -50],[0.5 0.5],[0 0],[1 1],[1 1]});
%! theta1=repmat(m1.theta,[K,1]);
%! theta1+=rand(size(theta1));
%! l1=calculate_likelihood_P(m1,theta1,z,d);
%! theta2=[theta1(:,1:numel(m1.p)) repmat([0.5 0.5 0 0],rows(theta1),1) theta1(:,end) theta1(:,end)];
%! theta2(:,3)*=-10;
%! l2=calculate_likelihood_P(m2,theta2,z,d);
%! assert(l1,l2,1e-10);



