function [d_u,rssi_u,traj_u,nts_u,ierr,t_rn]=filter_rssi(node_s,robot_s,filter_options)
% Filter the rssi measurements of the given node
%
% The filter type and its required parameters are specified in the filter_options cell array. Its first element 
% is a string identifying the filter type, and the second its required parameters. 
% 
% The following types of filter are supported.
%	TWA: Time Window Average. The RSSI signal is averaged using the time as the index. See indexed_waverage
%		function for the required parameters
%	SDWA: Sorted Distance Window Average. The RSSI signal is averaged using the node-robot 
%		distance as the index. See indexed_waverage function for the required parameters
%	RMWA: Robot Movement Window Average: the RSSI signal is averaged in windows defined by distance moved
%		by the robot (in any direction). The required argument is the distance moved by the robot that 
%		defines the averaging window.
%	GRID:
%	DPTRAJ: Distant Points in Trajectory: selects the points of the trajectory (and associated measurements)
%		that are distant of each other at least d_0 meters. The required argument is d_0.
%
% SYNTAX: [d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,filter_options)
%
% Arguments
%	- node_s: structure with the target node
%	- robot_s: structure with the position of the robot
%	- filter_options: cell array containing the filtering type and its required parameters
%
% Returns
%	d_u: 
%	rssi_u:
%	traj_u:
%	nts_u:
%	ierr: If requested, ierr contains the following status information
%		1 - Normal return
%		2 - No points found inside window for averaging.
%
% Examples
%	[d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,{'TWA',{"WINDOW",10,"CENTRE"});
%	[d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,{'TWA',{"REG_INTERVAL",0.1,"CENTRE",0.01});
%	[d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,{'SDWA',{"WINDOW",0.5,"CENTRE"}});
%	[d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,{'SDWA',{"REG_INTERVAL",0.1,"CENTRE",0.01}});
%	[d_u,rssi_u,traj_u,nts_u,ierr]=filter_rssi(node_s,robot_s,{'RMWA',0.5});
%

if (nargin != 3)
	error ("Wrong number of arguments");
endif

% filter_options must be a cell array with with 2 elements
if !iscell(filter_options)
  error('filter_options must be a cell array');
endif

nr_id=robot_s.nr_id; %node robot id

f_type=filter_options{1};
if strcmp(f_type,'Antenna')
  f_params=filter_options{2:end};
else
  f_params={filter_options{2:end}};
endif

% The average_rssi_measurements_rm function does all this internally, so we avoid doing it twice.
if !strcmp(f_type,'RMWA')
  %Measurements from the node of the robot
  mr_i=find(node_s.tx_id==nr_id); %indexes of the measurements
  if isempty(mr_i) error('Node with no measurements'); endif
  rssi=node_s.rssi(mr_i);
  nts=node_s.nts(mr_i);
  pos=robot_pos_at_time(nts,robot_s);%Position of the robot for each measurement
endif

switch(f_type)
  case 'TWA' %Time Window Average
    %The indexes for the averaging filter are the time of the measurements. They are already sorted.
    %f_params must be something like {"WINDOW",3,"CENTRE"} or {"REG_INTERVAL",4,"FRONT",0.5}
    [nts_u,rssi_u,ierr]=indexed_waverage(nts,rssi,f_params{:});
    pos=robot_pos_at_time(nts_u,robot_s);%Position of the robot for each measurement
    d_u= sqrt( (pos(:,1)-node_s.pos_x()).^2 + (pos(:,2)-node_s.pos_y).^2 );%Distances to the robot
    traj_u=[pos(:,1) pos(:,2)];
  case 'SDWA' % Sorted Distance Window Average
    d= sqrt( (pos(:,1)-node_s.pos_x()).^2 + (pos(:,2)-node_s.pos_y).^2 );%Distances to the robot
    %Sort the RSSI with the increasing distance, and filter with the distance as index
    [ds,idx]=sort(d);
    rssis=rssi(idx);
    [d_u,rssi_u,ierr]=indexed_waverage(ds,rssis,f_params{:});
    %NOTE that the nts_u and traj_u might not have the same number of components than the d_u and rssi_u when the
    %REG_INTERVAL type of filter is used
    nts_u=nts(idx);
    traj_u=[pos(idx,1) pos(idx,2)];
  case 'RMWA' %Robot Movement Window Average
    d_cum_max=f_params{:};
    [d_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,d_cum_max);
    ierr=1;
  case 'GRID' %Average RSSI on bins
    grd_res=f_params{:};
    [d_u,rssi_u,traj_u,bins]=average_rssi_measurements_grid(node_s,robot_s,grd_res);
    %The nts does not have a proper meaning here!
    nts_u=[];
    ierr=1;
  case 'DPTRAJ' %Distant points in trajectory
    %First we make sure that we have one RSS measurement per robot position
    [d_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,0);
    %Now we select the points of the trajectory
    if numel(f_params)==1
      d_0=f_params;
      seed=NA;
    elseif numel(f_params)==2
      d_0=f_params{1};
      seed=f_params{2};
    else
      error("filter_rssi:DPTRAJ needs one or two parameters.")
    endif
    [traj_u sel_i]=dptraj(traj_u, d_0,seed);
    d_u=d_u(sel_i);
    rssi_u=rssi_u(sel_i);
    nts_u=nts_u(sel_i);
    ierr=1;
  case 'Antenna'
    %The parameters is a string or handle to the function to calculate the antenna pattern
    if !ischar(f_params) && !strcmp(class(f_params),'function_handle')
      error("filter_rssi:Antenna needs a function handle as a unique parameter")
    endif
    %First we make sure that we have one RSS measurement per robot position
    [d_u,rssi_u,traj_u,nts_u]=average_rssi_measurements_rm(node_s,robot_s,0);
    %Calculate the heading of the robot
    %keyboard
    [heading,u]=get_heading(traj_u);
    %Calculate relative angle between the robot and the node
    r=node_s.pos-traj_u; %vector from robot to node[d
    q=atan2(r(:,2),r(:,1)); %Absolute angle of r
    t_rn=q-heading; %relative angle between the robot and the node
    gain=feval(f_params,t_rn);
    %Modify the RSS 
    rssi_u+=gain; 
    ierr=1;
  otherwise
    error(['Invalid filter type: ''' f_type '''.']);
end
